(*^
::[	Information =

	"This is a Mathematica Notebook file.  It contains ASCII text, and can be
	transferred by email, ftp, or other text-file transfer utility.  It should
	be read or edited using a copy of Mathematica or MathReader.  If you 
	received this as email, use your mail application or copy/paste to save 
	everything from the line containing (*^ down to the line containing ^*)
	into a plain text file.  On some systems you may have to give the file a 
	name ending with ".ma" to allow Mathematica to recognize it as a Notebook.
	The line below identifies what version of Mathematica created this file,
	but it can be opened using any other version as well.";

	FrontEndVersion = "Macintosh Mathematica Notebook Front End Version 2.2";

	MacintoshStandardFontEncoding; 
	
	fontset = title, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeTitle, center, M7, bold, e8,  24, "Times"; 
	fontset = subtitle, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeTitle, center, M7, bold, e6,  18, "Times"; 
	fontset = subsubtitle, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeTitle, center, M7, italic, e6,  14, "Times"; 
	fontset = section, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeSection, grayBox, M22, bold, a20,  18, "Times"; 
	fontset = subsection, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeSection, blackBox, M19, bold, a15,  14, "Times"; 
	fontset = subsubsection, inactive, noPageBreakBelow, nohscroll, preserveAspect, groupLikeSection, whiteBox, M18, bold, R65535, a12,  12, "Times"; 
	fontset = text, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = smalltext, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  10, "Times"; 
	fontset = input, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeInput, M42, N23, bold, L-5,  10, "Courier"; 
	fontset = output, output, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, L-5,  10, "Courier"; 
	fontset = message, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, R65535, L-5,  12, "Courier"; 
	fontset = print, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, L-5,  10, "Courier"; 
	fontset = info, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeOutput, M42, N23, B65535, L-5,  12, "Courier"; 
	fontset = postscript, PostScript, formatAsPostScript, output, inactive, noPageBreakInGroup, nowordwrap, preserveAspect, groupLikeGraphics, M7, l34, w245, h249,  12, "Courier"; 
	fontset = name, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7, italic,  10, "Geneva"; 
	fontset = header, inactive, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = leftheader, inactive, L2,  12, "Times"; 
	fontset = footer, inactive, noKeepOnOnePage, preserveAspect, center, M7,  12, "Times"; 
	fontset = leftfooter, inactive, L2,  12, "Times"; 
	fontset = help, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  10, "Times"; 
	fontset = clipboard, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = completions, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special1, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special2, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special3, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special4, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	fontset = special5, inactive, nohscroll, noKeepOnOnePage, preserveAspect, M7,  12, "Times"; 
	paletteColors = 128; automaticGrouping; currentKernel; 
]
:[font = title; inactive; locked; preserveAspect; startGroup]
Quantum Error Correction
:[font = subsubtitle; inactive; locked; preserveAspect]
Colin P. Williams
:[font = input; locked; initialization; preserveAspect]
*)
Off[General::spell1]
(*
:[font = section; inactive; locked; preserveAspect; startGroup]
Copyright Notice
:[font = text; inactive; locked; preserveAspect; endGroup]
Copyright Colin P. Williams (1997).

This Notebook is intended to be used in conjunction with "Explorations in Quantum Computing" by Colin P. Williams and Scott H. Clearwater, TELOS, Springer-Verlag (1997), ISBN:0-387-94768-X. Permission is hereby granted to copy and distribute this Notebook freely for any non-commercial activity provided you include this copyright notice at the beginning of all such copies. Please send suggestions and bug reports to Colin P. Williams at 
        colin@solstice.jpl.nasa.gov      (818) 306 6512 or 
        cpw@cs.stanford.edu               (415) 728 2118
For information on "Explorations in Quantum Computing" check out the TELOS web site:  http://www.telospub.com/catalog/PHYSICS/Explorations.html. To order call 1-800-777-4643.

All other rights reserved.
:[font = section; inactive; preserveAspect; startGroup]
How to use this Notebook
:[font = text; inactive; preserveAspect]
This Notebook contains a simulation of a circuit for quantum error correction i.e. the correction of an error in the state of a qubit without knowing (or measuring) what state that qubit is in (see "Explorations in Quantum Computing, Chapter 10).  This is important because errors will be a pervasive problem in quantum computers.  Even an idle quantum memory register will be affected by errors due to stray interactions with its "environment".  At the quantum level the concept of "environment" is quite general and could even include the supporting infrastructure of the memory register itself. Realistically, a quantum computer is likely, therefore, to have to operate by interleaving computational steps with error correction steps. See Chapter 10 for description of possible approaches to quantum error correction.

You are encouraged to work through the Notebook, executing commands labelled as "Try me!" as you go.  To execute a such a command, position your cursor over the cell containing the command, click the mouse to select the cell and type the SHIFT and RETURN keys simultaneously.

The following functions are available, each mimicking the action of a step in the error correction process. :
:[font = input; preserveAspect; plain]
CreateUnknownState (* creates an arbitrary state, |psi>, of a 2-state system *)

EncodeQubit    (* entangles |psi> with a 4 other qubits                  *)

BitFlipError              (* simulates a random bit flip error    *)
PhaseShiftError           (* simulates a random phase shift error *)
BitFlipAndPhaseShiftError (* simulates simultaneous  random bit shift and *)
                          (* random phase shift error                     *)

DecodeQubit    (* disentangles 5 qubits to retrieve original state *)

:[font = text; inactive; preserveAspect]
To do the encoding, we entangle a given state (the one to be protected) with 4 other qubits. This entangling operation smears out the information about the state we want to protect amongst many other qubits in such a way that if one of the qubits later becomes corrupted due to an error, there is still sufficient information amongst the other qubits to restore the original state. The entangling step is accomplished by pushing 5 qubits through the LaflammeEncodingCircuit[].
:[font = input; preserveAspect; plain]
LaflammeEncodingCircuit[]
:[font = text; inactive; preserveAspect]
An error is assumed to occur amongst the 5 entangled qubits but the buggy qubits are then pushed through the LaflammeDecodingCircuit[]
:[font = input; preserveAspect; plain; endGroup]
LaflammeDecodingCircuit[]
:[font = section; inactive; preserveAspect; startGroup]
Error Models
:[font = text; inactive; preserveAspect]
Before we can discuss error correction, we need to describe the effect of errors in a quantum computation.  Errors can be of four kinds: bit flips, phase shifts, combined bit flips and phase shifts and no error whatsoever. These can be characterized by the application of a linear combination of the Pauli spin matrices:
:[font = input; initialization; preserveAspect]
*)
sigmaX = {{0, 1},
          {1, 0}};     (* just bit flip *)

sigmaY = {{0,-I},
          {I, 0}};    (* both bit flip & phase shift *)

sigmaZ = {{1, 0},
          {0,-1}};    (* just phase shift *)
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
For example, to see the action of the Pauli spin matrix sigmaX (mimicking a bit flip error), try the following:
:[font = input; preserveAspect; startGroup]
(sigmaX . KetToColumnVector[a ket[0] + b ket[1]]) // ColumnVectorToKet
:[font = output; output; inactive; preserveAspect; endGroup]
b*ket[0] + a*ket[1]
;[o]
b ket[0] + a ket[1]
:[font = text; inactive; preserveAspect; endGroup]
Do you see how the bit has been flipped?
:[font = subsection; inactive; preserveAspect; startGroup]
Errors in a multi-particle state
:[font = text; inactive; preserveAspect]
Next we imagine something closer to a true quantum memory register. At the very least a memory register will consist of m 2-state systems. To describe the action of the Pauli spin matrices on the i-th of m qubits, we derive the operator corresponding to each Pauli spin matrix acting on the i-th of m qubits:
:[font = subsubsection; inactive; Cclosed; preserveAspect; startGroup]
Code
:[font = input; initialization; preserveAspect; endGroup]
*)
PauliX[i_, m_]:=
	Apply[Direct, 
	      ReplacePart[Table[IdentityMatrix[2], {m}], sigmaX, i]]

PauliY[i_, m_]:=
	Apply[Direct, 
	      ReplacePart[Table[IdentityMatrix[2], {m}], sigmaY, i]]
	      
PauliZ[i_, m_]:=
	Apply[Direct, 
	      ReplacePart[Table[IdentityMatrix[2], {m}], sigmaZ, i]]	      

(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
Let's determine the Pauli spin matrix sigmaZ acting on the 2nd of 3 qubits (mimicking a phase shift error in just the middle of 3 qubits in a 3 qubit memory register):
:[font = input; preserveAspect; startGroup]
PauliZ[2,3] // MatrixForm
:[font = output; output; inactive; preserveAspect; endGroup]
MatrixForm[{{1, 0, 0, 0, 0, 0, 0, 0}, {0, 1, 0, 0, 0, 0, 0, 0}, 
 
   {0, 0, -1, 0, 0, 0, 0, 0}, {0, 0, 0, -1, 0, 0, 0, 0}, {0, 0, 0, 0, 1, 0, 0, 0}, 
 
   {0, 0, 0, 0, 0, 1, 0, 0}, {0, 0, 0, 0, 0, 0, -1, 0}, {0, 0, 0, 0, 0, 0, 0, -1}}]
;[o]
1    0    0    0    0    0    0    0

0    1    0    0    0    0    0    0

0    0    -1   0    0    0    0    0

0    0    0    -1   0    0    0    0

0    0    0    0    1    0    0    0

0    0    0    0    0    1    0    0

0    0    0    0    0    0    -1   0

0    0    0    0    0    0    0    -1
:[font = text; inactive; preserveAspect; endGroup]
Compare this with sigmaZ alone:
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
sigmaZ // MatrixForm
:[font = output; output; inactive; preserveAspect; endGroup]
MatrixForm[{{1, 0}, {0, -1}}]
;[o]
1    0

0    -1
:[font = text; inactive; preserveAspect]
or, equivalently, 
:[font = input; preserveAspect; startGroup]
PauliZ[1,1] // MatrixForm
:[font = output; output; inactive; preserveAspect; endGroup; endGroup]
MatrixForm[{{1, 0}, {0, -1}}]
;[o]
1    0

0    -1
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
We can now examine the effect of the embedded Pauli spin matrices on the states of quantum memory regsiters. First, let's create an arbitrary symbolic description of the state of a 3-qubit register:
:[font = input; preserveAspect; startGroup]
SymbolicSuperposition[3]
:[font = output; output; inactive; preserveAspect; endGroup]
w0*ket[0, 0, 0] + w1*ket[0, 0, 1] + w2*ket[0, 1, 0] + w3*ket[0, 1, 1] + 
 
  w4*ket[1, 0, 0] + w5*ket[1, 0, 1] + w6*ket[1, 1, 0] + w7*ket[1, 1, 1]
;[o]
w0 ket[0, 0, 0] + w1 ket[0, 0, 1] + w2 ket[0, 1, 0] + w3 ket[0, 1, 1] + 
 
  w4 ket[1, 0, 0] + w5 ket[1, 0, 1] + w6 ket[1, 1, 0] + w7 ket[1, 1, 1]
:[font = text; inactive; preserveAspect]
Then we'll apply sigmaZ to the 2nd of 3 qubits.
:[font = input; preserveAspect; startGroup]
(PauliZ[2,3] . KetToColumnVector[SymbolicSuperposition[3]]) // ColumnVectorToKet
:[font = output; output; inactive; preserveAspect; endGroup]
w0*ket[0, 0, 0] + w1*ket[0, 0, 1] - w2*ket[0, 1, 0] - w3*ket[0, 1, 1] + 
 
  w4*ket[1, 0, 0] + w5*ket[1, 0, 1] - w6*ket[1, 1, 0] - w7*ket[1, 1, 1]
;[o]
w0 ket[0, 0, 0] + w1 ket[0, 0, 1] - w2 ket[0, 1, 0] - w3 ket[0, 1, 1] + 
 
  w4 ket[1, 0, 0] + w5 ket[1, 0, 1] - w6 ket[1, 1, 0] - w7 ket[1, 1, 1]
:[font = text; inactive; preserveAspect]
Do you see how some of the amplitudes have changed sign indicating a phase shift error?
:[font = text; inactive; preserveAspect]
Then, we can describe bit flips, phase shifts or both via:
:[font = input; preserveAspect; startGroup]
?BitFlipError
:[font = print; inactive; preserveAspect; endGroup]
BitFlipError[state,i,m] performs a bit flip on the i-th of m qubits.
:[font = input; preserveAspect; startGroup]
?PhaseShiftError
:[font = print; inactive; preserveAspect; endGroup]
PhaseShiftError[state,i,m] performs a phase shift on the i-th of m qubits.
:[font = input; preserveAspect; startGroup]
?BitFlipAndPhaseShiftError
:[font = print; inactive; preserveAspect; endGroup; endGroup]
BitFlipAndPhaseShiftError[state,i,m] performs a bit flip and a phase shift on the
   i-th of m qubits.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Code
:[font = input; initialization; preserveAspect; endGroup; endGroup]
*)
(* Bit flip on the i-th of m qubits *)
BitFlipError[state_, i_, m_]:=
	Chop[ColumnVectorToKet[PauliX[i, m] . KetToColumnVector[state]]]
BitFlipError::usage = 
	"BitFlipError[state,i,m] performs a bit flip on the i-th of m qubits.";
	
PhaseShiftError[state_, i_, m_]:=
	Chop[ColumnVectorToKet[PauliZ[i, m] . KetToColumnVector[state]]]
PhaseShiftError::usage =
	"PhaseShiftError[state,i,m] performs a phase shift on the i-th of m qubits.";
	
BitFlipAndPhaseShiftError[state_, i_, m_]:=
	Chop[ColumnVectorToKet[PauliY[i, m] . KetToColumnVector[state]]]
BitFlipAndPhaseShiftError::usage =
	"BitFlipAndPhaseShiftError[state,i,m] performs a bit flip and a phase shift on the i-th of m qubits.";
	
(*
:[font = subsection; inactive; preserveAspect; startGroup]
A Single Error in a 1 Qubit Memory Register
:[font = text; inactive; preserveAspect]
It is convenient to have a function that returns an arbitrary superposition of m qubits.
:[font = input; preserveAspect; startGroup]
?SymbolicSuperposition
:[font = print; inactive; preserveAspect; endGroup]
SymbolicSuperposition[m] creates a superposition of 2^m eigenstates whose  amplitudes
   are uninstantiated symbols. These eigenstates represent the  possible states of an
   m-bit memory register of a quantum computer. This  function is useful for
   exploring the effects of quantum mechanical  operations on arbitrary
   superpositions. Note that the general form does not  guarentee that the
   superposition is normalized.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
state1 = SymbolicSuperposition[1]
:[font = output; output; inactive; preserveAspect; endGroup]
w0*ket[0] + w1*ket[1]
;[o]
w0 ket[0] + w1 ket[1]
:[font = input; preserveAspect; startGroup]
BitFlipError[state1,1,1]
:[font = output; output; inactive; preserveAspect; endGroup]
w1*ket[0] + w0*ket[1]
;[o]
w1 ket[0] + w0 ket[1]
:[font = input; preserveAspect; startGroup]
PhaseShiftError[state1,1,1]
:[font = output; output; inactive; preserveAspect; endGroup]
w0*ket[0] - w1*ket[1]
;[o]
w0 ket[0] - w1 ket[1]
:[font = input; preserveAspect; startGroup]
BitFlipAndPhaseShiftError[state1,1,1]
:[font = output; output; inactive; preserveAspect; endGroup]
-I*w1*ket[0] + I*w0*ket[1]
;[o]
-I w1 ket[0] + I w0 ket[1]
:[font = text; inactive; preserveAspect]
Notice that we can compound the errors.
:[font = input; preserveAspect; startGroup]
BitFlipError[PhaseShiftError[state1,1,1],1,1]
:[font = output; output; inactive; preserveAspect; endGroup]
-(w1*ket[0]) + w0*ket[1]
;[o]
-(w1 ket[0]) + w0 ket[1]
:[font = input; preserveAspect; startGroup]
PhaseShiftError[BitFlipError[state1,1,1],1,1]
:[font = output; output; inactive; preserveAspect; endGroup]
w1*ket[0] - w0*ket[1]
;[o]
w1 ket[0] - w0 ket[1]
:[font = input; preserveAspect; startGroup]
BitFlipAndPhaseShiftError[state1, 1, 1]
:[font = output; output; inactive; preserveAspect; endGroup; endGroup; endGroup]
-I*w1*ket[0] + I*w0*ket[1]
;[o]
-I w1 ket[0] + I w0 ket[1]
:[font = subsection; inactive; preserveAspect; startGroup]
A Single Error in a 2-Qubit Memory Register
:[font = text; inactive; preserveAspect]
Now let's get more ambitious and examine errors in a 2-qubit register.  If you build up to the larger registers slowly you will get a better idea about what the various kinds of errors do to the state of the memory register.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
state2 = SymbolicSuperposition[2]
:[font = output; output; inactive; preserveAspect; endGroup]
w0*ket[0, 0] + w1*ket[0, 1] + w2*ket[1, 0] + w3*ket[1, 1]
;[o]
w0 ket[0, 0] + w1 ket[0, 1] + w2 ket[1, 0] + w3 ket[1, 1]
:[font = input; preserveAspect; startGroup]
BitFlipError[state2,1,2]
:[font = output; output; inactive; preserveAspect; endGroup]
w2*ket[0, 0] + w3*ket[0, 1] + w0*ket[1, 0] + w1*ket[1, 1]
;[o]
w2 ket[0, 0] + w3 ket[0, 1] + w0 ket[1, 0] + w1 ket[1, 1]
:[font = input; preserveAspect; startGroup]
BitFlipError[state2,2,2]
:[font = output; output; inactive; preserveAspect; endGroup]
w1*ket[0, 0] + w0*ket[0, 1] + w3*ket[1, 0] + w2*ket[1, 1]
;[o]
w1 ket[0, 0] + w0 ket[0, 1] + w3 ket[1, 0] + w2 ket[1, 1]
:[font = input; preserveAspect; startGroup]
PhaseShiftError[state2,1,2]
:[font = output; output; inactive; preserveAspect; endGroup]
w0*ket[0, 0] + w1*ket[0, 1] - w2*ket[1, 0] - w3*ket[1, 1]
;[o]
w0 ket[0, 0] + w1 ket[0, 1] - w2 ket[1, 0] - w3 ket[1, 1]
:[font = input; preserveAspect; startGroup]
PhaseShiftError[state2,2,2]
:[font = output; output; inactive; preserveAspect; endGroup]
w0*ket[0, 0] - w1*ket[0, 1] + w2*ket[1, 0] - w3*ket[1, 1]
;[o]
w0 ket[0, 0] - w1 ket[0, 1] + w2 ket[1, 0] - w3 ket[1, 1]
:[font = input; preserveAspect; startGroup]
BitFlipAndPhaseShiftError[state2,1,2]
:[font = output; output; inactive; preserveAspect; endGroup]
-I*w2*ket[0, 0] - I*w3*ket[0, 1] + I*w0*ket[1, 0] + I*w1*ket[1, 1]
;[o]
-I w2 ket[0, 0] - I w3 ket[0, 1] + I w0 ket[1, 0] + I w1 ket[1, 1]
:[font = input; preserveAspect; startGroup]
BitFlipAndPhaseShiftError[state2,2,2]
:[font = output; output; inactive; preserveAspect; endGroup; endGroup; endGroup]
-I*w1*ket[0, 0] + I*w0*ket[0, 1] - I*w3*ket[1, 0] + I*w2*ket[1, 1]
;[o]
-I w1 ket[0, 0] + I w0 ket[0, 1] - I w3 ket[1, 0] + I w2 ket[1, 1]
:[font = subsection; inactive; preserveAspect; startGroup]
A Single Error in a 3-Qubit Memory Register
:[font = text; inactive; preserveAspect]
Now we can invoke bit flip, phase shift,  combined bit flip and phase shift and "no errors" on the i-th of 3 qubits.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
Here is the general state of a 3-qubit register.
:[font = input; preserveAspect; startGroup]
state3 = SymbolicSuperposition[3]
:[font = output; output; inactive; preserveAspect; endGroup]
w0*ket[0, 0, 0] + w1*ket[0, 0, 1] + w2*ket[0, 1, 0] + w3*ket[0, 1, 1] + 
 
  w4*ket[1, 0, 0] + w5*ket[1, 0, 1] + w6*ket[1, 1, 0] + w7*ket[1, 1, 1]
;[o]
w0 ket[0, 0, 0] + w1 ket[0, 0, 1] + w2 ket[0, 1, 0] + w3 ket[0, 1, 1] + 
 
  w4 ket[1, 0, 0] + w5 ket[1, 0, 1] + w6 ket[1, 1, 0] + w7 ket[1, 1, 1]
:[font = text; inactive; preserveAspect]
A bit flip on the 1st qubit is as follows:
:[font = input; preserveAspect; startGroup]
BitFlipError[state3,1,3]
:[font = output; output; inactive; preserveAspect; endGroup]
w4*ket[0, 0, 0] + w5*ket[0, 0, 1] + w6*ket[0, 1, 0] + w7*ket[0, 1, 1] + 
 
  w0*ket[1, 0, 0] + w1*ket[1, 0, 1] + w2*ket[1, 1, 0] + w3*ket[1, 1, 1]
;[o]
w4 ket[0, 0, 0] + w5 ket[0, 0, 1] + w6 ket[0, 1, 0] + w7 ket[0, 1, 1] + 
 
  w0 ket[1, 0, 0] + w1 ket[1, 0, 1] + w2 ket[1, 1, 0] + w3 ket[1, 1, 1]
:[font = text; inactive; preserveAspect]
whereas a bit flip on the 2nd qubit is this:
:[font = input; preserveAspect; startGroup]
BitFlipError[state3,2,3]
:[font = output; output; inactive; preserveAspect; endGroup]
w2*ket[0, 0, 0] + w3*ket[0, 0, 1] + w0*ket[0, 1, 0] + w1*ket[0, 1, 1] + 
 
  w6*ket[1, 0, 0] + w7*ket[1, 0, 1] + w4*ket[1, 1, 0] + w5*ket[1, 1, 1]
;[o]
w2 ket[0, 0, 0] + w3 ket[0, 0, 1] + w0 ket[0, 1, 0] + w1 ket[0, 1, 1] + 
 
  w6 ket[1, 0, 0] + w7 ket[1, 0, 1] + w4 ket[1, 1, 0] + w5 ket[1, 1, 1]
:[font = text; inactive; preserveAspect]
Can you spot the difference? What would you predict for a bit flip on the 3rd qubit?
:[font = input; preserveAspect; startGroup]
BitFlipError[state3,3,3]
:[font = output; output; inactive; preserveAspect; endGroup]
w1*ket[0, 0, 0] + w0*ket[0, 0, 1] + w3*ket[0, 1, 0] + w2*ket[0, 1, 1] + 
 
  w5*ket[1, 0, 0] + w4*ket[1, 0, 1] + w7*ket[1, 1, 0] + w6*ket[1, 1, 1]
;[o]
w1 ket[0, 0, 0] + w0 ket[0, 0, 1] + w3 ket[0, 1, 0] + w2 ket[0, 1, 1] + 
 
  w5 ket[1, 0, 0] + w4 ket[1, 0, 1] + w7 ket[1, 1, 0] + w6 ket[1, 1, 1]
:[font = input; preserveAspect; startGroup]
PhaseShiftError[state3,1,3]
:[font = output; output; inactive; preserveAspect; endGroup]
w0*ket[0, 0, 0] + w1*ket[0, 0, 1] + w2*ket[0, 1, 0] + w3*ket[0, 1, 1] - 
 
  w4*ket[1, 0, 0] - w5*ket[1, 0, 1] - w6*ket[1, 1, 0] - w7*ket[1, 1, 1]
;[o]
w0 ket[0, 0, 0] + w1 ket[0, 0, 1] + w2 ket[0, 1, 0] + w3 ket[0, 1, 1] - 
 
  w4 ket[1, 0, 0] - w5 ket[1, 0, 1] - w6 ket[1, 1, 0] - w7 ket[1, 1, 1]
:[font = input; preserveAspect; startGroup]
PhaseShiftError[state3,2,3]
:[font = output; output; inactive; preserveAspect; endGroup]
w0*ket[0, 0, 0] + w1*ket[0, 0, 1] - w2*ket[0, 1, 0] - w3*ket[0, 1, 1] + 
 
  w4*ket[1, 0, 0] + w5*ket[1, 0, 1] - w6*ket[1, 1, 0] - w7*ket[1, 1, 1]
;[o]
w0 ket[0, 0, 0] + w1 ket[0, 0, 1] - w2 ket[0, 1, 0] - w3 ket[0, 1, 1] + 
 
  w4 ket[1, 0, 0] + w5 ket[1, 0, 1] - w6 ket[1, 1, 0] - w7 ket[1, 1, 1]
:[font = input; preserveAspect; startGroup]
PhaseShiftError[state3,3,3]
:[font = output; output; inactive; preserveAspect; endGroup]
w0*ket[0, 0, 0] - w1*ket[0, 0, 1] + w2*ket[0, 1, 0] - w3*ket[0, 1, 1] + 
 
  w4*ket[1, 0, 0] - w5*ket[1, 0, 1] + w6*ket[1, 1, 0] - w7*ket[1, 1, 1]
;[o]
w0 ket[0, 0, 0] - w1 ket[0, 0, 1] + w2 ket[0, 1, 0] - w3 ket[0, 1, 1] + 
 
  w4 ket[1, 0, 0] - w5 ket[1, 0, 1] + w6 ket[1, 1, 0] - w7 ket[1, 1, 1]
:[font = input; preserveAspect; startGroup]
BitFlipAndPhaseShiftError[state3,1,3]
:[font = output; output; inactive; preserveAspect; endGroup]
-I*w4*ket[0, 0, 0] - I*w5*ket[0, 0, 1] - I*w6*ket[0, 1, 0] - I*w7*ket[0, 1, 1] + 
 
  I*w0*ket[1, 0, 0] + I*w1*ket[1, 0, 1] + I*w2*ket[1, 1, 0] + I*w3*ket[1, 1, 1]
;[o]
-I w4 ket[0, 0, 0] - I w5 ket[0, 0, 1] - I w6 ket[0, 1, 0] - I w7 ket[0, 1, 1] + 
 
  I w0 ket[1, 0, 0] + I w1 ket[1, 0, 1] + I w2 ket[1, 1, 0] + I w3 ket[1, 1, 1]
:[font = input; preserveAspect; startGroup]
BitFlipAndPhaseShiftError[state3,2,3]
:[font = output; output; inactive; preserveAspect; endGroup]
-I*w2*ket[0, 0, 0] - I*w3*ket[0, 0, 1] + I*w0*ket[0, 1, 0] + I*w1*ket[0, 1, 1] - 
 
  I*w6*ket[1, 0, 0] - I*w7*ket[1, 0, 1] + I*w4*ket[1, 1, 0] + I*w5*ket[1, 1, 1]
;[o]
-I w2 ket[0, 0, 0] - I w3 ket[0, 0, 1] + I w0 ket[0, 1, 0] + I w1 ket[0, 1, 1] - 
 
  I w6 ket[1, 0, 0] - I w7 ket[1, 0, 1] + I w4 ket[1, 1, 0] + I w5 ket[1, 1, 1]
:[font = input; preserveAspect; startGroup]
BitFlipAndPhaseShiftError[state3,3,3]
:[font = output; output; inactive; preserveAspect; endGroup; endGroup; endGroup; endGroup]
-I*w1*ket[0, 0, 0] + I*w0*ket[0, 0, 1] - I*w3*ket[0, 1, 0] + I*w2*ket[0, 1, 1] - 
 
  I*w5*ket[1, 0, 0] + I*w4*ket[1, 0, 1] - I*w7*ket[1, 1, 0] + I*w6*ket[1, 1, 1]
;[o]
-I w1 ket[0, 0, 0] + I w0 ket[0, 0, 1] - I w3 ket[0, 1, 0] + I w2 ket[0, 1, 1] - 
 
  I w5 ket[1, 0, 0] + I w4 ket[1, 0, 1] - I w7 ket[1, 1, 0] + I w6 ket[1, 1, 1]
:[font = section; inactive; preserveAspect; startGroup]
Laflamme's Circuit for Correcting 1 General Error amongst 5 Entangled Qubits
:[font = text; inactive; preserveAspect]
In Chapter 10 of "Explorations in Quantum Computing", we describe a scheme by Laflamme, Miquel, Paz and Zurek for encoding a single (logical) qubit in an entangled set of 5 qubits such that the logical qubit can be recovered even if there is 1 general type error amongst the 5 entangled qubits (a general error is a phase shift plus a bit flip). Thus this scheme could be used to preserve the correctness of the state of a simple quantum memory register. We will simulate the scheme as follows:

1. Encode a given state, |psi>, as an entangled state of 5-qubits
2. Simulate the introduction of an error in the entanglement
3. Decode the (buggy) state of the entangled qubits
4. Determine the "error syndrome"
5. Given the error syndrome, apply the appropriate unitary operator (rotation)  to correct the state.
:[font = subsection; inactive; preserveAspect; startGroup]
Step 1: Encode a Qubit in a Set of 5 Entangled Qubits
:[font = text; inactive; preserveAspect]
The encoding we choose is to map the logical qubit state |0> and |1> into the following entangled states:
:[font = input; preserveAspect; startGroup]
EncodeQubit[ket[0]]
:[font = output; output; inactive; preserveAspect; endGroup]
0.3535533905932737*ket[0, 0, 0, 0, 0] + 0.3535533905932737*ket[0, 0, 1, 1, 0] + 
 
  0.3535533905932737*ket[0, 1, 0, 0, 1] - 0.3535533905932737*ket[0, 1, 1, 1, 1] + 
 
  0.3535533905932737*ket[1, 0, 0, 1, 1] + 0.3535533905932737*ket[1, 0, 1, 0, 1] + 
 
  0.3535533905932737*ket[1, 1, 0, 1, 0] - 0.3535533905932737*ket[1, 1, 1, 0, 0]
;[o]
0.353553 ket[0, 0, 0, 0, 0] + 0.353553 ket[0, 0, 1, 1, 0] + 
 
  0.353553 ket[0, 1, 0, 0, 1] - 0.353553 ket[0, 1, 1, 1, 1] + 
 
  0.353553 ket[1, 0, 0, 1, 1] + 0.353553 ket[1, 0, 1, 0, 1] + 
 
  0.353553 ket[1, 1, 0, 1, 0] - 0.353553 ket[1, 1, 1, 0, 0]
:[font = input; preserveAspect; startGroup]
EncodeQubit[ket[1]]
:[font = output; output; inactive; preserveAspect; endGroup]
0.3535533905932737*ket[0, 0, 0, 1, 1] - 0.3535533905932737*ket[0, 0, 1, 0, 1] - 
 
  0.3535533905932737*ket[0, 1, 0, 1, 0] - 0.3535533905932737*ket[0, 1, 1, 0, 0] - 
 
  0.3535533905932737*ket[1, 0, 0, 0, 0] + 0.3535533905932737*ket[1, 0, 1, 1, 0] + 
 
  0.3535533905932737*ket[1, 1, 0, 0, 1] + 0.3535533905932737*ket[1, 1, 1, 1, 1]
;[o]
0.353553 ket[0, 0, 0, 1, 1] - 0.353553 ket[0, 0, 1, 0, 1] - 
 
  0.353553 ket[0, 1, 0, 1, 0] - 0.353553 ket[0, 1, 1, 0, 0] - 
 
  0.353553 ket[1, 0, 0, 0, 0] + 0.353553 ket[1, 0, 1, 1, 0] + 
 
  0.353553 ket[1, 1, 0, 0, 1] + 0.353553 ket[1, 1, 1, 1, 1]
:[font = text; inactive; preserveAspect]
To do the encoding, we use the following circuit (this is a simplified version of the original Laflamme, Miquel, Paz and Zurek circuit invented by Samuel Braunstein). Here is Braunstein's circuit in Mathematica. (note that we embed each gate in the full 2^5 x 2^5 dimensional space).  Look in Chapter 10 of "Explorations in Quantum Computing" for a circuit diagram. 
;[s]
3:0,0;199,1;210,0;367,-1;
2:2,13,9,Times,0,12,0,0,0;1,13,9,Times,2,12,0,0,0;
:[font = input; initialization; preserveAspect]
*)
LaflammeEncodingCircuit[]:=
	N[XORGate[1,5,5] .
	  XORGate[1,4,5] .
	  LOP[1,5] .
	  XORGate[1,2,5] .
	  XORGate[1,3,5] .
	  XORGate[2,5,5] .
	  LOP[2,5] .
	  XORGate[2,3,5] .
	  XORGate[2,4,5] .
	  LDaggerOP[2,5]
	 ]
(*
:[font = text; inactive; preserveAspect]
where 
:[font = input; initialization; preserveAspect]
*)
L = 1/Sqrt[2] {{1, -1}, 
               {1,  1}};
LDagger = Conjugate[Transpose[L]];

XOR = {{1,0,0,0},
       {0,1,0,0},
       {0,0,0,1},
       {0,0,1,0}};
(*
:[font = text; inactive; preserveAspect]
However, we need to embed these in a larger circuit. Thus, as before we write:
:[font = input; initialization; preserveAspect]
*)
LOP[i_, m_]:=
	Apply[Direct,
	      ReplacePart[Table[IdentityMatrix[2], {m}], L, i]]
	      
LDaggerOP[i_, m_]:=
	Apply[Direct,
	      ReplacePart[Table[IdentityMatrix[2], {m}], LDagger, i]]
	      
XORGate[a_, b_, m_]:=
	CNGate[a, b, m]
(*
:[font = text; inactive; preserveAspect]
The XORGate is the same as a controlled-NOT gate (CNGate) defined at the end of this file.
:[font = text; inactive; preserveAspect]
The actual encoding is a direct application of the circuit to an arbitrary input state.
:[font = input; initialization; preserveAspect]
*)
EncodeQubit[state_]:=
	(LaflammeEncodingCircuit[] . 
	 KetToColumnVector[Direct[state, ket[0], ket[0], ket[0], ket[0]]]) // N // Chop // ColumnVectorToKet
(*
:[font = text; inactive; preserveAspect]
To simulate the action of the Laflamme et al. circuit on an arbitrary state, we first generate a random superposition and then encode the resulting state using the LaflammeEncodingCircuit[].
:[font = input; preserveAspect; startGroup]
?RandomSuperposition
:[font = print; inactive; preserveAspect; endGroup]
RandomSuperposition[m] creates a normalized superposition of 2^m eigenstates whose
   amplitudes are random complex numbers. These eigenstates represent the possible
   states of an m-bit memory register of a quantum computer. You can generate an
   un-normalized superposition by setting the option Normalized->False.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
To generate a random superposition of 1 qubit, enter the following command.
:[font = input; preserveAspect; startGroup]
state1 = RandomSuperposition[1]
:[font = output; output; inactive; preserveAspect; endGroup; endGroup]
(0.624781767448082 - 0.4262032497671136*I)*ket[0] + 
 
  (0.281198129343284 - 0.5906997079788001*I)*ket[1]
;[o]
(0.624782 - 0.426203 I) ket[0] + (0.281198 - 0.5907 I) ket[1]
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
To encode this state using the Laflamme et al. code enter the following command.
:[font = input; preserveAspect; startGroup]
encodedQubit = EncodeQubit[state1]
:[font = output; output; inactive; preserveAspect; endGroup; endGroup; endGroup]
(0.2208937122621276 - 0.1506856040370349*I)*ket[0, 0, 0, 0, 0] + 
 
  (0.099418552057804 - 0.2088438845783615*I)*ket[0, 0, 0, 1, 1] + 
 
  (-0.099418552057804 + 0.2088438845783615*I)*ket[0, 0, 1, 0, 1] + 
 
  (0.2208937122621276 - 0.1506856040370349*I)*ket[0, 0, 1, 1, 0] + 
 
  (0.2208937122621276 - 0.1506856040370349*I)*ket[0, 1, 0, 0, 1] + 
 
  (-0.099418552057804 + 0.2088438845783615*I)*ket[0, 1, 0, 1, 0] + 
 
  (-0.099418552057804 + 0.2088438845783615*I)*ket[0, 1, 1, 0, 0] + 
 
  (-0.2208937122621276 + 0.1506856040370349*I)*ket[0, 1, 1, 1, 1] + 
 
  (-0.099418552057804 + 0.2088438845783615*I)*ket[1, 0, 0, 0, 0] + 
 
  (0.2208937122621276 - 0.1506856040370349*I)*ket[1, 0, 0, 1, 1] + 
 
  (0.2208937122621276 - 0.1506856040370349*I)*ket[1, 0, 1, 0, 1] + 
 
  (0.099418552057804 - 0.2088438845783615*I)*ket[1, 0, 1, 1, 0] + 
 
  (0.099418552057804 - 0.2088438845783615*I)*ket[1, 1, 0, 0, 1] + 
 
  (0.2208937122621276 - 0.1506856040370349*I)*ket[1, 1, 0, 1, 0] + 
 
  (-0.2208937122621276 + 0.1506856040370349*I)*ket[1, 1, 1, 0, 0] + 
 
  (0.099418552057804 - 0.2088438845783615*I)*ket[1, 1, 1, 1, 1]
;[o]
(0.220894 - 0.150686 I) ket[0, 0, 0, 0, 0] + 
 
  (0.0994186 - 0.208844 I) ket[0, 0, 0, 1, 1] + 
 
  (-0.0994186 + 0.208844 I) ket[0, 0, 1, 0, 1] + 
 
  (0.220894 - 0.150686 I) ket[0, 0, 1, 1, 0] + 
 
  (0.220894 - 0.150686 I) ket[0, 1, 0, 0, 1] + 
 
  (-0.0994186 + 0.208844 I) ket[0, 1, 0, 1, 0] + 
 
  (-0.0994186 + 0.208844 I) ket[0, 1, 1, 0, 0] + 
 
  (-0.220894 + 0.150686 I) ket[0, 1, 1, 1, 1] + 
 
  (-0.0994186 + 0.208844 I) ket[1, 0, 0, 0, 0] + 
 
  (0.220894 - 0.150686 I) ket[1, 0, 0, 1, 1] + 
 
  (0.220894 - 0.150686 I) ket[1, 0, 1, 0, 1] + 
 
  (0.0994186 - 0.208844 I) ket[1, 0, 1, 1, 0] + 
 
  (0.0994186 - 0.208844 I) ket[1, 1, 0, 0, 1] + 
 
  (0.220894 - 0.150686 I) ket[1, 1, 0, 1, 0] + 
 
  (-0.220894 + 0.150686 I) ket[1, 1, 1, 0, 0] + 
 
  (0.0994186 - 0.208844 I) ket[1, 1, 1, 1, 1]
:[font = subsection; inactive; preserveAspect; startGroup]
Step 2: Simulate the Introduction of an Error
:[font = text; inactive; preserveAspect]
Next we will introduce one error.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
buggyState = BitFlipAndPhaseShiftError[encodedQubit, 4, 5]
:[font = output; output; inactive; preserveAspect; endGroup; endGroup; endGroup]
(-0.2088438845783615 - 0.099418552057804*I)*ket[0, 0, 0, 0, 1] + 
 
  (0.1506856040370349 + 0.2208937122621276*I)*ket[0, 0, 0, 1, 0] + 
 
  (-0.1506856040370349 - 0.2208937122621276*I)*ket[0, 0, 1, 0, 0] + 
 
  (-0.2088438845783615 - 0.099418552057804*I)*ket[0, 0, 1, 1, 1] + 
 
  (0.2088438845783615 + 0.099418552057804*I)*ket[0, 1, 0, 0, 0] + 
 
  (0.1506856040370349 + 0.2208937122621276*I)*ket[0, 1, 0, 1, 1] + 
 
  (0.1506856040370349 + 0.2208937122621276*I)*ket[0, 1, 1, 0, 1] + 
 
  (-0.2088438845783615 - 0.099418552057804*I)*ket[0, 1, 1, 1, 0] + 
 
  (-0.1506856040370349 - 0.2208937122621276*I)*ket[1, 0, 0, 0, 1] + 
 
  (-0.2088438845783615 - 0.099418552057804*I)*ket[1, 0, 0, 1, 0] + 
 
  (-0.2088438845783615 - 0.099418552057804*I)*ket[1, 0, 1, 0, 0] + 
 
  (0.1506856040370349 + 0.2208937122621276*I)*ket[1, 0, 1, 1, 1] + 
 
  (-0.1506856040370349 - 0.2208937122621276*I)*ket[1, 1, 0, 0, 0] + 
 
  (0.2088438845783615 + 0.099418552057804*I)*ket[1, 1, 0, 1, 1] + 
 
  (-0.2088438845783615 - 0.099418552057804*I)*ket[1, 1, 1, 0, 1] + 
 
  (-0.1506856040370349 - 0.2208937122621276*I)*ket[1, 1, 1, 1, 0]
;[o]
(-0.208844 - 0.0994186 I) ket[0, 0, 0, 0, 1] + 
 
  (0.150686 + 0.220894 I) ket[0, 0, 0, 1, 0] + 
 
  (-0.150686 - 0.220894 I) ket[0, 0, 1, 0, 0] + 
 
  (-0.208844 - 0.0994186 I) ket[0, 0, 1, 1, 1] + 
 
  (0.208844 + 0.0994186 I) ket[0, 1, 0, 0, 0] + 
 
  (0.150686 + 0.220894 I) ket[0, 1, 0, 1, 1] + 
 
  (0.150686 + 0.220894 I) ket[0, 1, 1, 0, 1] + 
 
  (-0.208844 - 0.0994186 I) ket[0, 1, 1, 1, 0] + 
 
  (-0.150686 - 0.220894 I) ket[1, 0, 0, 0, 1] + 
 
  (-0.208844 - 0.0994186 I) ket[1, 0, 0, 1, 0] + 
 
  (-0.208844 - 0.0994186 I) ket[1, 0, 1, 0, 0] + 
 
  (0.150686 + 0.220894 I) ket[1, 0, 1, 1, 1] + 
 
  (-0.150686 - 0.220894 I) ket[1, 1, 0, 0, 0] + 
 
  (0.208844 + 0.0994186 I) ket[1, 1, 0, 1, 1] + 
 
  (-0.208844 - 0.0994186 I) ket[1, 1, 1, 0, 1] + 
 
  (-0.150686 - 0.220894 I) ket[1, 1, 1, 1, 0]
:[font = subsection; inactive; preserveAspect; startGroup]
Step 3: Decode the (Buggy) Entangled Qubits
:[font = text; inactive; preserveAspect]
Next we attempt to decode the buggy state using the LaflammeDecodingCircuit[].
:[font = input; initialization; preserveAspect]
*)

LaflammeDecodingCircuit[]:=
	N[LDaggerOP[2,5] .
	  XORGate[2,4,5] .
	  XORGate[2,3,5] .
	  LOP[2,5] .
	  XORGate[2,5,5] .
	  XORGate[1,3,5] .
	  XORGate[1,2,5] .
	  LOP[1,5] .
	  XORGate[1,4,5] .
	  XORGate[1,5,5]
	]
(*
:[font = input; initialization; preserveAspect]
*)
DecodeQubit[buggyState_]:=
	(LaflammeDecodingCircuit[] . 
	 KetToColumnVector[buggyState]) // N // Chop // ColumnVectorToKet
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
decodedQubit = DecodeQubit[buggyState]
:[font = output; output; inactive; preserveAspect; endGroup; endGroup; endGroup]
(0.4262032497671135 + 0.624781767448082*I)*ket[0, 1, 1, 0, 0] + 
 
  (-0.5906997079788 - 0.2811981293432839*I)*ket[1, 1, 1, 0, 0]
;[o]
(0.426203 + 0.624782 I) ket[0, 1, 1, 0, 0] + 
 
  (-0.5907 - 0.281198 I) ket[1, 1, 1, 0, 0]
:[font = subsection; inactive; preserveAspect; startGroup]
Step 4: Determine the "Error Syndrome"
:[font = text; inactive; preserveAspect]
We determine the error syndrome by measuring the state of the qubits 2 through 5 inclusive.
:[font = input; initialization; preserveAspect]
*)
MeasureErrorSyndrome[decodedState_, bitsToRead_]:=
	ReadPartOfMemoryRegister[decodedState, bitsToRead];
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
{errorSyndrome, finalState} = 
	MeasureErrorSyndrome[decodedQubit, {2,3,4,5}]
:[font = output; output; inactive; preserveAspect; endGroup; endGroup; endGroup]
{{1, 1, 0, 0}, (0.4262032497671135 + 0.624781767448082*I)*ket[0, 1, 1, 0, 0] + 
 
   (-0.5906997079788 - 0.2811981293432839*I)*ket[1, 1, 1, 0, 0]}
;[o]
{{1, 1, 0, 0}, (0.426203 + 0.624782 I) ket[0, 1, 1, 0, 0] + 
 
   (-0.5907 - 0.281198 I) ket[1, 1, 1, 0, 0]}
:[font = subsection; inactive; preserveAspect; startGroup]
Step 5: Apply the Appropriate Rotation to Correct the Error
:[font = text; inactive; preserveAspect]
Given, knowledge of the error syndrome (the values of bits 2 through 5) we can then apply a rotation that restores the original (unknown) state of the qubit we are trying to preserve.
:[font = input; initialization; preserveAspect]
*)
RotateState[state_, errorSyndrome_]:=
	Module[{U},
		U = Lookup[errorSyndrome];
		ColumnVectorToKet[Apply[Direct, 
		                        Join[{U}, Table[IdentityMatrix[2], {4}]]] . 
		                  KetToColumnVector[state]
		                 ]
	]
(*
:[font = text; inactive; preserveAspect]
Here is the lookup table that relates each possible error syndrome to the correct rotation to be applied to the final state.
:[font = input; initialization; preserveAspect]
*)
Lookup[{0,0,0,0}]:= {{ 1,  0}, { 0, -1}} 	(* P5  *)
Lookup[{0,0,0,1}]:= {{-I,  0}, { 0,  I}} 	(* BP5 *)
Lookup[{0,0,1,0}]:= {{ 0,  1}, {-1,  0}} 	(* B2  *)
Lookup[{0,0,1,1}]:= {{ 0,  1}, { 1,  0}} 	(* P3  *)
Lookup[{0,1,0,0}]:= {{ 0, -I}, {-I,  0}} 	(* BP2 *)
Lookup[{0,1,0,1}]:= {{ I,  0}, { 0,  I}} 	(* BP1 *)
Lookup[{0,1,1,0}]:= {{ 1,  0}, { 0, -1}} 	(* P1  *)
Lookup[{0,1,1,1}]:= {{ 0, -I}, {-I,  0}} 	(* BP3 *)
Lookup[{1,0,0,0}]:= {{ 0,  1}, { 1,  0}} 	(* B1  *)
Lookup[{1,0,0,1}]:= {{ 0,  1}, {-1,  0}} 	(* B4  *)
Lookup[{1,0,1,0}]:= {{ 0,  1}, {-1,  0}} 	(* B5  *)
Lookup[{1,0,1,1}]:= {{ 0,  1}, {-1,  0}} (* No Error *)
Lookup[{1,1,0,0}]:= {{-I,  0}, { 0,  I}} 	(* BP4 *)
Lookup[{1,1,0,1}]:= {{ 0,  1}, { 1,  0}} 	(* P2  *)
Lookup[{1,1,1,0}]:= {{ 1,  0}, { 0, -1}} 	(* P4  *)
Lookup[{1,1,1,1}]:= {{ 0,  1}, {-1,  0}} 	(* B3  *)

	
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
RotateState[finalState, errorSyndrome]
:[font = output; output; inactive; preserveAspect; endGroup; endGroup]
(0.624781767448082 - 0.4262032497671135*I)*ket[0, 1, 1, 0, 0] + 
 
  (0.2811981293432839 - 0.5906997079788*I)*ket[1, 1, 1, 0, 0]
;[o]
(0.624782 - 0.426203 I) ket[0, 1, 1, 0, 0] + 
 
  (0.281198 - 0.5907 I) ket[1, 1, 1, 0, 0]
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = text; inactive; preserveAspect]
Now compare this with the original state (state1).
:[font = input; preserveAspect; startGroup]
state1
:[font = output; output; inactive; preserveAspect; endGroup]
(0.624781767448082 - 0.4262032497671136*I)*ket[0] + 
 
  (0.281198129343284 - 0.5906997079788001*I)*ket[1]
;[o]
(0.624782 - 0.426203 I) ket[0] + (0.281198 - 0.5907 I) ket[1]
:[font = text; inactive; preserveAspect; cellOutline; endGroup; endGroup; endGroup]
The original state has been restored depsite the random error that was introduced after the qubit was placed in the encoded form. At no time was the unknown state measured. The quantum error correction was accomplished without ever knowing the state being corrected.
:[font = section; inactive; preserveAspect; startGroup]
Utility: Tool to Determine the Error Syndrome and Necessary Rotation to Correct the Output
:[font = text; inactive; preserveAspect]
This section contains a tool that will automatically search for the appropriate rotation needed to restore a qubit to its original state for each possible error syndrome. This knowledge forms the basis for the lookup table in the error correction technique. The code currently presumes that you have a 5-qubit encoding and you obtain the error syndrome by measuring bits 2 through 5 inclusive. (I think this choice means, unfortunately, that the "no error" error syndrome is not {0,0,0,0}. Instead it is {1,0,1,1}. This is not wrong but perhaps there is a way of extracting a syndrome from the other lines such that the "no error" indication is {0,0,0,0}). 
:[font = subsubsection; inactive; Cclosed; preserveAspect; startGroup]
Code
:[font = input; initialization; preserveAspect]
*)
DetermineErrorSyndromeAndCorrection[inputState_, {errorType_, i_, j_}]:=
	Module[{encodedState, buggyState, decodedState, syndrome, finalState,
	        correction, error},
		encodedState      = EncodeQubit[inputState];
		buggyState        = If[errorType===NoError,
		                       encodedState,
		                       Apply[errorType, {encodedState, i, j}]
		                      ];
		decodedState      = DecodeQubit[buggyState];
		{syndrome, finalState} = MeasureErrorSyndrome[decodedState, {2,3,4,5}];
		correction = FindRotation[finalState, inputState];
		error = If[errorType===NoError, NoError, Apply[errorType,{i}]];
		{error, syndrome, correction}
	]
(*
:[font = input; initialization; preserveAspect]
*)
FindRotation[w0_. ket[bit0_,___] + w1_. ket[bit1_,___], correctState_]:=
	Module[{vec1, vec2, ans},
		vec1 = KetToColumnVector[w0 ket[bit0] + w1 ket[bit1]];
		vec2 = KetToColumnVector[correctState];
		ans = Select[$PossibleRotations, (N[# . vec1] == N[vec2])&];
		If[Length[ans]>1, 
		   Message[FindRotation::multipleAnswers, ans],
		   ans[[1]]
		  ]
	]
FindRotation::multipleAnswers = 
	"Warning: There seem to be multiple rotations that restore the state.\
	The answers are `1` .";
(*
:[font = input; initialization; preserveAspect]
*)
$PossibleRotations = {
		{{1,0},{0,1}},
		{{1,0},{0,-1}},
		{{-1,0},{0,1}},
		{{-1,0},{0,-1}},
		{{0,1},{1,0}},
		{{0,1},{-1,0}},
		{{0,-1},{1,0}},
		{{0,-1},{-1,0}},
		{{I,0},{0,I}},
		{{I,0},{0,-I}},
		{{-I,0},{0,I}},
		{{-I,0},{0,-I}},
		{{0,I},{I,0}},
		{{0,I},{-I,0}},
		{{0,-I},{I,0}},
		{{0,-I},{-I,0}}
		
	};
(*
:[font = text; inactive; preserveAspect]
I check that all these candidates are indeed unitary. This test shows they are:
:[font = input; preserveAspect; startGroup]
Map[UnitaryQ, $PossibleRotations]
:[font = output; output; inactive; preserveAspect; endGroup]
{True, True, True, True, True, True, True, True, True, True, True, True, True, True, 
 
  True, True}
;[o]
{True, True, True, True, True, True, True, True, True, True, True, True, True, True, 
 
  True, True}
:[font = text; inactive; preserveAspect; endGroup]
Here's an example of how to use this tool. The user must specify the kind of error they want to introduce and which bit is affected. The following places a bit flip error on the 3rd of 5 bits.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
DetermineErrorSyndromeAndCorrection[state1, {BitFlipError, 3, 5}]
:[font = output; output; inactive; preserveAspect; endGroup]
{BitFlipError[3], {1, 1, 1, 1}, {{0, 1}, {-1, 0}}}
;[o]
{BitFlipError[3], {1, 1, 1, 1}, {{0, 1}, {-1, 0}}}
:[font = text; inactive; preserveAspect; endGroup]
The output says that the error type was a bit flip on the 3rd bit,  the error syndrome was {1,1,1,1} and the appropriate action needed to restore the final state (i.e. the state after the error syndrome has been measured) is to apply the unitary matrix {{0,1},{-1,0}} to the first bit of the output state.

Here's another example, that determines the correct course of action if there is a phase shift error on the 5th of the 5 entangled qubits.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
DetermineErrorSyndromeAndCorrection[state1, {PhaseShiftError, 5, 5}]
:[font = output; output; inactive; preserveAspect; endGroup]
{PhaseShiftError[5], {0, 0, 0, 0}, {{1, 0}, {0, -1}}}
;[o]
{PhaseShiftError[5], {0, 0, 0, 0}, {{1, 0}, {0, -1}}}
:[font = text; inactive; preserveAspect; endGroup]
The error type is a phase shift, its syndrome is {0,0,0,0} and the corrective action is to apply the rotation {{1,0},{0,-1}} to the first bit in the output state.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
DetermineErrorSyndromeAndCorrection[state1, {BitFlipAndPhaseShiftError, 5, 5}]
:[font = output; output; inactive; preserveAspect; endGroup]
{BitFlipAndPhaseShiftError[5], {0, 0, 0, 1}, {{-I, 0}, {0, I}}}
;[o]
{BitFlipAndPhaseShiftError[5], {0, 0, 0, 1}, {{-I, 0}, {0, I}}}
:[font = text; inactive; preserveAspect; endGroup]
You can also introduce no error whatsoever!
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
DetermineErrorSyndromeAndCorrection[state1, {NoError, _, _}]
:[font = output; output; inactive; preserveAspect; endGroup]
{NoError, {1, 0, 1, 1}, {{0, 1}, {-1, 0}}}
;[o]
{NoError, {1, 0, 1, 1}, {{0, 1}, {-1, 0}}}
:[font = text; inactive; preserveAspect; endGroup; endGroup]
In this case the "error syndrome" (i.e. the output on bits 2 through 5) is {1,0,1,1}. In other words, if you measure the state of bits 2 through 5 and obtain the results 1,0,1,1 there was no error. However, you must rotate the final state by applying the matrix {{0,1},{-1,0}} (as the circuit mixes things up).
:[font = section; inactive; preserveAspect; startGroup]
A "Rule of Thumb" for Quantum Error-Correcting Codes
:[font = text; inactive; preserveAspect]
Ekert and Macchiavello give a rough quantum version of the Hamming bound for  quantum error correcting codes.  This bound can be inverted to determine some crude relationships between the number of qubits in a codeword, the number of (general type) errors that can be tolerated and the number of logical qubits that can be encoded using such a code. Their formula is neither necessary nor sufficient but nevertheless seems to be pretty good in practice!  See Chapter 10, "Explorations in Quantum Computer" for an explanation.
:[font = input; initialization; preserveAspect]
*)
(* n = number of qubits in a codeword
   t = number of (general) errors i.e. bit flips and/or phase shifts
   l = number of (logical) qubits being encoded
*)
QuantumHammingBoundQ[l_, n_, t_]:=
	2^l Sum[3^i Binomial[n, i], {i, 0, t}] <= 2^n
	
(*
:[font = text; inactive; preserveAspect]
We can use the quantum Hamming bound to determine the maximum possible number of general errors that we can tolerate and still be able to correct the code after such errors.
:[font = input; initialization; preserveAspect]
*)
MaxErrorsCorrectable[l_, n_]:=
	Module[{t},
		t=1;
		While[QuantumHammingBoundQ[l,n,t]===True, t++];
		t-1
	]
(*
:[font = text; inactive; preserveAspect]
Let's create some data that explores how the maximum number of correctable errors varies as a function of the number of logical qubits being encoded and the number of physical qubits used to do the encoding.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect]
data3D = Table[Table[MaxErrorsCorrectable[l, n], {n, 1, 30}], {l, 1, 25}];
:[font = text; inactive; preserveAspect; endGroup]
We can plot this using BarChart3D.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; initialization; preserveAspect]
*)
Needs["Graphics`Graphics3D`"]
(*
:[font = input; preserveAspect; startGroup]
BarChart3D[data3D, 
           FaceGrids->{{-1,0,0}, {0,1,0}},
           AxesLabel->{"l", "n", "tMax"},
           AxesEdge->{{-1,-1}, {1,-1}, {-1,-1}}];
:[font = postscript; PostScript; formatAsPostScript; output; inactive; preserveAspect; pictureLeft = 34; pictureWidth = 230; pictureHeight = 249; endGroup]
%!
%%Creator: Mathematica
%%AspectRatio: 1.08079 
MathPictureStart
%% Graphics3D
/Courier findfont 8  scalefont  setfont
% Scaling calculations
-0.0260524 1.10184 0.0275459 1.10184 [
[(0)] .10384 .24874 .97891 1 Msboxa
[(10)] .30021 .16579 .83912 1 Msboxa
[(20)] .51453 .07535 .69933 1 Msboxa
[(l)] .31959 .08564 .79718 1 Msboxa
[(0)] .67799 .02603 -1 .41952 Msboxa
[(10)] .76525 .16834 -1 .3848 Msboxa
[(20)] .84155 .2928 -1 .35539 Msboxa
[(30)] .90883 .40255 -1 .33016 Msboxa
[(n)] .87306 .21499 -1 .36805 Msboxa
[(0)] .09676 .27537 1 -0.41829 Msboxa
[(2)] .07311 .4234 1 -0.38733 Msboxa
[(4)] .04663 .58917 1 -0.35226 Msboxa
[(6)] .01678 .77609 1 -0.31222 Msboxa
[(tMax)] -0.00432 .52779 1 -0.37036 Msboxa
[ 0 0 0 0 ]
[ 1 1.08079 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
[ ] 0 setdash
0 g
p
p
.002 w
.11113 .25957 m
.66295 .02755 L
s
P
p
.002 w
.11347 .25859 m
.11829 .26351 L
s
P
[(0)] .10384 .24874 .97891 1 Mshowa
p
.002 w
.30906 .17634 m
.31349 .18162 L
s
P
[(10)] .30021 .16579 .83912 1 Mshowa
p
.002 w
.52243 .08663 m
.52637 .09228 L
s
P
[(20)] .51453 .07535 .69933 1 Mshowa
p
.001 w
.15128 .24269 m
.15412 .24568 L
s
P
p
.001 w
.18972 .22652 m
.19252 .22956 L
s
P
p
.001 w
.22882 .21008 m
.23158 .21316 L
s
P
p
.001 w
.2686 .19336 m
.2713 .19648 L
s
P
p
.001 w
.35024 .15903 m
.35284 .16224 L
s
P
p
.001 w
.39214 .14141 m
.39469 .14467 L
s
P
p
.001 w
.4348 .12348 m
.43729 .12678 L
s
P
p
.001 w
.47822 .10522 m
.48065 .10856 L
s
P
p
.001 w
.56744 .0677 m
.56975 .07113 L
s
P
p
.001 w
.6133 .04843 m
.61553 .0519 L
s
P
p
.001 w
.66001 .02879 m
.66217 .0323 L
s
P
[(l)] .31959 .08564 .79718 1 Mshowa
P
p
p
.002 w
.66295 .02755 m
.90359 .41965 L
s
P
p
.002 w
.66529 .03136 m
.65894 .03402 L
s
P
[(0)] .67799 .02603 -1 .41952 Mshowa
p
.002 w
.7524 .17329 m
.74597 .17576 L
s
P
[(10)] .76525 .16834 -1 .3848 Mshowa
p
.002 w
.82857 .29741 m
.82208 .29971 L
s
P
[(20)] .84155 .2928 -1 .35539 Mshowa
p
.002 w
.89575 .40687 m
.88921 .40903 L
s
P
[(30)] .90883 .40255 -1 .33016 Mshowa
p
.001 w
.6837 .06135 m
.67988 .06292 L
s
P
p
.001 w
.70159 .0905 m
.69776 .09205 L
s
P
p
.001 w
.71899 .11886 m
.71515 .12039 L
s
P
p
.001 w
.73592 .14644 m
.73207 .14795 L
s
P
p
.001 w
.76844 .19942 m
.76457 .20089 L
s
P
p
.001 w
.78406 .22488 m
.78019 .22632 L
s
P
p
.001 w
.79928 .24968 m
.7954 .2511 L
s
P
p
.001 w
.81411 .27385 m
.81022 .27525 L
s
P
p
.001 w
.84267 .32039 m
.83877 .32175 L
s
P
p
.001 w
.85643 .3428 m
.85252 .34415 L
s
P
p
.001 w
.86985 .36467 m
.86594 .366 L
s
P
p
.001 w
.88296 .38603 m
.87904 .38734 L
s
P
[(n)] .87306 .21499 -1 .36805 Mshowa
P
p
p
.002 w
.11113 .25957 m
.02755 .78701 L
s
P
p
.002 w
.10947 .27006 m
.11582 .2674 L
s
P
[(0)] .09676 .27537 1 -0.41829 Mshowa
p
.002 w
.08595 .41843 m
.09238 .41594 L
s
P
[(2)] .07311 .4234 1 -0.38733 Mshowa
p
.002 w
.05962 .5846 m
.06612 .58231 L
s
P
[(4)] .04663 .58917 1 -0.35226 Mshowa
p
.002 w
.02993 .77198 m
.0365 .76993 L
s
P
[(6)] .01678 .77609 1 -0.31222 Mshowa
p
.001 w
.10497 .29845 m
.10879 .29687 L
s
P
p
.001 w
.10037 .32746 m
.1042 .3259 L
s
P
p
.001 w
.09567 .35711 m
.09951 .35557 L
s
P
p
.001 w
.09087 .38743 m
.09471 .38591 L
s
P
p
.001 w
.08093 .45014 m
.08479 .44867 L
s
P
p
.001 w
.07579 .48259 m
.07966 .48114 L
s
P
p
.001 w
.07053 .51579 m
.0744 .51437 L
s
P
p
.001 w
.06514 .54979 m
.06903 .54839 L
s
P
p
.001 w
.05397 .62025 m
.05788 .61891 L
s
P
p
.001 w
.04818 .65679 m
.0521 .65547 L
s
P
p
.001 w
.04225 .69423 m
.04617 .69294 L
s
P
p
.001 w
.03616 .73262 m
.0401 .73135 L
s
P
[(tMax)] -0.00432 .52779 1 -0.37036 Mshowa
P
0 0 m
1 0 L
1 1.08079 L
0 1.08079 L
closepath
clip
newpath
p
p
.4 g
.002 w
.10947 .27006 m
.42216 .58687 L
s
P
p
.4 g
.002 w
.08595 .41843 m
.41738 .72386 L
s
P
p
.4 g
.002 w
.05962 .5846 m
.41212 .87425 L
s
P
p
.4 g
.002 w
.02993 .77198 m
.40632 1.04011 L
s
P
p
.4 g
.002 w
.11409 .26259 m
.03127 .78963 L
s
P
p
.4 g
.002 w
.22539 .37608 m
.16931 .88677 L
s
P
p
.4 g
.002 w
.32406 .4767 m
.28897 .97098 L
s
P
p
.4 g
.002 w
.41216 .56653 m
.39369 1.04467 L
s
P
.002 w
.11113 .25957 m
.02755 .78701 L
s
.02755 .78701 m
.40586 1.05324 L
s
.40586 1.05324 m
.42251 .57709 L
s
.42251 .57709 m
.11113 .25957 L
s
.66295 .02755 m
.90359 .41965 L
s
.90359 .41965 m
.97245 .92345 L
s
.97245 .92345 m
.69507 .58075 L
s
.69507 .58075 m
.66295 .02755 L
s
.11113 .25957 m
.02755 .78701 L
s
.02755 .78701 m
.69507 .58075 L
s
.69507 .58075 m
.66295 .02755 L
s
.66295 .02755 m
.11113 .25957 L
s
p
.4 g
.42459 .57641 m
.40828 1.05269 L
s
P
p
.4 g
.59733 .51988 m
.60961 1.00657 L
s
P
p
.4 g
.78293 .45914 m
.82859 .9564 L
s
P
p
.4 g
.42216 .58687 m
.90499 .42984 L
s
P
p
.4 g
.41738 .72386 m
.92458 .57319 L
s
P
p
.4 g
.41212 .87425 m
.9463 .73213 L
s
P
p
.4 g
.40632 1.04011 m
.97052 .90934 L
s
P
.42251 .57709 m
.90359 .41965 L
s
.90359 .41965 m
.97245 .92345 L
s
.97245 .92345 m
.40586 1.05324 L
s
.40586 1.05324 m
.42251 .57709 L
s
P
p
.002 w
.44334 .57162 m .42645 .57714 L .41812 .56863 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43062 1.02758 m .44334 .57162 L .42645 .57714 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.42645 .57714 m .41116 1.03212 L .43062 1.02758 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.41812 .56863 m .42645 .57714 L .41116 1.03212 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.46035 .56606 m .44334 .57162 L .43507 .56304 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45215 .93716 m .46035 .56606 L .44334 .57162 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.44334 .57162 m .43301 .94196 L .45215 .93716 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.41812 .56863 m .43507 .56304 L .44334 .57162 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43507 .56304 m .44334 .57162 L .43301 .94196 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.43507 .56304 m .44334 .57162 L .43062 1.02758 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.42098 1.02053 m .43507 .56304 L .41812 .56863 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.41116 1.03212 m .40143 1.02512 L .41812 .56863 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.41812 .56863 m .40143 1.02512 L .42098 1.02053 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.43507 .56304 m .41812 .56863 L .4097 .56002 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42363 .93456 m .43507 .56304 L .41812 .56863 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.41812 .56863 m .40456 .93938 L .42363 .93456 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.4097 .56002 m .41812 .56863 L .40456 .93938 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.47749 .56045 m .46035 .56606 L .45215 .55742 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47143 .93232 m .47749 .56045 L .46035 .56606 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.46035 .56606 m .45215 .93716 L .47143 .93232 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.43507 .56304 m .45215 .55742 L .46035 .56606 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45215 .55742 m .46035 .56606 L .45215 .93716 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.45215 .55742 m .46035 .56606 L .45215 .93716 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.44284 .9297 m .45215 .55742 L .43507 .56304 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.43301 .94196 m .42363 .93456 L .43507 .56304 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.43062 1.02758 m .42098 1.02053 L .43507 .56304 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.43507 .56304 m .42363 .93456 L .44284 .9297 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.45215 .55742 m .43507 .56304 L .42672 .55438 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44284 .9297 m .45215 .55742 L .43507 .56304 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.43507 .56304 m .42363 .93456 L .44284 .9297 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.4097 .56002 m .42672 .55438 L .43507 .56304 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42672 .55438 m .43507 .56304 L .42363 .93456 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.42672 .55438 m .43507 .56304 L .42363 .93456 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.49474 .55481 m .47749 .56045 L .46935 .55176 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49088 .92744 m .49474 .55481 L .47749 .56045 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.47749 .56045 m .47143 .93232 L .49088 .92744 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.45215 .55742 m .46935 .55176 L .47749 .56045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46935 .55176 m .47749 .56045 L .47143 .93232 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.46935 .55176 m .47749 .56045 L .47143 .93232 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.41413 .92707 m .42672 .55438 L .4097 .56002 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.40456 .93938 m .39499 .93195 L .4097 .56002 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.4097 .56002 m .39499 .93195 L .41413 .92707 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.42672 .55438 m .4097 .56002 L .40119 .55132 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41413 .92707 m .42672 .55438 L .4097 .56002 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.4097 .56002 m .39499 .93195 L .41413 .92707 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.40119 .55132 m .4097 .56002 L .39499 .93195 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.46221 .9248 m .46935 .55176 L .45215 .55742 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.45215 .93716 m .44284 .9297 L .45215 .55742 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.45215 .93716 m .44284 .9297 L .45215 .55742 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.45215 .55742 m .44284 .9297 L .46221 .9248 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.46935 .55176 m .45215 .55742 L .44386 .54869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46221 .9248 m .46935 .55176 L .45215 .55742 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.45215 .55742 m .44284 .9297 L .46221 .9248 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.42672 .55438 m .44386 .54869 L .45215 .55742 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44386 .54869 m .45215 .55742 L .44284 .9297 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.44386 .54869 m .45215 .55742 L .44284 .9297 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.51212 .54913 m .49474 .55481 L .48667 .54605 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51084 .84035 m .51212 .54913 L .49474 .55481 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.49474 .55481 m .49173 .84546 L .51084 .84035 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.46935 .55176 m .48667 .54605 L .49474 .55481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48667 .54605 m .49474 .55481 L .49173 .84546 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.48667 .54605 m .49474 .55481 L .49088 .92744 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.43343 .92215 m .44386 .54869 L .42672 .55438 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.42363 .93456 m .41413 .92707 L .42672 .55438 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.42363 .93456 m .41413 .92707 L .42672 .55438 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.42672 .55438 m .41413 .92707 L .43343 .92215 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.44386 .54869 m .42672 .55438 L .41827 .54561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43343 .92215 m .44386 .54869 L .42672 .55438 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.42672 .55438 m .41413 .92707 L .43343 .92215 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.40119 .55132 m .41827 .54561 L .42672 .55438 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41827 .54561 m .42672 .55438 L .41413 .92707 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.41827 .54561 m .42672 .55438 L .41413 .92707 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.48174 .91986 m .48667 .54605 L .46935 .55176 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.47143 .93232 m .46221 .9248 L .46935 .55176 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.47143 .93232 m .46221 .9248 L .46935 .55176 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.46935 .55176 m .46221 .9248 L .48174 .91986 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.48667 .54605 m .46935 .55176 L .46112 .54297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48174 .91986 m .48667 .54605 L .46935 .55176 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.46935 .55176 m .46221 .9248 L .48174 .91986 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.44386 .54869 m .46112 .54297 L .46935 .55176 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46112 .54297 m .46935 .55176 L .46221 .9248 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.46112 .54297 m .46935 .55176 L .46221 .9248 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.40451 .91948 m .41827 .54561 L .40119 .55132 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.39499 .93195 m .38529 .92442 L .40119 .55132 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.40119 .55132 m .38529 .92442 L .40451 .91948 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.41827 .54561 m .40119 .55132 L .39258 .54252 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40451 .91948 m .41827 .54561 L .40119 .55132 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.40119 .55132 m .38529 .92442 L .40451 .91948 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.39258 .54252 m .40119 .55132 L .38529 .92442 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.52962 .54341 m .51212 .54913 L .50411 .54031 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5301 .8352 m .52962 .54341 L .51212 .54913 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.51212 .54913 m .51084 .84035 L .5301 .8352 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.48667 .54605 m .50411 .54031 L .51212 .54913 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50411 .54031 m .51212 .54913 L .51084 .84035 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.50411 .54031 m .51212 .54913 L .51084 .84035 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.45288 .91719 m .46112 .54297 L .44386 .54869 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.44284 .9297 m .43343 .92215 L .44386 .54869 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.44284 .9297 m .43343 .92215 L .44386 .54869 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.44386 .54869 m .43343 .92215 L .45288 .91719 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.46112 .54297 m .44386 .54869 L .43547 .53987 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45288 .91719 m .46112 .54297 L .44386 .54869 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.44386 .54869 m .43343 .92215 L .45288 .91719 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.41827 .54561 m .43547 .53987 L .44386 .54869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43547 .53987 m .44386 .54869 L .43343 .92215 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.43547 .53987 m .44386 .54869 L .43343 .92215 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.50201 .8324 m .50411 .54031 L .48667 .54605 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.49173 .84546 m .48282 .83758 L .48667 .54605 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.49088 .92744 m .48174 .91986 L .48667 .54605 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.48667 .54605 m .48282 .83758 L .50201 .8324 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.50411 .54031 m .48667 .54605 L .47851 .5372 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50201 .8324 m .50411 .54031 L .48667 .54605 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.48667 .54605 m .48282 .83758 L .50201 .8324 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.46112 .54297 m .47851 .5372 L .48667 .54605 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47851 .5372 m .48667 .54605 L .48282 .83758 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.47851 .5372 m .48667 .54605 L .48174 .91986 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.42389 .9145 m .43547 .53987 L .41827 .54561 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.41413 .92707 m .40451 .91948 L .41827 .54561 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.41413 .92707 m .40451 .91948 L .41827 .54561 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.41827 .54561 m .40451 .91948 L .42389 .9145 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.43547 .53987 m .41827 .54561 L .40973 .53675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42389 .9145 m .43547 .53987 L .41827 .54561 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.41827 .54561 m .40451 .91948 L .42389 .9145 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.39258 .54252 m .40973 .53675 L .41827 .54561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40973 .53675 m .41827 .54561 L .40451 .91948 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.40973 .53675 m .41827 .54561 L .40451 .91948 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.54724 .53764 m .52962 .54341 L .52168 .53452 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54951 .83 m .54724 .53764 L .52962 .54341 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.52962 .54341 m .5301 .8352 L .54951 .83 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.50411 .54031 m .52168 .53452 L .52962 .54341 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52168 .53452 m .52962 .54341 L .5301 .8352 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.52168 .53452 m .52962 .54341 L .5301 .8352 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.47249 .91219 m .47851 .5372 L .46112 .54297 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.46221 .9248 m .45288 .91719 L .46112 .54297 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.46221 .9248 m .45288 .91719 L .46112 .54297 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.46112 .54297 m .45288 .91719 L .47249 .91219 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.47851 .5372 m .46112 .54297 L .4528 .53408 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47381 .8296 m .47851 .5372 L .46112 .54297 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.46112 .54297 m .45469 .8348 L .47381 .8296 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.43547 .53987 m .4528 .53408 L .46112 .54297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4528 .53408 m .46112 .54297 L .45469 .8348 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.4528 .53408 m .46112 .54297 L .45288 .91719 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.39478 .9118 m .40973 .53675 L .39258 .54252 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.38529 .92442 m .37547 .91681 L .39258 .54252 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.39258 .54252 m .37547 .91681 L .39478 .9118 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.40973 .53675 m .39258 .54252 L .38388 .53363 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39478 .9118 m .40973 .53675 L .39258 .54252 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.39258 .54252 m .37547 .91681 L .39478 .9118 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.38388 .53363 m .39258 .54252 L .37547 .91681 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.52136 .82719 m .52168 .53452 L .50411 .54031 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.51084 .84035 m .50201 .8324 L .50411 .54031 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.51084 .84035 m .50201 .8324 L .50411 .54031 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.50411 .54031 m .50201 .8324 L .52136 .82719 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.52168 .53452 m .50411 .54031 L .49602 .53139 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52136 .82719 m .52168 .53452 L .50411 .54031 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.50411 .54031 m .50201 .8324 L .52136 .82719 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.47851 .5372 m .49602 .53139 L .50411 .54031 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49602 .53139 m .50411 .54031 L .50201 .8324 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.49602 .53139 m .50411 .54031 L .50201 .8324 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.44343 .90948 m .4528 .53408 L .43547 .53987 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.43343 .92215 m .42389 .9145 L .43547 .53987 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.43343 .92215 m .42389 .9145 L .43547 .53987 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.43547 .53987 m .42389 .9145 L .44343 .90948 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.4528 .53408 m .43547 .53987 L .427 .53094 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44549 .82678 m .4528 .53408 L .43547 .53987 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.43547 .53987 m .42644 .832 L .44549 .82678 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.40973 .53675 m .427 .53094 L .43547 .53987 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.427 .53094 m .43547 .53987 L .42644 .832 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.427 .53094 m .43547 .53987 L .42389 .9145 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.565 .53184 m .54724 .53764 L .53938 .5287 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56908 .82476 m .565 .53184 L .54724 .53764 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.54724 .53764 m .54951 .83 L .56908 .82476 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.52168 .53452 m .53938 .5287 L .54724 .53764 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53938 .5287 m .54724 .53764 L .54951 .83 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.53938 .5287 m .54724 .53764 L .54951 .83 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.49308 .82436 m .49602 .53139 L .47851 .5372 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.48282 .83758 m .47381 .8296 L .47851 .5372 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.48174 .91986 m .47249 .91219 L .47851 .5372 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.47851 .5372 m .47381 .8296 L .49308 .82436 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.49602 .53139 m .47851 .5372 L .47025 .52825 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49308 .82436 m .49602 .53139 L .47851 .5372 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.47851 .5372 m .47381 .8296 L .49308 .82436 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.4528 .53408 m .47025 .52825 L .47851 .5372 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47025 .52825 m .47851 .5372 L .47381 .8296 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.47025 .52825 m .47851 .5372 L .47381 .8296 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.41424 .90676 m .427 .53094 L .40973 .53675 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.40451 .91948 m .39478 .9118 L .40973 .53675 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.40451 .91948 m .39478 .9118 L .40973 .53675 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.40973 .53675 m .39478 .9118 L .41424 .90676 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.427 .53094 m .40973 .53675 L .40109 .5278 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41705 .82396 m .427 .53094 L .40973 .53675 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.40973 .53675 m .39807 .8292 L .41705 .82396 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.38388 .53363 m .40109 .5278 L .40973 .53675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40109 .5278 m .40973 .53675 L .39807 .8292 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.40109 .5278 m .40973 .53675 L .39478 .9118 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.54085 .82193 m .53938 .5287 L .52168 .53452 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.5301 .8352 m .52136 .82719 L .52168 .53452 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.5301 .8352 m .52136 .82719 L .52168 .53452 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.52168 .53452 m .52136 .82719 L .54085 .82193 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.53938 .5287 m .52168 .53452 L .51365 .52554 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54085 .82193 m .53938 .5287 L .52168 .53452 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.52168 .53452 m .52136 .82719 L .54085 .82193 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.49602 .53139 m .51365 .52554 L .52168 .53452 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51365 .52554 m .52168 .53452 L .52136 .82719 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.51365 .52554 m .52168 .53452 L .52136 .82719 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.46469 .82152 m .47025 .52825 L .4528 .53408 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.45469 .8348 m .44549 .82678 L .4528 .53408 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.45288 .91719 m .44343 .90948 L .4528 .53408 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.4528 .53408 m .44549 .82678 L .46469 .82152 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.47025 .52825 m .4528 .53408 L .44439 .52509 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46469 .82152 m .47025 .52825 L .4528 .53408 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.4528 .53408 m .44549 .82678 L .46469 .82152 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.427 .53094 m .44439 .52509 L .4528 .53408 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44439 .52509 m .4528 .53408 L .44549 .82678 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.44439 .52509 m .4528 .53408 L .44549 .82678 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.38492 .90403 m .40109 .5278 L .38388 .53363 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.37547 .91681 m .36554 .90909 L .38388 .53363 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.38388 .53363 m .36554 .90909 L .38492 .90403 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.40109 .5278 m .38388 .53363 L .37509 .52464 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38492 .90403 m .40109 .5278 L .38388 .53363 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.38388 .53363 m .36554 .90909 L .38492 .90403 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.37509 .52464 m .38388 .53363 L .36554 .90909 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.58288 .52599 m .565 .53184 L .5572 .52283 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58721 .74064 m .58288 .52599 L .565 .53184 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.565 .53184 m .56798 .74609 L .58721 .74064 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.53938 .5287 m .5572 .52283 L .565 .53184 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5572 .52283 m .565 .53184 L .56798 .74609 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.5572 .52283 m .565 .53184 L .56908 .82476 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.51251 .81908 m .51365 .52554 L .49602 .53139 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.50201 .8324 m .49308 .82436 L .49602 .53139 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.50201 .8324 m .49308 .82436 L .49602 .53139 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.49602 .53139 m .49308 .82436 L .51251 .81908 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.51365 .52554 m .49602 .53139 L .48783 .52237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51251 .81908 m .51365 .52554 L .49602 .53139 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.49602 .53139 m .49308 .82436 L .51251 .81908 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.47025 .52825 m .48783 .52237 L .49602 .53139 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48783 .52237 m .49602 .53139 L .49308 .82436 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.48783 .52237 m .49602 .53139 L .49308 .82436 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.43618 .81867 m .44439 .52509 L .427 .53094 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.42644 .832 m .41705 .82396 L .427 .53094 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.42389 .9145 m .41424 .90676 L .427 .53094 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.427 .53094 m .41705 .82396 L .43618 .81867 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.44439 .52509 m .427 .53094 L .41843 .52192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43618 .81867 m .44439 .52509 L .427 .53094 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.427 .53094 m .41705 .82396 L .43618 .81867 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.40109 .5278 m .41843 .52192 L .427 .53094 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41843 .52192 m .427 .53094 L .41705 .82396 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.41843 .52192 m .427 .53094 L .41705 .82396 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.56051 .81663 m .5572 .52283 L .53938 .5287 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.54951 .83 m .54085 .82193 L .53938 .5287 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.54951 .83 m .54085 .82193 L .53938 .5287 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.53938 .5287 m .54085 .82193 L .56051 .81663 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.5572 .52283 m .53938 .5287 L .53142 .51965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56051 .81663 m .5572 .52283 L .53938 .5287 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.53938 .5287 m .54085 .82193 L .56051 .81663 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.51365 .52554 m .53142 .51965 L .53938 .5287 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53142 .51965 m .53938 .5287 L .54085 .82193 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.53142 .51965 m .53938 .5287 L .54085 .82193 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.48405 .81622 m .48783 .52237 L .47025 .52825 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.47381 .8296 m .46469 .82152 L .47025 .52825 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.47381 .8296 m .46469 .82152 L .47025 .52825 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.47025 .52825 m .46469 .82152 L .48405 .81622 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.48783 .52237 m .47025 .52825 L .46191 .51919 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48405 .81622 m .48783 .52237 L .47025 .52825 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.47025 .52825 m .46469 .82152 L .48405 .81622 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.44439 .52509 m .46191 .51919 L .47025 .52825 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46191 .51919 m .47025 .52825 L .46469 .82152 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.46191 .51919 m .47025 .52825 L .46469 .82152 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.40754 .81581 m .41843 .52192 L .40109 .5278 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.39807 .8292 m .38848 .82112 L .40109 .5278 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.39478 .9118 m .38492 .90403 L .40109 .5278 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.40109 .5278 m .38848 .82112 L .40754 .81581 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.41843 .52192 m .40109 .5278 L .39236 .51874 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40754 .81581 m .41843 .52192 L .40109 .5278 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.40109 .5278 m .38848 .82112 L .40754 .81581 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.37509 .52464 m .39236 .51874 L .40109 .5278 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39236 .51874 m .40109 .5278 L .38848 .82112 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.39236 .51874 m .40109 .5278 L .38492 .90403 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.60089 .5201 m .58288 .52599 L .57515 .51692 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60658 .73515 m .60089 .5201 L .58288 .52599 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.58288 .52599 m .58721 .74064 L .60658 .73515 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.5572 .52283 m .57515 .51692 L .58288 .52599 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57515 .51692 m .58288 .52599 L .58721 .74064 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.57515 .51692 m .58288 .52599 L .58721 .74064 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.53209 .81376 m .53142 .51965 L .51365 .52554 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.52136 .82719 m .51251 .81908 L .51365 .52554 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.52136 .82719 m .51251 .81908 L .51365 .52554 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.51365 .52554 m .51251 .81908 L .53209 .81376 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.53142 .51965 m .51365 .52554 L .50554 .51646 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53209 .81376 m .53142 .51965 L .51365 .52554 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.51365 .52554 m .51251 .81908 L .53209 .81376 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.48783 .52237 m .50554 .51646 L .51365 .52554 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50554 .51646 m .51365 .52554 L .51251 .81908 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.50554 .51646 m .51365 .52554 L .51251 .81908 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.45546 .81334 m .46191 .51919 L .44439 .52509 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.44549 .82678 m .43618 .81867 L .44439 .52509 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.44549 .82678 m .43618 .81867 L .44439 .52509 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.44439 .52509 m .43618 .81867 L .45546 .81334 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.46191 .51919 m .44439 .52509 L .43589 .516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45546 .81334 m .46191 .51919 L .44439 .52509 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.44439 .52509 m .43618 .81867 L .45546 .81334 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.41843 .52192 m .43589 .516 L .44439 .52509 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43589 .516 m .44439 .52509 L .43618 .81867 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.43589 .516 m .44439 .52509 L .43618 .81867 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.37494 .89615 m .39236 .51874 L .37509 .52464 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.36554 .90909 m .35547 .90128 L .37509 .52464 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.37509 .52464 m .35547 .90128 L .37494 .89615 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.39236 .51874 m .37509 .52464 L .36619 .51554 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37878 .81293 m .39236 .51874 L .37509 .52464 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.37509 .52464 m .3598 .81826 L .37878 .81293 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.36619 .51554 m .37509 .52464 L .3598 .81826 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.57893 .73217 m .57515 .51692 L .5572 .52283 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.56798 .74609 m .55962 .73769 L .5572 .52283 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.56908 .82476 m .56051 .81663 L .5572 .52283 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.5572 .52283 m .55962 .73769 L .57893 .73217 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.57515 .51692 m .5572 .52283 L .54931 .51371 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57893 .73217 m .57515 .51692 L .5572 .52283 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.5572 .52283 m .55962 .73769 L .57893 .73217 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.53142 .51965 m .54931 .51371 L .5572 .52283 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54931 .51371 m .5572 .52283 L .55962 .73769 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.54931 .51371 m .5572 .52283 L .56051 .81663 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.50355 .81087 m .50554 .51646 L .48783 .52237 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.49308 .82436 m .48405 .81622 L .48783 .52237 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.49308 .82436 m .48405 .81622 L .48783 .52237 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.48783 .52237 m .48405 .81622 L .50355 .81087 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.50554 .51646 m .48783 .52237 L .47956 .51326 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50355 .81087 m .50554 .51646 L .48783 .52237 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.48783 .52237 m .48405 .81622 L .50355 .81087 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.46191 .51919 m .47956 .51326 L .48783 .52237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47956 .51326 m .48783 .52237 L .48405 .81622 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.47956 .51326 m .48783 .52237 L .48405 .81622 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.42675 .81046 m .43589 .516 L .41843 .52192 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.41705 .82396 m .40754 .81581 L .41843 .52192 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.41705 .82396 m .40754 .81581 L .41843 .52192 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.41843 .52192 m .40754 .81581 L .42675 .81046 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.43589 .516 m .41843 .52192 L .40976 .5128 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42675 .81046 m .43589 .516 L .41843 .52192 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.41843 .52192 m .40754 .81581 L .42675 .81046 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.39236 .51874 m .40976 .5128 L .41843 .52192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40976 .5128 m .41843 .52192 L .40754 .81581 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.40976 .5128 m .41843 .52192 L .40754 .81581 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.61903 .51417 m .60089 .5201 L .59324 .51096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62611 .72961 m .61903 .51417 L .60089 .5201 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.60089 .5201 m .60658 .73515 L .62611 .72961 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.57515 .51692 m .59324 .51096 L .60089 .5201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59324 .51096 m .60089 .5201 L .60658 .73515 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.59324 .51096 m .60089 .5201 L .60658 .73515 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.55183 .80839 m .54931 .51371 L .53142 .51965 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.54085 .82193 m .53209 .81376 L .53142 .51965 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.54085 .82193 m .53209 .81376 L .53142 .51965 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.53142 .51965 m .53209 .81376 L .55183 .80839 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.54931 .51371 m .53142 .51965 L .52337 .5105 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55115 .72918 m .54931 .51371 L .53142 .51965 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.53142 .51965 m .53191 .73472 L .55115 .72918 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.50554 .51646 m .52337 .5105 L .53142 .51965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52337 .5105 m .53142 .51965 L .53191 .73472 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.52337 .5105 m .53142 .51965 L .53209 .81376 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.47489 .80798 m .47956 .51326 L .46191 .51919 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.46469 .82152 m .45546 .81334 L .46191 .51919 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.46469 .82152 m .45546 .81334 L .46191 .51919 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.46191 .51919 m .45546 .81334 L .47489 .80798 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.47956 .51326 m .46191 .51919 L .45347 .51004 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47489 .80798 m .47956 .51326 L .46191 .51919 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.46191 .51919 m .45546 .81334 L .47489 .80798 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.43589 .516 m .45347 .51004 L .46191 .51919 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45347 .51004 m .46191 .51919 L .45546 .81334 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.45347 .51004 m .46191 .51919 L .45546 .81334 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.39791 .80756 m .40976 .5128 L .39236 .51874 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.38848 .82112 m .37878 .81293 L .39236 .51874 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.38492 .90403 m .37494 .89615 L .39236 .51874 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.39236 .51874 m .37878 .81293 L .39791 .80756 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.40976 .5128 m .39236 .51874 L .38353 .50958 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39791 .80756 m .40976 .5128 L .39236 .51874 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.39236 .51874 m .37878 .81293 L .39791 .80756 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.36619 .51554 m .38353 .50958 L .39236 .51874 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38353 .50958 m .39236 .51874 L .37878 .81293 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.38353 .50958 m .39236 .51874 L .37878 .81293 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.59839 .72661 m .59324 .51096 L .57515 .51692 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.58721 .74064 m .57893 .73217 L .57515 .51692 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.58721 .74064 m .57893 .73217 L .57515 .51692 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.57515 .51692 m .57893 .73217 L .59839 .72661 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.59324 .51096 m .57515 .51692 L .56734 .50774 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59839 .72661 m .59324 .51096 L .57515 .51692 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.57515 .51692 m .57893 .73217 L .59839 .72661 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.54931 .51371 m .56734 .50774 L .57515 .51692 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56734 .50774 m .57515 .51692 L .57893 .73217 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.56734 .50774 m .57515 .51692 L .57893 .73217 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.52322 .80548 m .52337 .5105 L .50554 .51646 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.51251 .81908 m .50355 .81087 L .50554 .51646 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.51251 .81908 m .50355 .81087 L .50554 .51646 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.50554 .51646 m .50355 .81087 L .52322 .80548 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.52337 .5105 m .50554 .51646 L .49733 .50727 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52326 .72618 m .52337 .5105 L .50554 .51646 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.50554 .51646 m .50409 .73174 L .52326 .72618 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.47956 .51326 m .49733 .50727 L .50554 .51646 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49733 .50727 m .50554 .51646 L .50409 .73174 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.49733 .50727 m .50554 .51646 L .50355 .81087 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.44611 .80507 m .45347 .51004 L .43589 .516 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.43618 .81867 m .42675 .81046 L .43589 .516 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.43618 .81867 m .42675 .81046 L .43589 .516 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.43589 .516 m .42675 .81046 L .44611 .80507 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.45347 .51004 m .43589 .516 L .42728 .50681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44611 .80507 m .45347 .51004 L .43589 .516 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.43589 .516 m .42675 .81046 L .44611 .80507 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.40976 .5128 m .42728 .50681 L .43589 .516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42728 .50681 m .43589 .516 L .42675 .81046 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.42728 .50681 m .43589 .516 L .42675 .81046 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.36895 .80465 m .38353 .50958 L .36619 .51554 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.3598 .81826 m .34989 .81005 L .36619 .51554 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.36619 .51554 m .34989 .81005 L .36895 .80465 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.38353 .50958 m .36619 .51554 L .3572 .50635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36895 .80465 m .38353 .50958 L .36619 .51554 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.36619 .51554 m .34989 .81005 L .36895 .80465 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.3572 .50635 m .36619 .51554 L .34989 .81005 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.63731 .5082 m .61903 .51417 L .61145 .50496 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64579 .72403 m .63731 .5082 L .61903 .51417 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.61903 .51417 m .62611 .72961 L .64579 .72403 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.59324 .51096 m .61145 .50496 L .61903 .51417 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61145 .50496 m .61903 .51417 L .62611 .72961 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.61145 .50496 m .61903 .51417 L .62611 .72961 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.57055 .7236 m .56734 .50774 L .54931 .51371 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.55962 .73769 m .55115 .72918 L .54931 .51371 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.56051 .81663 m .55183 .80839 L .54931 .51371 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.54931 .51371 m .55115 .72918 L .57055 .7236 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.56734 .50774 m .54931 .51371 L .54134 .5045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57055 .7236 m .56734 .50774 L .54931 .51371 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.54931 .51371 m .55115 .72918 L .57055 .7236 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.52337 .5105 m .54134 .5045 L .54931 .51371 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54134 .5045 m .54931 .51371 L .55115 .72918 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.54134 .5045 m .54931 .51371 L .55115 .72918 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.49449 .80256 m .49733 .50727 L .47956 .51326 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.48405 .81622 m .47489 .80798 L .47956 .51326 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.48405 .81622 m .47489 .80798 L .47956 .51326 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.47956 .51326 m .47489 .80798 L .49449 .80256 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.49733 .50727 m .47956 .51326 L .47119 .50403 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49525 .72316 m .49733 .50727 L .47956 .51326 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.47956 .51326 m .47615 .72875 L .49525 .72316 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.45347 .51004 m .47119 .50403 L .47956 .51326 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47119 .50403 m .47956 .51326 L .47615 .72875 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.47119 .50403 m .47956 .51326 L .47489 .80798 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.4172 .80214 m .42728 .50681 L .40976 .5128 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.40754 .81581 m .39791 .80756 L .40976 .5128 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.40754 .81581 m .39791 .80756 L .40976 .5128 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.40976 .5128 m .39791 .80756 L .4172 .80214 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.42728 .50681 m .40976 .5128 L .40099 .50357 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4172 .80214 m .42728 .50681 L .40976 .5128 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.40976 .5128 m .39791 .80756 L .4172 .80214 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.38353 .50958 m .40099 .50357 L .40976 .5128 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40099 .50357 m .40976 .5128 L .39791 .80756 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.40099 .50357 m .40976 .5128 L .39791 .80756 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.618 .721 m .61145 .50496 L .59324 .51096 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.60658 .73515 m .59839 .72661 L .59324 .51096 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.60658 .73515 m .59839 .72661 L .59324 .51096 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.59324 .51096 m .59839 .72661 L .618 .721 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.61145 .50496 m .59324 .51096 L .5855 .50171 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.618 .721 m .61145 .50496 L .59324 .51096 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.59324 .51096 m .59839 .72661 L .618 .721 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.56734 .50774 m .5855 .50171 L .59324 .51096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5855 .50171 m .59324 .51096 L .59839 .72661 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.5855 .50171 m .59324 .51096 L .59839 .72661 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.54259 .72057 m .54134 .5045 L .52337 .5105 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.53191 .73472 m .52326 .72618 L .52337 .5105 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.53209 .81376 m .52322 .80548 L .52337 .5105 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.52337 .5105 m .52326 .72618 L .54259 .72057 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.54134 .5045 m .52337 .5105 L .51524 .50125 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54259 .72057 m .54134 .5045 L .52337 .5105 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.52337 .5105 m .52326 .72618 L .54259 .72057 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.49733 .50727 m .51524 .50125 L .52337 .5105 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51524 .50125 m .52337 .5105 L .52326 .72618 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.51524 .50125 m .52337 .5105 L .52326 .72618 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.46563 .79963 m .47119 .50403 L .45347 .51004 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.45546 .81334 m .44611 .80507 L .45347 .51004 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.45546 .81334 m .44611 .80507 L .45347 .51004 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.45347 .51004 m .44611 .80507 L .46563 .79963 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.47119 .50403 m .45347 .51004 L .44494 .50078 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46712 .72014 m .47119 .50403 L .45347 .51004 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.45347 .51004 m .44809 .72575 L .46712 .72014 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.42728 .50681 m .44494 .50078 L .45347 .51004 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44494 .50078 m .45347 .51004 L .44809 .72575 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.44494 .50078 m .45347 .51004 L .44611 .80507 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.38817 .79921 m .40099 .50357 L .38353 .50958 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.37878 .81293 m .36895 .80465 L .38353 .50958 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.37878 .81293 m .36895 .80465 L .38353 .50958 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.38353 .50958 m .36895 .80465 L .38817 .79921 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.40099 .50357 m .38353 .50958 L .3746 .50032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38817 .79921 m .40099 .50357 L .38353 .50958 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.38353 .50958 m .36895 .80465 L .38817 .79921 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.3572 .50635 m .3746 .50032 L .38353 .50958 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3746 .50032 m .38353 .50958 L .36895 .80465 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.3746 .50032 m .38353 .50958 L .36895 .80465 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.65572 .50218 m .63731 .5082 L .6298 .49892 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66563 .7184 m .65572 .50218 L .63731 .5082 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.63731 .5082 m .64579 .72403 L .66563 .7184 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.61145 .50496 m .6298 .49892 L .63731 .5082 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6298 .49892 m .63731 .5082 L .64579 .72403 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.6298 .49892 m .63731 .5082 L .64579 .72403 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.59009 .71797 m .5855 .50171 L .56734 .50774 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.57893 .73217 m .57055 .7236 L .56734 .50774 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.57893 .73217 m .57055 .7236 L .56734 .50774 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.56734 .50774 m .57055 .7236 L .59009 .71797 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.5855 .50171 m .56734 .50774 L .55944 .49845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59009 .71797 m .5855 .50171 L .56734 .50774 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.56734 .50774 m .57055 .7236 L .59009 .71797 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.54134 .5045 m .55944 .49845 L .56734 .50774 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55944 .49845 m .56734 .50774 L .57055 .7236 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.55944 .49845 m .56734 .50774 L .57055 .7236 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.51451 .71753 m .51524 .50125 L .49733 .50727 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.50409 .73174 m .49525 .72316 L .49733 .50727 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.50355 .81087 m .49449 .80256 L .49733 .50727 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.49733 .50727 m .49525 .72316 L .51451 .71753 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.51524 .50125 m .49733 .50727 L .48903 .49798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51451 .71753 m .51524 .50125 L .49733 .50727 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.49733 .50727 m .49525 .72316 L .51451 .71753 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.47119 .50403 m .48903 .49798 L .49733 .50727 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48903 .49798 m .49733 .50727 L .49525 .72316 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.48903 .49798 m .49733 .50727 L .49525 .72316 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.43665 .79668 m .44494 .50078 L .42728 .50681 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.42675 .81046 m .4172 .80214 L .42728 .50681 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.42675 .81046 m .4172 .80214 L .42728 .50681 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.42728 .50681 m .4172 .80214 L .43665 .79668 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.44494 .50078 m .42728 .50681 L .41859 .49752 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43665 .79668 m .44494 .50078 L .42728 .50681 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.42728 .50681 m .4172 .80214 L .43665 .79668 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.40099 .50357 m .41859 .49752 L .42728 .50681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41859 .49752 m .42728 .50681 L .4172 .80214 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.41859 .49752 m .42728 .50681 L .4172 .80214 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.35901 .79626 m .3746 .50032 L .3572 .50635 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.34989 .81005 m .33987 .80173 L .3572 .50635 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.3572 .50635 m .33987 .80173 L .35901 .79626 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.3746 .50032 m .3572 .50635 L .3481 .49705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35901 .79626 m .3746 .50032 L .3572 .50635 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.3572 .50635 m .33987 .80173 L .35901 .79626 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.3481 .49705 m .3572 .50635 L .33987 .80173 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.63777 .71536 m .6298 .49892 L .61145 .50496 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.62611 .72961 m .618 .721 L .61145 .50496 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.62611 .72961 m .618 .721 L .61145 .50496 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.61145 .50496 m .618 .721 L .63777 .71536 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.6298 .49892 m .61145 .50496 L .60379 .49565 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63777 .71536 m .6298 .49892 L .61145 .50496 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.61145 .50496 m .618 .721 L .63777 .71536 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.5855 .50171 m .60379 .49565 L .61145 .50496 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60379 .49565 m .61145 .50496 L .618 .721 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.60379 .49565 m .61145 .50496 L .618 .721 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.56206 .71492 m .55944 .49845 L .54134 .5045 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.55115 .72918 m .54259 .72057 L .54134 .5045 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.55115 .72918 m .54259 .72057 L .54134 .5045 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.54134 .5045 m .54259 .72057 L .56206 .71492 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.55944 .49845 m .54134 .5045 L .53327 .49518 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56206 .71492 m .55944 .49845 L .54134 .5045 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.54134 .5045 m .54259 .72057 L .56206 .71492 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.51524 .50125 m .53327 .49518 L .54134 .5045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53327 .49518 m .54134 .5045 L .54259 .72057 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.53327 .49518 m .54134 .5045 L .54259 .72057 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.48631 .71448 m .48903 .49798 L .47119 .50403 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.47615 .72875 m .46712 .72014 L .47119 .50403 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.47489 .80798 m .46563 .79963 L .47119 .50403 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.47119 .50403 m .46712 .72014 L .48631 .71448 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.48903 .49798 m .47119 .50403 L .46272 .49471 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48631 .71448 m .48903 .49798 L .47119 .50403 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.47119 .50403 m .46712 .72014 L .48631 .71448 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.44494 .50078 m .46272 .49471 L .47119 .50403 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46272 .49471 m .47119 .50403 L .46712 .72014 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.46272 .49471 m .47119 .50403 L .46712 .72014 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.40754 .79373 m .41859 .49752 L .40099 .50357 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.39791 .80756 m .38817 .79921 L .40099 .50357 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.39791 .80756 m .38817 .79921 L .40099 .50357 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.40099 .50357 m .38817 .79921 L .40754 .79373 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.41859 .49752 m .40099 .50357 L .39213 .49424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40754 .79373 m .41859 .49752 L .40099 .50357 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.40099 .50357 m .38817 .79921 L .40754 .79373 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.3746 .50032 m .39213 .49424 L .40099 .50357 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39213 .49424 m .40099 .50357 L .38817 .79921 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.39213 .49424 m .40099 .50357 L .38817 .79921 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.67426 .49611 m .65572 .50218 L .64829 .49283 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68165 .63696 m .67426 .49611 L .65572 .50218 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.65572 .50218 m .66216 .64278 L .68165 .63696 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.6298 .49892 m .64829 .49283 L .65572 .50218 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64829 .49283 m .65572 .50218 L .66216 .64278 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.64829 .49283 m .65572 .50218 L .66563 .7184 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.60979 .7123 m .60379 .49565 L .5855 .50171 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.59839 .72661 m .59009 .71797 L .5855 .50171 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.59839 .72661 m .59009 .71797 L .5855 .50171 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.5855 .50171 m .59009 .71797 L .60979 .7123 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.60379 .49565 m .5855 .50171 L .57767 .49236 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60979 .7123 m .60379 .49565 L .5855 .50171 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.5855 .50171 m .59009 .71797 L .60979 .7123 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.55944 .49845 m .57767 .49236 L .5855 .50171 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57767 .49236 m .5855 .50171 L .59009 .71797 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.57767 .49236 m .5855 .50171 L .59009 .71797 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.53392 .71186 m .53327 .49518 L .51524 .50125 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.52326 .72618 m .51451 .71753 L .51524 .50125 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.52326 .72618 m .51451 .71753 L .51524 .50125 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.51524 .50125 m .51451 .71753 L .53392 .71186 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.53327 .49518 m .51524 .50125 L .50701 .49189 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53392 .71186 m .53327 .49518 L .51524 .50125 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.51524 .50125 m .51451 .71753 L .53392 .71186 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.48903 .49798 m .50701 .49189 L .51524 .50125 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50701 .49189 m .51524 .50125 L .51451 .71753 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.50701 .49189 m .51524 .50125 L .51451 .71753 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.45799 .71142 m .46272 .49471 L .44494 .50078 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.44809 .72575 m .43888 .7171 L .44494 .50078 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.44611 .80507 m .43665 .79668 L .44494 .50078 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.44494 .50078 m .43888 .7171 L .45799 .71142 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.46272 .49471 m .44494 .50078 L .43631 .49142 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45799 .71142 m .46272 .49471 L .44494 .50078 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.44494 .50078 m .43888 .7171 L .45799 .71142 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.41859 .49752 m .43631 .49142 L .44494 .50078 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43631 .49142 m .44494 .50078 L .43888 .7171 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.43631 .49142 m .44494 .50078 L .43665 .79668 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3783 .79076 m .39213 .49424 L .3746 .50032 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.36895 .80465 m .35901 .79626 L .3746 .50032 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.36895 .80465 m .35901 .79626 L .3746 .50032 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.3746 .50032 m .35901 .79626 L .3783 .79076 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.39213 .49424 m .3746 .50032 L .36557 .49095 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3783 .79076 m .39213 .49424 L .3746 .50032 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.3746 .50032 m .35901 .79626 L .3783 .79076 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.3481 .49705 m .36557 .49095 L .3746 .50032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36557 .49095 m .3746 .50032 L .35901 .79626 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.36557 .49095 m .3746 .50032 L .35901 .79626 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.6577 .70966 m .64829 .49283 L .6298 .49892 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.64579 .72403 m .63777 .71536 L .6298 .49892 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.64579 .72403 m .63777 .71536 L .6298 .49892 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.6298 .49892 m .63777 .71536 L .6577 .70966 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.64829 .49283 m .6298 .49892 L .62221 .48953 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65441 .6338 m .64829 .49283 L .6298 .49892 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.6298 .49892 m .63498 .63965 L .65441 .6338 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.60379 .49565 m .62221 .48953 L .6298 .49892 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62221 .48953 m .6298 .49892 L .63498 .63965 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.62221 .48953 m .6298 .49892 L .63777 .71536 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.5817 .70922 m .57767 .49236 L .55944 .49845 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.57055 .7236 m .56206 .71492 L .55944 .49845 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.57055 .7236 m .56206 .71492 L .55944 .49845 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.55944 .49845 m .56206 .71492 L .5817 .70922 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.57767 .49236 m .55944 .49845 L .55145 .48906 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5817 .70922 m .57767 .49236 L .55944 .49845 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.55944 .49845 m .56206 .71492 L .5817 .70922 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.53327 .49518 m .55145 .48906 L .55944 .49845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55145 .48906 m .55944 .49845 L .56206 .71492 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.55145 .48906 m .55944 .49845 L .56206 .71492 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.50565 .70878 m .50701 .49189 L .48903 .49798 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.49525 .72316 m .48631 .71448 L .48903 .49798 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.49525 .72316 m .48631 .71448 L .48903 .49798 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.48903 .49798 m .48631 .71448 L .50565 .70878 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.50701 .49189 m .48903 .49798 L .48064 .48859 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50565 .70878 m .50701 .49189 L .48903 .49798 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.48903 .49798 m .48631 .71448 L .50565 .70878 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.46272 .49471 m .48064 .48859 L .48903 .49798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48064 .48859 m .48903 .49798 L .48631 .71448 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.48064 .48859 m .48903 .49798 L .48631 .71448 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.42706 .7882 m .43631 .49142 L .41859 .49752 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.4172 .80214 m .40754 .79373 L .41859 .49752 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.4172 .80214 m .40754 .79373 L .41859 .49752 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.41859 .49752 m .40754 .79373 L .42706 .7882 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.43631 .49142 m .41859 .49752 L .40979 .48812 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42955 .70834 m .43631 .49142 L .41859 .49752 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.41859 .49752 m .41051 .71405 L .42955 .70834 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.39213 .49424 m .40979 .48812 L .41859 .49752 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40979 .48812 m .41859 .49752 L .41051 .71405 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.40979 .48812 m .41859 .49752 L .40754 .79373 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.34893 .78777 m .36557 .49095 L .3481 .49705 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.33987 .80173 m .32972 .7933 L .3481 .49705 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.3481 .49705 m .32972 .7933 L .34893 .78777 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.36557 .49095 m .3481 .49705 L .3389 .48764 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34893 .78777 m .36557 .49095 L .3481 .49705 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.3481 .49705 m .32972 .7933 L .34893 .78777 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.3389 .48764 m .3481 .49705 L .32972 .7933 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.69294 .49001 m .67426 .49611 L .66691 .4867 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70129 .63109 m .69294 .49001 L .67426 .49611 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.67426 .49611 m .68165 .63696 L .70129 .63109 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.64829 .49283 m .66691 .4867 L .67426 .49611 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66691 .4867 m .67426 .49611 L .68165 .63696 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.66691 .4867 m .67426 .49611 L .68165 .63696 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.62965 .70658 m .62221 .48953 L .60379 .49565 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.618 .721 m .60979 .7123 L .60379 .49565 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.618 .721 m .60979 .7123 L .60379 .49565 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.60379 .49565 m .60979 .7123 L .62965 .70658 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.62221 .48953 m .60379 .49565 L .59603 .48622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62705 .63064 m .62221 .48953 L .60379 .49565 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.60379 .49565 m .60769 .63651 L .62705 .63064 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.57767 .49236 m .59603 .48622 L .60379 .49565 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59603 .48622 m .60379 .49565 L .60769 .63651 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.59603 .48622 m .60379 .49565 L .60979 .7123 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.55348 .70614 m .55145 .48906 L .53327 .49518 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.54259 .72057 m .53392 .71186 L .53327 .49518 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.54259 .72057 m .53392 .71186 L .53327 .49518 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.53327 .49518 m .53392 .71186 L .55348 .70614 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.55145 .48906 m .53327 .49518 L .52512 .48575 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55348 .70614 m .55145 .48906 L .53327 .49518 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.53327 .49518 m .53392 .71186 L .55348 .70614 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.50701 .49189 m .52512 .48575 L .53327 .49518 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52512 .48575 m .53327 .49518 L .53392 .71186 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.52512 .48575 m .53327 .49518 L .53392 .71186 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.47726 .70569 m .48064 .48859 L .46272 .49471 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.46712 .72014 m .45799 .71142 L .46272 .49471 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.46712 .72014 m .45799 .71142 L .46272 .49471 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.46272 .49471 m .45799 .71142 L .47726 .70569 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.48064 .48859 m .46272 .49471 L .45416 .48528 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47726 .70569 m .48064 .48859 L .46272 .49471 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.46272 .49471 m .45799 .71142 L .47726 .70569 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.43631 .49142 m .45416 .48528 L .46272 .49471 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45416 .48528 m .46272 .49471 L .45799 .71142 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.45416 .48528 m .46272 .49471 L .45799 .71142 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.39775 .7852 m .40979 .48812 L .39213 .49424 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.38817 .79921 m .3783 .79076 L .39213 .49424 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.38817 .79921 m .3783 .79076 L .39213 .49424 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.39213 .49424 m .3783 .79076 L .39775 .7852 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.40979 .48812 m .39213 .49424 L .38317 .4848 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40099 .70525 m .40979 .48812 L .39213 .49424 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.39213 .49424 m .38202 .71098 L .40099 .70525 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.36557 .49095 m .38317 .4848 L .39213 .49424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38317 .4848 m .39213 .49424 L .38202 .71098 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.38317 .4848 m .39213 .49424 L .3783 .79076 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.67398 .62791 m .66691 .4867 L .64829 .49283 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.66216 .64278 m .65441 .6338 L .64829 .49283 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.66563 .7184 m .6577 .70966 L .64829 .49283 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.64829 .49283 m .65441 .6338 L .67398 .62791 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.66691 .4867 m .64829 .49283 L .64078 .48338 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67398 .62791 m .66691 .4867 L .64829 .49283 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.64829 .49283 m .65441 .6338 L .67398 .62791 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.62221 .48953 m .64078 .48338 L .64829 .49283 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64078 .48338 m .64829 .49283 L .65441 .6338 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.64078 .48338 m .64829 .49283 L .65441 .6338 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.60148 .70348 m .59603 .48622 L .57767 .49236 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.59009 .71797 m .5817 .70922 L .57767 .49236 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.59009 .71797 m .5817 .70922 L .57767 .49236 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.57767 .49236 m .5817 .70922 L .60148 .70348 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.59603 .48622 m .57767 .49236 L .56975 .4829 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59958 .62746 m .59603 .48622 L .57767 .49236 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.57767 .49236 m .58029 .63335 L .59958 .62746 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.55145 .48906 m .56975 .4829 L .57767 .49236 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56975 .4829 m .57767 .49236 L .58029 .63335 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.56975 .4829 m .57767 .49236 L .5817 .70922 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.52514 .70304 m .52512 .48575 L .50701 .49189 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.51451 .71753 m .50565 .70878 L .50701 .49189 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.51451 .71753 m .50565 .70878 L .50701 .49189 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.50701 .49189 m .50565 .70878 L .52514 .70304 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.52512 .48575 m .50701 .49189 L .49869 .48243 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52514 .70304 m .52512 .48575 L .50701 .49189 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.50701 .49189 m .50565 .70878 L .52514 .70304 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.48064 .48859 m .49869 .48243 L .50701 .49189 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49869 .48243 m .50701 .49189 L .50565 .70878 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.49869 .48243 m .50701 .49189 L .50565 .70878 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.44875 .70259 m .45416 .48528 L .43631 .49142 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.43888 .7171 m .42955 .70834 L .43631 .49142 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.43665 .79668 m .42706 .7882 L .43631 .49142 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.43631 .49142 m .42955 .70834 L .44875 .70259 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.45416 .48528 m .43631 .49142 L .42758 .48195 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44875 .70259 m .45416 .48528 L .43631 .49142 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.43631 .49142 m .42955 .70834 L .44875 .70259 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.40979 .48812 m .42758 .48195 L .43631 .49142 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42758 .48195 m .43631 .49142 L .42955 .70834 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.42758 .48195 m .43631 .49142 L .42955 .70834 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.36831 .78219 m .38317 .4848 L .36557 .49095 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.35901 .79626 m .34893 .78777 L .36557 .49095 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.35901 .79626 m .34893 .78777 L .36557 .49095 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.36557 .49095 m .34893 .78777 L .36831 .78219 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.38317 .4848 m .36557 .49095 L .35643 .48147 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36831 .78219 m .38317 .4848 L .36557 .49095 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.36557 .49095 m .34893 .78777 L .36831 .78219 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.3389 .48764 m .35643 .48147 L .36557 .49095 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35643 .48147 m .36557 .49095 L .34893 .78777 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.35643 .48147 m .36557 .49095 L .34893 .78777 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.71176 .48385 m .69294 .49001 L .68567 .48052 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72108 .62518 m .71176 .48385 L .69294 .49001 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.69294 .49001 m .70129 .63109 L .72108 .62518 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.66691 .4867 m .68567 .48052 L .69294 .49001 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68567 .48052 m .69294 .49001 L .70129 .63109 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.68567 .48052 m .69294 .49001 L .70129 .63109 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.64656 .62472 m .64078 .48338 L .62221 .48953 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.63498 .63965 m .62705 .63064 L .62221 .48953 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.63777 .71536 m .62965 .70658 L .62221 .48953 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.62221 .48953 m .62705 .63064 L .64656 .62472 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.64078 .48338 m .62221 .48953 L .61454 .48004 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64656 .62472 m .64078 .48338 L .62221 .48953 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.62221 .48953 m .62705 .63064 L .64656 .62472 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.59603 .48622 m .61454 .48004 L .62221 .48953 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61454 .48004 m .62221 .48953 L .62705 .63064 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.61454 .48004 m .62221 .48953 L .62705 .63064 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.5732 .70037 m .56975 .4829 L .55145 .48906 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.56206 .71492 m .55348 .70614 L .55145 .48906 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.56206 .71492 m .55348 .70614 L .55145 .48906 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.55145 .48906 m .55348 .70614 L .5732 .70037 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.56975 .4829 m .55145 .48906 L .54336 .47957 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57199 .62426 m .56975 .4829 L .55145 .48906 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.55145 .48906 m .55277 .63018 L .57199 .62426 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.52512 .48575 m .54336 .47957 L .55145 .48906 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54336 .47957 m .55145 .48906 L .55277 .63018 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.54336 .47957 m .55145 .48906 L .55348 .70614 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.49668 .69992 m .49869 .48243 L .48064 .48859 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.48631 .71448 m .47726 .70569 L .48064 .48859 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.48631 .71448 m .47726 .70569 L .48064 .48859 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.48064 .48859 m .47726 .70569 L .49668 .69992 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.49869 .48243 m .48064 .48859 L .47215 .47909 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49668 .69992 m .49869 .48243 L .48064 .48859 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.48064 .48859 m .47726 .70569 L .49668 .69992 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.45416 .48528 m .47215 .47909 L .48064 .48859 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47215 .47909 m .48064 .48859 L .47726 .70569 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.47215 .47909 m .48064 .48859 L .47726 .70569 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.42011 .69948 m .42758 .48195 L .40979 .48812 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.41051 .71405 m .40099 .70525 L .40979 .48812 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.40754 .79373 m .39775 .7852 L .40979 .48812 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.40979 .48812 m .40099 .70525 L .42011 .69948 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.42758 .48195 m .40979 .48812 L .40089 .47861 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42011 .69948 m .42758 .48195 L .40979 .48812 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.40979 .48812 m .40099 .70525 L .42011 .69948 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.38317 .4848 m .40089 .47861 L .40979 .48812 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40089 .47861 m .40979 .48812 L .40099 .70525 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.40089 .47861 m .40979 .48812 L .40099 .70525 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.33873 .77917 m .35643 .48147 L .3389 .48764 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.32972 .7933 m .31944 .78477 L .3389 .48764 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.3389 .48764 m .31944 .78477 L .33873 .77917 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.35643 .48147 m .3389 .48764 L .3296 .47813 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33873 .77917 m .35643 .48147 L .3389 .48764 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.3389 .48764 m .31944 .78477 L .33873 .77917 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.3296 .47813 m .3389 .48764 L .31944 .78477 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.6937 .62198 m .68567 .48052 L .66691 .4867 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.68165 .63696 m .67398 .62791 L .66691 .4867 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.68165 .63696 m .67398 .62791 L .66691 .4867 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.66691 .4867 m .67398 .62791 L .6937 .62198 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.68567 .48052 m .66691 .4867 L .65948 .47718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6937 .62198 m .68567 .48052 L .66691 .4867 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.66691 .4867 m .67398 .62791 L .6937 .62198 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.64078 .48338 m .65948 .47718 L .66691 .4867 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65948 .47718 m .66691 .4867 L .67398 .62791 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.65948 .47718 m .66691 .4867 L .67398 .62791 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.61902 .62152 m .61454 .48004 L .59603 .48622 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.60769 .63651 m .59958 .62746 L .59603 .48622 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.60979 .7123 m .60148 .70348 L .59603 .48622 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.59603 .48622 m .59958 .62746 L .61902 .62152 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.61454 .48004 m .59603 .48622 L .58819 .4767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61902 .62152 m .61454 .48004 L .59603 .48622 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.59603 .48622 m .59958 .62746 L .61902 .62152 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.56975 .4829 m .58819 .4767 L .59603 .48622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58819 .4767 m .59603 .48622 L .59958 .62746 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.58819 .4767 m .59603 .48622 L .59958 .62746 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.54478 .69724 m .54336 .47957 L .52512 .48575 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.53392 .71186 m .52514 .70304 L .52512 .48575 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.53392 .71186 m .52514 .70304 L .52512 .48575 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.52512 .48575 m .52514 .70304 L .54478 .69724 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.54336 .47957 m .52512 .48575 L .51687 .47622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54478 .69724 m .54336 .47957 L .52512 .48575 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.52512 .48575 m .52514 .70304 L .54478 .69724 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.49869 .48243 m .51687 .47622 L .52512 .48575 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51687 .47622 m .52512 .48575 L .52514 .70304 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.51687 .47622 m .52512 .48575 L .52514 .70304 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.46809 .6968 m .47215 .47909 L .45416 .48528 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.45799 .71142 m .44875 .70259 L .45416 .48528 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.45799 .71142 m .44875 .70259 L .45416 .48528 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.45416 .48528 m .44875 .70259 L .46809 .6968 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.47215 .47909 m .45416 .48528 L .4455 .47574 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46809 .6968 m .47215 .47909 L .45416 .48528 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.45416 .48528 m .44875 .70259 L .46809 .6968 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.42758 .48195 m .4455 .47574 L .45416 .48528 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4455 .47574 m .45416 .48528 L .44875 .70259 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.4455 .47574 m .45416 .48528 L .44875 .70259 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.39135 .69635 m .40089 .47861 L .38317 .4848 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.38202 .71098 m .37231 .70215 L .38317 .4848 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.3783 .79076 m .36831 .78219 L .38317 .4848 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.38317 .4848 m .37231 .70215 L .39135 .69635 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.40089 .47861 m .38317 .4848 L .3741 .47526 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39135 .69635 m .40089 .47861 L .38317 .4848 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.38317 .4848 m .37231 .70215 L .39135 .69635 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.35643 .48147 m .3741 .47526 L .38317 .4848 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3741 .47526 m .38317 .4848 L .37231 .70215 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.3741 .47526 m .38317 .4848 L .36831 .78219 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.73072 .47765 m .71176 .48385 L .70457 .4743 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74102 .61922 m .73072 .47765 L .71176 .48385 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.71176 .48385 m .72108 .62518 L .74102 .61922 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.68567 .48052 m .70457 .4743 L .71176 .48385 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70457 .4743 m .71176 .48385 L .72108 .62518 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.70457 .4743 m .71176 .48385 L .72108 .62518 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.66621 .61876 m .65948 .47718 L .64078 .48338 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.65441 .6338 m .64656 .62472 L .64078 .48338 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.65441 .6338 m .64656 .62472 L .64078 .48338 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.64078 .48338 m .64656 .62472 L .66621 .61876 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.65948 .47718 m .64078 .48338 L .63318 .47382 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66621 .61876 m .65948 .47718 L .64078 .48338 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.64078 .48338 m .64656 .62472 L .66621 .61876 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.61454 .48004 m .63318 .47382 L .64078 .48338 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63318 .47382 m .64078 .48338 L .64656 .62472 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.63318 .47382 m .64078 .48338 L .64656 .62472 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.59136 .6183 m .58819 .4767 L .56975 .4829 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.58029 .63335 m .57199 .62426 L .56975 .4829 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.5817 .70922 m .5732 .70037 L .56975 .4829 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.56975 .4829 m .57199 .62426 L .59136 .6183 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.58819 .4767 m .56975 .4829 L .56174 .47334 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59136 .6183 m .58819 .4767 L .56975 .4829 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.56975 .4829 m .57199 .62426 L .59136 .6183 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.54336 .47957 m .56174 .47334 L .56975 .4829 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56174 .47334 m .56975 .4829 L .57199 .62426 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.56174 .47334 m .56975 .4829 L .57199 .62426 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.51625 .69411 m .51687 .47622 L .49869 .48243 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.50565 .70878 m .49668 .69992 L .49869 .48243 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.50565 .70878 m .49668 .69992 L .49869 .48243 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.49869 .48243 m .49668 .69992 L .51625 .69411 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.51687 .47622 m .49869 .48243 L .49027 .47285 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51625 .69411 m .51687 .47622 L .49869 .48243 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.49869 .48243 m .49668 .69992 L .51625 .69411 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.47215 .47909 m .49027 .47285 L .49869 .48243 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49027 .47285 m .49869 .48243 L .49668 .69992 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.49027 .47285 m .49869 .48243 L .49668 .69992 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.43939 .69366 m .4455 .47574 L .42758 .48195 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.42955 .70834 m .42011 .69948 L .42758 .48195 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.42955 .70834 m .42011 .69948 L .42758 .48195 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.42758 .48195 m .42011 .69948 L .43939 .69366 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.4455 .47574 m .42758 .48195 L .41875 .47237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43939 .69366 m .4455 .47574 L .42758 .48195 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.42758 .48195 m .42011 .69948 L .43939 .69366 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.40089 .47861 m .41875 .47237 L .42758 .48195 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41875 .47237 m .42758 .48195 L .42011 .69948 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.41875 .47237 m .42758 .48195 L .42011 .69948 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.35819 .77352 m .3741 .47526 L .35643 .48147 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.34893 .78777 m .33873 .77917 L .35643 .48147 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.34893 .78777 m .33873 .77917 L .35643 .48147 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.35643 .48147 m .33873 .77917 L .35819 .77352 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.3741 .47526 m .35643 .48147 L .3472 .47189 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36247 .69321 m .3741 .47526 L .35643 .48147 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.35643 .48147 m .3435 .69903 L .36247 .69321 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.3296 .47813 m .3472 .47189 L .35643 .48147 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3472 .47189 m .35643 .48147 L .3435 .69903 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.3472 .47189 m .35643 .48147 L .33873 .77917 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.71358 .61599 m .70457 .4743 L .68567 .48052 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.70129 .63109 m .6937 .62198 L .68567 .48052 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.70129 .63109 m .6937 .62198 L .68567 .48052 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.68567 .48052 m .6937 .62198 L .71358 .61599 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.70457 .4743 m .68567 .48052 L .67832 .47093 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71358 .61599 m .70457 .4743 L .68567 .48052 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.68567 .48052 m .6937 .62198 L .71358 .61599 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.65948 .47718 m .67832 .47093 L .68567 .48052 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67832 .47093 m .68567 .48052 L .6937 .62198 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.67832 .47093 m .68567 .48052 L .6937 .62198 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.63861 .61553 m .63318 .47382 L .61454 .48004 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.62705 .63064 m .61902 .62152 L .61454 .48004 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.62705 .63064 m .61902 .62152 L .61454 .48004 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.61454 .48004 m .61902 .62152 L .63861 .61553 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.63318 .47382 m .61454 .48004 L .60677 .47044 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63861 .61553 m .63318 .47382 L .61454 .48004 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.61454 .48004 m .61902 .62152 L .63861 .61553 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.58819 .4767 m .60677 .47044 L .61454 .48004 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60677 .47044 m .61454 .48004 L .61902 .62152 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.60677 .47044 m .61454 .48004 L .61902 .62152 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.56359 .61507 m .56174 .47334 L .54336 .47957 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.55277 .63018 m .54429 .62106 L .54336 .47957 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.55348 .70614 m .54478 .69724 L .54336 .47957 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.54336 .47957 m .54429 .62106 L .56359 .61507 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.56174 .47334 m .54336 .47957 L .53519 .46996 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56359 .61507 m .56174 .47334 L .54336 .47957 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.54336 .47957 m .54429 .62106 L .56359 .61507 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.51687 .47622 m .53519 .46996 L .54336 .47957 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53519 .46996 m .54336 .47957 L .54429 .62106 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.53519 .46996 m .54336 .47957 L .54478 .69724 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.4876 .69096 m .49027 .47285 L .47215 .47909 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.47726 .70569 m .46809 .6968 L .47215 .47909 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.47726 .70569 m .46809 .6968 L .47215 .47909 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.47215 .47909 m .46809 .6968 L .4876 .69096 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.49027 .47285 m .47215 .47909 L .46356 .46948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4876 .69096 m .49027 .47285 L .47215 .47909 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.47215 .47909 m .46809 .6968 L .4876 .69096 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.4455 .47574 m .46356 .46948 L .47215 .47909 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46356 .46948 m .47215 .47909 L .46809 .6968 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.46356 .46948 m .47215 .47909 L .46809 .6968 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.41055 .6905 m .41875 .47237 L .40089 .47861 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.40099 .70525 m .39135 .69635 L .40089 .47861 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.40099 .70525 m .39135 .69635 L .40089 .47861 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.40089 .47861 m .39135 .69635 L .41055 .6905 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.41875 .47237 m .40089 .47861 L .39189 .46899 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41055 .6905 m .41875 .47237 L .40089 .47861 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.40089 .47861 m .39135 .69635 L .41055 .6905 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.3741 .47526 m .39189 .46899 L .40089 .47861 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39189 .46899 m .40089 .47861 L .39135 .69635 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.39189 .46899 m .40089 .47861 L .39135 .69635 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.32841 .77046 m .3472 .47189 L .3296 .47813 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.31944 .78477 m .30903 .77614 L .3296 .47813 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.3296 .47813 m .30903 .77614 L .32841 .77046 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.3472 .47189 m .3296 .47813 L .32018 .46851 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33347 .69005 m .3472 .47189 L .3296 .47813 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.3296 .47813 m .31457 .6959 L .33347 .69005 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.32018 .46851 m .3296 .47813 L .31457 .6959 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.74982 .47141 m .73072 .47765 L .72361 .46803 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76112 .61322 m .74982 .47141 L .73072 .47765 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.73072 .47765 m .74102 .61922 L .76112 .61322 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.70457 .4743 m .72361 .46803 L .73072 .47765 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72361 .46803 m .73072 .47765 L .74102 .61922 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.72361 .46803 m .73072 .47765 L .74102 .61922 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.68603 .61275 m .67832 .47093 L .65948 .47718 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.67398 .62791 m .66621 .61876 L .65948 .47718 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.67398 .62791 m .66621 .61876 L .65948 .47718 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.65948 .47718 m .66621 .61876 L .68603 .61275 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.67832 .47093 m .65948 .47718 L .65196 .46754 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68603 .61275 m .67832 .47093 L .65948 .47718 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.65948 .47718 m .66621 .61876 L .68603 .61275 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.63318 .47382 m .65196 .46754 L .65948 .47718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65196 .46754 m .65948 .47718 L .66621 .61876 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.65196 .46754 m .65948 .47718 L .66621 .61876 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.61089 .61229 m .60677 .47044 L .58819 .4767 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.59958 .62746 m .59136 .6183 L .58819 .4767 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.59958 .62746 m .59136 .6183 L .58819 .4767 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.58819 .4767 m .59136 .6183 L .61089 .61229 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.60677 .47044 m .58819 .4767 L .58026 .46706 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61089 .61229 m .60677 .47044 L .58819 .4767 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.58819 .4767 m .59136 .6183 L .61089 .61229 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.56174 .47334 m .58026 .46706 L .58819 .4767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58026 .46706 m .58819 .4767 L .59136 .6183 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.58026 .46706 m .58819 .4767 L .59136 .6183 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.53599 .68824 m .53519 .46996 L .51687 .47622 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.52514 .70304 m .51625 .69411 L .51687 .47622 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.52514 .70304 m .51625 .69411 L .51687 .47622 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.51687 .47622 m .51625 .69411 L .53599 .68824 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.53519 .46996 m .51687 .47622 L .50852 .46657 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53571 .61182 m .53519 .46996 L .51687 .47622 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.51687 .47622 m .51647 .61784 L .53571 .61182 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.49027 .47285 m .50852 .46657 L .51687 .47622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50852 .46657 m .51687 .47622 L .51647 .61784 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.50852 .46657 m .51687 .47622 L .51625 .69411 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.45882 .68779 m .46356 .46948 L .4455 .47574 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.44875 .70259 m .43939 .69366 L .4455 .47574 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.44875 .70259 m .43939 .69366 L .4455 .47574 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.4455 .47574 m .43939 .69366 L .45882 .68779 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.46356 .46948 m .4455 .47574 L .43675 .46609 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45882 .68779 m .46356 .46948 L .4455 .47574 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.4455 .47574 m .43939 .69366 L .45882 .68779 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.41875 .47237 m .43675 .46609 L .4455 .47574 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43675 .46609 m .4455 .47574 L .43939 .69366 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.43675 .46609 m .4455 .47574 L .43939 .69366 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.3816 .68734 m .39189 .46899 L .3741 .47526 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.37231 .70215 m .36247 .69321 L .3741 .47526 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.36831 .78219 m .35819 .77352 L .3741 .47526 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.3741 .47526 m .36247 .69321 L .3816 .68734 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.39189 .46899 m .3741 .47526 L .36493 .4656 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3816 .68734 m .39189 .46899 L .3741 .47526 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.3741 .47526 m .36247 .69321 L .3816 .68734 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.3472 .47189 m .36493 .4656 L .3741 .47526 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36493 .4656 m .3741 .47526 L .36247 .69321 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.36493 .4656 m .3741 .47526 L .36247 .69321 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.73362 .60996 m .72361 .46803 L .70457 .4743 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.72108 .62518 m .71358 .61599 L .70457 .4743 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.72108 .62518 m .71358 .61599 L .70457 .4743 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.70457 .4743 m .71358 .61599 L .73362 .60996 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.72361 .46803 m .70457 .4743 L .6973 .46463 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73362 .60996 m .72361 .46803 L .70457 .4743 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.70457 .4743 m .71358 .61599 L .73362 .60996 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.67832 .47093 m .6973 .46463 L .70457 .4743 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6973 .46463 m .70457 .4743 L .71358 .61599 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.6973 .46463 m .70457 .4743 L .71358 .61599 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.65836 .6095 m .65196 .46754 L .63318 .47382 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.64656 .62472 m .63861 .61553 L .63318 .47382 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.64656 .62472 m .63861 .61553 L .63318 .47382 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.63318 .47382 m .63861 .61553 L .65836 .6095 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.65196 .46754 m .63318 .47382 L .62549 .46414 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65836 .6095 m .65196 .46754 L .63318 .47382 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.63318 .47382 m .63861 .61553 L .65836 .6095 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.60677 .47044 m .62549 .46414 L .63318 .47382 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62549 .46414 m .63318 .47382 L .63861 .61553 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.62549 .46414 m .63318 .47382 L .63861 .61553 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.58305 .60903 m .58026 .46706 L .56174 .47334 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.57199 .62426 m .56359 .61507 L .56174 .47334 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.57199 .62426 m .56359 .61507 L .56174 .47334 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.56174 .47334 m .56359 .61507 L .58305 .60903 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.58026 .46706 m .56174 .47334 L .55364 .46366 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58305 .60903 m .58026 .46706 L .56174 .47334 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.56174 .47334 m .56359 .61507 L .58305 .60903 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.53519 .46996 m .55364 .46366 L .56174 .47334 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55364 .46366 m .56174 .47334 L .56359 .61507 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.55364 .46366 m .56174 .47334 L .56359 .61507 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.50726 .68507 m .50852 .46657 L .49027 .47285 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.49668 .69992 m .4876 .69096 L .49027 .47285 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.49668 .69992 m .4876 .69096 L .49027 .47285 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.49027 .47285 m .4876 .69096 L .50726 .68507 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.50852 .46657 m .49027 .47285 L .48175 .46317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5077 .60857 m .50852 .46657 L .49027 .47285 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.49027 .47285 m .48853 .61461 L .5077 .60857 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.46356 .46948 m .48175 .46317 L .49027 .47285 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48175 .46317 m .49027 .47285 L .48853 .61461 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.48175 .46317 m .49027 .47285 L .4876 .69096 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.42991 .68461 m .43675 .46609 L .41875 .47237 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.42011 .69948 m .41055 .6905 L .41875 .47237 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.42011 .69948 m .41055 .6905 L .41875 .47237 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.41875 .47237 m .41055 .6905 L .42991 .68461 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.43675 .46609 m .41875 .47237 L .40982 .46268 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42991 .68461 m .43675 .46609 L .41875 .47237 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.41875 .47237 m .41055 .6905 L .42991 .68461 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.39189 .46899 m .40982 .46268 L .41875 .47237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40982 .46268 m .41875 .47237 L .41055 .6905 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.40982 .46268 m .41875 .47237 L .41055 .6905 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.35252 .68416 m .36493 .4656 L .3472 .47189 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.3435 .69903 m .33347 .69005 L .3472 .47189 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.33873 .77917 m .32841 .77046 L .3472 .47189 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.3472 .47189 m .33347 .69005 L .35252 .68416 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.36493 .4656 m .3472 .47189 L .33785 .4622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35252 .68416 m .36493 .4656 L .3472 .47189 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.3472 .47189 m .33347 .69005 L .35252 .68416 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.32018 .46851 m .33785 .4622 L .3472 .47189 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33785 .4622 m .3472 .47189 L .33347 .69005 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.33785 .4622 m .3472 .47189 L .33347 .69005 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.76906 .46512 m .74982 .47141 L .74279 .46171 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77507 .53439 m .76906 .46512 L .74982 .47141 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.74982 .47141 m .75533 .54057 L .77507 .53439 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.72361 .46803 m .74279 .46171 L .74982 .47141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74279 .46171 m .74982 .47141 L .75533 .54057 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.74279 .46171 m .74982 .47141 L .76112 .61322 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.706 .6067 m .6973 .46463 L .67832 .47093 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.6937 .62198 m .68603 .61275 L .67832 .47093 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.6937 .62198 m .68603 .61275 L .67832 .47093 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.67832 .47093 m .68603 .61275 L .706 .6067 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.6973 .46463 m .67832 .47093 L .67088 .46122 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.706 .6067 m .6973 .46463 L .67832 .47093 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.67832 .47093 m .68603 .61275 L .706 .6067 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.65196 .46754 m .67088 .46122 L .67832 .47093 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67088 .46122 m .67832 .47093 L .68603 .61275 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.67088 .46122 m .67832 .47093 L .68603 .61275 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.63057 .60623 m .62549 .46414 L .60677 .47044 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.61902 .62152 m .61089 .61229 L .60677 .47044 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.61902 .62152 m .61089 .61229 L .60677 .47044 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.60677 .47044 m .61089 .61229 L .63057 .60623 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.62549 .46414 m .60677 .47044 L .59892 .46073 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63057 .60623 m .62549 .46414 L .60677 .47044 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.60677 .47044 m .61089 .61229 L .63057 .60623 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.58026 .46706 m .59892 .46073 L .60677 .47044 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59892 .46073 m .60677 .47044 L .61089 .61229 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.59892 .46073 m .60677 .47044 L .61089 .61229 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.5551 .60576 m .55364 .46366 L .53519 .46996 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.54429 .62106 m .53571 .61182 L .53519 .46996 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.54478 .69724 m .53599 .68824 L .53519 .46996 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.53519 .46996 m .53571 .61182 L .5551 .60576 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.55364 .46366 m .53519 .46996 L .52692 .46024 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5551 .60576 m .55364 .46366 L .53519 .46996 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.53519 .46996 m .53571 .61182 L .5551 .60576 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.50852 .46657 m .52692 .46024 L .53519 .46996 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52692 .46024 m .53519 .46996 L .53571 .61182 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.52692 .46024 m .53519 .46996 L .53571 .61182 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.4784 .68188 m .48175 .46317 L .46356 .46948 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.46809 .6968 m .45882 .68779 L .46356 .46948 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.46809 .6968 m .45882 .68779 L .46356 .46948 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.46356 .46948 m .45882 .68779 L .4784 .68188 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.48175 .46317 m .46356 .46948 L .45487 .45976 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47958 .60529 m .48175 .46317 L .46356 .46948 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.46356 .46948 m .46048 .61136 L .47958 .60529 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.43675 .46609 m .45487 .45976 L .46356 .46948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45487 .45976 m .46356 .46948 L .46048 .61136 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.45487 .45976 m .46356 .46948 L .45882 .68779 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.40088 .68142 m .40982 .46268 L .39189 .46899 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.39135 .69635 m .3816 .68734 L .39189 .46899 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.39135 .69635 m .3816 .68734 L .39189 .46899 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.39189 .46899 m .3816 .68734 L .40088 .68142 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.40982 .46268 m .39189 .46899 L .38279 .45927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40088 .68142 m .40982 .46268 L .39189 .46899 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.39189 .46899 m .3816 .68734 L .40088 .68142 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.36493 .4656 m .38279 .45927 L .39189 .46899 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38279 .45927 m .39189 .46899 L .3816 .68734 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.38279 .45927 m .39189 .46899 L .3816 .68734 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.32331 .68096 m .33785 .4622 L .32018 .46851 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.31457 .6959 m .30433 .68688 L .32018 .46851 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.32018 .46851 m .30433 .68688 L .32331 .68096 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.33785 .4622 m .32018 .46851 L .31066 .45878 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32331 .68096 m .33785 .4622 L .32018 .46851 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.32018 .46851 m .30433 .68688 L .32331 .68096 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.31066 .45878 m .32018 .46851 L .30433 .68688 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.75381 .60389 m .74279 .46171 L .72361 .46803 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.74102 .61922 m .73362 .60996 L .72361 .46803 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.74102 .61922 m .73362 .60996 L .72361 .46803 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.72361 .46803 m .73362 .60996 L .75381 .60389 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.74279 .46171 m .72361 .46803 L .71642 .45829 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74817 .53105 m .74279 .46171 L .72361 .46803 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.72361 .46803 m .72849 .53725 L .74817 .53105 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.6973 .46463 m .71642 .45829 L .72361 .46803 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71642 .45829 m .72361 .46803 L .72849 .53725 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.71642 .45829 m .72361 .46803 L .73362 .60996 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.67826 .60342 m .67088 .46122 L .65196 .46754 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.66621 .61876 m .65836 .6095 L .65196 .46754 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.66621 .61876 m .65836 .6095 L .65196 .46754 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.65196 .46754 m .65836 .6095 L .67826 .60342 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.67088 .46122 m .65196 .46754 L .64435 .4578 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67826 .60342 m .67088 .46122 L .65196 .46754 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.65196 .46754 m .65836 .6095 L .67826 .60342 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.62549 .46414 m .64435 .4578 L .65196 .46754 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64435 .4578 m .65196 .46754 L .65836 .6095 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.64435 .4578 m .65196 .46754 L .65836 .6095 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.60266 .60295 m .59892 .46073 L .58026 .46706 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.59136 .6183 m .58305 .60903 L .58026 .46706 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.59136 .6183 m .58305 .60903 L .58026 .46706 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.58026 .46706 m .58305 .60903 L .60266 .60295 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.59892 .46073 m .58026 .46706 L .57224 .45731 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60266 .60295 m .59892 .46073 L .58026 .46706 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.58026 .46706 m .58305 .60903 L .60266 .60295 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.55364 .46366 m .57224 .45731 L .58026 .46706 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57224 .45731 m .58026 .46706 L .58305 .60903 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.57224 .45731 m .58026 .46706 L .58305 .60903 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.52702 .60248 m .52692 .46024 L .50852 .46657 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.51647 .61784 m .5077 .60857 L .50852 .46657 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.51625 .69411 m .50726 .68507 L .50852 .46657 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.50852 .46657 m .5077 .60857 L .52702 .60248 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.52692 .46024 m .50852 .46657 L .50008 .45682 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52702 .60248 m .52692 .46024 L .50852 .46657 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.50852 .46657 m .5077 .60857 L .52702 .60248 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.48175 .46317 m .50008 .45682 L .50852 .46657 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50008 .45682 m .50852 .46657 L .5077 .60857 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.50008 .45682 m .50852 .46657 L .5077 .60857 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.44942 .67867 m .45487 .45976 L .43675 .46609 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.43939 .69366 m .42991 .68461 L .43675 .46609 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.43939 .69366 m .42991 .68461 L .43675 .46609 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.43675 .46609 m .42991 .68461 L .44942 .67867 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.45487 .45976 m .43675 .46609 L .42789 .45633 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45133 .60201 m .45487 .45976 L .43675 .46609 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.43675 .46609 m .4323 .6081 L .45133 .60201 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.40982 .46268 m .42789 .45633 L .43675 .46609 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42789 .45633 m .43675 .46609 L .4323 .6081 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.42789 .45633 m .43675 .46609 L .42991 .68461 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.37172 .67821 m .38279 .45927 L .36493 .4656 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.36247 .69321 m .35252 .68416 L .36493 .4656 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.36247 .69321 m .35252 .68416 L .36493 .4656 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.36493 .4656 m .35252 .68416 L .37172 .67821 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.38279 .45927 m .36493 .4656 L .35565 .45584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37172 .67821 m .38279 .45927 L .36493 .4656 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.36493 .4656 m .35252 .68416 L .37172 .67821 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.33785 .4622 m .35565 .45584 L .36493 .4656 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35565 .45584 m .36493 .4656 L .35252 .68416 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.35565 .45584 m .36493 .4656 L .35252 .68416 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.78845 .45878 m .76906 .46512 L .76212 .45534 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79496 .52816 m .78845 .45878 L .76906 .46512 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.76906 .46512 m .77507 .53439 L .79496 .52816 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.74279 .46171 m .76212 .45534 L .76906 .46512 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76212 .45534 m .76906 .46512 L .77507 .53439 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.76212 .45534 m .76906 .46512 L .77507 .53439 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.72612 .60059 m .71642 .45829 L .6973 .46463 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.71358 .61599 m .706 .6067 L .6973 .46463 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.71358 .61599 m .706 .6067 L .6973 .46463 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.6973 .46463 m .706 .6067 L .72612 .60059 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.71642 .45829 m .6973 .46463 L .68994 .45485 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72115 .52768 m .71642 .45829 L .6973 .46463 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.6973 .46463 m .70154 .53392 L .72115 .52768 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.67088 .46122 m .68994 .45485 L .6973 .46463 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68994 .45485 m .6973 .46463 L .70154 .53392 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.68994 .45485 m .6973 .46463 L .706 .6067 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.6504 .60012 m .64435 .4578 L .62549 .46414 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.63861 .61553 m .63057 .60623 L .62549 .46414 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.63861 .61553 m .63057 .60623 L .62549 .46414 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.62549 .46414 m .63057 .60623 L .6504 .60012 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.64435 .4578 m .62549 .46414 L .61771 .45436 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6504 .60012 m .64435 .4578 L .62549 .46414 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.62549 .46414 m .63057 .60623 L .6504 .60012 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.59892 .46073 m .61771 .45436 L .62549 .46414 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61771 .45436 m .62549 .46414 L .63057 .60623 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.61771 .45436 m .62549 .46414 L .63057 .60623 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.57464 .59965 m .57224 .45731 L .55364 .46366 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.56359 .61507 m .5551 .60576 L .55364 .46366 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.56359 .61507 m .5551 .60576 L .55364 .46366 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.55364 .46366 m .5551 .60576 L .57464 .59965 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.57224 .45731 m .55364 .46366 L .54545 .45387 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57464 .59965 m .57224 .45731 L .55364 .46366 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.55364 .46366 m .5551 .60576 L .57464 .59965 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.52692 .46024 m .54545 .45387 L .55364 .46366 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54545 .45387 m .55364 .46366 L .5551 .60576 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.54545 .45387 m .55364 .46366 L .5551 .60576 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.49883 .59918 m .50008 .45682 L .48175 .46317 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.48853 .61461 m .47958 .60529 L .48175 .46317 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.4876 .69096 m .4784 .68188 L .48175 .46317 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.48175 .46317 m .47958 .60529 L .49883 .59918 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.50008 .45682 m .48175 .46317 L .47314 .45338 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49883 .59918 m .50008 .45682 L .48175 .46317 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.48175 .46317 m .47958 .60529 L .49883 .59918 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.45487 .45976 m .47314 .45338 L .48175 .46317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47314 .45338 m .48175 .46317 L .47958 .60529 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.47314 .45338 m .48175 .46317 L .47958 .60529 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.42032 .67545 m .42789 .45633 L .40982 .46268 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.41055 .6905 m .40088 .68142 L .40982 .46268 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.41055 .6905 m .40088 .68142 L .40982 .46268 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.40982 .46268 m .40088 .68142 L .42032 .67545 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.42789 .45633 m .40982 .46268 L .40079 .45288 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42032 .67545 m .42789 .45633 L .40982 .46268 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.40982 .46268 m .40088 .68142 L .42032 .67545 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.38279 .45927 m .40079 .45288 L .40982 .46268 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40079 .45288 m .40982 .46268 L .40088 .68142 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.40079 .45288 m .40982 .46268 L .40088 .68142 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.34244 .67499 m .35565 .45584 L .33785 .4622 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.33347 .69005 m .32331 .68096 L .33785 .4622 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.33347 .69005 m .32331 .68096 L .33785 .4622 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.33785 .4622 m .32331 .68096 L .34244 .67499 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.35565 .45584 m .33785 .4622 L .3284 .45239 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34244 .67499 m .35565 .45584 L .33785 .4622 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.33785 .4622 m .32331 .68096 L .34244 .67499 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.31066 .45878 m .3284 .45239 L .33785 .4622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3284 .45239 m .33785 .4622 L .32331 .68096 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.3284 .45239 m .33785 .4622 L .32331 .68096 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.76799 .52479 m .76212 .45534 L .74279 .46171 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.75533 .54057 m .74817 .53105 L .74279 .46171 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.76112 .61322 m .75381 .60389 L .74279 .46171 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.74279 .46171 m .74817 .53105 L .76799 .52479 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.76212 .45534 m .74279 .46171 L .73569 .4519 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76799 .52479 m .76212 .45534 L .74279 .46171 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.74279 .46171 m .74817 .53105 L .76799 .52479 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.71642 .45829 m .73569 .4519 L .74279 .46171 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73569 .4519 m .74279 .46171 L .74817 .53105 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.73569 .4519 m .74279 .46171 L .74817 .53105 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.69832 .59729 m .68994 .45485 L .67088 .46122 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.68603 .61275 m .67826 .60342 L .67088 .46122 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.68603 .61275 m .67826 .60342 L .67088 .46122 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.67088 .46122 m .67826 .60342 L .69832 .59729 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.68994 .45485 m .67088 .46122 L .66335 .4514 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69403 .52431 m .68994 .45485 L .67088 .46122 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.67088 .46122 m .67448 .53057 L .69403 .52431 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.64435 .4578 m .66335 .4514 L .67088 .46122 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66335 .4514 m .67088 .46122 L .67448 .53057 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.66335 .4514 m .67088 .46122 L .67826 .60342 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.62243 .59681 m .61771 .45436 L .59892 .46073 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.61089 .61229 m .60266 .60295 L .59892 .46073 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.61089 .61229 m .60266 .60295 L .59892 .46073 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.59892 .46073 m .60266 .60295 L .62243 .59681 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.61771 .45436 m .59892 .46073 L .59097 .45091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62243 .59681 m .61771 .45436 L .59892 .46073 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.59892 .46073 m .60266 .60295 L .62243 .59681 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.57224 .45731 m .59097 .45091 L .59892 .46073 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59097 .45091 m .59892 .46073 L .60266 .60295 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.59097 .45091 m .59892 .46073 L .60266 .60295 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.54649 .59634 m .54545 .45387 L .52692 .46024 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.53571 .61182 m .52702 .60248 L .52692 .46024 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.53571 .61182 m .52702 .60248 L .52692 .46024 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.52692 .46024 m .52702 .60248 L .54649 .59634 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.54545 .45387 m .52692 .46024 L .51855 .45041 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54649 .59634 m .54545 .45387 L .52692 .46024 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.52692 .46024 m .52702 .60248 L .54649 .59634 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.50008 .45682 m .51855 .45041 L .52692 .46024 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51855 .45041 m .52692 .46024 L .52702 .60248 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.51855 .45041 m .52692 .46024 L .52702 .60248 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.47051 .59587 m .47314 .45338 L .45487 .45976 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.46048 .61136 m .45133 .60201 L .45487 .45976 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.45882 .68779 m .44942 .67867 L .45487 .45976 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.45487 .45976 m .45133 .60201 L .47051 .59587 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.47314 .45338 m .45487 .45976 L .44609 .44992 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47051 .59587 m .47314 .45338 L .45487 .45976 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.45487 .45976 m .45133 .60201 L .47051 .59587 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.42789 .45633 m .44609 .44992 L .45487 .45976 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44609 .44992 m .45487 .45976 L .45133 .60201 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.44609 .44992 m .45487 .45976 L .45133 .60201 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.39108 .67222 m .40079 .45288 L .38279 .45927 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.3816 .68734 m .37172 .67821 L .38279 .45927 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.3816 .68734 m .37172 .67821 L .38279 .45927 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.38279 .45927 m .37172 .67821 L .39108 .67222 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.40079 .45288 m .38279 .45927 L .37358 .44943 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39108 .67222 m .40079 .45288 L .38279 .45927 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.38279 .45927 m .37172 .67821 L .39108 .67222 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.35565 .45584 m .37358 .44943 L .38279 .45927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37358 .44943 m .38279 .45927 L .37172 .67821 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.37358 .44943 m .38279 .45927 L .37172 .67821 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.80798 .45239 m .78845 .45878 L .7816 .44893 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.815 .52189 m .80798 .45239 L .78845 .45878 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.78845 .45878 m .79496 .52816 L .815 .52189 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.76212 .45534 m .7816 .44893 L .78845 .45878 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7816 .44893 m .78845 .45878 L .79496 .52816 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.7816 .44893 m .78845 .45878 L .79496 .52816 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.31302 .67176 m .3284 .45239 L .31066 .45878 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.30433 .68688 m .29397 .67775 L .31066 .45878 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.31066 .45878 m .29397 .67775 L .31302 .67176 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.3284 .45239 m .31066 .45878 L .30103 .44893 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31302 .67176 m .3284 .45239 L .31066 .45878 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.31066 .45878 m .29397 .67775 L .31302 .67176 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.30103 .44893 m .31066 .45878 L .29397 .67775 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.74092 .5214 m .73569 .4519 L .71642 .45829 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.72849 .53725 m .72115 .52768 L .71642 .45829 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.73362 .60996 m .72612 .60059 L .71642 .45829 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.71642 .45829 m .72115 .52768 L .74092 .5214 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.73569 .4519 m .71642 .45829 L .70915 .44844 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74092 .5214 m .73569 .4519 L .71642 .45829 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.71642 .45829 m .72115 .52768 L .74092 .5214 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.68994 .45485 m .70915 .44844 L .71642 .45829 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70915 .44844 m .71642 .45829 L .72115 .52768 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.70915 .44844 m .71642 .45829 L .72115 .52768 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.6704 .59397 m .66335 .4514 L .64435 .4578 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.65836 .6095 m .6504 .60012 L .64435 .4578 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.65836 .6095 m .6504 .60012 L .64435 .4578 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.64435 .4578 m .6504 .60012 L .6704 .59397 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.66335 .4514 m .64435 .4578 L .63665 .44794 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66679 .52092 m .66335 .4514 L .64435 .4578 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.64435 .4578 m .6473 .5272 L .66679 .52092 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.61771 .45436 m .63665 .44794 L .64435 .4578 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63665 .44794 m .64435 .4578 L .6473 .5272 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.63665 .44794 m .64435 .4578 L .6504 .60012 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.59434 .59349 m .59097 .45091 L .57224 .45731 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.58305 .60903 m .57464 .59965 L .57224 .45731 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.58305 .60903 m .57464 .59965 L .57224 .45731 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.57224 .45731 m .57464 .59965 L .59434 .59349 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.59097 .45091 m .57224 .45731 L .56412 .44744 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59434 .59349 m .59097 .45091 L .57224 .45731 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.57224 .45731 m .57464 .59965 L .59434 .59349 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.54545 .45387 m .56412 .44744 L .57224 .45731 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56412 .44744 m .57224 .45731 L .57464 .59965 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.56412 .44744 m .57224 .45731 L .57464 .59965 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.51823 .59302 m .51855 .45041 L .50008 .45682 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.5077 .60857 m .49883 .59918 L .50008 .45682 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.5077 .60857 m .49883 .59918 L .50008 .45682 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.50008 .45682 m .49883 .59918 L .51823 .59302 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.51855 .45041 m .50008 .45682 L .49154 .44695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51823 .59302 m .51855 .45041 L .50008 .45682 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.50008 .45682 m .49883 .59918 L .51823 .59302 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.47314 .45338 m .49154 .44695 L .50008 .45682 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49154 .44695 m .50008 .45682 L .49883 .59918 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.49154 .44695 m .50008 .45682 L .49883 .59918 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.44208 .59254 m .44609 .44992 L .42789 .45633 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.4323 .6081 m .42297 .59871 L .42789 .45633 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.42991 .68461 m .42032 .67545 L .42789 .45633 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.42789 .45633 m .42297 .59871 L .44208 .59254 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.44609 .44992 m .42789 .45633 L .41892 .44645 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44208 .59254 m .44609 .44992 L .42789 .45633 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.42789 .45633 m .42297 .59871 L .44208 .59254 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.40079 .45288 m .41892 .44645 L .42789 .45633 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41892 .44645 m .42789 .45633 L .42297 .59871 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.41892 .44645 m .42789 .45633 L .42032 .67545 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.36172 .66897 m .37358 .44943 L .35565 .45584 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.35252 .68416 m .34244 .67499 L .35565 .45584 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.35252 .68416 m .34244 .67499 L .35565 .45584 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.35565 .45584 m .34244 .67499 L .36172 .66897 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.37358 .44943 m .35565 .45584 L .34626 .44595 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36172 .66897 m .37358 .44943 L .35565 .45584 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.35565 .45584 m .34244 .67499 L .36172 .66897 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.3284 .45239 m .34626 .44595 L .35565 .45584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34626 .44595 m .35565 .45584 L .34244 .67499 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.34626 .44595 m .35565 .45584 L .34244 .67499 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.78798 .51849 m .7816 .44893 L .76212 .45534 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.77507 .53439 m .76799 .52479 L .76212 .45534 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.77507 .53439 m .76799 .52479 L .76212 .45534 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.76212 .45534 m .76799 .52479 L .78798 .51849 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.7816 .44893 m .76212 .45534 L .7551 .44546 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78798 .51849 m .7816 .44893 L .76212 .45534 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.76212 .45534 m .76799 .52479 L .78798 .51849 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.73569 .4519 m .7551 .44546 L .76212 .45534 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7551 .44546 m .76212 .45534 L .76799 .52479 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.7551 .44546 m .76212 .45534 L .76799 .52479 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.71373 .518 m .70915 .44844 L .68994 .45485 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.70154 .53392 m .69403 .52431 L .68994 .45485 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.706 .6067 m .69832 .59729 L .68994 .45485 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.68994 .45485 m .69403 .52431 L .71373 .518 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.70915 .44844 m .68994 .45485 L .6825 .44496 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71373 .518 m .70915 .44844 L .68994 .45485 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.68994 .45485 m .69403 .52431 L .71373 .518 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.66335 .4514 m .6825 .44496 L .68994 .45485 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6825 .44496 m .68994 .45485 L .69403 .52431 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.6825 .44496 m .68994 .45485 L .69403 .52431 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.64235 .59063 m .63665 .44794 L .61771 .45436 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.63057 .60623 m .62243 .59681 L .61771 .45436 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.63057 .60623 m .62243 .59681 L .61771 .45436 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.61771 .45436 m .62243 .59681 L .64235 .59063 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.63665 .44794 m .61771 .45436 L .60985 .44446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63943 .51751 m .63665 .44794 L .61771 .45436 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.61771 .45436 m .62001 .52382 L .63943 .51751 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.59097 .45091 m .60985 .44446 L .61771 .45436 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60985 .44446 m .61771 .45436 L .62001 .52382 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.60985 .44446 m .61771 .45436 L .62243 .59681 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.56612 .59016 m .56412 .44744 L .54545 .45387 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.5551 .60576 m .54649 .59634 L .54545 .45387 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.5551 .60576 m .54649 .59634 L .54545 .45387 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.54545 .45387 m .54649 .59634 L .56612 .59016 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.56412 .44744 m .54545 .45387 L .53716 .44396 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56612 .59016 m .56412 .44744 L .54545 .45387 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.54545 .45387 m .54649 .59634 L .56612 .59016 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.51855 .45041 m .53716 .44396 L .54545 .45387 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53716 .44396 m .54545 .45387 L .54649 .59634 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.53716 .44396 m .54545 .45387 L .54649 .59634 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.48984 .58968 m .49154 .44695 L .47314 .45338 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.47958 .60529 m .47051 .59587 L .47314 .45338 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.47958 .60529 m .47051 .59587 L .47314 .45338 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.47314 .45338 m .47051 .59587 L .48984 .58968 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.49154 .44695 m .47314 .45338 L .46442 .44347 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48984 .58968 m .49154 .44695 L .47314 .45338 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.47314 .45338 m .47051 .59587 L .48984 .58968 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.44609 .44992 m .46442 .44347 L .47314 .45338 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46442 .44347 m .47314 .45338 L .47051 .59587 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.46442 .44347 m .47314 .45338 L .47051 .59587 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.4106 .66618 m .41892 .44645 L .40079 .45288 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.40088 .68142 m .39108 .67222 L .40079 .45288 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.40088 .68142 m .39108 .67222 L .40079 .45288 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.40079 .45288 m .39108 .67222 L .4106 .66618 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.41892 .44645 m .40079 .45288 L .39165 .44297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41352 .5892 m .41892 .44645 L .40079 .45288 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.40079 .45288 m .39448 .59539 L .41352 .5892 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.37358 .44943 m .39165 .44297 L .40079 .45288 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39165 .44297 m .40079 .45288 L .39448 .59539 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.39165 .44297 m .40079 .45288 L .39108 .67222 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.82767 .44595 m .80798 .45239 L .80122 .44247 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.8352 .51556 m .82767 .44595 L .80798 .45239 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.80798 .45239 m .815 .52189 L .8352 .51556 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.7816 .44893 m .80122 .44247 L .80798 .45239 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80122 .44247 m .80798 .45239 L .815 .52189 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.80122 .44247 m .80798 .45239 L .815 .52189 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.33223 .66571 m .34626 .44595 L .3284 .45239 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.32331 .68096 m .31302 .67176 L .3284 .45239 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.32331 .68096 m .31302 .67176 L .3284 .45239 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.3284 .45239 m .31302 .67176 L .33223 .66571 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.34626 .44595 m .3284 .45239 L .31883 .44247 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33223 .66571 m .34626 .44595 L .3284 .45239 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.3284 .45239 m .31302 .67176 L .33223 .66571 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.30103 .44893 m .31883 .44247 L .3284 .45239 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31883 .44247 m .3284 .45239 L .31302 .67176 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.31883 .44247 m .3284 .45239 L .31302 .67176 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.76083 .51507 m .7551 .44546 L .73569 .4519 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.74817 .53105 m .74092 .5214 L .73569 .4519 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.74817 .53105 m .74092 .5214 L .73569 .4519 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.73569 .4519 m .74092 .5214 L .76083 .51507 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.7551 .44546 m .73569 .4519 L .7285 .44197 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76083 .51507 m .7551 .44546 L .73569 .4519 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.73569 .4519 m .74092 .5214 L .76083 .51507 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.70915 .44844 m .7285 .44197 L .73569 .4519 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7285 .44197 m .73569 .4519 L .74092 .5214 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.7285 .44197 m .73569 .4519 L .74092 .5214 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.68642 .51458 m .6825 .44496 L .66335 .4514 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.67448 .53057 m .66679 .52092 L .66335 .4514 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.67826 .60342 m .6704 .59397 L .66335 .4514 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.66335 .4514 m .66679 .52092 L .68642 .51458 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.6825 .44496 m .66335 .4514 L .65574 .44147 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68642 .51458 m .6825 .44496 L .66335 .4514 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.66335 .4514 m .66679 .52092 L .68642 .51458 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.63665 .44794 m .65574 .44147 L .66335 .4514 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65574 .44147 m .66335 .4514 L .66679 .52092 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.65574 .44147 m .66335 .4514 L .66679 .52092 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.61419 .58728 m .60985 .44446 L .59097 .45091 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.60266 .60295 m .59434 .59349 L .59097 .45091 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.60266 .60295 m .59434 .59349 L .59097 .45091 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.59097 .45091 m .59434 .59349 L .61419 .58728 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.60985 .44446 m .59097 .45091 L .58293 .44097 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61197 .51409 m .60985 .44446 L .59097 .45091 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.59097 .45091 m .59261 .52043 L .61197 .51409 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.56412 .44744 m .58293 .44097 L .59097 .45091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58293 .44097 m .59097 .45091 L .59261 .52043 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.58293 .44097 m .59097 .45091 L .59434 .59349 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.53779 .5868 m .53716 .44396 L .51855 .45041 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.52702 .60248 m .51823 .59302 L .51855 .45041 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.52702 .60248 m .51823 .59302 L .51855 .45041 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.51855 .45041 m .51823 .59302 L .53779 .5868 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.53716 .44396 m .51855 .45041 L .51009 .44047 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53779 .5868 m .53716 .44396 L .51855 .45041 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.51855 .45041 m .51823 .59302 L .53779 .5868 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.49154 .44695 m .51009 .44047 L .51855 .45041 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51009 .44047 m .51855 .45041 L .51823 .59302 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.51009 .44047 m .51855 .45041 L .51823 .59302 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.46133 .58632 m .46442 .44347 L .44609 .44992 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.45133 .60201 m .44208 .59254 L .44609 .44992 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.45133 .60201 m .44208 .59254 L .44609 .44992 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.44609 .44992 m .44208 .59254 L .46133 .58632 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.46442 .44347 m .44609 .44992 L .4372 .43997 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46133 .58632 m .46442 .44347 L .44609 .44992 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.44609 .44992 m .44208 .59254 L .46133 .58632 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.41892 .44645 m .4372 .43997 L .44609 .44992 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4372 .43997 m .44609 .44992 L .44208 .59254 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4372 .43997 m .44609 .44992 L .44208 .59254 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.38116 .66291 m .39165 .44297 L .37358 .44943 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.37172 .67821 m .36172 .66897 L .37358 .44943 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.37172 .67821 m .36172 .66897 L .37358 .44943 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.37358 .44943 m .36172 .66897 L .38116 .66291 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.39165 .44297 m .37358 .44943 L .36426 .43947 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38484 .58584 m .39165 .44297 L .37358 .44943 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.37358 .44943 m .36588 .59206 L .38484 .58584 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.34626 .44595 m .36426 .43947 L .37358 .44943 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36426 .43947 m .37358 .44943 L .36588 .59206 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.36426 .43947 m .37358 .44943 L .36172 .66897 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.80811 .51213 m .80122 .44247 L .7816 .44893 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.79496 .52816 m .78798 .51849 L .7816 .44893 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.79496 .52816 m .78798 .51849 L .7816 .44893 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.7816 .44893 m .78798 .51849 L .80811 .51213 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.80122 .44247 m .7816 .44893 L .77467 .43897 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80811 .51213 m .80122 .44247 L .7816 .44893 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.7816 .44893 m .78798 .51849 L .80811 .51213 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.7551 .44546 m .77467 .43897 L .7816 .44893 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77467 .43897 m .7816 .44893 L .78798 .51849 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.77467 .43897 m .7816 .44893 L .78798 .51849 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.30261 .66244 m .31883 .44247 L .30103 .44893 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.29397 .67775 m .28348 .66851 L .30103 .44893 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.30103 .44893 m .28348 .66851 L .30261 .66244 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.31883 .44247 m .30103 .44893 L .29129 .43897 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30261 .66244 m .31883 .44247 L .30103 .44893 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.30103 .44893 m .28348 .66851 L .30261 .66244 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.29129 .43897 m .30103 .44893 L .28348 .66851 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.73358 .51164 m .7285 .44197 L .70915 .44844 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.72115 .52768 m .71373 .518 L .70915 .44844 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.72115 .52768 m .71373 .518 L .70915 .44844 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.70915 .44844 m .71373 .518 L .73358 .51164 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.7285 .44197 m .70915 .44844 L .70179 .43847 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73358 .51164 m .7285 .44197 L .70915 .44844 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.70915 .44844 m .71373 .518 L .73358 .51164 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.6825 .44496 m .70179 .43847 L .70915 .44844 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70179 .43847 m .70915 .44844 L .71373 .518 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.70179 .43847 m .70915 .44844 L .71373 .518 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.659 .51115 m .65574 .44147 L .63665 .44794 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.6473 .5272 m .63943 .51751 L .63665 .44794 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.6504 .60012 m .64235 .59063 L .63665 .44794 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.63665 .44794 m .63943 .51751 L .659 .51115 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.65574 .44147 m .63665 .44794 L .62887 .43797 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.659 .51115 m .65574 .44147 L .63665 .44794 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.63665 .44794 m .63943 .51751 L .659 .51115 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.60985 .44446 m .62887 .43797 L .63665 .44794 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62887 .43797 m .63665 .44794 L .63943 .51751 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.62887 .43797 m .63665 .44794 L .63943 .51751 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.58591 .58392 m .58293 .44097 L .56412 .44744 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.57464 .59965 m .56612 .59016 L .56412 .44744 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.57464 .59965 m .56612 .59016 L .56412 .44744 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.56412 .44744 m .56612 .59016 L .58591 .58392 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.58293 .44097 m .56412 .44744 L .55591 .43746 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58438 .51066 m .58293 .44097 L .56412 .44744 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.56412 .44744 m .5651 .51703 L .58438 .51066 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.53716 .44396 m .55591 .43746 L .56412 .44744 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55591 .43746 m .56412 .44744 L .5651 .51703 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.55591 .43746 m .56412 .44744 L .56612 .59016 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.50933 .58344 m .51009 .44047 L .49154 .44695 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.49883 .59918 m .48984 .58968 L .49154 .44695 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.49883 .59918 m .48984 .58968 L .49154 .44695 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.49154 .44695 m .48984 .58968 L .50933 .58344 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.51009 .44047 m .49154 .44695 L .4829 .43696 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50933 .58344 m .51009 .44047 L .49154 .44695 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.49154 .44695 m .48984 .58968 L .50933 .58344 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.46442 .44347 m .4829 .43696 L .49154 .44695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4829 .43696 m .49154 .44695 L .48984 .58968 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.4829 .43696 m .49154 .44695 L .48984 .58968 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.4327 .58296 m .4372 .43997 L .41892 .44645 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.42297 .59871 m .41352 .5892 L .41892 .44645 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.42032 .67545 m .4106 .66618 L .41892 .44645 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.41892 .44645 m .41352 .5892 L .4327 .58296 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.4372 .43997 m .41892 .44645 L .40986 .43646 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4327 .58296 m .4372 .43997 L .41892 .44645 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.41892 .44645 m .41352 .5892 L .4327 .58296 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.39165 .44297 m .40986 .43646 L .41892 .44645 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40986 .43646 m .41892 .44645 L .41352 .5892 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.40986 .43646 m .41892 .44645 L .41352 .5892 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.8475 .43947 m .82767 .44595 L .82099 .43596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.85556 .50919 m .8475 .43947 L .82767 .44595 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.82767 .44595 m .8352 .51556 L .85556 .50919 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.80122 .44247 m .82099 .43596 L .82767 .44595 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82099 .43596 m .82767 .44595 L .8352 .51556 L p
.976 .674 .353 r
F P
s
P
p
.002 w
.82099 .43596 m .82767 .44595 L .8352 .51556 L p
.976 .674 .353 r
F P
s
P
p
.002 w
.35159 .65962 m .36426 .43947 L .34626 .44595 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.34244 .67499 m .33223 .66571 L .34626 .44595 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.34244 .67499 m .33223 .66571 L .34626 .44595 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.34626 .44595 m .33223 .66571 L .35159 .65962 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.36426 .43947 m .34626 .44595 L .33676 .43596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35603 .58247 m .36426 .43947 L .34626 .44595 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.34626 .44595 m .33715 .58872 L .35603 .58247 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.31883 .44247 m .33676 .43596 L .34626 .44595 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33676 .43596 m .34626 .44595 L .33715 .58872 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.33676 .43596 m .34626 .44595 L .33223 .66571 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.78091 .50869 m .77467 .43897 L .7551 .44546 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.76799 .52479 m .76083 .51507 L .7551 .44546 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.76799 .52479 m .76083 .51507 L .7551 .44546 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.7551 .44546 m .76083 .51507 L .78091 .50869 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.77467 .43897 m .7551 .44546 L .748 .43545 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78091 .50869 m .77467 .43897 L .7551 .44546 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.7551 .44546 m .76083 .51507 L .78091 .50869 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.7285 .44197 m .748 .43545 L .7551 .44546 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.748 .43545 m .7551 .44546 L .76083 .51507 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.748 .43545 m .7551 .44546 L .76083 .51507 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.70621 .5082 m .70179 .43847 L .6825 .44496 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.69403 .52431 m .68642 .51458 L .6825 .44496 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.69403 .52431 m .68642 .51458 L .6825 .44496 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.6825 .44496 m .68642 .51458 L .70621 .5082 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.70179 .43847 m .6825 .44496 L .67497 .43495 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70621 .5082 m .70179 .43847 L .6825 .44496 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.6825 .44496 m .68642 .51458 L .70621 .5082 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.65574 .44147 m .67497 .43495 L .6825 .44496 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67497 .43495 m .6825 .44496 L .68642 .51458 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.67497 .43495 m .6825 .44496 L .68642 .51458 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.63147 .50771 m .62887 .43797 L .60985 .44446 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.62001 .52382 m .61197 .51409 L .60985 .44446 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.62243 .59681 m .61419 .58728 L .60985 .44446 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.60985 .44446 m .61197 .51409 L .63147 .50771 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.62887 .43797 m .60985 .44446 L .60189 .43445 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63147 .50771 m .62887 .43797 L .60985 .44446 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.60985 .44446 m .61197 .51409 L .63147 .50771 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.58293 .44097 m .60189 .43445 L .60985 .44446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60189 .43445 m .60985 .44446 L .61197 .51409 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.60189 .43445 m .60985 .44446 L .61197 .51409 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.5575 .58054 m .55591 .43746 L .53716 .44396 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.54649 .59634 m .53779 .5868 L .53716 .44396 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.54649 .59634 m .53779 .5868 L .53716 .44396 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.53716 .44396 m .53779 .5868 L .5575 .58054 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.55591 .43746 m .53716 .44396 L .52877 .43394 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55668 .50721 m .55591 .43746 L .53716 .44396 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.53716 .44396 m .53746 .5136 L .55668 .50721 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.51009 .44047 m .52877 .43394 L .53716 .44396 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52877 .43394 m .53716 .44396 L .53746 .5136 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.52877 .43394 m .53716 .44396 L .53779 .5868 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.48075 .58006 m .4829 .43696 L .46442 .44347 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.47051 .59587 m .46133 .58632 L .46442 .44347 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.47051 .59587 m .46133 .58632 L .46442 .44347 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.46442 .44347 m .46133 .58632 L .48075 .58006 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.4829 .43696 m .46442 .44347 L .45561 .43344 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48075 .58006 m .4829 .43696 L .46442 .44347 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.46442 .44347 m .46133 .58632 L .48075 .58006 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.4372 .43997 m .45561 .43344 L .46442 .44347 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45561 .43344 m .46442 .44347 L .46133 .58632 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.45561 .43344 m .46442 .44347 L .46133 .58632 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.40395 .57957 m .40986 .43646 L .39165 .44297 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.39448 .59539 m .38484 .58584 L .39165 .44297 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.39108 .67222 m .38116 .66291 L .39165 .44297 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.39165 .44297 m .38484 .58584 L .40395 .57957 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.40986 .43646 m .39165 .44297 L .3824 .43294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40395 .57957 m .40986 .43646 L .39165 .44297 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.39165 .44297 m .38484 .58584 L .40395 .57957 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.36426 .43947 m .3824 .43294 L .39165 .44297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3824 .43294 m .39165 .44297 L .38484 .58584 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3824 .43294 m .39165 .44297 L .38484 .58584 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.82841 .50573 m .82099 .43596 L .80122 .44247 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.815 .52189 m .80811 .51213 L .80122 .44247 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.815 .52189 m .80811 .51213 L .80122 .44247 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.80122 .44247 m .80811 .51213 L .82841 .50573 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.82099 .43596 m .80122 .44247 L .79438 .43243 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82841 .50573 m .82099 .43596 L .80122 .44247 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.80122 .44247 m .80811 .51213 L .82841 .50573 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.77467 .43897 m .79438 .43243 L .80122 .44247 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79438 .43243 m .80122 .44247 L .80811 .51213 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.79438 .43243 m .80122 .44247 L .80811 .51213 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.32189 .65631 m .33676 .43596 L .31883 .44247 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.31302 .67176 m .30261 .66244 L .31883 .44247 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.31302 .67176 m .30261 .66244 L .31883 .44247 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.31883 .44247 m .30261 .66244 L .32189 .65631 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.33676 .43596 m .31883 .44247 L .30915 .43243 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32189 .65631 m .33676 .43596 L .31883 .44247 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.31883 .44247 m .30261 .66244 L .32189 .65631 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.29129 .43897 m .30915 .43243 L .31883 .44247 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30915 .43243 m .31883 .44247 L .30261 .66244 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.30915 .43243 m .31883 .44247 L .30261 .66244 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.75359 .50524 m .748 .43545 L .7285 .44197 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.74092 .5214 m .73358 .51164 L .7285 .44197 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.74092 .5214 m .73358 .51164 L .7285 .44197 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.7285 .44197 m .73358 .51164 L .75359 .50524 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.748 .43545 m .7285 .44197 L .72123 .43193 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75359 .50524 m .748 .43545 L .7285 .44197 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.7285 .44197 m .73358 .51164 L .75359 .50524 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.70179 .43847 m .72123 .43193 L .7285 .44197 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72123 .43193 m .7285 .44197 L .73358 .51164 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.72123 .43193 m .7285 .44197 L .73358 .51164 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.67873 .50474 m .67497 .43495 L .65574 .44147 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.66679 .52092 m .659 .51115 L .65574 .44147 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.66679 .52092 m .659 .51115 L .65574 .44147 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.65574 .44147 m .659 .51115 L .67873 .50474 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.67497 .43495 m .65574 .44147 L .64804 .43142 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67873 .50474 m .67497 .43495 L .65574 .44147 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.65574 .44147 m .659 .51115 L .67873 .50474 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.62887 .43797 m .64804 .43142 L .65574 .44147 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64804 .43142 m .65574 .44147 L .659 .51115 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.64804 .43142 m .65574 .44147 L .659 .51115 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.60382 .50425 m .60189 .43445 L .58293 .44097 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.59261 .52043 m .58438 .51066 L .58293 .44097 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.59434 .59349 m .58591 .58392 L .58293 .44097 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.58293 .44097 m .58438 .51066 L .60382 .50425 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.60189 .43445 m .58293 .44097 L .5748 .43092 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60382 .50425 m .60189 .43445 L .58293 .44097 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.58293 .44097 m .58438 .51066 L .60382 .50425 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.55591 .43746 m .5748 .43092 L .58293 .44097 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5748 .43092 m .58293 .44097 L .58438 .51066 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.5748 .43092 m .58293 .44097 L .58438 .51066 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.52897 .57715 m .52877 .43394 L .51009 .44047 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.51823 .59302 m .50933 .58344 L .51009 .44047 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.51823 .59302 m .50933 .58344 L .51009 .44047 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.51009 .44047 m .50933 .58344 L .52897 .57715 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.52877 .43394 m .51009 .44047 L .50152 .43041 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52897 .57715 m .52877 .43394 L .51009 .44047 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.51009 .44047 m .50933 .58344 L .52897 .57715 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.4829 .43696 m .50152 .43041 L .51009 .44047 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50152 .43041 m .51009 .44047 L .50933 .58344 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.50152 .43041 m .51009 .44047 L .50933 .58344 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.45205 .57666 m .45561 .43344 L .4372 .43997 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.44208 .59254 m .4327 .58296 L .4372 .43997 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.44208 .59254 m .4327 .58296 L .4372 .43997 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4372 .43997 m .4327 .58296 L .45205 .57666 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.45561 .43344 m .4372 .43997 L .4282 .4299 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45205 .57666 m .45561 .43344 L .4372 .43997 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.4372 .43997 m .4327 .58296 L .45205 .57666 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.40986 .43646 m .4282 .4299 L .4372 .43997 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4282 .4299 m .4372 .43997 L .4327 .58296 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4282 .4299 m .4372 .43997 L .4327 .58296 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.86748 .43294 m .8475 .43947 L .84092 .4294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.86748 .43294 m .8475 .43947 L .84092 .4294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82099 .43596 m .84092 .4294 L .8475 .43947 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84092 .4294 m .8475 .43947 L .85556 .50919 L p
.976 .671 .339 r
F P
s
P
p
.002 w
.37507 .57618 m .3824 .43294 L .36426 .43947 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.36588 .59206 m .35603 .58247 L .36426 .43947 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.36172 .66897 m .35159 .65962 L .36426 .43947 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.36426 .43947 m .35603 .58247 L .37507 .57618 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.3824 .43294 m .36426 .43947 L .35484 .4294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37507 .57618 m .3824 .43294 L .36426 .43947 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.36426 .43947 m .35603 .58247 L .37507 .57618 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.33676 .43596 m .35484 .4294 L .36426 .43947 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35484 .4294 m .36426 .43947 L .35603 .58247 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.35484 .4294 m .36426 .43947 L .35603 .58247 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.80114 .50226 m .79438 .43243 L .77467 .43897 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.78798 .51849 m .78091 .50869 L .77467 .43897 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.78798 .51849 m .78091 .50869 L .77467 .43897 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.77467 .43897 m .78091 .50869 L .80114 .50226 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.79438 .43243 m .77467 .43897 L .76765 .42889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80114 .50226 m .79438 .43243 L .77467 .43897 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.77467 .43897 m .78091 .50869 L .80114 .50226 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.748 .43545 m .76765 .42889 L .77467 .43897 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76765 .42889 m .77467 .43897 L .78091 .50869 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.76765 .42889 m .77467 .43897 L .78091 .50869 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.29206 .653 m .30915 .43243 L .29129 .43897 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.28348 .66851 m .27286 .65915 L .29129 .43897 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.29129 .43897 m .27286 .65915 L .29206 .653 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.30915 .43243 m .29129 .43897 L .28143 .42889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29206 .653 m .30915 .43243 L .29129 .43897 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.29129 .43897 m .27286 .65915 L .29206 .653 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.28143 .42889 m .29129 .43897 L .27286 .65915 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.72615 .50177 m .72123 .43193 L .70179 .43847 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.71373 .518 m .70621 .5082 L .70179 .43847 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.71373 .518 m .70621 .5082 L .70179 .43847 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.70179 .43847 m .70621 .5082 L .72615 .50177 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.72123 .43193 m .70179 .43847 L .69434 .42838 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72615 .50177 m .72123 .43193 L .70179 .43847 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.70179 .43847 m .70621 .5082 L .72615 .50177 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.67497 .43495 m .69434 .42838 L .70179 .43847 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69434 .42838 m .70179 .43847 L .70621 .5082 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.69434 .42838 m .70179 .43847 L .70621 .5082 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.65113 .50127 m .64804 .43142 L .62887 .43797 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.63943 .51751 m .63147 .50771 L .62887 .43797 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.63943 .51751 m .63147 .50771 L .62887 .43797 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.62887 .43797 m .63147 .50771 L .65113 .50127 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.64804 .43142 m .62887 .43797 L .62099 .42788 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65113 .50127 m .64804 .43142 L .62887 .43797 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.62887 .43797 m .63147 .50771 L .65113 .50127 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.60189 .43445 m .62099 .42788 L .62887 .43797 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62099 .42788 m .62887 .43797 L .63147 .50771 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.62099 .42788 m .62887 .43797 L .63147 .50771 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.57605 .50077 m .5748 .43092 L .55591 .43746 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.5651 .51703 m .55668 .50721 L .55591 .43746 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.56612 .59016 m .5575 .58054 L .55591 .43746 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.55591 .43746 m .55668 .50721 L .57605 .50077 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.5748 .43092 m .55591 .43746 L .5476 .42737 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57605 .50077 m .5748 .43092 L .55591 .43746 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.55591 .43746 m .55668 .50721 L .57605 .50077 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.52877 .43394 m .5476 .42737 L .55591 .43746 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5476 .42737 m .55591 .43746 L .55668 .50721 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.5476 .42737 m .55591 .43746 L .55668 .50721 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.50032 .57374 m .50152 .43041 L .4829 .43696 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.48984 .58968 m .48075 .58006 L .4829 .43696 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.48984 .58968 m .48075 .58006 L .4829 .43696 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.4829 .43696 m .48075 .58006 L .50032 .57374 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.50152 .43041 m .4829 .43696 L .47416 .42686 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50032 .57374 m .50152 .43041 L .4829 .43696 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.4829 .43696 m .48075 .58006 L .50032 .57374 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.45561 .43344 m .47416 .42686 L .4829 .43696 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47416 .42686 m .4829 .43696 L .48075 .58006 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.47416 .42686 m .4829 .43696 L .48075 .58006 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.42322 .57326 m .4282 .4299 L .40986 .43646 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.41352 .5892 m .40395 .57957 L .40986 .43646 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.41352 .5892 m .40395 .57957 L .40986 .43646 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.40986 .43646 m .40395 .57957 L .42322 .57326 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.4282 .4299 m .40986 .43646 L .40068 .42635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42322 .57326 m .4282 .4299 L .40986 .43646 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.40986 .43646 m .40395 .57957 L .42322 .57326 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.3824 .43294 m .40068 .42635 L .40986 .43646 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40068 .42635 m .40986 .43646 L .40395 .57957 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.40068 .42635 m .40986 .43646 L .40395 .57957 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.84886 .49928 m .84092 .4294 L .82099 .43596 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.8352 .51556 m .82841 .50573 L .82099 .43596 L p
.976 .674 .353 r
F P
s
P
p
.002 w
.8352 .51556 m .82841 .50573 L .82099 .43596 L p
.976 .674 .353 r
F P
s
P
p
.002 w
.82099 .43596 m .82841 .50573 L .84886 .49928 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.84092 .4294 m .82099 .43596 L .81424 .42584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84092 .4294 m .82099 .43596 L .81424 .42584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79438 .43243 m .81424 .42584 L .82099 .43596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81424 .42584 m .82099 .43596 L .82841 .50573 L p
.976 .674 .353 r
F P
s
P
p
.002 w
.34607 .57277 m .35484 .4294 L .33676 .43596 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.33715 .58872 m .3271 .57909 L .33676 .43596 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.33223 .66571 m .32189 .65631 L .33676 .43596 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.33676 .43596 m .3271 .57909 L .34607 .57277 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.35484 .4294 m .33676 .43596 L .32716 .42584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34607 .57277 m .35484 .4294 L .33676 .43596 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.33676 .43596 m .3271 .57909 L .34607 .57277 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.30915 .43243 m .32716 .42584 L .33676 .43596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32716 .42584 m .33676 .43596 L .3271 .57909 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.32716 .42584 m .33676 .43596 L .32189 .65631 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.77375 .49878 m .76765 .42889 L .748 .43545 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.76083 .51507 m .75359 .50524 L .748 .43545 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.76083 .51507 m .75359 .50524 L .748 .43545 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.748 .43545 m .75359 .50524 L .77375 .49878 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.76765 .42889 m .748 .43545 L .74082 .42533 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77375 .49878 m .76765 .42889 L .748 .43545 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.748 .43545 m .75359 .50524 L .77375 .49878 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.72123 .43193 m .74082 .42533 L .748 .43545 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74082 .42533 m .748 .43545 L .75359 .50524 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.74082 .42533 m .748 .43545 L .75359 .50524 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.69861 .49828 m .69434 .42838 L .67497 .43495 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.68642 .51458 m .67873 .50474 L .67497 .43495 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.68642 .51458 m .67873 .50474 L .67497 .43495 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.67497 .43495 m .67873 .50474 L .69861 .49828 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.69434 .42838 m .67497 .43495 L .66735 .42482 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69861 .49828 m .69434 .42838 L .67497 .43495 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.67497 .43495 m .67873 .50474 L .69861 .49828 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.64804 .43142 m .66735 .42482 L .67497 .43495 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66735 .42482 m .67497 .43495 L .67873 .50474 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.66735 .42482 m .67497 .43495 L .67873 .50474 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.62341 .49778 m .62099 .42788 L .60189 .43445 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.61197 .51409 m .60382 .50425 L .60189 .43445 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.61197 .51409 m .60382 .50425 L .60189 .43445 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.60189 .43445 m .60382 .50425 L .62341 .49778 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.62099 .42788 m .60189 .43445 L .59384 .42432 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62341 .49778 m .62099 .42788 L .60189 .43445 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.60189 .43445 m .60382 .50425 L .62341 .49778 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.5748 .43092 m .59384 .42432 L .60189 .43445 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59384 .42432 m .60189 .43445 L .60382 .50425 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.59384 .42432 m .60189 .43445 L .60382 .50425 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.54817 .49728 m .5476 .42737 L .52877 .43394 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.53746 .5136 m .52887 .50375 L .52877 .43394 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.53779 .5868 m .52897 .57715 L .52877 .43394 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.52877 .43394 m .52887 .50375 L .54817 .49728 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.5476 .42737 m .52877 .43394 L .52028 .42381 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54817 .49728 m .5476 .42737 L .52877 .43394 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.52877 .43394 m .52887 .50375 L .54817 .49728 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.50152 .43041 m .52028 .42381 L .52877 .43394 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52028 .42381 m .52877 .43394 L .52887 .50375 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.52028 .42381 m .52877 .43394 L .52897 .57715 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.47154 .57032 m .47416 .42686 L .45561 .43344 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.46133 .58632 m .45205 .57666 L .45561 .43344 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.46133 .58632 m .45205 .57666 L .45561 .43344 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.45561 .43344 m .45205 .57666 L .47154 .57032 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.47416 .42686 m .45561 .43344 L .44669 .4233 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47154 .57032 m .47416 .42686 L .45561 .43344 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.45561 .43344 m .45205 .57666 L .47154 .57032 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.4282 .4299 m .44669 .4233 L .45561 .43344 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44669 .4233 m .45561 .43344 L .45205 .57666 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.44669 .4233 m .45561 .43344 L .45205 .57666 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.88762 .42635 m .86748 .43294 L .86099 .42279 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.88762 .42635 m .86748 .43294 L .86099 .42279 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84092 .4294 m .86099 .42279 L .86748 .43294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84092 .4294 m .86099 .42279 L .86748 .43294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39427 .56983 m .40068 .42635 L .3824 .43294 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.38484 .58584 m .37507 .57618 L .3824 .43294 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.38484 .58584 m .37507 .57618 L .3824 .43294 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3824 .43294 m .37507 .57618 L .39427 .56983 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.40068 .42635 m .3824 .43294 L .37305 .42279 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39427 .56983 m .40068 .42635 L .3824 .43294 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.3824 .43294 m .37507 .57618 L .39427 .56983 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.35484 .4294 m .37305 .42279 L .3824 .43294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37305 .42279 m .3824 .43294 L .37507 .57618 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.37305 .42279 m .3824 .43294 L .37507 .57618 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.82153 .49578 m .81424 .42584 L .79438 .43243 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.80811 .51213 m .80114 .50226 L .79438 .43243 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.80811 .51213 m .80114 .50226 L .79438 .43243 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.79438 .43243 m .80114 .50226 L .82153 .49578 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.81424 .42584 m .79438 .43243 L .78745 .42227 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81424 .42584 m .79438 .43243 L .78745 .42227 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76765 .42889 m .78745 .42227 L .79438 .43243 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78745 .42227 m .79438 .43243 L .80114 .50226 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.31143 .6468 m .32716 .42584 L .30915 .43243 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.30261 .66244 m .29206 .653 L .30915 .43243 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.30261 .66244 m .29206 .653 L .30915 .43243 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.30915 .43243 m .29206 .653 L .31143 .6468 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.32716 .42584 m .30915 .43243 L .29936 .42227 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31694 .56934 m .32716 .42584 L .30915 .43243 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.30915 .43243 m .29805 .57569 L .31694 .56934 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.28143 .42889 m .29936 .42227 L .30915 .43243 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29936 .42227 m .30915 .43243 L .29805 .57569 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.29936 .42227 m .30915 .43243 L .29206 .653 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.74626 .49528 m .74082 .42533 L .72123 .43193 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.73358 .51164 m .72615 .50177 L .72123 .43193 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.73358 .51164 m .72615 .50177 L .72123 .43193 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.72123 .43193 m .72615 .50177 L .74626 .49528 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.74082 .42533 m .72123 .43193 L .71387 .42176 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74626 .49528 m .74082 .42533 L .72123 .43193 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.72123 .43193 m .72615 .50177 L .74626 .49528 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.69434 .42838 m .71387 .42176 L .72123 .43193 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71387 .42176 m .72123 .43193 L .72615 .50177 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.71387 .42176 m .72123 .43193 L .72615 .50177 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.67094 .49478 m .66735 .42482 L .64804 .43142 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.659 .51115 m .65113 .50127 L .64804 .43142 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.659 .51115 m .65113 .50127 L .64804 .43142 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.64804 .43142 m .65113 .50127 L .67094 .49478 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.66735 .42482 m .64804 .43142 L .64024 .42125 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67094 .49478 m .66735 .42482 L .64804 .43142 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.64804 .43142 m .65113 .50127 L .67094 .49478 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.62099 .42788 m .64024 .42125 L .64804 .43142 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64024 .42125 m .64804 .43142 L .65113 .50127 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.64024 .42125 m .64804 .43142 L .65113 .50127 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.59558 .49428 m .59384 .42432 L .5748 .43092 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.58438 .51066 m .57605 .50077 L .5748 .43092 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.58438 .51066 m .57605 .50077 L .5748 .43092 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.5748 .43092 m .57605 .50077 L .59558 .49428 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.59384 .42432 m .5748 .43092 L .56657 .42074 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59558 .49428 m .59384 .42432 L .5748 .43092 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.5748 .43092 m .57605 .50077 L .59558 .49428 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.5476 .42737 m .56657 .42074 L .5748 .43092 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56657 .42074 m .5748 .43092 L .57605 .50077 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.56657 .42074 m .5748 .43092 L .57605 .50077 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.52005 .56738 m .52028 .42381 L .50152 .43041 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.50933 .58344 m .50032 .57374 L .50152 .43041 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.50933 .58344 m .50032 .57374 L .50152 .43041 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.50152 .43041 m .50032 .57374 L .52005 .56738 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.52028 .42381 m .50152 .43041 L .49286 .42023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52017 .49378 m .52028 .42381 L .50152 .43041 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.50152 .43041 m .50094 .50028 L .52017 .49378 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.47416 .42686 m .49286 .42023 L .50152 .43041 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49286 .42023 m .50152 .43041 L .50094 .50028 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.49286 .42023 m .50152 .43041 L .50032 .57374 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.77507 .53439 m .75533 .54057 L .74817 .53105 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.44264 .56688 m .44669 .4233 L .4282 .4299 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.4327 .58296 m .42322 .57326 L .4282 .4299 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4327 .58296 m .42322 .57326 L .4282 .4299 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4282 .4299 m .42322 .57326 L .44264 .56688 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.44669 .4233 m .4282 .4299 L .4191 .41972 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44264 .56688 m .44669 .4233 L .4282 .4299 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.4282 .4299 m .42322 .57326 L .44264 .56688 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.40068 .42635 m .4191 .41972 L .4282 .4299 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4191 .41972 m .4282 .4299 L .42322 .57326 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4191 .41972 m .4282 .4299 L .42322 .57326 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.86099 .42279 m .84092 .4294 L .83425 .4192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.86099 .42279 m .84092 .4294 L .83425 .4192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.85556 .50919 m .84886 .49928 L .84092 .4294 L p
.976 .671 .339 r
F P
s
P
p
.002 w
.81424 .42584 m .83425 .4192 L .84092 .4294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81424 .42584 m .83425 .4192 L .84092 .4294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36519 .56639 m .37305 .42279 L .35484 .4294 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.35603 .58247 m .34607 .57277 L .35484 .4294 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.35603 .58247 m .34607 .57277 L .35484 .4294 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.35484 .4294 m .34607 .57277 L .36519 .56639 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.37305 .42279 m .35484 .4294 L .3453 .4192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36519 .56639 m .37305 .42279 L .35484 .4294 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.35484 .4294 m .34607 .57277 L .36519 .56639 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.32716 .42584 m .3453 .4192 L .35484 .4294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3453 .4192 m .35484 .4294 L .34607 .57277 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.3453 .4192 m .35484 .4294 L .34607 .57277 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.79408 .49227 m .78745 .42227 L .76765 .42889 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.78091 .50869 m .77375 .49878 L .76765 .42889 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.78091 .50869 m .77375 .49878 L .76765 .42889 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.76765 .42889 m .77375 .49878 L .79408 .49227 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.78745 .42227 m .76765 .42889 L .76055 .41869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78745 .42227 m .76765 .42889 L .76055 .41869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74082 .42533 m .76055 .41869 L .76765 .42889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76055 .41869 m .76765 .42889 L .77375 .49878 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.28138 .64344 m .29936 .42227 L .28143 .42889 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.27286 .65915 m .2621 .64967 L .28143 .42889 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.28143 .42889 m .2621 .64967 L .28138 .64344 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.29936 .42227 m .28143 .42889 L .27145 .41869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28768 .5659 m .29936 .42227 L .28143 .42889 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.28143 .42889 m .26887 .57228 L .28768 .5659 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.27145 .41869 m .28143 .42889 L .26887 .57228 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.71864 .49177 m .71387 .42176 L .69434 .42838 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.70621 .5082 m .69861 .49828 L .69434 .42838 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.70621 .5082 m .69861 .49828 L .69434 .42838 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.69434 .42838 m .69861 .49828 L .71864 .49177 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.71387 .42176 m .69434 .42838 L .68681 .41818 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71864 .49177 m .71387 .42176 L .69434 .42838 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.69434 .42838 m .69861 .49828 L .71864 .49177 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.66735 .42482 m .68681 .41818 L .69434 .42838 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68681 .41818 m .69434 .42838 L .69861 .49828 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.68681 .41818 m .69434 .42838 L .69861 .49828 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.64316 .49127 m .64024 .42125 L .62099 .42788 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.63147 .50771 m .62341 .49778 L .62099 .42788 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.63147 .50771 m .62341 .49778 L .62099 .42788 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.62099 .42788 m .62341 .49778 L .64316 .49127 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.64024 .42125 m .62099 .42788 L .61302 .41766 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64316 .49127 m .64024 .42125 L .62099 .42788 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.62099 .42788 m .62341 .49778 L .64316 .49127 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.59384 .42432 m .61302 .41766 L .62099 .42788 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61302 .41766 m .62099 .42788 L .62341 .49778 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.61302 .41766 m .62099 .42788 L .62341 .49778 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.56762 .49077 m .56657 .42074 L .5476 .42737 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.55668 .50721 m .54817 .49728 L .5476 .42737 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.55668 .50721 m .54817 .49728 L .5476 .42737 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.5476 .42737 m .54817 .49728 L .56762 .49077 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.56657 .42074 m .5476 .42737 L .53919 .41715 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56762 .49077 m .56657 .42074 L .5476 .42737 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.5476 .42737 m .54817 .49728 L .56762 .49077 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.52028 .42381 m .53919 .41715 L .5476 .42737 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53919 .41715 m .5476 .42737 L .54817 .49728 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.53919 .41715 m .5476 .42737 L .54817 .49728 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.4912 .56393 m .49286 .42023 L .47416 .42686 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.48075 .58006 m .47154 .57032 L .47416 .42686 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.48075 .58006 m .47154 .57032 L .47416 .42686 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.47416 .42686 m .47154 .57032 L .4912 .56393 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.49286 .42023 m .47416 .42686 L .46532 .41664 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49205 .49026 m .49286 .42023 L .47416 .42686 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.47416 .42686 m .47288 .49678 L .49205 .49026 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.44669 .4233 m .46532 .41664 L .47416 .42686 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46532 .41664 m .47416 .42686 L .47288 .49678 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.46532 .41664 m .47416 .42686 L .47154 .57032 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.74817 .53105 m .72849 .53725 L .72115 .52768 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.86099 .42279 m .88122 .41612 L .88762 .42635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.86099 .42279 m .88122 .41612 L .88762 .42635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41361 .56343 m .4191 .41972 L .40068 .42635 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.40395 .57957 m .39427 .56983 L .40068 .42635 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.40395 .57957 m .39427 .56983 L .40068 .42635 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.40068 .42635 m .39427 .56983 L .41361 .56343 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.4191 .41972 m .40068 .42635 L .3914 .41612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41361 .56343 m .4191 .41972 L .40068 .42635 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.40068 .42635 m .39427 .56983 L .41361 .56343 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.37305 .42279 m .3914 .41612 L .40068 .42635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3914 .41612 m .40068 .42635 L .39427 .56983 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.3914 .41612 m .40068 .42635 L .39427 .56983 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.83425 .4192 m .81424 .42584 L .80741 .41561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83425 .4192 m .81424 .42584 L .80741 .41561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82841 .50573 m .82153 .49578 L .81424 .42584 L p
.976 .674 .353 r
F P
s
P
p
.002 w
.78745 .42227 m .80741 .41561 L .81424 .42584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78745 .42227 m .80741 .41561 L .81424 .42584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33598 .56294 m .3453 .4192 L .32716 .42584 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.3271 .57909 m .31694 .56934 L .32716 .42584 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.32189 .65631 m .31143 .6468 L .32716 .42584 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.32716 .42584 m .31694 .56934 L .33598 .56294 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.3453 .4192 m .32716 .42584 L .31743 .41561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33598 .56294 m .3453 .4192 L .32716 .42584 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.32716 .42584 m .31694 .56934 L .33598 .56294 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.29936 .42227 m .31743 .41561 L .32716 .42584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31743 .41561 m .32716 .42584 L .31694 .56934 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.31743 .41561 m .32716 .42584 L .31694 .56934 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.76651 .48875 m .76055 .41869 L .74082 .42533 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.75359 .50524 m .74626 .49528 L .74082 .42533 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.75359 .50524 m .74626 .49528 L .74082 .42533 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.74082 .42533 m .74626 .49528 L .76651 .48875 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.76055 .41869 m .74082 .42533 L .73354 .41509 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76055 .41869 m .74082 .42533 L .73354 .41509 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71387 .42176 m .73354 .41509 L .74082 .42533 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73354 .41509 m .74082 .42533 L .74626 .49528 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.69091 .48824 m .68681 .41818 L .66735 .42482 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.67873 .50474 m .67094 .49478 L .66735 .42482 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.67873 .50474 m .67094 .49478 L .66735 .42482 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.66735 .42482 m .67094 .49478 L .69091 .48824 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.68681 .41818 m .66735 .42482 L .65964 .41458 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69091 .48824 m .68681 .41818 L .66735 .42482 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.66735 .42482 m .67094 .49478 L .69091 .48824 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.64024 .42125 m .65964 .41458 L .66735 .42482 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65964 .41458 m .66735 .42482 L .67094 .49478 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.65964 .41458 m .66735 .42482 L .67094 .49478 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.61525 .48774 m .61302 .41766 L .59384 .42432 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.60382 .50425 m .59558 .49428 L .59384 .42432 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.60382 .50425 m .59558 .49428 L .59384 .42432 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.59384 .42432 m .59558 .49428 L .61525 .48774 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.61302 .41766 m .59384 .42432 L .58569 .41406 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61525 .48774 m .61302 .41766 L .59384 .42432 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.59384 .42432 m .59558 .49428 L .61525 .48774 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.56657 .42074 m .58569 .41406 L .59384 .42432 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58569 .41406 m .59384 .42432 L .59558 .49428 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.58569 .41406 m .59384 .42432 L .59558 .49428 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.53955 .48723 m .53919 .41715 L .52028 .42381 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.52887 .50375 m .52017 .49378 L .52028 .42381 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.52897 .57715 m .52005 .56738 L .52028 .42381 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.52028 .42381 m .52017 .49378 L .53955 .48723 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.53919 .41715 m .52028 .42381 L .5117 .41355 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53955 .48723 m .53919 .41715 L .52028 .42381 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.52028 .42381 m .52017 .49378 L .53955 .48723 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.49286 .42023 m .5117 .41355 L .52028 .42381 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5117 .41355 m .52028 .42381 L .52017 .49378 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.5117 .41355 m .52028 .42381 L .52017 .49378 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.79496 .52816 m .77507 .53439 L .76799 .52479 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.74817 .53105 m .76799 .52479 L .77507 .53439 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.46222 .56046 m .46532 .41664 L .44669 .4233 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.45205 .57666 m .44264 .56688 L .44669 .4233 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.45205 .57666 m .44264 .56688 L .44669 .4233 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.44669 .4233 m .44264 .56688 L .46222 .56046 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.46532 .41664 m .44669 .4233 L .43766 .41303 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46381 .48673 m .46532 .41664 L .44669 .4233 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.44669 .4233 m .44471 .49328 L .46381 .48673 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.4191 .41972 m .43766 .41303 L .44669 .4233 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43766 .41303 m .44669 .4233 L .44471 .49328 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.43766 .41303 m .44669 .4233 L .44264 .56688 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.72115 .52768 m .70154 .53392 L .69403 .52431 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.88122 .41612 m .86099 .42279 L .85442 .41251 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.88122 .41612 m .86099 .42279 L .85442 .41251 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83425 .4192 m .85442 .41251 L .86099 .42279 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83425 .4192 m .85442 .41251 L .86099 .42279 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38446 .55997 m .3914 .41612 L .37305 .42279 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.37507 .57618 m .36519 .56639 L .37305 .42279 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.37507 .57618 m .36519 .56639 L .37305 .42279 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.37305 .42279 m .36519 .56639 L .38446 .55997 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.3914 .41612 m .37305 .42279 L .36358 .41251 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38446 .55997 m .3914 .41612 L .37305 .42279 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.37305 .42279 m .36519 .56639 L .38446 .55997 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.3453 .4192 m .36358 .41251 L .37305 .42279 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36358 .41251 m .37305 .42279 L .36519 .56639 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.36358 .41251 m .37305 .42279 L .36519 .56639 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.80741 .41561 m .78745 .42227 L .78044 .412 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80741 .41561 m .78745 .42227 L .78044 .412 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80114 .50226 m .79408 .49227 L .78745 .42227 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.76055 .41869 m .78044 .412 L .78745 .42227 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76055 .41869 m .78044 .412 L .78745 .42227 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30665 .55947 m .31743 .41561 L .29936 .42227 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.29805 .57569 m .28768 .5659 L .29936 .42227 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.29206 .653 m .28138 .64344 L .29936 .42227 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.29936 .42227 m .28768 .5659 L .30665 .55947 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.31743 .41561 m .29936 .42227 L .28945 .412 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30665 .55947 m .31743 .41561 L .29936 .42227 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.29936 .42227 m .28768 .5659 L .30665 .55947 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.27145 .41869 m .28945 .412 L .29936 .42227 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28945 .412 m .29936 .42227 L .28768 .5659 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.28945 .412 m .29936 .42227 L .28768 .5659 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.73883 .48521 m .73354 .41509 L .71387 .42176 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.72615 .50177 m .71864 .49177 L .71387 .42176 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.72615 .50177 m .71864 .49177 L .71387 .42176 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.71387 .42176 m .71864 .49177 L .73883 .48521 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.73354 .41509 m .71387 .42176 L .70642 .41148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73354 .41509 m .71387 .42176 L .70642 .41148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68681 .41818 m .70642 .41148 L .71387 .42176 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70642 .41148 m .71387 .42176 L .71864 .49177 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.66306 .4847 m .65964 .41458 L .64024 .42125 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.65113 .50127 m .64316 .49127 L .64024 .42125 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.65113 .50127 m .64316 .49127 L .64024 .42125 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.64024 .42125 m .64316 .49127 L .66306 .4847 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.65964 .41458 m .64024 .42125 L .63235 .41096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66306 .4847 m .65964 .41458 L .64024 .42125 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.64024 .42125 m .64316 .49127 L .66306 .4847 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.61302 .41766 m .63235 .41096 L .64024 .42125 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63235 .41096 m .64024 .42125 L .64316 .49127 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.63235 .41096 m .64024 .42125 L .64316 .49127 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.68165 .63696 m .66216 .64278 L .65441 .6338 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.58723 .4842 m .58569 .41406 L .56657 .42074 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.57605 .50077 m .56762 .49077 L .56657 .42074 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.57605 .50077 m .56762 .49077 L .56657 .42074 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.56657 .42074 m .56762 .49077 L .58723 .4842 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.58569 .41406 m .56657 .42074 L .55824 .41045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58723 .4842 m .58569 .41406 L .56657 .42074 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.56657 .42074 m .56762 .49077 L .58723 .4842 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.53919 .41715 m .55824 .41045 L .56657 .42074 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55824 .41045 m .56657 .42074 L .56762 .49077 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.55824 .41045 m .56657 .42074 L .56762 .49077 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.51136 .48369 m .5117 .41355 L .49286 .42023 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.50094 .50028 m .49205 .49026 L .49286 .42023 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.50032 .57374 m .4912 .56393 L .49286 .42023 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.49286 .42023 m .49205 .49026 L .51136 .48369 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.5117 .41355 m .49286 .42023 L .48409 .40993 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51136 .48369 m .5117 .41355 L .49286 .42023 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.49286 .42023 m .49205 .49026 L .51136 .48369 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.46532 .41664 m .48409 .40993 L .49286 .42023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48409 .40993 m .49286 .42023 L .49205 .49026 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.48409 .40993 m .49286 .42023 L .49205 .49026 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.76799 .52479 m .74817 .53105 L .74092 .5214 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.72115 .52768 m .74092 .5214 L .74817 .53105 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.43312 .55698 m .43766 .41303 L .4191 .41972 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.42322 .57326 m .41361 .56343 L .4191 .41972 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.42322 .57326 m .41361 .56343 L .4191 .41972 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4191 .41972 m .41361 .56343 L .43312 .55698 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.43766 .41303 m .4191 .41972 L .40989 .40941 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43545 .48318 m .43766 .41303 L .4191 .41972 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.4191 .41972 m .41643 .48976 L .43545 .48318 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.3914 .41612 m .40989 .40941 L .4191 .41972 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40989 .40941 m .4191 .41972 L .41643 .48976 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.40989 .40941 m .4191 .41972 L .41361 .56343 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.69403 .52431 m .67448 .53057 L .66679 .52092 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.85442 .41251 m .83425 .4192 L .82751 .40889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.85442 .41251 m .83425 .4192 L .82751 .40889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80741 .41561 m .82751 .40889 L .83425 .4192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80741 .41561 m .82751 .40889 L .83425 .4192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35518 .55648 m .36358 .41251 L .3453 .4192 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.34607 .57277 m .33598 .56294 L .3453 .4192 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.34607 .57277 m .33598 .56294 L .3453 .4192 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.3453 .4192 m .33598 .56294 L .35518 .55648 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.36358 .41251 m .3453 .4192 L .33565 .40889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35518 .55648 m .36358 .41251 L .3453 .4192 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.3453 .4192 m .33598 .56294 L .35518 .55648 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.31743 .41561 m .33565 .40889 L .3453 .4192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33565 .40889 m .3453 .4192 L .33598 .56294 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.33565 .40889 m .3453 .4192 L .33598 .56294 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.78044 .412 m .76055 .41869 L .75337 .40837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78044 .412 m .76055 .41869 L .75337 .40837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77375 .49878 m .76651 .48875 L .76055 .41869 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.73354 .41509 m .75337 .40837 L .76055 .41869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73354 .41509 m .75337 .40837 L .76055 .41869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27719 .55599 m .28945 .412 L .27145 .41869 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.26887 .57228 m .2583 .56244 L .27145 .41869 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.27145 .41869 m .2583 .56244 L .27719 .55599 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.28945 .412 m .27145 .41869 L .26136 .40837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27719 .55599 m .28945 .412 L .27145 .41869 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.27145 .41869 m .2583 .56244 L .27719 .55599 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.26136 .40837 m .27145 .41869 L .2583 .56244 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.71103 .48165 m .70642 .41148 L .68681 .41818 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.69861 .49828 m .69091 .48824 L .68681 .41818 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.69861 .49828 m .69091 .48824 L .68681 .41818 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.68681 .41818 m .69091 .48824 L .71103 .48165 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.70642 .41148 m .68681 .41818 L .67919 .40785 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70642 .41148 m .68681 .41818 L .67919 .40785 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65964 .41458 m .67919 .40785 L .68681 .41818 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67919 .40785 m .68681 .41818 L .69091 .48824 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.63509 .48115 m .63235 .41096 L .61302 .41766 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.62341 .49778 m .61525 .48774 L .61302 .41766 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.62341 .49778 m .61525 .48774 L .61302 .41766 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.61302 .41766 m .61525 .48774 L .63509 .48115 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.63235 .41096 m .61302 .41766 L .60496 .40733 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63509 .48115 m .63235 .41096 L .61302 .41766 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.61302 .41766 m .61525 .48774 L .63509 .48115 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.58569 .41406 m .60496 .40733 L .61302 .41766 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60496 .40733 m .61302 .41766 L .61525 .48774 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.60496 .40733 m .61302 .41766 L .61525 .48774 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.65441 .6338 m .63498 .63965 L .62705 .63064 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.55909 .48064 m .55824 .41045 L .53919 .41715 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.54817 .49728 m .53955 .48723 L .53919 .41715 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.54817 .49728 m .53955 .48723 L .53919 .41715 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.53919 .41715 m .53955 .48723 L .55909 .48064 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.55824 .41045 m .53919 .41715 L .53068 .40681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55909 .48064 m .55824 .41045 L .53919 .41715 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.53919 .41715 m .53955 .48723 L .55909 .48064 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.5117 .41355 m .53068 .40681 L .53919 .41715 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53068 .40681 m .53919 .41715 L .53955 .48723 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.53068 .40681 m .53919 .41715 L .53955 .48723 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.815 .52189 m .79496 .52816 L .78798 .51849 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.76799 .52479 m .78798 .51849 L .79496 .52816 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.48305 .48013 m .48409 .40993 L .46532 .41664 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.47288 .49678 m .46381 .48673 L .46532 .41664 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.47154 .57032 m .46222 .56046 L .46532 .41664 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.46532 .41664 m .46381 .48673 L .48305 .48013 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.48409 .40993 m .46532 .41664 L .45636 .40629 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48305 .48013 m .48409 .40993 L .46532 .41664 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.46532 .41664 m .46381 .48673 L .48305 .48013 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.43766 .41303 m .45636 .40629 L .46532 .41664 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45636 .40629 m .46532 .41664 L .46381 .48673 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.45636 .40629 m .46532 .41664 L .46381 .48673 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.74092 .5214 m .72115 .52768 L .71373 .518 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.69403 .52431 m .71373 .518 L .72115 .52768 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.85442 .41251 m .87475 .40577 L .88122 .41612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.85442 .41251 m .87475 .40577 L .88122 .41612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40389 .55349 m .40989 .40941 L .3914 .41612 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.39427 .56983 m .38446 .55997 L .3914 .41612 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.39427 .56983 m .38446 .55997 L .3914 .41612 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.3914 .41612 m .38446 .55997 L .40389 .55349 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.40989 .40941 m .3914 .41612 L .382 .40577 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40696 .47962 m .40989 .40941 L .3914 .41612 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.3914 .41612 m .38802 .48622 L .40696 .47962 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.36358 .41251 m .382 .40577 L .3914 .41612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.382 .40577 m .3914 .41612 L .38802 .48622 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.382 .40577 m .3914 .41612 L .38446 .55997 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.66679 .52092 m .6473 .5272 L .63943 .51751 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.82751 .40889 m .80741 .41561 L .80049 .40525 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82751 .40889 m .80741 .41561 L .80049 .40525 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78044 .412 m .80049 .40525 L .80741 .41561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78044 .412 m .80049 .40525 L .80741 .41561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32577 .55299 m .33565 .40889 L .31743 .41561 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.31694 .56934 m .30665 .55947 L .31743 .41561 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.31694 .56934 m .30665 .55947 L .31743 .41561 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.31743 .41561 m .30665 .55947 L .32577 .55299 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.33565 .40889 m .31743 .41561 L .3076 .40525 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32577 .55299 m .33565 .40889 L .31743 .41561 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.31743 .41561 m .30665 .55947 L .32577 .55299 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.28945 .412 m .3076 .40525 L .31743 .41561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3076 .40525 m .31743 .41561 L .30665 .55947 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.3076 .40525 m .31743 .41561 L .30665 .55947 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.75337 .40837 m .73354 .41509 L .72619 .40473 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75337 .40837 m .73354 .41509 L .72619 .40473 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74626 .49528 m .73883 .48521 L .73354 .41509 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.70642 .41148 m .72619 .40473 L .73354 .41509 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70642 .41148 m .72619 .40473 L .73354 .41509 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68311 .47808 m .67919 .40785 L .65964 .41458 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.67094 .49478 m .66306 .4847 L .65964 .41458 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.67094 .49478 m .66306 .4847 L .65964 .41458 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.65964 .41458 m .66306 .4847 L .68311 .47808 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.67919 .40785 m .65964 .41458 L .65184 .40421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67919 .40785 m .65964 .41458 L .65184 .40421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63235 .41096 m .65184 .40421 L .65964 .41458 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65184 .40421 m .65964 .41458 L .66306 .4847 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.70129 .63109 m .68165 .63696 L .67398 .62791 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.65441 .6338 m .67398 .62791 L .68165 .63696 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.607 .47757 m .60496 .40733 L .58569 .41406 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.59558 .49428 m .58723 .4842 L .58569 .41406 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.59558 .49428 m .58723 .4842 L .58569 .41406 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.58569 .41406 m .58723 .4842 L .607 .47757 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.60496 .40733 m .58569 .41406 L .57744 .40369 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.607 .47757 m .60496 .40733 L .58569 .41406 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.58569 .41406 m .58723 .4842 L .607 .47757 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.55824 .41045 m .57744 .40369 L .58569 .41406 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57744 .40369 m .58569 .41406 L .58723 .4842 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.57744 .40369 m .58569 .41406 L .58723 .4842 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.62705 .63064 m .60769 .63651 L .59958 .62746 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.53083 .47706 m .53068 .40681 L .5117 .41355 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.52017 .49378 m .51136 .48369 L .5117 .41355 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.52017 .49378 m .51136 .48369 L .5117 .41355 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.5117 .41355 m .51136 .48369 L .53083 .47706 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.53068 .40681 m .5117 .41355 L .50301 .40317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53083 .47706 m .53068 .40681 L .5117 .41355 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.5117 .41355 m .51136 .48369 L .53083 .47706 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.48409 .40993 m .50301 .40317 L .5117 .41355 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50301 .40317 m .5117 .41355 L .51136 .48369 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.50301 .40317 m .5117 .41355 L .51136 .48369 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.78798 .51849 m .76799 .52479 L .76083 .51507 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.74092 .5214 m .76083 .51507 L .76799 .52479 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.45462 .47655 m .45636 .40629 L .43766 .41303 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.44471 .49328 m .43545 .48318 L .43766 .41303 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.44264 .56688 m .43312 .55698 L .43766 .41303 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.43766 .41303 m .43545 .48318 L .45462 .47655 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.45636 .40629 m .43766 .41303 L .42853 .40264 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45462 .47655 m .45636 .40629 L .43766 .41303 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.43766 .41303 m .43545 .48318 L .45462 .47655 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.40989 .40941 m .42853 .40264 L .43766 .41303 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42853 .40264 m .43766 .41303 L .43545 .48318 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.42853 .40264 m .43766 .41303 L .43545 .48318 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.71373 .518 m .69403 .52431 L .68642 .51458 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.66679 .52092 m .68642 .51458 L .69403 .52431 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.87475 .40577 m .85442 .41251 L .84778 .40212 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.87475 .40577 m .85442 .41251 L .84778 .40212 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82751 .40889 m .84778 .40212 L .85442 .41251 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82751 .40889 m .84778 .40212 L .85442 .41251 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37453 .54998 m .382 .40577 L .36358 .41251 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.36519 .56639 m .35518 .55648 L .36358 .41251 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.36519 .56639 m .35518 .55648 L .36358 .41251 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.36358 .41251 m .35518 .55648 L .37453 .54998 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.382 .40577 m .36358 .41251 L .354 .40212 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37453 .54998 m .382 .40577 L .36358 .41251 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.36358 .41251 m .35518 .55648 L .37453 .54998 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.33565 .40889 m .354 .40212 L .36358 .41251 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.354 .40212 m .36358 .41251 L .35518 .55648 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.354 .40212 m .36358 .41251 L .35518 .55648 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.63943 .51751 m .62001 .52382 L .61197 .51409 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.80049 .40525 m .78044 .412 L .77335 .4016 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80049 .40525 m .78044 .412 L .77335 .4016 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75337 .40837 m .77335 .4016 L .78044 .412 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75337 .40837 m .77335 .4016 L .78044 .412 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29623 .54947 m .3076 .40525 L .28945 .412 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.28768 .5659 m .27719 .55599 L .28945 .412 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.28768 .5659 m .27719 .55599 L .28945 .412 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.28945 .412 m .27719 .55599 L .29623 .54947 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.3076 .40525 m .28945 .412 L .27943 .4016 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29623 .54947 m .3076 .40525 L .28945 .412 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.28945 .412 m .27719 .55599 L .29623 .54947 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.26136 .40837 m .27943 .4016 L .28945 .412 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27943 .4016 m .28945 .412 L .27719 .55599 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.27943 .4016 m .28945 .412 L .27719 .55599 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.72619 .40473 m .70642 .41148 L .69889 .40107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72619 .40473 m .70642 .41148 L .69889 .40107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71864 .49177 m .71103 .48165 L .70642 .41148 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.67919 .40785 m .69889 .40107 L .70642 .41148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67919 .40785 m .69889 .40107 L .70642 .41148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58721 .74064 m .56798 .74609 L .55962 .73769 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.65508 .4745 m .65184 .40421 L .63235 .41096 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.64316 .49127 m .63509 .48115 L .63235 .41096 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.64316 .49127 m .63509 .48115 L .63235 .41096 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.63235 .41096 m .63509 .48115 L .65508 .4745 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.65184 .40421 m .63235 .41096 L .62437 .40055 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65184 .40421 m .63235 .41096 L .62437 .40055 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60496 .40733 m .62437 .40055 L .63235 .41096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62437 .40055 m .63235 .41096 L .63509 .48115 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.67398 .62791 m .65441 .6338 L .64656 .62472 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.62705 .63064 m .64656 .62472 L .65441 .6338 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.57879 .47398 m .57744 .40369 L .55824 .41045 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.56762 .49077 m .55909 .48064 L .55824 .41045 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.56762 .49077 m .55909 .48064 L .55824 .41045 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.55824 .41045 m .55909 .48064 L .57879 .47398 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.57744 .40369 m .55824 .41045 L .54982 .40003 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57879 .47398 m .57744 .40369 L .55824 .41045 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.55824 .41045 m .55909 .48064 L .57879 .47398 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.53068 .40681 m .54982 .40003 L .55824 .41045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54982 .40003 m .55824 .41045 L .55909 .48064 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.54982 .40003 m .55824 .41045 L .55909 .48064 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.8352 .51556 m .815 .52189 L .80811 .51213 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.78798 .51849 m .80811 .51213 L .815 .52189 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.59958 .62746 m .58029 .63335 L .57199 .62426 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.50245 .47347 m .50301 .40317 L .48409 .40993 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.49205 .49026 m .48305 .48013 L .48409 .40993 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.49205 .49026 m .48305 .48013 L .48409 .40993 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.48409 .40993 m .48305 .48013 L .50245 .47347 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.50301 .40317 m .48409 .40993 L .47522 .3995 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50245 .47347 m .50301 .40317 L .48409 .40993 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.48409 .40993 m .48305 .48013 L .50245 .47347 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.45636 .40629 m .47522 .3995 L .48409 .40993 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47522 .3995 m .48409 .40993 L .48305 .48013 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.47522 .3995 m .48409 .40993 L .48305 .48013 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.76083 .51507 m .74092 .5214 L .73358 .51164 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.71373 .518 m .73358 .51164 L .74092 .5214 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.42607 .47296 m .42853 .40264 L .40989 .40941 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.41643 .48976 m .40696 .47962 L .40989 .40941 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.41361 .56343 m .40389 .55349 L .40989 .40941 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.40989 .40941 m .40696 .47962 L .42607 .47296 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.42853 .40264 m .40989 .40941 L .40057 .39898 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42607 .47296 m .42853 .40264 L .40989 .40941 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.40989 .40941 m .40696 .47962 L .42607 .47296 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.382 .40577 m .40057 .39898 L .40989 .40941 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40057 .39898 m .40989 .40941 L .40696 .47962 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.40057 .39898 m .40989 .40941 L .40696 .47962 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.68642 .51458 m .66679 .52092 L .659 .51115 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.63943 .51751 m .659 .51115 L .66679 .52092 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.84778 .40212 m .82751 .40889 L .82069 .39845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84778 .40212 m .82751 .40889 L .82069 .39845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80049 .40525 m .82069 .39845 L .82751 .40889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80049 .40525 m .82069 .39845 L .82751 .40889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34504 .54645 m .354 .40212 L .33565 .40889 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.33598 .56294 m .32577 .55299 L .33565 .40889 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.33598 .56294 m .32577 .55299 L .33565 .40889 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.33565 .40889 m .32577 .55299 L .34504 .54645 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.354 .40212 m .33565 .40889 L .32588 .39845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34504 .54645 m .354 .40212 L .33565 .40889 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.33565 .40889 m .32577 .55299 L .34504 .54645 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.3076 .40525 m .32588 .39845 L .33565 .40889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32588 .39845 m .33565 .40889 L .32577 .55299 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.32588 .39845 m .33565 .40889 L .32577 .55299 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.61197 .51409 m .59261 .52043 L .58438 .51066 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.77335 .4016 m .75337 .40837 L .7461 .39793 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77335 .4016 m .75337 .40837 L .7461 .39793 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72619 .40473 m .7461 .39793 L .75337 .40837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72619 .40473 m .7461 .39793 L .75337 .40837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26656 .54595 m .27943 .4016 L .26136 .40837 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.2583 .56244 m .2476 .55249 L .26136 .40837 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.26136 .40837 m .2476 .55249 L .26656 .54595 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.27943 .4016 m .26136 .40837 L .25114 .39793 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26656 .54595 m .27943 .4016 L .26136 .40837 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.26136 .40837 m .2476 .55249 L .26656 .54595 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.25114 .39793 m .26136 .40837 L .2476 .55249 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.69889 .40107 m .67919 .40785 L .67147 .3974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69889 .40107 m .67919 .40785 L .67147 .3974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69091 .48824 m .68311 .47808 L .67919 .40785 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.65184 .40421 m .67147 .3974 L .67919 .40785 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65184 .40421 m .67147 .3974 L .67919 .40785 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72108 .62518 m .70129 .63109 L .6937 .62198 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.67398 .62791 m .6937 .62198 L .70129 .63109 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.62692 .4709 m .62437 .40055 L .60496 .40733 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.61525 .48774 m .607 .47757 L .60496 .40733 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.61525 .48774 m .607 .47757 L .60496 .40733 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.60496 .40733 m .607 .47757 L .62692 .4709 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.62437 .40055 m .60496 .40733 L .59679 .39688 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62692 .4709 m .62437 .40055 L .60496 .40733 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.60496 .40733 m .607 .47757 L .62692 .4709 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.57744 .40369 m .59679 .39688 L .60496 .40733 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59679 .39688 m .60496 .40733 L .607 .47757 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.59679 .39688 m .60496 .40733 L .607 .47757 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.64656 .62472 m .62705 .63064 L .61902 .62152 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.59958 .62746 m .61902 .62152 L .62705 .63064 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.55045 .47038 m .54982 .40003 L .53068 .40681 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.53955 .48723 m .53083 .47706 L .53068 .40681 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.53955 .48723 m .53083 .47706 L .53068 .40681 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.53068 .40681 m .53083 .47706 L .55045 .47038 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.54982 .40003 m .53068 .40681 L .52207 .39635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55045 .47038 m .54982 .40003 L .53068 .40681 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.53068 .40681 m .53083 .47706 L .55045 .47038 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.50301 .40317 m .52207 .39635 L .53068 .40681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52207 .39635 m .53068 .40681 L .53083 .47706 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.52207 .39635 m .53068 .40681 L .53083 .47706 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.80811 .51213 m .78798 .51849 L .78091 .50869 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.76083 .51507 m .78091 .50869 L .78798 .51849 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.57199 .62426 m .55277 .63018 L .54429 .62106 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.47395 .46987 m .47522 .3995 L .45636 .40629 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.46381 .48673 m .45462 .47655 L .45636 .40629 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.46381 .48673 m .45462 .47655 L .45636 .40629 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.45636 .40629 m .45462 .47655 L .47395 .46987 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.47522 .3995 m .45636 .40629 L .44731 .39582 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47395 .46987 m .47522 .3995 L .45636 .40629 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.45636 .40629 m .45462 .47655 L .47395 .46987 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.42853 .40264 m .44731 .39582 L .45636 .40629 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44731 .39582 m .45636 .40629 L .45462 .47655 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.44731 .39582 m .45636 .40629 L .45462 .47655 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.73358 .51164 m .71373 .518 L .70621 .5082 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.68642 .51458 m .70621 .5082 L .71373 .518 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.84778 .40212 m .8682 .3953 L .87475 .40577 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84778 .40212 m .8682 .3953 L .87475 .40577 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39739 .46935 m .40057 .39898 L .382 .40577 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.38802 .48622 m .37836 .47604 L .382 .40577 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.38446 .55997 m .37453 .54998 L .382 .40577 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.382 .40577 m .37836 .47604 L .39739 .46935 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.40057 .39898 m .382 .40577 L .3725 .3953 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39739 .46935 m .40057 .39898 L .382 .40577 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.382 .40577 m .37836 .47604 L .39739 .46935 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.354 .40212 m .3725 .3953 L .382 .40577 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3725 .3953 m .382 .40577 L .37836 .47604 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.3725 .3953 m .382 .40577 L .37453 .54998 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.659 .51115 m .63943 .51751 L .63147 .50771 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.61197 .51409 m .63147 .50771 L .63943 .51751 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.82069 .39845 m .80049 .40525 L .79349 .39477 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82069 .39845 m .80049 .40525 L .79349 .39477 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77335 .4016 m .79349 .39477 L .80049 .40525 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77335 .4016 m .79349 .39477 L .80049 .40525 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31542 .54291 m .32588 .39845 L .3076 .40525 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.30665 .55947 m .29623 .54947 L .3076 .40525 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.30665 .55947 m .29623 .54947 L .3076 .40525 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.3076 .40525 m .29623 .54947 L .31542 .54291 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.32588 .39845 m .3076 .40525 L .29764 .39477 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31542 .54291 m .32588 .39845 L .3076 .40525 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.3076 .40525 m .29623 .54947 L .31542 .54291 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.27943 .4016 m .29764 .39477 L .3076 .40525 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29764 .39477 m .3076 .40525 L .29623 .54947 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.29764 .39477 m .3076 .40525 L .29623 .54947 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.58438 .51066 m .5651 .51703 L .55668 .50721 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.7461 .39793 m .72619 .40473 L .71874 .39424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7461 .39793 m .72619 .40473 L .71874 .39424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69889 .40107 m .71874 .39424 L .72619 .40473 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69889 .40107 m .71874 .39424 L .72619 .40473 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60658 .73515 m .58721 .74064 L .57893 .73217 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.55962 .73769 m .57893 .73217 L .58721 .74064 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.67147 .3974 m .65184 .40421 L .64394 .39372 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67147 .3974 m .65184 .40421 L .64394 .39372 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66306 .4847 m .65508 .4745 L .65184 .40421 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.62437 .40055 m .64394 .39372 L .65184 .40421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62437 .40055 m .64394 .39372 L .65184 .40421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6937 .62198 m .67398 .62791 L .66621 .61876 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.64656 .62472 m .66621 .61876 L .67398 .62791 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.59864 .46728 m .59679 .39688 L .57744 .40369 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.58723 .4842 m .57879 .47398 L .57744 .40369 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.58723 .4842 m .57879 .47398 L .57744 .40369 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.57744 .40369 m .57879 .47398 L .59864 .46728 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.59679 .39688 m .57744 .40369 L .5691 .39319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59864 .46728 m .59679 .39688 L .57744 .40369 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.57744 .40369 m .57879 .47398 L .59864 .46728 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.54982 .40003 m .5691 .39319 L .57744 .40369 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5691 .39319 m .57744 .40369 L .57879 .47398 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.5691 .39319 m .57744 .40369 L .57879 .47398 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.85556 .50919 m .8352 .51556 L .82841 .50573 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.80811 .51213 m .82841 .50573 L .8352 .51556 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.61902 .62152 m .59958 .62746 L .59136 .6183 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.57199 .62426 m .59136 .6183 L .59958 .62746 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.522 .46676 m .52207 .39635 L .50301 .40317 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.51136 .48369 m .50245 .47347 L .50301 .40317 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.51136 .48369 m .50245 .47347 L .50301 .40317 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.50301 .40317 m .50245 .47347 L .522 .46676 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.52207 .39635 m .50301 .40317 L .49421 .39266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.522 .46676 m .52207 .39635 L .50301 .40317 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.50301 .40317 m .50245 .47347 L .522 .46676 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.47522 .3995 m .49421 .39266 L .50301 .40317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49421 .39266 m .50301 .40317 L .50245 .47347 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.49421 .39266 m .50301 .40317 L .50245 .47347 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.78091 .50869 m .76083 .51507 L .75359 .50524 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.73358 .51164 m .75359 .50524 L .76083 .51507 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.44532 .46624 m .44731 .39582 L .42853 .40264 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.43545 .48318 m .42607 .47296 L .42853 .40264 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.43545 .48318 m .42607 .47296 L .42853 .40264 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.42853 .40264 m .42607 .47296 L .44532 .46624 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.44731 .39582 m .42853 .40264 L .41928 .39213 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44532 .46624 m .44731 .39582 L .42853 .40264 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.42853 .40264 m .42607 .47296 L .44532 .46624 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.40057 .39898 m .41928 .39213 L .42853 .40264 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41928 .39213 m .42853 .40264 L .42607 .47296 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.41928 .39213 m .42853 .40264 L .42607 .47296 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.70621 .5082 m .68642 .51458 L .67873 .50474 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.659 .51115 m .67873 .50474 L .68642 .51458 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.8682 .3953 m .84778 .40212 L .84105 .3916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.8682 .3953 m .84778 .40212 L .84105 .3916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82069 .39845 m .84105 .3916 L .84778 .40212 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82069 .39845 m .84105 .3916 L .84778 .40212 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36448 .53986 m .3725 .3953 L .354 .40212 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.35518 .55648 m .34504 .54645 L .354 .40212 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.35518 .55648 m .34504 .54645 L .354 .40212 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.354 .40212 m .34504 .54645 L .36448 .53986 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.3725 .3953 m .354 .40212 L .3443 .3916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36859 .46573 m .3725 .3953 L .354 .40212 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.354 .40212 m .34963 .47244 L .36859 .46573 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.32588 .39845 m .3443 .3916 L .354 .40212 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3443 .3916 m .354 .40212 L .34963 .47244 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3443 .3916 m .354 .40212 L .34504 .54645 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.63147 .50771 m .61197 .51409 L .60382 .50425 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.58438 .51066 m .60382 .50425 L .61197 .51409 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.79349 .39477 m .77335 .4016 L .76618 .39107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79349 .39477 m .77335 .4016 L .76618 .39107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7461 .39793 m .76618 .39107 L .77335 .4016 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7461 .39793 m .76618 .39107 L .77335 .4016 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28567 .53935 m .29764 .39477 L .27943 .4016 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.27719 .55599 m .26656 .54595 L .27943 .4016 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.27719 .55599 m .26656 .54595 L .27943 .4016 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.27943 .4016 m .26656 .54595 L .28567 .53935 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.29764 .39477 m .27943 .4016 L .26928 .39107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28567 .53935 m .29764 .39477 L .27943 .4016 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.27943 .4016 m .26656 .54595 L .28567 .53935 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.25114 .39793 m .26928 .39107 L .27943 .4016 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26928 .39107 m .27943 .4016 L .26656 .54595 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.26928 .39107 m .27943 .4016 L .26656 .54595 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.55668 .50721 m .53746 .5136 L .52887 .50375 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.71874 .39424 m .69889 .40107 L .69126 .39054 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71874 .39424 m .69889 .40107 L .69126 .39054 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67147 .3974 m .69126 .39054 L .69889 .40107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67147 .3974 m .69126 .39054 L .69889 .40107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57893 .73217 m .55962 .73769 L .55115 .72918 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.74102 .61922 m .72108 .62518 L .71358 .61599 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.6937 .62198 m .71358 .61599 L .72108 .62518 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.64394 .39372 m .62437 .40055 L .6163 .39001 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64394 .39372 m .62437 .40055 L .6163 .39001 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63509 .48115 m .62692 .4709 L .62437 .40055 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.59679 .39688 m .6163 .39001 L .62437 .40055 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6163 .39001 m .62437 .40055 L .62692 .4709 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.66621 .61876 m .64656 .62472 L .63861 .61553 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.61902 .62152 m .63861 .61553 L .64656 .62472 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.57024 .46365 m .5691 .39319 L .54982 .40003 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.55909 .48064 m .55045 .47038 L .54982 .40003 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.55909 .48064 m .55045 .47038 L .54982 .40003 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.54982 .40003 m .55045 .47038 L .57024 .46365 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.5691 .39319 m .54982 .40003 L .54129 .38948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57024 .46365 m .5691 .39319 L .54982 .40003 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.54982 .40003 m .55045 .47038 L .57024 .46365 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.52207 .39635 m .54129 .38948 L .54982 .40003 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54129 .38948 m .54982 .40003 L .55045 .47038 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.54129 .38948 m .54982 .40003 L .55045 .47038 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.82841 .50573 m .80811 .51213 L .80114 .50226 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.78091 .50869 m .80114 .50226 L .80811 .51213 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.59136 .6183 m .57199 .62426 L .56359 .61507 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.54429 .62106 m .56359 .61507 L .57199 .62426 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.49343 .46313 m .49421 .39266 L .47522 .3995 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.48305 .48013 m .47395 .46987 L .47522 .3995 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.48305 .48013 m .47395 .46987 L .47522 .3995 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.47522 .3995 m .47395 .46987 L .49343 .46313 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.49421 .39266 m .47522 .3995 L .46623 .38895 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49343 .46313 m .49421 .39266 L .47522 .3995 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.47522 .3995 m .47395 .46987 L .49343 .46313 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.44731 .39582 m .46623 .38895 L .47522 .3995 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46623 .38895 m .47522 .3995 L .47395 .46987 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.46623 .38895 m .47522 .3995 L .47395 .46987 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.75359 .50524 m .73358 .51164 L .72615 .50177 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.70621 .5082 m .72615 .50177 L .73358 .51164 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.41657 .46261 m .41928 .39213 L .40057 .39898 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.40696 .47962 m .39739 .46935 L .40057 .39898 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.40696 .47962 m .39739 .46935 L .40057 .39898 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.40057 .39898 m .39739 .46935 L .41657 .46261 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.41928 .39213 m .40057 .39898 L .39114 .38842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41657 .46261 m .41928 .39213 L .40057 .39898 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.40057 .39898 m .39739 .46935 L .41657 .46261 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.3725 .3953 m .39114 .38842 L .40057 .39898 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39114 .38842 m .40057 .39898 L .39739 .46935 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.39114 .38842 m .40057 .39898 L .39739 .46935 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.67873 .50474 m .659 .51115 L .65113 .50127 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.63147 .50771 m .65113 .50127 L .659 .51115 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.84105 .3916 m .82069 .39845 L .81379 .38789 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84105 .3916 m .82069 .39845 L .81379 .38789 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79349 .39477 m .81379 .38789 L .82069 .39845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79349 .39477 m .81379 .38789 L .82069 .39845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33478 .53629 m .3443 .3916 L .32588 .39845 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.32577 .55299 m .31542 .54291 L .32588 .39845 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.32577 .55299 m .31542 .54291 L .32588 .39845 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.32588 .39845 m .31542 .54291 L .33478 .53629 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.3443 .3916 m .32588 .39845 L .31599 .38789 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33966 .46209 m .3443 .3916 L .32588 .39845 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.32588 .39845 m .32078 .46883 L .33966 .46209 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.29764 .39477 m .31599 .38789 L .32588 .39845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31599 .38789 m .32588 .39845 L .32078 .46883 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.31599 .38789 m .32588 .39845 L .31542 .54291 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.60382 .50425 m .58438 .51066 L .57605 .50077 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.55668 .50721 m .57605 .50077 L .58438 .51066 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.76618 .39107 m .7461 .39793 L .73875 .38736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76618 .39107 m .7461 .39793 L .73875 .38736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71874 .39424 m .73875 .38736 L .7461 .39793 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71874 .39424 m .73875 .38736 L .7461 .39793 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25579 .53578 m .26928 .39107 L .25114 .39793 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.2476 .55249 m .23676 .5424 L .25114 .39793 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.25114 .39793 m .23676 .5424 L .25579 .53578 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.26928 .39107 m .25114 .39793 L .2408 .38736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25579 .53578 m .26928 .39107 L .25114 .39793 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.25114 .39793 m .23676 .5424 L .25579 .53578 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.2408 .38736 m .25114 .39793 L .23676 .5424 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.62611 .72961 m .60658 .73515 L .59839 .72661 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.57893 .73217 m .59839 .72661 L .60658 .73515 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.69126 .39054 m .67147 .3974 L .66366 .38683 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69126 .39054 m .67147 .3974 L .66366 .38683 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64394 .39372 m .66366 .38683 L .67147 .3974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64394 .39372 m .66366 .38683 L .67147 .3974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55115 .72918 m .53191 .73472 L .52326 .72618 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.71358 .61599 m .6937 .62198 L .68603 .61275 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.66621 .61876 m .68603 .61275 L .6937 .62198 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.61865 .46052 m .6163 .39001 L .59679 .39688 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.607 .47757 m .59864 .46728 L .59679 .39688 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.607 .47757 m .59864 .46728 L .59679 .39688 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.59679 .39688 m .59864 .46728 L .61865 .46052 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.6163 .39001 m .59679 .39688 L .58853 .38629 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6163 .39001 m .59679 .39688 L .58853 .38629 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5691 .39319 m .58853 .38629 L .59679 .39688 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58853 .38629 m .59679 .39688 L .59864 .46728 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.82841 .50573 m .84886 .49928 L .85556 .50919 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.63861 .61553 m .61902 .62152 L .61089 .61229 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.59136 .6183 m .61089 .61229 L .61902 .62152 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.54171 .46 m .54129 .38948 L .52207 .39635 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.53083 .47706 m .522 .46676 L .52207 .39635 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.53083 .47706 m .522 .46676 L .52207 .39635 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.52207 .39635 m .522 .46676 L .54171 .46 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.54129 .38948 m .52207 .39635 L .51336 .38576 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54171 .46 m .54129 .38948 L .52207 .39635 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.52207 .39635 m .522 .46676 L .54171 .46 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.49421 .39266 m .51336 .38576 L .52207 .39635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51336 .38576 m .52207 .39635 L .522 .46676 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.51336 .38576 m .52207 .39635 L .522 .46676 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.80114 .50226 m .78091 .50869 L .77375 .49878 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.75359 .50524 m .77375 .49878 L .78091 .50869 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.56359 .61507 m .54429 .62106 L .53571 .61182 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.46473 .45948 m .46623 .38895 L .44731 .39582 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.45462 .47655 m .44532 .46624 L .44731 .39582 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.45462 .47655 m .44532 .46624 L .44731 .39582 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.44731 .39582 m .44532 .46624 L .46473 .45948 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.46623 .38895 m .44731 .39582 L .43814 .38523 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46473 .45948 m .46623 .38895 L .44731 .39582 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.44731 .39582 m .44532 .46624 L .46473 .45948 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.41928 .39213 m .43814 .38523 L .44731 .39582 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43814 .38523 m .44731 .39582 L .44532 .46624 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.43814 .38523 m .44731 .39582 L .44532 .46624 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.72615 .50177 m .70621 .5082 L .69861 .49828 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.67873 .50474 m .69861 .49828 L .70621 .5082 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.84105 .3916 m .86157 .3847 L .8682 .3953 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84105 .3916 m .86157 .3847 L .8682 .3953 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38769 .45896 m .39114 .38842 L .3725 .3953 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.37836 .47604 m .36859 .46573 L .3725 .3953 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.37453 .54998 m .36448 .53986 L .3725 .3953 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.3725 .3953 m .36859 .46573 L .38769 .45896 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.39114 .38842 m .3725 .3953 L .36287 .3847 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38769 .45896 m .39114 .38842 L .3725 .3953 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.3725 .3953 m .36859 .46573 L .38769 .45896 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.3443 .3916 m .36287 .3847 L .3725 .3953 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36287 .3847 m .3725 .3953 L .36859 .46573 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.36287 .3847 m .3725 .3953 L .36859 .46573 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.65113 .50127 m .63147 .50771 L .62341 .49778 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.60382 .50425 m .62341 .49778 L .63147 .50771 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.51084 .84035 m .49173 .84546 L .48282 .83758 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.81379 .38789 m .79349 .39477 L .78641 .38416 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81379 .38789 m .79349 .39477 L .78641 .38416 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76618 .39107 m .78641 .38416 L .79349 .39477 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76618 .39107 m .78641 .38416 L .79349 .39477 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30495 .5327 m .31599 .38789 L .29764 .39477 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.29623 .54947 m .28567 .53935 L .29764 .39477 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.29623 .54947 m .28567 .53935 L .29764 .39477 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.29764 .39477 m .28567 .53935 L .30495 .5327 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.31599 .38789 m .29764 .39477 L .28756 .38416 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31061 .45843 m .31599 .38789 L .29764 .39477 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.29764 .39477 m .29181 .46521 L .31061 .45843 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.26928 .39107 m .28756 .38416 L .29764 .39477 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28756 .38416 m .29764 .39477 L .29181 .46521 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.28756 .38416 m .29764 .39477 L .28567 .53935 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.57605 .50077 m .55668 .50721 L .54817 .49728 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.52887 .50375 m .54817 .49728 L .55668 .50721 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.73875 .38736 m .71874 .39424 L .7112 .38363 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73875 .38736 m .71874 .39424 L .7112 .38363 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69126 .39054 m .7112 .38363 L .71874 .39424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69126 .39054 m .7112 .38363 L .71874 .39424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59839 .72661 m .57893 .73217 L .57055 .7236 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.55115 .72918 m .57055 .7236 L .57893 .73217 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.76112 .61322 m .74102 .61922 L .73362 .60996 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.71358 .61599 m .73362 .60996 L .74102 .61922 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.66366 .38683 m .64394 .39372 L .63595 .3831 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66366 .38683 m .64394 .39372 L .63595 .3831 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6163 .39001 m .63595 .3831 L .64394 .39372 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6163 .39001 m .63595 .3831 L .64394 .39372 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52326 .72618 m .50409 .73174 L .49525 .72316 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.68603 .61275 m .66621 .61876 L .65836 .6095 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.63861 .61553 m .65836 .6095 L .66621 .61876 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.59018 .45686 m .58853 .38629 L .5691 .39319 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.57879 .47398 m .57024 .46365 L .5691 .39319 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.57879 .47398 m .57024 .46365 L .5691 .39319 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.5691 .39319 m .57024 .46365 L .59018 .45686 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.58853 .38629 m .5691 .39319 L .56065 .38256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58853 .38629 m .5691 .39319 L .56065 .38256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54129 .38948 m .56065 .38256 L .5691 .39319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56065 .38256 m .5691 .39319 L .57024 .46365 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.80114 .50226 m .82153 .49578 L .82841 .50573 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.61089 .61229 m .59136 .6183 L .58305 .60903 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.56359 .61507 m .58305 .60903 L .59136 .6183 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.51306 .45634 m .51336 .38576 L .49421 .39266 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.50245 .47347 m .49343 .46313 L .49421 .39266 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.50245 .47347 m .49343 .46313 L .49421 .39266 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.49421 .39266 m .49343 .46313 L .51306 .45634 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.51336 .38576 m .49421 .39266 L .48531 .38203 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51306 .45634 m .51336 .38576 L .49421 .39266 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.49421 .39266 m .49343 .46313 L .51306 .45634 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.46623 .38895 m .48531 .38203 L .49421 .39266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48531 .38203 m .49421 .39266 L .49343 .46313 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.48531 .38203 m .49421 .39266 L .49343 .46313 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.77375 .49878 m .75359 .50524 L .74626 .49528 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.72615 .50177 m .74626 .49528 L .75359 .50524 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.53571 .61182 m .51647 .61784 L .5077 .60857 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.4359 .45581 m .43814 .38523 L .41928 .39213 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.42607 .47296 m .41657 .46261 L .41928 .39213 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.42607 .47296 m .41657 .46261 L .41928 .39213 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.41928 .39213 m .41657 .46261 L .4359 .45581 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.43814 .38523 m .41928 .39213 L .40992 .38149 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4359 .45581 m .43814 .38523 L .41928 .39213 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.41928 .39213 m .41657 .46261 L .4359 .45581 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.39114 .38842 m .40992 .38149 L .41928 .39213 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40992 .38149 m .41928 .39213 L .41657 .46261 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.40992 .38149 m .41928 .39213 L .41657 .46261 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.69861 .49828 m .67873 .50474 L .67094 .49478 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.65113 .50127 m .67094 .49478 L .67873 .50474 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.86157 .3847 m .84105 .3916 L .83424 .38096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.86157 .3847 m .84105 .3916 L .83424 .38096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81379 .38789 m .83424 .38096 L .84105 .3916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81379 .38789 m .83424 .38096 L .84105 .3916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35869 .45529 m .36287 .3847 L .3443 .3916 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.34963 .47244 m .33966 .46209 L .3443 .3916 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.34504 .54645 m .33478 .53629 L .3443 .3916 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3443 .3916 m .33966 .46209 L .35869 .45529 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.36287 .3847 m .3443 .3916 L .33449 .38096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35869 .45529 m .36287 .3847 L .3443 .3916 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.3443 .3916 m .33966 .46209 L .35869 .45529 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.31599 .38789 m .33449 .38096 L .3443 .3916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33449 .38096 m .3443 .3916 L .33966 .46209 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.33449 .38096 m .3443 .3916 L .33966 .46209 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.62341 .49778 m .60382 .50425 L .59558 .49428 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.57605 .50077 m .59558 .49428 L .60382 .50425 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.78641 .38416 m .76618 .39107 L .75892 .38042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78641 .38416 m .76618 .39107 L .75892 .38042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73875 .38736 m .75892 .38042 L .76618 .39107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73875 .38736 m .75892 .38042 L .76618 .39107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27499 .5291 m .28756 .38416 L .26928 .39107 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.26656 .54595 m .25579 .53578 L .26928 .39107 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.26656 .54595 m .25579 .53578 L .26928 .39107 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.26928 .39107 m .25579 .53578 L .27499 .5291 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.28756 .38416 m .26928 .39107 L .25901 .38042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28144 .45476 m .28756 .38416 L .26928 .39107 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.26928 .39107 m .26271 .46157 L .28144 .45476 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.2408 .38736 m .25901 .38042 L .26928 .39107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25901 .38042 m .26928 .39107 L .26271 .46157 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.25901 .38042 m .26928 .39107 L .25579 .53578 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.64579 .72403 m .62611 .72961 L .618 .721 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.59839 .72661 m .618 .721 L .62611 .72961 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.54817 .49728 m .52887 .50375 L .52017 .49378 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.7112 .38363 m .69126 .39054 L .68354 .37988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7112 .38363 m .69126 .39054 L .68354 .37988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66366 .38683 m .68354 .37988 L .69126 .39054 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66366 .38683 m .68354 .37988 L .69126 .39054 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57055 .7236 m .55115 .72918 L .54259 .72057 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.52326 .72618 m .54259 .72057 L .55115 .72918 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.73362 .60996 m .71358 .61599 L .706 .6067 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.68603 .61275 m .706 .6067 L .71358 .61599 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.63595 .3831 m .6163 .39001 L .60812 .37935 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63595 .3831 m .6163 .39001 L .60812 .37935 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62692 .4709 m .61865 .46052 L .6163 .39001 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.58853 .38629 m .60812 .37935 L .6163 .39001 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58853 .38629 m .60812 .37935 L .6163 .39001 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49525 .72316 m .47615 .72875 L .46712 .72014 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.65836 .6095 m .63861 .61553 L .63057 .60623 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.61089 .61229 m .63057 .60623 L .63861 .61553 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.56158 .45318 m .56065 .38256 L .54129 .38948 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.55045 .47038 m .54171 .46 L .54129 .38948 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.55045 .47038 m .54171 .46 L .54129 .38948 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.54129 .38948 m .54171 .46 L .56158 .45318 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.56065 .38256 m .54129 .38948 L .53266 .37881 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56065 .38256 m .54129 .38948 L .53266 .37881 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51336 .38576 m .53266 .37881 L .54129 .38948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53266 .37881 m .54129 .38948 L .54171 .46 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.77375 .49878 m .79408 .49227 L .80114 .50226 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.58305 .60903 m .56359 .61507 L .5551 .60576 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.53571 .61182 m .5551 .60576 L .56359 .61507 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.48429 .45266 m .48531 .38203 L .46623 .38895 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.47395 .46987 m .46473 .45948 L .46623 .38895 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.47395 .46987 m .46473 .45948 L .46623 .38895 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.46623 .38895 m .46473 .45948 L .48429 .45266 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.48531 .38203 m .46623 .38895 L .45715 .37827 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48429 .45266 m .48531 .38203 L .46623 .38895 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.46623 .38895 m .46473 .45948 L .48429 .45266 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.43814 .38523 m .45715 .37827 L .46623 .38895 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45715 .37827 m .46623 .38895 L .46473 .45948 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.45715 .37827 m .46623 .38895 L .46473 .45948 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.74626 .49528 m .72615 .50177 L .71864 .49177 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.69861 .49828 m .71864 .49177 L .72615 .50177 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.5077 .60857 m .48853 .61461 L .47958 .60529 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.40695 .45213 m .40992 .38149 L .39114 .38842 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.39739 .46935 m .38769 .45896 L .39114 .38842 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.39739 .46935 m .38769 .45896 L .39114 .38842 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.39114 .38842 m .38769 .45896 L .40695 .45213 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.40992 .38149 m .39114 .38842 L .38159 .37774 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40695 .45213 m .40992 .38149 L .39114 .38842 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.39114 .38842 m .38769 .45896 L .40695 .45213 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.36287 .3847 m .38159 .37774 L .39114 .38842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38159 .37774 m .39114 .38842 L .38769 .45896 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.38159 .37774 m .39114 .38842 L .38769 .45896 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.67094 .49478 m .65113 .50127 L .64316 .49127 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.62341 .49778 m .64316 .49127 L .65113 .50127 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.5301 .8352 m .51084 .84035 L .50201 .8324 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.48282 .83758 m .50201 .8324 L .51084 .84035 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.83424 .38096 m .81379 .38789 L .8068 .3772 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83424 .38096 m .81379 .38789 L .8068 .3772 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78641 .38416 m .8068 .3772 L .81379 .38789 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78641 .38416 m .8068 .3772 L .81379 .38789 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32957 .4516 m .33449 .38096 L .31599 .38789 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.32078 .46883 m .31061 .45843 L .31599 .38789 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.31542 .54291 m .30495 .5327 L .31599 .38789 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.31599 .38789 m .31061 .45843 L .32957 .4516 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.33449 .38096 m .31599 .38789 L .30599 .3772 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32957 .4516 m .33449 .38096 L .31599 .38789 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.31599 .38789 m .31061 .45843 L .32957 .4516 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.28756 .38416 m .30599 .3772 L .31599 .38789 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30599 .3772 m .31599 .38789 L .31061 .45843 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.30599 .3772 m .31599 .38789 L .31061 .45843 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.59558 .49428 m .57605 .50077 L .56762 .49077 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.54817 .49728 m .56762 .49077 L .57605 .50077 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.75892 .38042 m .73875 .38736 L .7313 .37666 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75892 .38042 m .73875 .38736 L .7313 .37666 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7112 .38363 m .7313 .37666 L .73875 .38736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7112 .38363 m .7313 .37666 L .73875 .38736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24489 .52548 m .25901 .38042 L .2408 .38736 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.23676 .5424 m .22578 .53219 L .2408 .38736 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.2408 .38736 m .22578 .53219 L .24489 .52548 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.25901 .38042 m .2408 .38736 L .23034 .37666 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24489 .52548 m .25901 .38042 L .2408 .38736 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.2408 .38736 m .22578 .53219 L .24489 .52548 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.23034 .37666 m .2408 .38736 L .22578 .53219 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.618 .721 m .59839 .72661 L .59009 .71797 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.57055 .7236 m .59009 .71797 L .59839 .72661 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.52017 .49378 m .50094 .50028 L .49205 .49026 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.73362 .60996 m .75381 .60389 L .76112 .61322 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.68354 .37988 m .66366 .38683 L .65576 .37612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68354 .37988 m .66366 .38683 L .65576 .37612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63595 .3831 m .65576 .37612 L .66366 .38683 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63595 .3831 m .65576 .37612 L .66366 .38683 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54259 .72057 m .52326 .72618 L .51451 .71753 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.49525 .72316 m .51451 .71753 L .52326 .72618 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.706 .6067 m .68603 .61275 L .67826 .60342 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.65836 .6095 m .67826 .60342 L .68603 .61275 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.60812 .37935 m .58853 .38629 L .58017 .37558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60812 .37935 m .58853 .38629 L .58017 .37558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59864 .46728 m .59018 .45686 L .58853 .38629 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.56065 .38256 m .58017 .37558 L .58853 .38629 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56065 .38256 m .58017 .37558 L .58853 .38629 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46712 .72014 m .44809 .72575 L .43888 .7171 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.63057 .60623 m .61089 .61229 L .60266 .60295 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.58305 .60903 m .60266 .60295 L .61089 .61229 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.53286 .44949 m .53266 .37881 L .51336 .38576 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.522 .46676 m .51306 .45634 L .51336 .38576 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.522 .46676 m .51306 .45634 L .51336 .38576 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.51336 .38576 m .51306 .45634 L .53286 .44949 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.53266 .37881 m .51336 .38576 L .50454 .37505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53266 .37881 m .51336 .38576 L .50454 .37505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48531 .38203 m .50454 .37505 L .51336 .38576 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50454 .37505 m .51336 .38576 L .51306 .45634 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.74626 .49528 m .76651 .48875 L .77375 .49878 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.5551 .60576 m .53571 .61182 L .52702 .60248 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.5077 .60857 m .52702 .60248 L .53571 .61182 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.45539 .44896 m .45715 .37827 L .43814 .38523 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.44532 .46624 m .4359 .45581 L .43814 .38523 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.44532 .46624 m .4359 .45581 L .43814 .38523 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.43814 .38523 m .4359 .45581 L .45539 .44896 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.45715 .37827 m .43814 .38523 L .42886 .37451 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45539 .44896 m .45715 .37827 L .43814 .38523 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.43814 .38523 m .4359 .45581 L .45539 .44896 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.40992 .38149 m .42886 .37451 L .43814 .38523 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42886 .37451 m .43814 .38523 L .4359 .45581 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.42886 .37451 m .43814 .38523 L .4359 .45581 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.71864 .49177 m .69861 .49828 L .69091 .48824 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.67094 .49478 m .69091 .48824 L .69861 .49828 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.47958 .60529 m .46048 .61136 L .45133 .60201 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.83424 .38096 m .85486 .37397 L .86157 .3847 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83424 .38096 m .85486 .37397 L .86157 .3847 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37788 .44843 m .38159 .37774 L .36287 .3847 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.36859 .46573 m .35869 .45529 L .36287 .3847 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.36859 .46573 m .35869 .45529 L .36287 .3847 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.36287 .3847 m .35869 .45529 L .37788 .44843 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.38159 .37774 m .36287 .3847 L .35313 .37397 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37788 .44843 m .38159 .37774 L .36287 .3847 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.36287 .3847 m .35869 .45529 L .37788 .44843 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.33449 .38096 m .35313 .37397 L .36287 .3847 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35313 .37397 m .36287 .3847 L .35869 .45529 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.35313 .37397 m .36287 .3847 L .35869 .45529 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.64316 .49127 m .62341 .49778 L .61525 .48774 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.59558 .49428 m .61525 .48774 L .62341 .49778 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.50201 .8324 m .48282 .83758 L .47381 .8296 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.8068 .3772 m .78641 .38416 L .77924 .37343 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.8068 .3772 m .78641 .38416 L .77924 .37343 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75892 .38042 m .77924 .37343 L .78641 .38416 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75892 .38042 m .77924 .37343 L .78641 .38416 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30032 .4479 m .30599 .3772 L .28756 .38416 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.29181 .46521 m .28144 .45476 L .28756 .38416 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.28567 .53935 m .27499 .5291 L .28756 .38416 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.28756 .38416 m .28144 .45476 L .30032 .4479 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.30599 .3772 m .28756 .38416 L .27736 .37343 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30032 .4479 m .30599 .3772 L .28756 .38416 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.28756 .38416 m .28144 .45476 L .30032 .4479 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.25901 .38042 m .27736 .37343 L .28756 .38416 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27736 .37343 m .28756 .38416 L .28144 .45476 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.27736 .37343 m .28756 .38416 L .28144 .45476 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.66563 .7184 m .64579 .72403 L .63777 .71536 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.618 .721 m .63777 .71536 L .64579 .72403 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.56762 .49077 m .54817 .49728 L .53955 .48723 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.52017 .49378 m .53955 .48723 L .54817 .49728 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.7313 .37666 m .7112 .38363 L .70358 .37289 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7313 .37666 m .7112 .38363 L .70358 .37289 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68354 .37988 m .70358 .37289 L .7112 .38363 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68354 .37988 m .70358 .37289 L .7112 .38363 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59009 .71797 m .57055 .7236 L .56206 .71492 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.54259 .72057 m .56206 .71492 L .57055 .7236 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.49205 .49026 m .47288 .49678 L .46381 .48673 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.706 .6067 m .72612 .60059 L .73362 .60996 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.65576 .37612 m .63595 .3831 L .62786 .37235 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65576 .37612 m .63595 .3831 L .62786 .37235 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60812 .37935 m .62786 .37235 L .63595 .3831 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60812 .37935 m .62786 .37235 L .63595 .3831 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51451 .71753 m .49525 .72316 L .48631 .71448 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.46712 .72014 m .48631 .71448 L .49525 .72316 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.67826 .60342 m .65836 .6095 L .6504 .60012 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.63057 .60623 m .6504 .60012 L .65836 .6095 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.58017 .37558 m .56065 .38256 L .55211 .3718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58017 .37558 m .56065 .38256 L .55211 .3718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57024 .46365 m .56158 .45318 L .56065 .38256 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.53266 .37881 m .55211 .3718 L .56065 .38256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53266 .37881 m .55211 .3718 L .56065 .38256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60266 .60295 m .58305 .60903 L .57464 .59965 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.5551 .60576 m .57464 .59965 L .58305 .60903 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.50401 .44578 m .50454 .37505 L .48531 .38203 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.49343 .46313 m .48429 .45266 L .48531 .38203 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.49343 .46313 m .48429 .45266 L .48531 .38203 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.48531 .38203 m .48429 .45266 L .50401 .44578 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.50454 .37505 m .48531 .38203 L .4763 .37126 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50454 .37505 m .48531 .38203 L .4763 .37126 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45715 .37827 m .4763 .37126 L .48531 .38203 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4763 .37126 m .48531 .38203 L .48429 .45266 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.71864 .49177 m .73883 .48521 L .74626 .49528 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.52702 .60248 m .5077 .60857 L .49883 .59918 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.47958 .60529 m .49883 .59918 L .5077 .60857 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.42637 .44525 m .42886 .37451 L .40992 .38149 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.41657 .46261 m .40695 .45213 L .40992 .38149 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.41657 .46261 m .40695 .45213 L .40992 .38149 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.40992 .38149 m .40695 .45213 L .42637 .44525 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.42886 .37451 m .40992 .38149 L .40045 .37072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42637 .44525 m .42886 .37451 L .40992 .38149 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.40992 .38149 m .40695 .45213 L .42637 .44525 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.38159 .37774 m .40045 .37072 L .40992 .38149 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40045 .37072 m .40992 .38149 L .40695 .45213 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.40045 .37072 m .40992 .38149 L .40695 .45213 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.69091 .48824 m .67094 .49478 L .66306 .4847 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.64316 .49127 m .66306 .4847 L .67094 .49478 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.54951 .83 m .5301 .8352 L .52136 .82719 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.50201 .8324 m .52136 .82719 L .5301 .8352 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.45133 .60201 m .4323 .6081 L .42297 .59871 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.85486 .37397 m .83424 .38096 L .82735 .37018 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.85486 .37397 m .83424 .38096 L .82735 .37018 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.8068 .3772 m .82735 .37018 L .83424 .38096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.8068 .3772 m .82735 .37018 L .83424 .38096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34868 .44472 m .35313 .37397 L .33449 .38096 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.33966 .46209 m .32957 .4516 L .33449 .38096 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.33966 .46209 m .32957 .4516 L .33449 .38096 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.33449 .38096 m .32957 .4516 L .34868 .44472 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.35313 .37397 m .33449 .38096 L .32456 .37018 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34868 .44472 m .35313 .37397 L .33449 .38096 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.33449 .38096 m .32957 .4516 L .34868 .44472 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.30599 .3772 m .32456 .37018 L .33449 .38096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32456 .37018 m .33449 .38096 L .32957 .4516 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.32456 .37018 m .33449 .38096 L .32957 .4516 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.61525 .48774 m .59558 .49428 L .58723 .4842 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.56762 .49077 m .58723 .4842 L .59558 .49428 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.47381 .8296 m .45469 .8348 L .44549 .82678 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.77924 .37343 m .75892 .38042 L .75156 .36964 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77924 .37343 m .75892 .38042 L .75156 .36964 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7313 .37666 m .75156 .36964 L .75892 .38042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7313 .37666 m .75156 .36964 L .75892 .38042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27093 .44419 m .27736 .37343 L .25901 .38042 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.26271 .46157 m .25213 .45108 L .25901 .38042 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.25579 .53578 m .24489 .52548 L .25901 .38042 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.25901 .38042 m .25213 .45108 L .27093 .44419 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.27736 .37343 m .25901 .38042 L .24862 .36964 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27093 .44419 m .27736 .37343 L .25901 .38042 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.25901 .38042 m .25213 .45108 L .27093 .44419 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.23034 .37666 m .24862 .36964 L .25901 .38042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24862 .36964 m .25901 .38042 L .25213 .45108 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.24862 .36964 m .25901 .38042 L .24489 .52548 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.63777 .71536 m .618 .721 L .60979 .7123 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.59009 .71797 m .60979 .7123 L .618 .721 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.53955 .48723 m .52017 .49378 L .51136 .48369 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.49205 .49026 m .51136 .48369 L .52017 .49378 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.70358 .37289 m .68354 .37988 L .67573 .3691 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70358 .37289 m .68354 .37988 L .67573 .3691 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65576 .37612 m .67573 .3691 L .68354 .37988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65576 .37612 m .67573 .3691 L .68354 .37988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56206 .71492 m .54259 .72057 L .53392 .71186 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.51451 .71753 m .53392 .71186 L .54259 .72057 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.46381 .48673 m .44471 .49328 L .43545 .48318 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.67826 .60342 m .69832 .59729 L .706 .6067 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.62786 .37235 m .60812 .37935 L .59985 .36855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62786 .37235 m .60812 .37935 L .59985 .36855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58017 .37558 m .59985 .36855 L .60812 .37935 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58017 .37558 m .59985 .36855 L .60812 .37935 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48631 .71448 m .46712 .72014 L .45799 .71142 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.43888 .7171 m .45799 .71142 L .46712 .72014 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.6504 .60012 m .63057 .60623 L .62243 .59681 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.60266 .60295 m .62243 .59681 L .63057 .60623 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.55211 .3718 m .53266 .37881 L .52392 .36801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55211 .3718 m .53266 .37881 L .52392 .36801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54171 .46 m .53286 .44949 L .53266 .37881 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.50454 .37505 m .52392 .36801 L .53266 .37881 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50454 .37505 m .52392 .36801 L .53266 .37881 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57464 .59965 m .5551 .60576 L .54649 .59634 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.52702 .60248 m .54649 .59634 L .5551 .60576 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.47504 .44206 m .4763 .37126 L .45715 .37827 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.46473 .45948 m .45539 .44896 L .45715 .37827 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.46473 .45948 m .45539 .44896 L .45715 .37827 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.45715 .37827 m .45539 .44896 L .47504 .44206 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.4763 .37126 m .45715 .37827 L .44795 .36747 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4763 .37126 m .45715 .37827 L .44795 .36747 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42886 .37451 m .44795 .36747 L .45715 .37827 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44795 .36747 m .45715 .37827 L .45539 .44896 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.69091 .48824 m .71103 .48165 L .71864 .49177 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.49883 .59918 m .47958 .60529 L .47051 .59587 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.45133 .60201 m .47051 .59587 L .47958 .60529 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.39722 .44152 m .40045 .37072 L .38159 .37774 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.38769 .45896 m .37788 .44843 L .38159 .37774 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.38769 .45896 m .37788 .44843 L .38159 .37774 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.38159 .37774 m .37788 .44843 L .39722 .44152 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.40045 .37072 m .38159 .37774 L .37193 .36692 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39722 .44152 m .40045 .37072 L .38159 .37774 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.38159 .37774 m .37788 .44843 L .39722 .44152 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.35313 .37397 m .37193 .36692 L .38159 .37774 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37193 .36692 m .38159 .37774 L .37788 .44843 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.37193 .36692 m .38159 .37774 L .37788 .44843 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.66306 .4847 m .64316 .49127 L .63509 .48115 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.61525 .48774 m .63509 .48115 L .64316 .49127 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.52136 .82719 m .50201 .8324 L .49308 .82436 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.47381 .8296 m .49308 .82436 L .50201 .8324 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.82735 .37018 m .8068 .3772 L .79973 .36638 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82735 .37018 m .8068 .3772 L .79973 .36638 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77924 .37343 m .79973 .36638 L .8068 .3772 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77924 .37343 m .79973 .36638 L .8068 .3772 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31935 .44099 m .32456 .37018 L .30599 .3772 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.31061 .45843 m .30032 .4479 L .30599 .3772 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.31061 .45843 m .30032 .4479 L .30599 .3772 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.30599 .3772 m .30032 .4479 L .31935 .44099 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.32456 .37018 m .30599 .3772 L .29586 .36638 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31935 .44099 m .32456 .37018 L .30599 .3772 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.30599 .3772 m .30032 .4479 L .31935 .44099 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.27736 .37343 m .29586 .36638 L .30599 .3772 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29586 .36638 m .30599 .3772 L .30032 .4479 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.29586 .36638 m .30599 .3772 L .30032 .4479 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.63777 .71536 m .6577 .70966 L .66563 .7184 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.58723 .4842 m .56762 .49077 L .55909 .48064 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.53955 .48723 m .55909 .48064 L .56762 .49077 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.44549 .82678 m .42644 .832 L .41705 .82396 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.75156 .36964 m .7313 .37666 L .72377 .36583 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75156 .36964 m .7313 .37666 L .72377 .36583 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70358 .37289 m .72377 .36583 L .7313 .37666 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70358 .37289 m .72377 .36583 L .7313 .37666 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23385 .51506 m .24862 .36964 L .23034 .37666 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.22578 .53219 m .21466 .52185 L .23034 .37666 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.23034 .37666 m .21466 .52185 L .23385 .51506 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.24862 .36964 m .23034 .37666 L .21975 .36583 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24143 .44045 m .24862 .36964 L .23034 .37666 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.23034 .37666 m .2227 .44737 L .24143 .44045 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.21975 .36583 m .23034 .37666 L .2227 .44737 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.60979 .7123 m .59009 .71797 L .5817 .70922 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.56206 .71492 m .5817 .70922 L .59009 .71797 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.51136 .48369 m .49205 .49026 L .48305 .48013 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.46381 .48673 m .48305 .48013 L .49205 .49026 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.67573 .3691 m .65576 .37612 L .64776 .36529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67573 .3691 m .65576 .37612 L .64776 .36529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62786 .37235 m .64776 .36529 L .65576 .37612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62786 .37235 m .64776 .36529 L .65576 .37612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53392 .71186 m .51451 .71753 L .50565 .70878 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.48631 .71448 m .50565 .70878 L .51451 .71753 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.43545 .48318 m .41643 .48976 L .40696 .47962 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.6504 .60012 m .6704 .59397 L .67826 .60342 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.59985 .36855 m .58017 .37558 L .57171 .36474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59985 .36855 m .58017 .37558 L .57171 .36474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55211 .3718 m .57171 .36474 L .58017 .37558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55211 .3718 m .57171 .36474 L .58017 .37558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45799 .71142 m .43888 .7171 L .42955 .70834 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.62243 .59681 m .60266 .60295 L .59434 .59349 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.57464 .59965 m .59434 .59349 L .60266 .60295 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.52392 .36801 m .50454 .37505 L .49561 .3642 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52392 .36801 m .50454 .37505 L .49561 .3642 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51306 .45634 m .50401 .44578 L .50454 .37505 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.4763 .37126 m .49561 .3642 L .50454 .37505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4763 .37126 m .49561 .3642 L .50454 .37505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54649 .59634 m .52702 .60248 L .51823 .59302 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.49883 .59918 m .51823 .59302 L .52702 .60248 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.44594 .43831 m .44795 .36747 L .42886 .37451 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.4359 .45581 m .42637 .44525 L .42886 .37451 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.4359 .45581 m .42637 .44525 L .42886 .37451 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.42886 .37451 m .42637 .44525 L .44594 .43831 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.44795 .36747 m .42886 .37451 L .41947 .36365 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44795 .36747 m .42886 .37451 L .41947 .36365 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40045 .37072 m .41947 .36365 L .42886 .37451 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41947 .36365 m .42886 .37451 L .42637 .44525 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.66306 .4847 m .68311 .47808 L .69091 .48824 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.56908 .82476 m .54951 .83 L .54085 .82193 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.52136 .82719 m .54085 .82193 L .54951 .83 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.47051 .59587 m .45133 .60201 L .44208 .59254 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.42297 .59871 m .44208 .59254 L .45133 .60201 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.82735 .37018 m .84807 .36311 L .85486 .37397 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82735 .37018 m .84807 .36311 L .85486 .37397 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36794 .43778 m .37193 .36692 L .35313 .37397 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.35869 .45529 m .34868 .44472 L .35313 .37397 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.35869 .45529 m .34868 .44472 L .35313 .37397 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.35313 .37397 m .34868 .44472 L .36794 .43778 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.37193 .36692 m .35313 .37397 L .34328 .36311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36794 .43778 m .37193 .36692 L .35313 .37397 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.35313 .37397 m .34868 .44472 L .36794 .43778 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.32456 .37018 m .34328 .36311 L .35313 .37397 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34328 .36311 m .35313 .37397 L .34868 .44472 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.34328 .36311 m .35313 .37397 L .34868 .44472 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.63509 .48115 m .61525 .48774 L .607 .47757 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.58723 .4842 m .607 .47757 L .61525 .48774 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.49308 .82436 m .47381 .8296 L .46469 .82152 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.44549 .82678 m .46469 .82152 L .47381 .8296 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.79973 .36638 m .77924 .37343 L .77198 .36256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79973 .36638 m .77924 .37343 L .77198 .36256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75156 .36964 m .77198 .36256 L .77924 .37343 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75156 .36964 m .77198 .36256 L .77924 .37343 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60979 .7123 m .62965 .70658 L .63777 .71536 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.28989 .43724 m .29586 .36638 L .27736 .37343 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.28144 .45476 m .27093 .44419 L .27736 .37343 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.28144 .45476 m .27093 .44419 L .27736 .37343 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.27736 .37343 m .27093 .44419 L .28989 .43724 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.29586 .36638 m .27736 .37343 L .26704 .36256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28989 .43724 m .29586 .36638 L .27736 .37343 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.27736 .37343 m .27093 .44419 L .28989 .43724 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.24862 .36964 m .26704 .36256 L .27736 .37343 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26704 .36256 m .27736 .37343 L .27093 .44419 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.26704 .36256 m .27736 .37343 L .27093 .44419 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.55909 .48064 m .53955 .48723 L .53083 .47706 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.51136 .48369 m .53083 .47706 L .53955 .48723 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.41705 .82396 m .39807 .8292 L .38848 .82112 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.72377 .36583 m .70358 .37289 L .69585 .36201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72377 .36583 m .70358 .37289 L .69585 .36201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67573 .3691 m .69585 .36201 L .70358 .37289 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67573 .3691 m .69585 .36201 L .70358 .37289 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5817 .70922 m .56206 .71492 L .55348 .70614 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.53392 .71186 m .55348 .70614 L .56206 .71492 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.48305 .48013 m .46381 .48673 L .45462 .47655 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.43545 .48318 m .45462 .47655 L .46381 .48673 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.64776 .36529 m .62786 .37235 L .61968 .36146 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64776 .36529 m .62786 .37235 L .61968 .36146 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59985 .36855 m .61968 .36146 L .62786 .37235 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59985 .36855 m .61968 .36146 L .62786 .37235 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50565 .70878 m .48631 .71448 L .47726 .70569 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.45799 .71142 m .47726 .70569 L .48631 .71448 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.40696 .47962 m .38802 .48622 L .37836 .47604 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.62243 .59681 m .64235 .59063 L .6504 .60012 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.57171 .36474 m .55211 .3718 L .54345 .36092 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57171 .36474 m .55211 .3718 L .54345 .36092 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52392 .36801 m .54345 .36092 L .55211 .3718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52392 .36801 m .54345 .36092 L .55211 .3718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42955 .70834 m .41051 .71405 L .40099 .70525 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.59434 .59349 m .57464 .59965 L .56612 .59016 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.54649 .59634 m .56612 .59016 L .57464 .59965 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.45215 .93716 m .43301 .94196 L .42363 .93456 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.49561 .3642 m .4763 .37126 L .46718 .36037 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49561 .3642 m .4763 .37126 L .46718 .36037 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48429 .45266 m .47504 .44206 L .4763 .37126 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.44795 .36747 m .46718 .36037 L .4763 .37126 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44795 .36747 m .46718 .36037 L .4763 .37126 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51823 .59302 m .49883 .59918 L .48984 .58968 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.47051 .59587 m .48984 .58968 L .49883 .59918 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.41672 .43456 m .41947 .36365 L .40045 .37072 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.40695 .45213 m .39722 .44152 L .40045 .37072 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.40695 .45213 m .39722 .44152 L .40045 .37072 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.40045 .37072 m .39722 .44152 L .41672 .43456 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.41947 .36365 m .40045 .37072 L .39087 .35982 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41947 .36365 m .40045 .37072 L .39087 .35982 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37193 .36692 m .39087 .35982 L .40045 .37072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39087 .35982 m .40045 .37072 L .39722 .44152 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.63509 .48115 m .65508 .4745 L .66306 .4847 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.54085 .82193 m .52136 .82719 L .51251 .81908 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.49308 .82436 m .51251 .81908 L .52136 .82719 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.44208 .59254 m .42297 .59871 L .41352 .5892 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.84807 .36311 m .82735 .37018 L .82038 .35927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84807 .36311 m .82735 .37018 L .82038 .35927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79973 .36638 m .82038 .35927 L .82735 .37018 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79973 .36638 m .82038 .35927 L .82735 .37018 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33853 .43402 m .34328 .36311 L .32456 .37018 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.32957 .4516 m .31935 .44099 L .32456 .37018 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.32957 .4516 m .31935 .44099 L .32456 .37018 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.32456 .37018 m .31935 .44099 L .33853 .43402 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.34328 .36311 m .32456 .37018 L .3145 .35927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33853 .43402 m .34328 .36311 L .32456 .37018 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.32456 .37018 m .31935 .44099 L .33853 .43402 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.29586 .36638 m .3145 .35927 L .32456 .37018 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3145 .35927 m .32456 .37018 L .31935 .44099 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3145 .35927 m .32456 .37018 L .31935 .44099 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.607 .47757 m .58723 .4842 L .57879 .47398 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.55909 .48064 m .57879 .47398 L .58723 .4842 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.46469 .82152 m .44549 .82678 L .43618 .81867 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.41705 .82396 m .43618 .81867 L .44549 .82678 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.77198 .36256 m .75156 .36964 L .74412 .35872 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77198 .36256 m .75156 .36964 L .74412 .35872 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72377 .36583 m .74412 .35872 L .75156 .36964 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72377 .36583 m .74412 .35872 L .75156 .36964 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2603 .43348 m .26704 .36256 L .24862 .36964 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.25213 .45108 m .24143 .44045 L .24862 .36964 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.24489 .52548 m .23385 .51506 L .24862 .36964 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.24862 .36964 m .24143 .44045 L .2603 .43348 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.26704 .36256 m .24862 .36964 L .23809 .35872 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2603 .43348 m .26704 .36256 L .24862 .36964 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.24862 .36964 m .24143 .44045 L .2603 .43348 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.21975 .36583 m .23809 .35872 L .24862 .36964 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23809 .35872 m .24862 .36964 L .24143 .44045 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.23809 .35872 m .24862 .36964 L .24143 .44045 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.5817 .70922 m .60148 .70348 L .60979 .7123 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.53083 .47706 m .51136 .48369 L .50245 .47347 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.48305 .48013 m .50245 .47347 L .51136 .48369 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.69585 .36201 m .67573 .3691 L .66782 .35817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69585 .36201 m .67573 .3691 L .66782 .35817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64776 .36529 m .66782 .35817 L .67573 .3691 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64776 .36529 m .66782 .35817 L .67573 .3691 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55348 .70614 m .53392 .71186 L .52514 .70304 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.50565 .70878 m .52514 .70304 L .53392 .71186 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.45462 .47655 m .43545 .48318 L .42607 .47296 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.40696 .47962 m .42607 .47296 L .43545 .48318 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.61968 .36146 m .59985 .36855 L .59147 .35762 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61968 .36146 m .59985 .36855 L .59147 .35762 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57171 .36474 m .59147 .35762 L .59985 .36855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57171 .36474 m .59147 .35762 L .59985 .36855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47726 .70569 m .45799 .71142 L .44875 .70259 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.42955 .70834 m .44875 .70259 L .45799 .71142 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.59434 .59349 m .61419 .58728 L .62243 .59681 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.54345 .36092 m .52392 .36801 L .51508 .35707 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54345 .36092 m .52392 .36801 L .51508 .35707 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49561 .3642 m .51508 .35707 L .52392 .36801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49561 .3642 m .51508 .35707 L .52392 .36801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40099 .70525 m .38202 .71098 L .37231 .70215 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.56612 .59016 m .54649 .59634 L .53779 .5868 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.51823 .59302 m .53779 .5868 L .54649 .59634 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.42363 .93456 m .40456 .93938 L .39499 .93195 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.46718 .36037 m .44795 .36747 L .43863 .35652 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46718 .36037 m .44795 .36747 L .43863 .35652 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45539 .44896 m .44594 .43831 L .44795 .36747 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.41947 .36365 m .43863 .35652 L .44795 .36747 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41947 .36365 m .43863 .35652 L .44795 .36747 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54085 .82193 m .56051 .81663 L .56908 .82476 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.48984 .58968 m .47051 .59587 L .46133 .58632 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.44208 .59254 m .46133 .58632 L .47051 .59587 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.38736 .43078 m .39087 .35982 L .37193 .36692 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.37788 .44843 m .36794 .43778 L .37193 .36692 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.37788 .44843 m .36794 .43778 L .37193 .36692 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.37193 .36692 m .36794 .43778 L .38736 .43078 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.39087 .35982 m .37193 .36692 L .36214 .35597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39087 .35982 m .37193 .36692 L .36214 .35597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34328 .36311 m .36214 .35597 L .37193 .36692 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36214 .35597 m .37193 .36692 L .36794 .43778 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.607 .47757 m .62692 .4709 L .63509 .48115 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.51251 .81908 m .49308 .82436 L .48405 .81622 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.46469 .82152 m .48405 .81622 L .49308 .82436 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.41352 .5892 m .39448 .59539 L .38484 .58584 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.82038 .35927 m .79973 .36638 L .79257 .35542 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82038 .35927 m .79973 .36638 L .79257 .35542 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77198 .36256 m .79257 .35542 L .79973 .36638 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77198 .36256 m .79257 .35542 L .79973 .36638 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.309 .43024 m .3145 .35927 L .29586 .36638 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.30032 .4479 m .28989 .43724 L .29586 .36638 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.30032 .4479 m .28989 .43724 L .29586 .36638 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.29586 .36638 m .28989 .43724 L .309 .43024 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.3145 .35927 m .29586 .36638 L .28561 .35542 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.309 .43024 m .3145 .35927 L .29586 .36638 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.29586 .36638 m .28989 .43724 L .309 .43024 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.26704 .36256 m .28561 .35542 L .29586 .36638 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28561 .35542 m .29586 .36638 L .28989 .43724 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.28561 .35542 m .29586 .36638 L .28989 .43724 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.57879 .47398 m .55909 .48064 L .55045 .47038 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.53083 .47706 m .55045 .47038 L .55909 .48064 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.43618 .81867 m .41705 .82396 L .40754 .81581 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.38848 .82112 m .40754 .81581 L .41705 .82396 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.74412 .35872 m .72377 .36583 L .71614 .35487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74412 .35872 m .72377 .36583 L .71614 .35487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69585 .36201 m .71614 .35487 L .72377 .36583 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69585 .36201 m .71614 .35487 L .72377 .36583 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23058 .4297 m .23809 .35872 L .21975 .36583 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.2227 .44737 m .21179 .43671 L .21975 .36583 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.21975 .36583 m .21179 .43671 L .23058 .4297 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.23809 .35872 m .21975 .36583 L .20903 .35487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23058 .4297 m .23809 .35872 L .21975 .36583 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.21975 .36583 m .21179 .43671 L .23058 .4297 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.20903 .35487 m .21975 .36583 L .21179 .43671 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.55348 .70614 m .5732 .70037 L .5817 .70922 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.50245 .47347 m .48305 .48013 L .47395 .46987 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.45462 .47655 m .47395 .46987 L .48305 .48013 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.66782 .35817 m .64776 .36529 L .63967 .35432 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66782 .35817 m .64776 .36529 L .63967 .35432 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61968 .36146 m .63967 .35432 L .64776 .36529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61968 .36146 m .63967 .35432 L .64776 .36529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52514 .70304 m .50565 .70878 L .49668 .69992 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.47726 .70569 m .49668 .69992 L .50565 .70878 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.42607 .47296 m .40696 .47962 L .39739 .46935 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.37836 .47604 m .39739 .46935 L .40696 .47962 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.59147 .35762 m .57171 .36474 L .56314 .35377 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59147 .35762 m .57171 .36474 L .56314 .35377 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54345 .36092 m .56314 .35377 L .57171 .36474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54345 .36092 m .56314 .35377 L .57171 .36474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44875 .70259 m .42955 .70834 L .42011 .69948 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.40099 .70525 m .42011 .69948 L .42955 .70834 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.56612 .59016 m .58591 .58392 L .59434 .59349 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.47143 .93232 m .45215 .93716 L .44284 .9297 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.42363 .93456 m .44284 .9297 L .45215 .93716 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.51508 .35707 m .49561 .3642 L .48658 .35322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51508 .35707 m .49561 .3642 L .48658 .35322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46718 .36037 m .48658 .35322 L .49561 .3642 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46718 .36037 m .48658 .35322 L .49561 .3642 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53779 .5868 m .51823 .59302 L .50933 .58344 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.48984 .58968 m .50933 .58344 L .51823 .59302 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.43863 .35652 m .41947 .36365 L .40996 .35266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43863 .35652 m .41947 .36365 L .40996 .35266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42637 .44525 m .41672 .43456 L .41947 .36365 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.39087 .35982 m .40996 .35266 L .41947 .36365 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39087 .35982 m .40996 .35266 L .41947 .36365 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56051 .81663 m .54085 .82193 L .53209 .81376 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.51251 .81908 m .53209 .81376 L .54085 .82193 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.46133 .58632 m .44208 .59254 L .4327 .58296 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.41352 .5892 m .4327 .58296 L .44208 .59254 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.82038 .35927 m .84119 .35211 L .84807 .36311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82038 .35927 m .84119 .35211 L .84807 .36311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35788 .42699 m .36214 .35597 L .34328 .36311 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.34868 .44472 m .33853 .43402 L .34328 .36311 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.34868 .44472 m .33853 .43402 L .34328 .36311 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.34328 .36311 m .33853 .43402 L .35788 .42699 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.36214 .35597 m .34328 .36311 L .3333 .35211 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36214 .35597 m .34328 .36311 L .3333 .35211 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3145 .35927 m .3333 .35211 L .34328 .36311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3333 .35211 m .34328 .36311 L .33853 .43402 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.62692 .4709 m .607 .47757 L .59864 .46728 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.57879 .47398 m .59864 .46728 L .607 .47757 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.48405 .81622 m .46469 .82152 L .45546 .81334 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.43618 .81867 m .45546 .81334 L .46469 .82152 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.38484 .58584 m .36588 .59206 L .35603 .58247 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.79257 .35542 m .77198 .36256 L .76464 .35156 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79257 .35542 m .77198 .36256 L .76464 .35156 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74412 .35872 m .76464 .35156 L .77198 .36256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74412 .35872 m .76464 .35156 L .77198 .36256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27933 .42645 m .28561 .35542 L .26704 .36256 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.27093 .44419 m .2603 .43348 L .26704 .36256 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.27093 .44419 m .2603 .43348 L .26704 .36256 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.26704 .36256 m .2603 .43348 L .27933 .42645 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.28561 .35542 m .26704 .36256 L .25659 .35156 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27933 .42645 m .28561 .35542 L .26704 .36256 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.26704 .36256 m .2603 .43348 L .27933 .42645 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.23809 .35872 m .25659 .35156 L .26704 .36256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25659 .35156 m .26704 .36256 L .2603 .43348 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.25659 .35156 m .26704 .36256 L .2603 .43348 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.55045 .47038 m .53083 .47706 L .522 .46676 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.50245 .47347 m .522 .46676 L .53083 .47706 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.40754 .81581 m .38848 .82112 L .37878 .81293 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.71614 .35487 m .69585 .36201 L .68804 .351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71614 .35487 m .69585 .36201 L .68804 .351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66782 .35817 m .68804 .351 L .69585 .36201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66782 .35817 m .68804 .351 L .69585 .36201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52514 .70304 m .54478 .69724 L .55348 .70614 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.47395 .46987 m .45462 .47655 L .44532 .46624 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.42607 .47296 m .44532 .46624 L .45462 .47655 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.63967 .35432 m .61968 .36146 L .61139 .35045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63967 .35432 m .61968 .36146 L .61139 .35045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59147 .35762 m .61139 .35045 L .61968 .36146 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59147 .35762 m .61139 .35045 L .61968 .36146 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49668 .69992 m .47726 .70569 L .46809 .6968 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.44875 .70259 m .46809 .6968 L .47726 .70569 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.39739 .46935 m .37836 .47604 L .36859 .46573 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.56314 .35377 m .54345 .36092 L .5347 .3499 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56314 .35377 m .54345 .36092 L .5347 .3499 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51508 .35707 m .5347 .3499 L .54345 .36092 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51508 .35707 m .5347 .3499 L .54345 .36092 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42011 .69948 m .40099 .70525 L .39135 .69635 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.37231 .70215 m .39135 .69635 L .40099 .70525 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.53779 .5868 m .5575 .58054 L .56612 .59016 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.44284 .9297 m .42363 .93456 L .41413 .92707 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.39499 .93195 m .41413 .92707 L .42363 .93456 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.48658 .35322 m .46718 .36037 L .45795 .34934 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48658 .35322 m .46718 .36037 L .45795 .34934 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43863 .35652 m .45795 .34934 L .46718 .36037 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43863 .35652 m .45795 .34934 L .46718 .36037 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50933 .58344 m .48984 .58968 L .48075 .58006 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.46133 .58632 m .48075 .58006 L .48984 .58968 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.40996 .35266 m .39087 .35982 L .38116 .34879 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40996 .35266 m .39087 .35982 L .38116 .34879 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39722 .44152 m .38736 .43078 L .39087 .35982 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.36214 .35597 m .38116 .34879 L .39087 .35982 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36214 .35597 m .38116 .34879 L .39087 .35982 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53209 .81376 m .51251 .81908 L .50355 .81087 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.48405 .81622 m .50355 .81087 L .51251 .81908 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.4327 .58296 m .41352 .5892 L .40395 .57957 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.38484 .58584 m .40395 .57957 L .41352 .5892 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.84119 .35211 m .82038 .35927 L .81332 .34823 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84119 .35211 m .82038 .35927 L .81332 .34823 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79257 .35542 m .81332 .34823 L .82038 .35927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79257 .35542 m .81332 .34823 L .82038 .35927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32826 .42318 m .3333 .35211 L .3145 .35927 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.31935 .44099 m .309 .43024 L .3145 .35927 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.31935 .44099 m .309 .43024 L .3145 .35927 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3145 .35927 m .309 .43024 L .32826 .42318 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.3333 .35211 m .3145 .35927 L .30433 .34823 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32826 .42318 m .3333 .35211 L .3145 .35927 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.3145 .35927 m .309 .43024 L .32826 .42318 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.28561 .35542 m .30433 .34823 L .3145 .35927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30433 .34823 m .3145 .35927 L .309 .43024 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.30433 .34823 m .3145 .35927 L .309 .43024 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.59864 .46728 m .57879 .47398 L .57024 .46365 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.55045 .47038 m .57024 .46365 L .57879 .47398 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.45546 .81334 m .43618 .81867 L .42675 .81046 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.40754 .81581 m .42675 .81046 L .43618 .81867 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.35603 .58247 m .33715 .58872 L .3271 .57909 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.76464 .35156 m .74412 .35872 L .73659 .34767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76464 .35156 m .74412 .35872 L .73659 .34767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71614 .35487 m .73659 .34767 L .74412 .35872 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71614 .35487 m .73659 .34767 L .74412 .35872 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24953 .42264 m .25659 .35156 L .23809 .35872 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.24143 .44045 m .23058 .4297 L .23809 .35872 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.24143 .44045 m .23058 .4297 L .23809 .35872 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.23809 .35872 m .23058 .4297 L .24953 .42264 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.25659 .35156 m .23809 .35872 L .22744 .34767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24953 .42264 m .25659 .35156 L .23809 .35872 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.23809 .35872 m .23058 .4297 L .24953 .42264 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.20903 .35487 m .22744 .34767 L .23809 .35872 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22744 .34767 m .23809 .35872 L .23058 .4297 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.22744 .34767 m .23809 .35872 L .23058 .4297 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.522 .46676 m .50245 .47347 L .49343 .46313 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.47395 .46987 m .49343 .46313 L .50245 .47347 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.37878 .81293 m .3598 .81826 L .34989 .81005 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.68804 .351 m .66782 .35817 L .65981 .34712 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68804 .351 m .66782 .35817 L .65981 .34712 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63967 .35432 m .65981 .34712 L .66782 .35817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63967 .35432 m .65981 .34712 L .66782 .35817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54478 .69724 m .52514 .70304 L .51625 .69411 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.49668 .69992 m .51625 .69411 L .52514 .70304 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.44532 .46624 m .42607 .47296 L .41657 .46261 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.39739 .46935 m .41657 .46261 L .42607 .47296 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.61139 .35045 m .59147 .35762 L .58299 .34656 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61139 .35045 m .59147 .35762 L .58299 .34656 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56314 .35377 m .58299 .34656 L .59147 .35762 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56314 .35377 m .58299 .34656 L .59147 .35762 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46809 .6968 m .44875 .70259 L .43939 .69366 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.42011 .69948 m .43939 .69366 L .44875 .70259 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.36859 .46573 m .34963 .47244 L .33966 .46209 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.49088 .92744 m .47143 .93232 L .46221 .9248 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.44284 .9297 m .46221 .9248 L .47143 .93232 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.5347 .3499 m .51508 .35707 L .50612 .34601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5347 .3499 m .51508 .35707 L .50612 .34601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48658 .35322 m .50612 .34601 L .51508 .35707 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48658 .35322 m .50612 .34601 L .51508 .35707 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39135 .69635 m .37231 .70215 L .36247 .69321 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.50933 .58344 m .52897 .57715 L .53779 .5868 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.41413 .92707 m .39499 .93195 L .38529 .92442 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.45795 .34934 m .43863 .35652 L .42921 .34545 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45795 .34934 m .43863 .35652 L .42921 .34545 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40996 .35266 m .42921 .34545 L .43863 .35652 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40996 .35266 m .42921 .34545 L .43863 .35652 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53209 .81376 m .55183 .80839 L .56051 .81663 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.48075 .58006 m .46133 .58632 L .45205 .57666 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.4327 .58296 m .45205 .57666 L .46133 .58632 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.38116 .34879 m .36214 .35597 L .35224 .34489 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38116 .34879 m .36214 .35597 L .35224 .34489 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36794 .43778 m .35788 .42699 L .36214 .35597 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.3333 .35211 m .35224 .34489 L .36214 .35597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3333 .35211 m .35224 .34489 L .36214 .35597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59864 .46728 m .61865 .46052 L .62692 .4709 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.50355 .81087 m .48405 .81622 L .47489 .80798 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.45546 .81334 m .47489 .80798 L .48405 .81622 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.40395 .57957 m .38484 .58584 L .37507 .57618 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.35603 .58247 m .37507 .57618 L .38484 .58584 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.81332 .34823 m .79257 .35542 L .78532 .34433 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81332 .34823 m .79257 .35542 L .78532 .34433 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76464 .35156 m .78532 .34433 L .79257 .35542 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76464 .35156 m .78532 .34433 L .79257 .35542 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29852 .41936 m .30433 .34823 L .28561 .35542 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.28989 .43724 m .27933 .42645 L .28561 .35542 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.28989 .43724 m .27933 .42645 L .28561 .35542 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.28561 .35542 m .27933 .42645 L .29852 .41936 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.30433 .34823 m .28561 .35542 L .27523 .34433 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29852 .41936 m .30433 .34823 L .28561 .35542 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.28561 .35542 m .27933 .42645 L .29852 .41936 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.25659 .35156 m .27523 .34433 L .28561 .35542 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27523 .34433 m .28561 .35542 L .27933 .42645 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.27523 .34433 m .28561 .35542 L .27933 .42645 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.57024 .46365 m .55045 .47038 L .54171 .46 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.522 .46676 m .54171 .46 L .55045 .47038 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.42675 .81046 m .40754 .81581 L .39791 .80756 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.37878 .81293 m .39791 .80756 L .40754 .81581 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.73659 .34767 m .71614 .35487 L .70842 .34378 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73659 .34767 m .71614 .35487 L .70842 .34378 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68804 .351 m .70842 .34378 L .71614 .35487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68804 .351 m .70842 .34378 L .71614 .35487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2196 .41881 m .22744 .34767 L .20903 .35487 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.21179 .43671 m .20073 .4259 L .20903 .35487 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.20903 .35487 m .20073 .4259 L .2196 .41881 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.22744 .34767 m .20903 .35487 L .19817 .34378 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2196 .41881 m .22744 .34767 L .20903 .35487 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.20903 .35487 m .20073 .4259 L .2196 .41881 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.19817 .34378 m .20903 .35487 L .20073 .4259 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.49343 .46313 m .47395 .46987 L .46473 .45948 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.44532 .46624 m .46473 .45948 L .47395 .46987 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.65981 .34712 m .63967 .35432 L .63147 .34322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65981 .34712 m .63967 .35432 L .63147 .34322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61139 .35045 m .63147 .34322 L .63967 .35432 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61139 .35045 m .63147 .34322 L .63967 .35432 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51625 .69411 m .49668 .69992 L .4876 .69096 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.46809 .6968 m .4876 .69096 L .49668 .69992 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.41657 .46261 m .39739 .46935 L .38769 .45896 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.36859 .46573 m .38769 .45896 L .39739 .46935 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.58299 .34656 m .56314 .35377 L .55447 .34266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58299 .34656 m .56314 .35377 L .55447 .34266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5347 .3499 m .55447 .34266 L .56314 .35377 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5347 .3499 m .55447 .34266 L .56314 .35377 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43939 .69366 m .42011 .69948 L .41055 .6905 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.39135 .69635 m .41055 .6905 L .42011 .69948 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.33966 .46209 m .32078 .46883 L .31061 .45843 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.46221 .9248 m .44284 .9297 L .43343 .92215 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.41413 .92707 m .43343 .92215 L .44284 .9297 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.50612 .34601 m .48658 .35322 L .47743 .3421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50612 .34601 m .48658 .35322 L .47743 .3421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45795 .34934 m .47743 .3421 L .48658 .35322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45795 .34934 m .47743 .3421 L .48658 .35322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36247 .69321 m .3435 .69903 L .33347 .69005 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.52897 .57715 m .50933 .58344 L .50032 .57374 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.48075 .58006 m .50032 .57374 L .50933 .58344 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.42921 .34545 m .40996 .35266 L .40034 .34154 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42921 .34545 m .40996 .35266 L .40034 .34154 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38116 .34879 m .40034 .34154 L .40996 .35266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38116 .34879 m .40034 .34154 L .40996 .35266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50355 .81087 m .52322 .80548 L .53209 .81376 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.45205 .57666 m .4327 .58296 L .42322 .57326 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.40395 .57957 m .42322 .57326 L .4327 .58296 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.81332 .34823 m .83423 .34098 L .84119 .35211 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81332 .34823 m .83423 .34098 L .84119 .35211 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35224 .34489 m .3333 .35211 L .3232 .34098 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35224 .34489 m .3333 .35211 L .3232 .34098 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33853 .43402 m .32826 .42318 L .3333 .35211 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.30433 .34823 m .3232 .34098 L .3333 .35211 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3232 .34098 m .3333 .35211 L .32826 .42318 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.57024 .46365 m .59018 .45686 L .59864 .46728 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.47489 .80798 m .45546 .81334 L .44611 .80507 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.42675 .81046 m .44611 .80507 L .45546 .81334 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.37507 .57618 m .35603 .58247 L .34607 .57277 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.3271 .57909 m .34607 .57277 L .35603 .58247 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.78532 .34433 m .76464 .35156 L .7572 .34042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78532 .34433 m .76464 .35156 L .7572 .34042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73659 .34767 m .7572 .34042 L .76464 .35156 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73659 .34767 m .7572 .34042 L .76464 .35156 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26864 .41551 m .27523 .34433 L .25659 .35156 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.2603 .43348 m .24953 .42264 L .25659 .35156 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.2603 .43348 m .24953 .42264 L .25659 .35156 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.25659 .35156 m .24953 .42264 L .26864 .41551 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.27523 .34433 m .25659 .35156 L .24601 .34042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26864 .41551 m .27523 .34433 L .25659 .35156 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.25659 .35156 m .24953 .42264 L .26864 .41551 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.22744 .34767 m .24601 .34042 L .25659 .35156 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24601 .34042 m .25659 .35156 L .24953 .42264 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.24601 .34042 m .25659 .35156 L .24953 .42264 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.54171 .46 m .522 .46676 L .51306 .45634 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.49343 .46313 m .51306 .45634 L .522 .46676 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.39791 .80756 m .37878 .81293 L .36895 .80465 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.34989 .81005 m .36895 .80465 L .37878 .81293 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.70842 .34378 m .68804 .351 L .68013 .33986 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70842 .34378 m .68804 .351 L .68013 .33986 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65981 .34712 m .68013 .33986 L .68804 .351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65981 .34712 m .68013 .33986 L .68804 .351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51625 .69411 m .53599 .68824 L .54478 .69724 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.46473 .45948 m .44532 .46624 L .4359 .45581 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.41657 .46261 m .4359 .45581 L .44532 .46624 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.63147 .34322 m .61139 .35045 L .603 .3393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63147 .34322 m .61139 .35045 L .603 .3393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58299 .34656 m .603 .3393 L .61139 .35045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58299 .34656 m .603 .3393 L .61139 .35045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4876 .69096 m .46809 .6968 L .45882 .68779 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.43939 .69366 m .45882 .68779 L .46809 .6968 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.38769 .45896 m .36859 .46573 L .35869 .45529 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.33966 .46209 m .35869 .45529 L .36859 .46573 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.46221 .9248 m .48174 .91986 L .49088 .92744 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.55447 .34266 m .5347 .3499 L .52583 .33874 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55447 .34266 m .5347 .3499 L .52583 .33874 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50612 .34601 m .52583 .33874 L .5347 .3499 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50612 .34601 m .52583 .33874 L .5347 .3499 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41055 .6905 m .39135 .69635 L .3816 .68734 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.36247 .69321 m .3816 .68734 L .39135 .69635 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.31061 .45843 m .29181 .46521 L .28144 .45476 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.43343 .92215 m .41413 .92707 L .40451 .91948 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.38529 .92442 m .40451 .91948 L .41413 .92707 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.47743 .3421 m .45795 .34934 L .44861 .33818 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47743 .3421 m .45795 .34934 L .44861 .33818 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42921 .34545 m .44861 .33818 L .45795 .34934 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42921 .34545 m .44861 .33818 L .45795 .34934 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33347 .69005 m .31457 .6959 L .30433 .68688 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.50032 .57374 m .48075 .58006 L .47154 .57032 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.45205 .57666 m .47154 .57032 L .48075 .58006 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.40034 .34154 m .38116 .34879 L .37134 .33761 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40034 .34154 m .38116 .34879 L .37134 .33761 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35224 .34489 m .37134 .33761 L .38116 .34879 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35224 .34489 m .37134 .33761 L .38116 .34879 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47489 .80798 m .49449 .80256 L .50355 .81087 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.42322 .57326 m .40395 .57957 L .39427 .56983 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.37507 .57618 m .39427 .56983 L .40395 .57957 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.83423 .34098 m .81332 .34823 L .80617 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83423 .34098 m .81332 .34823 L .80617 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78532 .34433 m .80617 .33705 L .81332 .34823 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78532 .34433 m .80617 .33705 L .81332 .34823 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31786 .41221 m .3232 .34098 L .30433 .34823 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.309 .43024 m .29852 .41936 L .30433 .34823 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.309 .43024 m .29852 .41936 L .30433 .34823 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.30433 .34823 m .29852 .41936 L .31786 .41221 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.3232 .34098 m .30433 .34823 L .29402 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3232 .34098 m .30433 .34823 L .29402 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27523 .34433 m .29402 .33705 L .30433 .34823 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29402 .33705 m .30433 .34823 L .29852 .41936 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.54171 .46 m .56158 .45318 L .57024 .46365 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.44611 .80507 m .42675 .81046 L .4172 .80214 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.39791 .80756 m .4172 .80214 L .42675 .81046 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.34607 .57277 m .3271 .57909 L .31694 .56934 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.7572 .34042 m .73659 .34767 L .72896 .33649 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7572 .34042 m .73659 .34767 L .72896 .33649 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70842 .34378 m .72896 .33649 L .73659 .34767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70842 .34378 m .72896 .33649 L .73659 .34767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23863 .41165 m .24601 .34042 L .22744 .34767 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.23058 .4297 m .2196 .41881 L .22744 .34767 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.23058 .4297 m .2196 .41881 L .22744 .34767 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.22744 .34767 m .2196 .41881 L .23863 .41165 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.24601 .34042 m .22744 .34767 L .21666 .33649 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23863 .41165 m .24601 .34042 L .22744 .34767 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.22744 .34767 m .2196 .41881 L .23863 .41165 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.19817 .34378 m .21666 .33649 L .22744 .34767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21666 .33649 m .22744 .34767 L .2196 .41881 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.21666 .33649 m .22744 .34767 L .2196 .41881 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.51306 .45634 m .49343 .46313 L .48429 .45266 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.46473 .45948 m .48429 .45266 L .49343 .46313 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.36895 .80465 m .34989 .81005 L .33987 .80173 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.68013 .33986 m .65981 .34712 L .65171 .33593 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68013 .33986 m .65981 .34712 L .65171 .33593 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63147 .34322 m .65171 .33593 L .65981 .34712 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63147 .34322 m .65171 .33593 L .65981 .34712 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4876 .69096 m .50726 .68507 L .51625 .69411 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.4359 .45581 m .41657 .46261 L .40695 .45213 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.38769 .45896 m .40695 .45213 L .41657 .46261 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.603 .3393 m .58299 .34656 L .57441 .33536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.603 .3393 m .58299 .34656 L .57441 .33536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55447 .34266 m .57441 .33536 L .58299 .34656 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55447 .34266 m .57441 .33536 L .58299 .34656 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45882 .68779 m .43939 .69366 L .42991 .68461 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.41055 .6905 m .42991 .68461 L .43939 .69366 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.35869 .45529 m .33966 .46209 L .32957 .4516 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.31061 .45843 m .32957 .4516 L .33966 .46209 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.48174 .91986 m .46221 .9248 L .45288 .91719 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.43343 .92215 m .45288 .91719 L .46221 .9248 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.52583 .33874 m .50612 .34601 L .49706 .3348 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52583 .33874 m .50612 .34601 L .49706 .3348 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47743 .3421 m .49706 .3348 L .50612 .34601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47743 .3421 m .49706 .3348 L .50612 .34601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3816 .68734 m .36247 .69321 L .35252 .68416 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.33347 .69005 m .35252 .68416 L .36247 .69321 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.28144 .45476 m .26271 .46157 L .25213 .45108 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.50032 .57374 m .52005 .56738 L .52897 .57715 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.40451 .91948 m .38529 .92442 L .37547 .91681 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.44861 .33818 m .42921 .34545 L .41966 .33424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44861 .33818 m .42921 .34545 L .41966 .33424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40034 .34154 m .41966 .33424 L .42921 .34545 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40034 .34154 m .41966 .33424 L .42921 .34545 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47154 .57032 m .45205 .57666 L .44264 .56688 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.42322 .57326 m .44264 .56688 L .45205 .57666 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.37134 .33761 m .35224 .34489 L .34222 .33367 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37134 .33761 m .35224 .34489 L .34222 .33367 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3232 .34098 m .34222 .33367 L .35224 .34489 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3232 .34098 m .34222 .33367 L .35224 .34489 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44611 .80507 m .46563 .79963 L .47489 .80798 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.39427 .56983 m .37507 .57618 L .36519 .56639 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.34607 .57277 m .36519 .56639 L .37507 .57618 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.80617 .33705 m .78532 .34433 L .77798 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80617 .33705 m .78532 .34433 L .77798 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7572 .34042 m .77798 .33311 L .78532 .34433 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7572 .34042 m .77798 .33311 L .78532 .34433 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2879 .40833 m .29402 .33705 L .27523 .34433 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.27933 .42645 m .26864 .41551 L .27523 .34433 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.27933 .42645 m .26864 .41551 L .27523 .34433 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.27523 .34433 m .26864 .41551 L .2879 .40833 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.29402 .33705 m .27523 .34433 L .26472 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29402 .33705 m .27523 .34433 L .26472 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24601 .34042 m .26472 .33311 L .27523 .34433 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26472 .33311 m .27523 .34433 L .26864 .41551 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.51306 .45634 m .53286 .44949 L .54171 .46 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.4172 .80214 m .39791 .80756 L .38817 .79921 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.36895 .80465 m .38817 .79921 L .39791 .80756 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.31694 .56934 m .29805 .57569 L .28768 .5659 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.72896 .33649 m .70842 .34378 L .7006 .33254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72896 .33649 m .70842 .34378 L .7006 .33254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68013 .33986 m .7006 .33254 L .70842 .34378 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68013 .33986 m .7006 .33254 L .70842 .34378 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20848 .40778 m .21666 .33649 L .19817 .34378 L p
.52 .29 .506 r
F P
s
P
p
.002 w
.20073 .4259 m .18954 .41496 L .19817 .34378 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.19817 .34378 m .18954 .41496 L .20848 .40778 L p
.52 .29 .506 r
F P
s
P
p
.002 w
.21666 .33649 m .19817 .34378 L .18718 .33254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20848 .40778 m .21666 .33649 L .19817 .34378 L p
.52 .29 .506 r
F P
s
P
p
.002 w
.19817 .34378 m .18954 .41496 L .20848 .40778 L p
.52 .29 .506 r
F P
s
P
p
.002 w
.18718 .33254 m .19817 .34378 L .18954 .41496 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.48429 .45266 m .46473 .45948 L .45539 .44896 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.4359 .45581 m .45539 .44896 L .46473 .45948 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.65171 .33593 m .63147 .34322 L .62317 .33198 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65171 .33593 m .63147 .34322 L .62317 .33198 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.603 .3393 m .62317 .33198 L .63147 .34322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.603 .3393 m .62317 .33198 L .63147 .34322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45882 .68779 m .4784 .68188 L .4876 .69096 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.40695 .45213 m .38769 .45896 L .37788 .44843 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.35869 .45529 m .37788 .44843 L .38769 .45896 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.57441 .33536 m .55447 .34266 L .54569 .33141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57441 .33536 m .55447 .34266 L .54569 .33141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52583 .33874 m .54569 .33141 L .55447 .34266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52583 .33874 m .54569 .33141 L .55447 .34266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42991 .68461 m .41055 .6905 L .40088 .68142 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.3816 .68734 m .40088 .68142 L .41055 .6905 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.32957 .4516 m .31061 .45843 L .30032 .4479 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.28144 .45476 m .30032 .4479 L .31061 .45843 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.45288 .91719 m .43343 .92215 L .42389 .9145 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.40451 .91948 m .42389 .9145 L .43343 .92215 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.49706 .3348 m .47743 .3421 L .46816 .33084 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49706 .3348 m .47743 .3421 L .46816 .33084 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44861 .33818 m .46816 .33084 L .47743 .3421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44861 .33818 m .46816 .33084 L .47743 .3421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35252 .68416 m .33347 .69005 L .32331 .68096 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.30433 .68688 m .32331 .68096 L .33347 .69005 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.47154 .57032 m .4912 .56393 L .50032 .57374 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.41966 .33424 m .40034 .34154 L .39059 .33028 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41966 .33424 m .40034 .34154 L .39059 .33028 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37134 .33761 m .39059 .33028 L .40034 .34154 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37134 .33761 m .39059 .33028 L .40034 .34154 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44264 .56688 m .42322 .57326 L .41361 .56343 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.39427 .56983 m .41361 .56343 L .42322 .57326 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.80617 .33705 m .82718 .32971 L .83423 .34098 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80617 .33705 m .82718 .32971 L .83423 .34098 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34222 .33367 m .3232 .34098 L .31297 .32971 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34222 .33367 m .3232 .34098 L .31297 .32971 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32826 .42318 m .31786 .41221 L .3232 .34098 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.29402 .33705 m .31297 .32971 L .3232 .34098 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29402 .33705 m .31297 .32971 L .3232 .34098 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4172 .80214 m .43665 .79668 L .44611 .80507 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.36519 .56639 m .34607 .57277 L .33598 .56294 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.31694 .56934 m .33598 .56294 L .34607 .57277 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.77798 .33311 m .7572 .34042 L .74967 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77798 .33311 m .7572 .34042 L .74967 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72896 .33649 m .74967 .32914 L .7572 .34042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72896 .33649 m .74967 .32914 L .7572 .34042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25781 .40444 m .26472 .33311 L .24601 .34042 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.24953 .42264 m .23863 .41165 L .24601 .34042 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.24953 .42264 m .23863 .41165 L .24601 .34042 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.24601 .34042 m .23863 .41165 L .25781 .40444 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.26472 .33311 m .24601 .34042 L .2353 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26472 .33311 m .24601 .34042 L .2353 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21666 .33649 m .2353 .32914 L .24601 .34042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2353 .32914 m .24601 .34042 L .23863 .41165 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.48429 .45266 m .50401 .44578 L .51306 .45634 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.38817 .79921 m .36895 .80465 L .35901 .79626 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.33987 .80173 m .35901 .79626 L .36895 .80465 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.28768 .5659 m .26887 .57228 L .2583 .56244 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.7006 .33254 m .68013 .33986 L .67211 .32858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7006 .33254 m .68013 .33986 L .67211 .32858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65171 .33593 m .67211 .32858 L .68013 .33986 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65171 .33593 m .67211 .32858 L .68013 .33986 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45539 .44896 m .4359 .45581 L .42637 .44525 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.40695 .45213 m .42637 .44525 L .4359 .45581 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.62317 .33198 m .603 .3393 L .59451 .32801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62317 .33198 m .603 .3393 L .59451 .32801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57441 .33536 m .59451 .32801 L .603 .3393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57441 .33536 m .59451 .32801 L .603 .3393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42991 .68461 m .44942 .67867 L .45882 .68779 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.37788 .44843 m .35869 .45529 L .34868 .44472 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.32957 .4516 m .34868 .44472 L .35869 .45529 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.45288 .91719 m .47249 .91219 L .48174 .91986 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.54569 .33141 m .52583 .33874 L .51685 .32744 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54569 .33141 m .52583 .33874 L .51685 .32744 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49706 .3348 m .51685 .32744 L .52583 .33874 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49706 .3348 m .51685 .32744 L .52583 .33874 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40088 .68142 m .3816 .68734 L .37172 .67821 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.35252 .68416 m .37172 .67821 L .3816 .68734 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.30032 .4479 m .28144 .45476 L .27093 .44419 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.25213 .45108 m .27093 .44419 L .28144 .45476 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.42389 .9145 m .40451 .91948 L .39478 .9118 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.37547 .91681 m .39478 .9118 L .40451 .91948 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.46816 .33084 m .44861 .33818 L .43914 .32687 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46816 .33084 m .44861 .33818 L .43914 .32687 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41966 .33424 m .43914 .32687 L .44861 .33818 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41966 .33424 m .43914 .32687 L .44861 .33818 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32331 .68096 m .30433 .68688 L .29397 .67775 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.44264 .56688 m .46222 .56046 L .47154 .57032 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.39059 .33028 m .37134 .33761 L .36139 .3263 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39059 .33028 m .37134 .33761 L .36139 .3263 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34222 .33367 m .36139 .3263 L .37134 .33761 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34222 .33367 m .36139 .3263 L .37134 .33761 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41361 .56343 m .39427 .56983 L .38446 .55997 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.36519 .56639 m .38446 .55997 L .39427 .56983 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.82718 .32971 m .80617 .33705 L .79893 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82718 .32971 m .80617 .33705 L .79893 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77798 .33311 m .79893 .32573 L .80617 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77798 .33311 m .79893 .32573 L .80617 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31297 .32971 m .29402 .33705 L .28359 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31297 .32971 m .29402 .33705 L .28359 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29852 .41936 m .2879 .40833 L .29402 .33705 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.26472 .33311 m .28359 .32573 L .29402 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26472 .33311 m .28359 .32573 L .29402 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43665 .79668 m .4172 .80214 L .40754 .79373 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.38817 .79921 m .40754 .79373 L .4172 .80214 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.33598 .56294 m .31694 .56934 L .30665 .55947 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.28768 .5659 m .30665 .55947 L .31694 .56934 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.74967 .32914 m .72896 .33649 L .72124 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74967 .32914 m .72896 .33649 L .72124 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7006 .33254 m .72124 .32516 L .72896 .33649 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7006 .33254 m .72124 .32516 L .72896 .33649 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22758 .40053 m .2353 .32914 L .21666 .33649 L p
.52 .29 .506 r
F P
s
P
p
.002 w
.2196 .41881 m .20848 .40778 L .21666 .33649 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.2196 .41881 m .20848 .40778 L .21666 .33649 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.21666 .33649 m .20848 .40778 L .22758 .40053 L p
.52 .29 .506 r
F P
s
P
p
.002 w
.2353 .32914 m .21666 .33649 L .20574 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2353 .32914 m .21666 .33649 L .20574 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18718 .33254 m .20574 .32516 L .21666 .33649 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20574 .32516 m .21666 .33649 L .20848 .40778 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.45539 .44896 m .47504 .44206 L .48429 .45266 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.35901 .79626 m .33987 .80173 L .32972 .7933 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.67211 .32858 m .65171 .33593 L .6435 .32459 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67211 .32858 m .65171 .33593 L .6435 .32459 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62317 .33198 m .6435 .32459 L .65171 .33593 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62317 .33198 m .6435 .32459 L .65171 .33593 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42637 .44525 m .40695 .45213 L .39722 .44152 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.37788 .44843 m .39722 .44152 L .40695 .45213 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.59451 .32801 m .57441 .33536 L .56572 .32402 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59451 .32801 m .57441 .33536 L .56572 .32402 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54569 .33141 m .56572 .32402 L .57441 .33536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54569 .33141 m .56572 .32402 L .57441 .33536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40088 .68142 m .42032 .67545 L .42991 .68461 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.34868 .44472 m .32957 .4516 L .31935 .44099 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.30032 .4479 m .31935 .44099 L .32957 .4516 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.42389 .9145 m .44343 .90948 L .45288 .91719 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.51685 .32744 m .49706 .3348 L .48788 .32345 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51685 .32744 m .49706 .3348 L .48788 .32345 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46816 .33084 m .48788 .32345 L .49706 .3348 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46816 .33084 m .48788 .32345 L .49706 .3348 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37172 .67821 m .35252 .68416 L .34244 .67499 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.32331 .68096 m .34244 .67499 L .35252 .68416 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.27093 .44419 m .25213 .45108 L .24143 .44045 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.39478 .9118 m .37547 .91681 L .36554 .90909 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.43914 .32687 m .41966 .33424 L .41 .32288 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43914 .32687 m .41966 .33424 L .41 .32288 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39059 .33028 m .41 .32288 L .41966 .33424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39059 .33028 m .41 .32288 L .41966 .33424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41361 .56343 m .43312 .55698 L .44264 .56688 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.36139 .3263 m .34222 .33367 L .33207 .32231 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36139 .3263 m .34222 .33367 L .33207 .32231 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31297 .32971 m .33207 .32231 L .34222 .33367 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31297 .32971 m .33207 .32231 L .34222 .33367 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38446 .55997 m .36519 .56639 L .35518 .55648 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.33598 .56294 m .35518 .55648 L .36519 .56639 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.79893 .32573 m .77798 .33311 L .77055 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79893 .32573 m .77798 .33311 L .77055 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74967 .32914 m .77055 .32174 L .77798 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74967 .32914 m .77055 .32174 L .77798 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28359 .32573 m .26472 .33311 L .25408 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28359 .32573 m .26472 .33311 L .25408 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26864 .41551 m .25781 .40444 L .26472 .33311 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.2353 .32914 m .25408 .32174 L .26472 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2353 .32914 m .25408 .32174 L .26472 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40754 .79373 m .38817 .79921 L .3783 .79076 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.35901 .79626 m .3783 .79076 L .38817 .79921 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.30665 .55947 m .28768 .5659 L .27719 .55599 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.2583 .56244 m .27719 .55599 L .28768 .5659 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.72124 .32516 m .7006 .33254 L .69268 .32117 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72124 .32516 m .7006 .33254 L .69268 .32117 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67211 .32858 m .69268 .32117 L .7006 .33254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67211 .32858 m .69268 .32117 L .7006 .33254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19722 .39661 m .20574 .32516 L .18718 .33254 L p
.517 .285 .503 r
F P
s
P
p
.002 w
.18954 .41496 m .1782 .40388 L .18718 .33254 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.18718 .33254 m .1782 .40388 L .19722 .39661 L p
.517 .285 .503 r
F P
s
P
p
.002 w
.20574 .32516 m .18718 .33254 L .17605 .32117 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19722 .39661 m .20574 .32516 L .18718 .33254 L p
.517 .285 .503 r
F P
s
P
p
.002 w
.18718 .33254 m .1782 .40388 L .19722 .39661 L p
.517 .285 .503 r
F P
s
P
p
.002 w
.17605 .32117 m .18718 .33254 L .1782 .40388 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.43062 1.02758 m .41116 1.03212 L .40143 1.02512 L p
.625 .78 .956 r
F P
s
P
p
.002 w
.42637 .44525 m .44594 .43831 L .45539 .44896 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.6435 .32459 m .62317 .33198 L .61477 .32059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6435 .32459 m .62317 .33198 L .61477 .32059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59451 .32801 m .61477 .32059 L .62317 .33198 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59451 .32801 m .61477 .32059 L .62317 .33198 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39722 .44152 m .37788 .44843 L .36794 .43778 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.34868 .44472 m .36794 .43778 L .37788 .44843 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.56572 .32402 m .54569 .33141 L .5368 .32002 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56572 .32402 m .54569 .33141 L .5368 .32002 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51685 .32744 m .5368 .32002 L .54569 .33141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51685 .32744 m .5368 .32002 L .54569 .33141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42032 .67545 m .40088 .68142 L .39108 .67222 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.37172 .67821 m .39108 .67222 L .40088 .68142 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.31935 .44099 m .30032 .4479 L .28989 .43724 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.27093 .44419 m .28989 .43724 L .30032 .4479 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.39478 .9118 m .41424 .90676 L .42389 .9145 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.48788 .32345 m .46816 .33084 L .45879 .31945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48788 .32345 m .46816 .33084 L .45879 .31945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43914 .32687 m .45879 .31945 L .46816 .33084 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43914 .32687 m .45879 .31945 L .46816 .33084 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34244 .67499 m .32331 .68096 L .31302 .67176 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.29397 .67775 m .31302 .67176 L .32331 .68096 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.24143 .44045 m .2227 .44737 L .21179 .43671 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.41 .32288 m .39059 .33028 L .38072 .31887 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41 .32288 m .39059 .33028 L .38072 .31887 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36139 .3263 m .38072 .31887 L .39059 .33028 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36139 .3263 m .38072 .31887 L .39059 .33028 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38446 .55997 m .40389 .55349 L .41361 .56343 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.79893 .32573 m .82005 .3183 L .82718 .32971 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79893 .32573 m .82005 .3183 L .82718 .32971 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33207 .32231 m .31297 .32971 L .30261 .3183 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33207 .32231 m .31297 .32971 L .30261 .3183 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28359 .32573 m .30261 .3183 L .31297 .32971 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28359 .32573 m .30261 .3183 L .31297 .32971 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40754 .79373 m .42706 .7882 L .43665 .79668 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.35518 .55648 m .33598 .56294 L .32577 .55299 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.30665 .55947 m .32577 .55299 L .33598 .56294 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.77055 .32174 m .74967 .32914 L .74205 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77055 .32174 m .74967 .32914 L .74205 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72124 .32516 m .74205 .31773 L .74967 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72124 .32516 m .74205 .31773 L .74967 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25408 .32174 m .2353 .32914 L .22445 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25408 .32174 m .2353 .32914 L .22445 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23863 .41165 m .22758 .40053 L .2353 .32914 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.20574 .32516 m .22445 .31773 L .2353 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20574 .32516 m .22445 .31773 L .2353 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3783 .79076 m .35901 .79626 L .34893 .78777 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.32972 .7933 m .34893 .78777 L .35901 .79626 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.27719 .55599 m .2583 .56244 L .2476 .55249 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.69268 .32117 m .67211 .32858 L .664 .31715 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69268 .32117 m .67211 .32858 L .664 .31715 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6435 .32459 m .664 .31715 L .67211 .32858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6435 .32459 m .664 .31715 L .67211 .32858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39722 .44152 m .41672 .43456 L .42637 .44525 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.61477 .32059 m .59451 .32801 L .5859 .31658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61477 .32059 m .59451 .32801 L .5859 .31658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56572 .32402 m .5859 .31658 L .59451 .32801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56572 .32402 m .5859 .31658 L .59451 .32801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36794 .43778 m .34868 .44472 L .33853 .43402 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.31935 .44099 m .33853 .43402 L .34868 .44472 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.5368 .32002 m .51685 .32744 L .50776 .316 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5368 .32002 m .51685 .32744 L .50776 .316 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48788 .32345 m .50776 .316 L .51685 .32744 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48788 .32345 m .50776 .316 L .51685 .32744 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39108 .67222 m .37172 .67821 L .36172 .66897 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.34244 .67499 m .36172 .66897 L .37172 .67821 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.28989 .43724 m .27093 .44419 L .2603 .43348 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.24143 .44045 m .2603 .43348 L .27093 .44419 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.36554 .90909 m .38492 .90403 L .39478 .9118 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.45879 .31945 m .43914 .32687 L .42956 .31543 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45879 .31945 m .43914 .32687 L .42956 .31543 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41 .32288 m .42956 .31543 L .43914 .32687 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41 .32288 m .42956 .31543 L .43914 .32687 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31302 .67176 m .29397 .67775 L .28348 .66851 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.38072 .31887 m .36139 .3263 L .35132 .31485 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38072 .31887 m .36139 .3263 L .35132 .31485 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33207 .32231 m .35132 .31485 L .36139 .3263 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33207 .32231 m .35132 .31485 L .36139 .3263 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35518 .55648 m .37453 .54998 L .38446 .55997 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.82005 .3183 m .79893 .32573 L .7916 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82005 .3183 m .79893 .32573 L .7916 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77055 .32174 m .7916 .31427 L .79893 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77055 .32174 m .7916 .31427 L .79893 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30261 .3183 m .28359 .32573 L .27303 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30261 .3183 m .28359 .32573 L .27303 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25408 .32174 m .27303 .31427 L .28359 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25408 .32174 m .27303 .31427 L .28359 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3783 .79076 m .39775 .7852 L .40754 .79373 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.32577 .55299 m .30665 .55947 L .29623 .54947 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.27719 .55599 m .29623 .54947 L .30665 .55947 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.74205 .31773 m .72124 .32516 L .71342 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74205 .31773 m .72124 .32516 L .71342 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69268 .32117 m .71342 .3137 L .72124 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69268 .32117 m .71342 .3137 L .72124 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22445 .31773 m .20574 .32516 L .19468 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22445 .31773 m .20574 .32516 L .19468 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20848 .40778 m .19722 .39661 L .20574 .32516 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.17605 .32117 m .19468 .3137 L .20574 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19468 .3137 m .20574 .32516 L .19722 .39661 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.40143 1.02512 m .42098 1.02053 L .43062 1.02758 L p
.625 .78 .956 r
F P
s
P
p
.002 w
.34893 .78777 m .32972 .7933 L .31944 .78477 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.664 .31715 m .6435 .32459 L .63519 .31312 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.664 .31715 m .6435 .32459 L .63519 .31312 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61477 .32059 m .63519 .31312 L .6435 .32459 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61477 .32059 m .63519 .31312 L .6435 .32459 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36794 .43778 m .38736 .43078 L .39722 .44152 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.5859 .31658 m .56572 .32402 L .55692 .31254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5859 .31658 m .56572 .32402 L .55692 .31254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5368 .32002 m .55692 .31254 L .56572 .32402 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5368 .32002 m .55692 .31254 L .56572 .32402 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39108 .67222 m .4106 .66618 L .42032 .67545 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.33853 .43402 m .31935 .44099 L .309 .43024 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.28989 .43724 m .309 .43024 L .31935 .44099 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.50776 .316 m .48788 .32345 L .47859 .31196 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50776 .316 m .48788 .32345 L .47859 .31196 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45879 .31945 m .47859 .31196 L .48788 .32345 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45879 .31945 m .47859 .31196 L .48788 .32345 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36172 .66897 m .34244 .67499 L .33223 .66571 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.31302 .67176 m .33223 .66571 L .34244 .67499 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.2603 .43348 m .24143 .44045 L .23058 .4297 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.21179 .43671 m .23058 .4297 L .24143 .44045 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.38492 .90403 m .36554 .90909 L .35547 .90128 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.42956 .31543 m .41 .32288 L .40021 .31139 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42956 .31543 m .41 .32288 L .40021 .31139 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38072 .31887 m .40021 .31139 L .41 .32288 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38072 .31887 m .40021 .31139 L .41 .32288 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35132 .31485 m .33207 .32231 L .32179 .31081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35132 .31485 m .33207 .32231 L .32179 .31081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30261 .3183 m .32179 .31081 L .33207 .32231 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30261 .3183 m .32179 .31081 L .33207 .32231 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37453 .54998 m .35518 .55648 L .34504 .54645 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.32577 .55299 m .34504 .54645 L .35518 .55648 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.7916 .31427 m .77055 .32174 L .76303 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7916 .31427 m .77055 .32174 L .76303 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74205 .31773 m .76303 .31023 L .77055 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74205 .31773 m .76303 .31023 L .77055 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27303 .31427 m .25408 .32174 L .24331 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27303 .31427 m .25408 .32174 L .24331 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22445 .31773 m .24331 .31023 L .25408 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22445 .31773 m .24331 .31023 L .25408 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34893 .78777 m .36831 .78219 L .3783 .79076 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.29623 .54947 m .27719 .55599 L .26656 .54595 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.2476 .55249 m .26656 .54595 L .27719 .55599 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.71342 .3137 m .69268 .32117 L .68467 .30965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71342 .3137 m .69268 .32117 L .68467 .30965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.664 .31715 m .68467 .30965 L .69268 .32117 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.664 .31715 m .68467 .30965 L .69268 .32117 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18581 .38529 m .19468 .3137 L .17605 .32117 L p
.513 .281 .499 r
F P
s
P
p
.002 w
.1782 .40388 m .16672 .39266 L .17605 .32117 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.17605 .32117 m .16672 .39266 L .18581 .38529 L p
.513 .281 .499 r
F P
s
P
p
.002 w
.19468 .3137 m .17605 .32117 L .16479 .30965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19468 .3137 m .17605 .32117 L .16479 .30965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63519 .31312 m .61477 .32059 L .60626 .30907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63519 .31312 m .61477 .32059 L .60626 .30907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5859 .31658 m .60626 .30907 L .61477 .32059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5859 .31658 m .60626 .30907 L .61477 .32059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33853 .43402 m .35788 .42699 L .36794 .43778 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.55692 .31254 m .5368 .32002 L .5278 .30849 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55692 .31254 m .5368 .32002 L .5278 .30849 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50776 .316 m .5278 .30849 L .5368 .32002 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50776 .316 m .5278 .30849 L .5368 .32002 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36172 .66897 m .38116 .66291 L .39108 .67222 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.309 .43024 m .28989 .43724 L .27933 .42645 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.2603 .43348 m .27933 .42645 L .28989 .43724 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.47859 .31196 m .45879 .31945 L .44929 .30791 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47859 .31196 m .45879 .31945 L .44929 .30791 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42956 .31543 m .44929 .30791 L .45879 .31945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42956 .31543 m .44929 .30791 L .45879 .31945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33223 .66571 m .31302 .67176 L .30261 .66244 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.28348 .66851 m .30261 .66244 L .31302 .67176 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.23058 .4297 m .21179 .43671 L .20073 .4259 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.40021 .31139 m .38072 .31887 L .37073 .30733 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40021 .31139 m .38072 .31887 L .37073 .30733 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35132 .31485 m .37073 .30733 L .38072 .31887 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35132 .31485 m .37073 .30733 L .38072 .31887 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7916 .31427 m .81282 .30675 L .82005 .3183 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7916 .31427 m .81282 .30675 L .82005 .3183 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32179 .31081 m .30261 .3183 L .29212 .30675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32179 .31081 m .30261 .3183 L .29212 .30675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27303 .31427 m .29212 .30675 L .30261 .3183 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27303 .31427 m .29212 .30675 L .30261 .3183 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34504 .54645 m .32577 .55299 L .31542 .54291 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.29623 .54947 m .31542 .54291 L .32577 .55299 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.76303 .31023 m .74205 .31773 L .73433 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76303 .31023 m .74205 .31773 L .73433 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71342 .3137 m .73433 .30616 L .74205 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71342 .3137 m .73433 .30616 L .74205 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24331 .31023 m .22445 .31773 L .21347 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24331 .31023 m .22445 .31773 L .21347 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19468 .3137 m .21347 .30616 L .22445 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19468 .3137 m .21347 .30616 L .22445 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36831 .78219 m .34893 .78777 L .33873 .77917 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.31944 .78477 m .33873 .77917 L .34893 .78777 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.26656 .54595 m .2476 .55249 L .23676 .5424 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.68467 .30965 m .664 .31715 L .65579 .30558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68467 .30965 m .664 .31715 L .65579 .30558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63519 .31312 m .65579 .30558 L .664 .31715 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63519 .31312 m .65579 .30558 L .664 .31715 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60626 .30907 m .5859 .31658 L .5772 .305 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60626 .30907 m .5859 .31658 L .5772 .305 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55692 .31254 m .5772 .305 L .5859 .31658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55692 .31254 m .5772 .305 L .5859 .31658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.309 .43024 m .32826 .42318 L .33853 .43402 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.5278 .30849 m .50776 .316 L .49855 .30442 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5278 .30849 m .50776 .316 L .49855 .30442 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47859 .31196 m .49855 .30442 L .50776 .316 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47859 .31196 m .49855 .30442 L .50776 .316 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33223 .66571 m .35159 .65962 L .36172 .66897 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.27933 .42645 m .2603 .43348 L .24953 .42264 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.23058 .4297 m .24953 .42264 L .2603 .43348 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.35547 .90128 m .37494 .89615 L .38492 .90403 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.44929 .30791 m .42956 .31543 L .41986 .30383 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44929 .30791 m .42956 .31543 L .41986 .30383 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40021 .31139 m .41986 .30383 L .42956 .31543 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40021 .31139 m .41986 .30383 L .42956 .31543 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30261 .66244 m .28348 .66851 L .27286 .65915 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.37073 .30733 m .35132 .31485 L .34112 .30325 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37073 .30733 m .35132 .31485 L .34112 .30325 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32179 .31081 m .34112 .30325 L .35132 .31485 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32179 .31081 m .34112 .30325 L .35132 .31485 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34504 .54645 m .36448 .53986 L .37453 .54998 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.81282 .30675 m .7916 .31427 L .78418 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81282 .30675 m .7916 .31427 L .78418 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76303 .31023 m .78418 .30267 L .7916 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76303 .31023 m .78418 .30267 L .7916 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29212 .30675 m .27303 .31427 L .26233 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29212 .30675 m .27303 .31427 L .26233 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24331 .31023 m .26233 .30267 L .27303 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24331 .31023 m .26233 .30267 L .27303 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31542 .54291 m .29623 .54947 L .28567 .53935 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.26656 .54595 m .28567 .53935 L .29623 .54947 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.73433 .30616 m .71342 .3137 L .7055 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73433 .30616 m .71342 .3137 L .7055 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68467 .30965 m .7055 .30208 L .71342 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68467 .30965 m .7055 .30208 L .71342 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21347 .30616 m .19468 .3137 L .18349 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21347 .30616 m .19468 .3137 L .18349 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19722 .39661 m .18581 .38529 L .19468 .3137 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.16479 .30965 m .18349 .30208 L .19468 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16479 .30965 m .18349 .30208 L .19468 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33873 .77917 m .31944 .78477 L .30903 .77614 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.65579 .30558 m .63519 .31312 L .62678 .3015 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65579 .30558 m .63519 .31312 L .62678 .3015 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60626 .30907 m .62678 .3015 L .63519 .31312 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60626 .30907 m .62678 .3015 L .63519 .31312 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5772 .305 m .55692 .31254 L .548 .30091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5772 .305 m .55692 .31254 L .548 .30091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5278 .30849 m .548 .30091 L .55692 .31254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5278 .30849 m .548 .30091 L .55692 .31254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32826 .42318 m .309 .43024 L .29852 .41936 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.27933 .42645 m .29852 .41936 L .309 .43024 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.49855 .30442 m .47859 .31196 L .46918 .30033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49855 .30442 m .47859 .31196 L .46918 .30033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44929 .30791 m .46918 .30033 L .47859 .31196 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44929 .30791 m .46918 .30033 L .47859 .31196 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30261 .66244 m .32189 .65631 L .33223 .66571 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.24953 .42264 m .23058 .4297 L .2196 .41881 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.20073 .4259 m .2196 .41881 L .23058 .4297 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.41986 .30383 m .40021 .31139 L .3903 .29974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41986 .30383 m .40021 .31139 L .3903 .29974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37073 .30733 m .3903 .29974 L .40021 .31139 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37073 .30733 m .3903 .29974 L .40021 .31139 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34112 .30325 m .32179 .31081 L .31138 .29916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34112 .30325 m .32179 .31081 L .31138 .29916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29212 .30675 m .31138 .29916 L .32179 .31081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29212 .30675 m .31138 .29916 L .32179 .31081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31542 .54291 m .33478 .53629 L .34504 .54645 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.78418 .30267 m .76303 .31023 L .75541 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78418 .30267 m .76303 .31023 L .75541 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73433 .30616 m .75541 .29857 L .76303 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73433 .30616 m .75541 .29857 L .76303 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26233 .30267 m .24331 .31023 L .2324 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26233 .30267 m .24331 .31023 L .2324 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21347 .30616 m .2324 .29857 L .24331 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21347 .30616 m .2324 .29857 L .24331 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33873 .77917 m .35819 .77352 L .36831 .78219 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.28567 .53935 m .26656 .54595 L .25579 .53578 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.23676 .5424 m .25579 .53578 L .26656 .54595 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.7055 .30208 m .68467 .30965 L .67655 .29798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7055 .30208 m .68467 .30965 L .67655 .29798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65579 .30558 m .67655 .29798 L .68467 .30965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65579 .30558 m .67655 .29798 L .68467 .30965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18349 .30208 m .16479 .30965 L .15338 .29798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18349 .30208 m .16479 .30965 L .15338 .29798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62678 .3015 m .60626 .30907 L .59764 .2974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62678 .3015 m .60626 .30907 L .59764 .2974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5772 .305 m .59764 .2974 L .60626 .30907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5772 .305 m .59764 .2974 L .60626 .30907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.548 .30091 m .5278 .30849 L .51868 .29681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.548 .30091 m .5278 .30849 L .51868 .29681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49855 .30442 m .51868 .29681 L .5278 .30849 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49855 .30442 m .51868 .29681 L .5278 .30849 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29852 .41936 m .27933 .42645 L .26864 .41551 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.24953 .42264 m .26864 .41551 L .27933 .42645 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.46918 .30033 m .44929 .30791 L .43967 .29622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46918 .30033 m .44929 .30791 L .43967 .29622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41986 .30383 m .43967 .29622 L .44929 .30791 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41986 .30383 m .43967 .29622 L .44929 .30791 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32189 .65631 m .30261 .66244 L .29206 .653 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.27286 .65915 m .29206 .653 L .30261 .66244 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.2196 .41881 m .20073 .4259 L .18954 .41496 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.3903 .29974 m .37073 .30733 L .36061 .29563 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3903 .29974 m .37073 .30733 L .36061 .29563 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34112 .30325 m .36061 .29563 L .37073 .30733 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34112 .30325 m .36061 .29563 L .37073 .30733 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78418 .30267 m .80551 .29505 L .81282 .30675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78418 .30267 m .80551 .29505 L .81282 .30675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31138 .29916 m .29212 .30675 L .2815 .29505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31138 .29916 m .29212 .30675 L .2815 .29505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26233 .30267 m .2815 .29505 L .29212 .30675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26233 .30267 m .2815 .29505 L .29212 .30675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28567 .53935 m .30495 .5327 L .31542 .54291 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.75541 .29857 m .73433 .30616 L .72651 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75541 .29857 m .73433 .30616 L .72651 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7055 .30208 m .72651 .29446 L .73433 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7055 .30208 m .72651 .29446 L .73433 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2324 .29857 m .21347 .30616 L .20234 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2324 .29857 m .21347 .30616 L .20234 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18349 .30208 m .20234 .29446 L .21347 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18349 .30208 m .20234 .29446 L .21347 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30903 .77614 m .32841 .77046 L .33873 .77917 L p
.648 .761 .931 r
F P
s
P
p
.002 w
.25579 .53578 m .23676 .5424 L .22578 .53219 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.67655 .29798 m .65579 .30558 L .64747 .29387 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67655 .29798 m .65579 .30558 L .64747 .29387 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62678 .3015 m .64747 .29387 L .65579 .30558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62678 .3015 m .64747 .29387 L .65579 .30558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59764 .2974 m .5772 .305 L .56837 .29328 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59764 .2974 m .5772 .305 L .56837 .29328 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.548 .30091 m .56837 .29328 L .5772 .305 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.548 .30091 m .56837 .29328 L .5772 .305 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29852 .41936 m .31786 .41221 L .32826 .42318 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.51868 .29681 m .49855 .30442 L .48923 .29269 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51868 .29681 m .49855 .30442 L .48923 .29269 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46918 .30033 m .48923 .29269 L .49855 .30442 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46918 .30033 m .48923 .29269 L .49855 .30442 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26864 .41551 m .24953 .42264 L .23863 .41165 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.2196 .41881 m .23863 .41165 L .24953 .42264 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.43967 .29622 m .41986 .30383 L .41004 .2921 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43967 .29622 m .41986 .30383 L .41004 .2921 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3903 .29974 m .41004 .2921 L .41986 .30383 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3903 .29974 m .41004 .2921 L .41986 .30383 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29206 .653 m .27286 .65915 L .2621 .64967 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.36061 .29563 m .34112 .30325 L .33079 .29151 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36061 .29563 m .34112 .30325 L .33079 .29151 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31138 .29916 m .33079 .29151 L .34112 .30325 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31138 .29916 m .33079 .29151 L .34112 .30325 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80551 .29505 m .78418 .30267 L .77666 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80551 .29505 m .78418 .30267 L .77666 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75541 .29857 m .77666 .29091 L .78418 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75541 .29857 m .77666 .29091 L .78418 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2815 .29505 m .26233 .30267 L .2515 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2815 .29505 m .26233 .30267 L .2515 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2324 .29857 m .2515 .29091 L .26233 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2324 .29857 m .2515 .29091 L .26233 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25579 .53578 m .27499 .5291 L .28567 .53935 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.72651 .29446 m .7055 .30208 L .69749 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72651 .29446 m .7055 .30208 L .69749 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67655 .29798 m .69749 .29032 L .7055 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67655 .29798 m .69749 .29032 L .7055 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20234 .29446 m .18349 .30208 L .17215 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20234 .29446 m .18349 .30208 L .17215 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.15338 .29798 m .17215 .29032 L .18349 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.15338 .29798 m .17215 .29032 L .18349 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64747 .29387 m .62678 .3015 L .61826 .28973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64747 .29387 m .62678 .3015 L .61826 .28973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59764 .2974 m .61826 .28973 L .62678 .3015 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59764 .2974 m .61826 .28973 L .62678 .3015 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56837 .29328 m .548 .30091 L .53898 .28914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56837 .29328 m .548 .30091 L .53898 .28914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51868 .29681 m .53898 .28914 L .548 .30091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51868 .29681 m .53898 .28914 L .548 .30091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26864 .41551 m .2879 .40833 L .29852 .41936 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.48923 .29269 m .46918 .30033 L .45965 .28855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48923 .29269 m .46918 .30033 L .45965 .28855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43967 .29622 m .45965 .28855 L .46918 .30033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43967 .29622 m .45965 .28855 L .46918 .30033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29206 .653 m .31143 .6468 L .32189 .65631 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.23863 .41165 m .2196 .41881 L .20848 .40778 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.18954 .41496 m .20848 .40778 L .2196 .41881 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.41004 .2921 m .3903 .29974 L .38027 .28795 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41004 .2921 m .3903 .29974 L .38027 .28795 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36061 .29563 m .38027 .28795 L .3903 .29974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36061 .29563 m .38027 .28795 L .3903 .29974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33079 .29151 m .31138 .29916 L .30084 .28736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33079 .29151 m .31138 .29916 L .30084 .28736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2815 .29505 m .30084 .28736 L .31138 .29916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2815 .29505 m .30084 .28736 L .31138 .29916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77666 .29091 m .75541 .29857 L .74769 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77666 .29091 m .75541 .29857 L .74769 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72651 .29446 m .74769 .28677 L .75541 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72651 .29446 m .74769 .28677 L .75541 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2515 .29091 m .2324 .29857 L .22136 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2515 .29091 m .2324 .29857 L .22136 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20234 .29446 m .22136 .28677 L .2324 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20234 .29446 m .22136 .28677 L .2324 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22578 .53219 m .24489 .52548 L .25579 .53578 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.69749 .29032 m .67655 .29798 L .66833 .28617 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69749 .29032 m .67655 .29798 L .66833 .28617 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64747 .29387 m .66833 .28617 L .67655 .29798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64747 .29387 m .66833 .28617 L .67655 .29798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17215 .29032 m .15338 .29798 L .14182 .28617 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17215 .29032 m .15338 .29798 L .14182 .28617 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61826 .28973 m .59764 .2974 L .58892 .28558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61826 .28973 m .59764 .2974 L .58892 .28558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56837 .29328 m .58892 .28558 L .59764 .2974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56837 .29328 m .58892 .28558 L .59764 .2974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53898 .28914 m .51868 .29681 L .50945 .28498 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53898 .28914 m .51868 .29681 L .50945 .28498 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48923 .29269 m .50945 .28498 L .51868 .29681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48923 .29269 m .50945 .28498 L .51868 .29681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23863 .41165 m .25781 .40444 L .26864 .41551 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.45965 .28855 m .43967 .29622 L .42993 .28439 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45965 .28855 m .43967 .29622 L .42993 .28439 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41004 .2921 m .42993 .28439 L .43967 .29622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41004 .2921 m .42993 .28439 L .43967 .29622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2621 .64967 m .28138 .64344 L .29206 .653 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.20848 .40778 m .18954 .41496 L .1782 .40388 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.38027 .28795 m .36061 .29563 L .35037 .28379 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38027 .28795 m .36061 .29563 L .35037 .28379 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33079 .29151 m .35037 .28379 L .36061 .29563 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33079 .29151 m .35037 .28379 L .36061 .29563 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77666 .29091 m .7981 .28319 L .80551 .29505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77666 .29091 m .7981 .28319 L .80551 .29505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30084 .28736 m .2815 .29505 L .27075 .28319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30084 .28736 m .2815 .29505 L .27075 .28319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2515 .29091 m .27075 .28319 L .2815 .29505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2515 .29091 m .27075 .28319 L .2815 .29505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74769 .28677 m .72651 .29446 L .7186 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74769 .28677 m .72651 .29446 L .7186 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69749 .29032 m .7186 .2826 L .72651 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69749 .29032 m .7186 .2826 L .72651 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22136 .28677 m .20234 .29446 L .19108 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22136 .28677 m .20234 .29446 L .19108 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17215 .29032 m .19108 .2826 L .20234 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17215 .29032 m .19108 .2826 L .20234 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24489 .52548 m .22578 .53219 L .21466 .52185 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.66833 .28617 m .64747 .29387 L .63904 .282 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66833 .28617 m .64747 .29387 L .63904 .282 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61826 .28973 m .63904 .282 L .64747 .29387 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61826 .28973 m .63904 .282 L .64747 .29387 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58892 .28558 m .56837 .29328 L .55944 .2814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58892 .28558 m .56837 .29328 L .55944 .2814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53898 .28914 m .55944 .2814 L .56837 .29328 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53898 .28914 m .55944 .2814 L .56837 .29328 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50945 .28498 m .48923 .29269 L .47979 .28081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50945 .28498 m .48923 .29269 L .47979 .28081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45965 .28855 m .47979 .28081 L .48923 .29269 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45965 .28855 m .47979 .28081 L .48923 .29269 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20848 .40778 m .22758 .40053 L .23863 .41165 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.42993 .28439 m .41004 .2921 L .40009 .28021 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42993 .28439 m .41004 .2921 L .40009 .28021 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38027 .28795 m .40009 .28021 L .41004 .2921 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38027 .28795 m .40009 .28021 L .41004 .2921 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35037 .28379 m .33079 .29151 L .32033 .27961 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35037 .28379 m .33079 .29151 L .32033 .27961 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30084 .28736 m .32033 .27961 L .33079 .29151 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30084 .28736 m .32033 .27961 L .33079 .29151 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27075 .28319 m .2515 .29091 L .24052 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27075 .28319 m .2515 .29091 L .24052 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22136 .28677 m .24052 .27901 L .2515 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22136 .28677 m .24052 .27901 L .2515 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7981 .28319 m .77666 .29091 L .76905 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7981 .28319 m .77666 .29091 L .76905 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74769 .28677 m .76905 .27901 L .77666 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74769 .28677 m .76905 .27901 L .77666 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7186 .2826 m .69749 .29032 L .68937 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7186 .2826 m .69749 .29032 L .68937 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66833 .28617 m .68937 .27841 L .69749 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66833 .28617 m .68937 .27841 L .69749 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19108 .2826 m .17215 .29032 L .16067 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19108 .2826 m .17215 .29032 L .16067 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.14182 .28617 m .16067 .27841 L .17215 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.14182 .28617 m .16067 .27841 L .17215 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63904 .282 m .61826 .28973 L .60963 .27781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63904 .282 m .61826 .28973 L .60963 .27781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58892 .28558 m .60963 .27781 L .61826 .28973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58892 .28558 m .60963 .27781 L .61826 .28973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55944 .2814 m .53898 .28914 L .52984 .27721 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55944 .2814 m .53898 .28914 L .52984 .27721 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50945 .28498 m .52984 .27721 L .53898 .28914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50945 .28498 m .52984 .27721 L .53898 .28914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47979 .28081 m .45965 .28855 L .45 .27661 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47979 .28081 m .45965 .28855 L .45 .27661 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42993 .28439 m .45 .27661 L .45965 .28855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42993 .28439 m .45 .27661 L .45965 .28855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.1782 .40388 m .19722 .39661 L .20848 .40778 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.40009 .28021 m .38027 .28795 L .3701 .27601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40009 .28021 m .38027 .28795 L .3701 .27601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35037 .28379 m .3701 .27601 L .38027 .28795 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35037 .28379 m .3701 .27601 L .38027 .28795 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32033 .27961 m .30084 .28736 L .29016 .27541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32033 .27961 m .30084 .28736 L .29016 .27541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27075 .28319 m .29016 .27541 L .30084 .28736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27075 .28319 m .29016 .27541 L .30084 .28736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76905 .27901 m .74769 .28677 L .73988 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76905 .27901 m .74769 .28677 L .73988 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7186 .2826 m .73988 .27481 L .74769 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7186 .2826 m .73988 .27481 L .74769 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24052 .27901 m .22136 .28677 L .21017 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24052 .27901 m .22136 .28677 L .21017 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19108 .2826 m .21017 .27481 L .22136 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19108 .2826 m .21017 .27481 L .22136 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21466 .52185 m .23385 .51506 L .24489 .52548 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.68937 .27841 m .66833 .28617 L .66 .27421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68937 .27841 m .66833 .28617 L .66 .27421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63904 .282 m .66 .27421 L .66833 .28617 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63904 .282 m .66 .27421 L .66833 .28617 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16067 .27841 m .14182 .28617 L .13012 .27421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16067 .27841 m .14182 .28617 L .13012 .27421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60963 .27781 m .58892 .28558 L .58008 .27361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60963 .27781 m .58892 .28558 L .58008 .27361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55944 .2814 m .58008 .27361 L .58892 .28558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55944 .2814 m .58008 .27361 L .58892 .28558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52984 .27721 m .50945 .28498 L .5001 .273 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52984 .27721 m .50945 .28498 L .5001 .273 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47979 .28081 m .5001 .273 L .50945 .28498 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47979 .28081 m .5001 .273 L .50945 .28498 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45 .27661 m .42993 .28439 L .42007 .2724 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45 .27661 m .42993 .28439 L .42007 .2724 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40009 .28021 m .42007 .2724 L .42993 .28439 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40009 .28021 m .42007 .2724 L .42993 .28439 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19722 .39661 m .1782 .40388 L .16672 .39266 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.3701 .27601 m .35037 .28379 L .33999 .2718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3701 .27601 m .35037 .28379 L .33999 .2718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32033 .27961 m .33999 .2718 L .35037 .28379 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32033 .27961 m .33999 .2718 L .35037 .28379 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76905 .27901 m .79059 .27119 L .7981 .28319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76905 .27901 m .79059 .27119 L .7981 .28319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29016 .27541 m .27075 .28319 L .25985 .27119 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29016 .27541 m .27075 .28319 L .25985 .27119 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24052 .27901 m .25985 .27119 L .27075 .28319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24052 .27901 m .25985 .27119 L .27075 .28319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73988 .27481 m .7186 .2826 L .71058 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73988 .27481 m .7186 .2826 L .71058 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68937 .27841 m .71058 .27059 L .7186 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68937 .27841 m .71058 .27059 L .7186 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21017 .27481 m .19108 .2826 L .17967 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21017 .27481 m .19108 .2826 L .17967 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16067 .27841 m .17967 .27059 L .19108 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16067 .27841 m .17967 .27059 L .19108 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66 .27421 m .63904 .282 L .63051 .26998 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66 .27421 m .63904 .282 L .63051 .26998 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60963 .27781 m .63051 .26998 L .63904 .282 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60963 .27781 m .63051 .26998 L .63904 .282 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58008 .27361 m .55944 .2814 L .55039 .26938 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58008 .27361 m .55944 .2814 L .55039 .26938 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52984 .27721 m .55039 .26938 L .55944 .2814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52984 .27721 m .55039 .26938 L .55944 .2814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5001 .273 m .47979 .28081 L .47023 .26877 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5001 .273 m .47979 .28081 L .47023 .26877 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45 .27661 m .47023 .26877 L .47979 .28081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45 .27661 m .47023 .26877 L .47979 .28081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42007 .2724 m .40009 .28021 L .39001 .26817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42007 .2724 m .40009 .28021 L .39001 .26817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3701 .27601 m .39001 .26817 L .40009 .28021 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3701 .27601 m .39001 .26817 L .40009 .28021 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33999 .2718 m .32033 .27961 L .30974 .26756 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33999 .2718 m .32033 .27961 L .30974 .26756 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29016 .27541 m .30974 .26756 L .32033 .27961 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29016 .27541 m .30974 .26756 L .32033 .27961 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79059 .27119 m .76905 .27901 L .76134 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79059 .27119 m .76905 .27901 L .76134 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73988 .27481 m .76134 .26695 L .76905 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73988 .27481 m .76134 .26695 L .76905 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25985 .27119 m .24052 .27901 L .22941 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25985 .27119 m .24052 .27901 L .22941 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21017 .27481 m .22941 .26695 L .24052 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21017 .27481 m .22941 .26695 L .24052 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71058 .27059 m .68937 .27841 L .68114 .26635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71058 .27059 m .68937 .27841 L .68114 .26635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66 .27421 m .68114 .26635 L .68937 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66 .27421 m .68114 .26635 L .68937 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17967 .27059 m .16067 .27841 L .14904 .26635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17967 .27059 m .16067 .27841 L .14904 .26635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.13012 .27421 m .14904 .26635 L .16067 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.13012 .27421 m .14904 .26635 L .16067 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63051 .26998 m .60963 .27781 L .60088 .26574 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63051 .26998 m .60963 .27781 L .60088 .26574 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58008 .27361 m .60088 .26574 L .60963 .27781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58008 .27361 m .60088 .26574 L .60963 .27781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55039 .26938 m .52984 .27721 L .52058 .26513 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55039 .26938 m .52984 .27721 L .52058 .26513 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5001 .273 m .52058 .26513 L .52984 .27721 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5001 .273 m .52058 .26513 L .52984 .27721 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47023 .26877 m .45 .27661 L .44022 .26453 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47023 .26877 m .45 .27661 L .44022 .26453 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42007 .2724 m .44022 .26453 L .45 .27661 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42007 .2724 m .44022 .26453 L .45 .27661 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16672 .39266 m .18581 .38529 L .19722 .39661 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.39001 .26817 m .3701 .27601 L .35981 .26392 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39001 .26817 m .3701 .27601 L .35981 .26392 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33999 .2718 m .35981 .26392 L .3701 .27601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33999 .2718 m .35981 .26392 L .3701 .27601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30974 .26756 m .29016 .27541 L .27935 .26331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30974 .26756 m .29016 .27541 L .27935 .26331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25985 .27119 m .27935 .26331 L .29016 .27541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25985 .27119 m .27935 .26331 L .29016 .27541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76134 .26695 m .73988 .27481 L .73196 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76134 .26695 m .73988 .27481 L .73196 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71058 .27059 m .73196 .2627 L .73988 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71058 .27059 m .73196 .2627 L .73988 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22941 .26695 m .21017 .27481 L .19883 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22941 .26695 m .21017 .27481 L .19883 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17967 .27059 m .19883 .2627 L .21017 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17967 .27059 m .19883 .2627 L .21017 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68114 .26635 m .66 .27421 L .65157 .26209 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68114 .26635 m .66 .27421 L .65157 .26209 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63051 .26998 m .65157 .26209 L .66 .27421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63051 .26998 m .65157 .26209 L .66 .27421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60088 .26574 m .58008 .27361 L .57112 .26148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60088 .26574 m .58008 .27361 L .57112 .26148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55039 .26938 m .57112 .26148 L .58008 .27361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55039 .26938 m .57112 .26148 L .58008 .27361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52058 .26513 m .5001 .273 L .49063 .26087 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52058 .26513 m .5001 .273 L .49063 .26087 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47023 .26877 m .49063 .26087 L .5001 .273 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47023 .26877 m .49063 .26087 L .5001 .273 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44022 .26453 m .42007 .2724 L .41008 .26026 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44022 .26453 m .42007 .2724 L .41008 .26026 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39001 .26817 m .41008 .26026 L .42007 .2724 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39001 .26817 m .41008 .26026 L .42007 .2724 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35981 .26392 m .33999 .2718 L .32947 .25965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35981 .26392 m .33999 .2718 L .32947 .25965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30974 .26756 m .32947 .25965 L .33999 .2718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30974 .26756 m .32947 .25965 L .33999 .2718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76134 .26695 m .78299 .25903 L .79059 .27119 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76134 .26695 m .78299 .25903 L .79059 .27119 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27935 .26331 m .25985 .27119 L .24882 .25903 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27935 .26331 m .25985 .27119 L .24882 .25903 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22941 .26695 m .24882 .25903 L .25985 .27119 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22941 .26695 m .24882 .25903 L .25985 .27119 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19883 .2627 m .17967 .27059 L .16811 .25842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19883 .2627 m .17967 .27059 L .16811 .25842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.14904 .26635 m .16811 .25842 L .17967 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.14904 .26635 m .16811 .25842 L .17967 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73196 .2627 m .71058 .27059 L .70245 .25842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73196 .2627 m .71058 .27059 L .70245 .25842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68114 .26635 m .70245 .25842 L .71058 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68114 .26635 m .70245 .25842 L .71058 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65157 .26209 m .63051 .26998 L .62187 .25781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65157 .26209 m .63051 .26998 L .62187 .25781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60088 .26574 m .62187 .25781 L .63051 .26998 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60088 .26574 m .62187 .25781 L .63051 .26998 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57112 .26148 m .55039 .26938 L .54123 .2572 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57112 .26148 m .55039 .26938 L .54123 .2572 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52058 .26513 m .54123 .2572 L .55039 .26938 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52058 .26513 m .54123 .2572 L .55039 .26938 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49063 .26087 m .47023 .26877 L .46054 .25658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49063 .26087 m .47023 .26877 L .46054 .25658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44022 .26453 m .46054 .25658 L .47023 .26877 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44022 .26453 m .46054 .25658 L .47023 .26877 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41008 .26026 m .39001 .26817 L .3798 .25597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41008 .26026 m .39001 .26817 L .3798 .25597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35981 .26392 m .3798 .25597 L .39001 .26817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35981 .26392 m .3798 .25597 L .39001 .26817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32947 .25965 m .30974 .26756 L .299 .25536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32947 .25965 m .30974 .26756 L .299 .25536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27935 .26331 m .299 .25536 L .30974 .26756 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27935 .26331 m .299 .25536 L .30974 .26756 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78299 .25903 m .76134 .26695 L .75353 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78299 .25903 m .76134 .26695 L .75353 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73196 .2627 m .75353 .25474 L .76134 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73196 .2627 m .75353 .25474 L .76134 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24882 .25903 m .22941 .26695 L .21816 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24882 .25903 m .22941 .26695 L .21816 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19883 .2627 m .21816 .25474 L .22941 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19883 .2627 m .21816 .25474 L .22941 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70245 .25842 m .68114 .26635 L .67281 .25413 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70245 .25842 m .68114 .26635 L .67281 .25413 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65157 .26209 m .67281 .25413 L .68114 .26635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65157 .26209 m .67281 .25413 L .68114 .26635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62187 .25781 m .60088 .26574 L .59203 .25351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62187 .25781 m .60088 .26574 L .59203 .25351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57112 .26148 m .59203 .25351 L .60088 .26574 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57112 .26148 m .59203 .25351 L .60088 .26574 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54123 .2572 m .52058 .26513 L .5112 .2529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54123 .2572 m .52058 .26513 L .5112 .2529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49063 .26087 m .5112 .2529 L .52058 .26513 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49063 .26087 m .5112 .2529 L .52058 .26513 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46054 .25658 m .44022 .26453 L .43032 .25228 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46054 .25658 m .44022 .26453 L .43032 .25228 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41008 .26026 m .43032 .25228 L .44022 .26453 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41008 .26026 m .43032 .25228 L .44022 .26453 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3798 .25597 m .35981 .26392 L .34938 .25166 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3798 .25597 m .35981 .26392 L .34938 .25166 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32947 .25965 m .34938 .25166 L .35981 .26392 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32947 .25965 m .34938 .25166 L .35981 .26392 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.299 .25536 m .27935 .26331 L .26839 .25105 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.299 .25536 m .27935 .26331 L .26839 .25105 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24882 .25903 m .26839 .25105 L .27935 .26331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24882 .25903 m .26839 .25105 L .27935 .26331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75353 .25474 m .73196 .2627 L .72395 .25043 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75353 .25474 m .73196 .2627 L .72395 .25043 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70245 .25842 m .72395 .25043 L .73196 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70245 .25842 m .72395 .25043 L .73196 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21816 .25474 m .19883 .2627 L .18735 .25043 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21816 .25474 m .19883 .2627 L .18735 .25043 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16811 .25842 m .18735 .25043 L .19883 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16811 .25842 m .18735 .25043 L .19883 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67281 .25413 m .65157 .26209 L .64303 .24981 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67281 .25413 m .65157 .26209 L .64303 .24981 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62187 .25781 m .64303 .24981 L .65157 .26209 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62187 .25781 m .64303 .24981 L .65157 .26209 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59203 .25351 m .57112 .26148 L .56206 .24919 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59203 .25351 m .57112 .26148 L .56206 .24919 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54123 .2572 m .56206 .24919 L .57112 .26148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54123 .2572 m .56206 .24919 L .57112 .26148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5112 .2529 m .49063 .26087 L .48103 .24858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5112 .2529 m .49063 .26087 L .48103 .24858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46054 .25658 m .48103 .24858 L .49063 .26087 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46054 .25658 m .48103 .24858 L .49063 .26087 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43032 .25228 m .41008 .26026 L .39995 .24796 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43032 .25228 m .41008 .26026 L .39995 .24796 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3798 .25597 m .39995 .24796 L .41008 .26026 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3798 .25597 m .39995 .24796 L .41008 .26026 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34938 .25166 m .32947 .25965 L .31882 .24734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34938 .25166 m .32947 .25965 L .31882 .24734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.299 .25536 m .31882 .24734 L .32947 .25965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.299 .25536 m .31882 .24734 L .32947 .25965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75353 .25474 m .77529 .24672 L .78299 .25903 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75353 .25474 m .77529 .24672 L .78299 .25903 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26839 .25105 m .24882 .25903 L .23764 .24672 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26839 .25105 m .24882 .25903 L .23764 .24672 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21816 .25474 m .23764 .24672 L .24882 .25903 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21816 .25474 m .23764 .24672 L .24882 .25903 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72395 .25043 m .70245 .25842 L .69422 .2461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72395 .25043 m .70245 .25842 L .69422 .2461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67281 .25413 m .69422 .2461 L .70245 .25842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67281 .25413 m .69422 .2461 L .70245 .25842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64303 .24981 m .62187 .25781 L .61311 .24548 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64303 .24981 m .62187 .25781 L .61311 .24548 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59203 .25351 m .61311 .24548 L .62187 .25781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59203 .25351 m .61311 .24548 L .62187 .25781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56206 .24919 m .54123 .2572 L .53195 .24486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56206 .24919 m .54123 .2572 L .53195 .24486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5112 .2529 m .53195 .24486 L .54123 .2572 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5112 .2529 m .53195 .24486 L .54123 .2572 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48103 .24858 m .46054 .25658 L .45073 .24424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48103 .24858 m .46054 .25658 L .45073 .24424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43032 .25228 m .45073 .24424 L .46054 .25658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43032 .25228 m .45073 .24424 L .46054 .25658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39995 .24796 m .3798 .25597 L .36945 .24361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39995 .24796 m .3798 .25597 L .36945 .24361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34938 .25166 m .36945 .24361 L .3798 .25597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34938 .25166 m .36945 .24361 L .3798 .25597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31882 .24734 m .299 .25536 L .28813 .24299 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31882 .24734 m .299 .25536 L .28813 .24299 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26839 .25105 m .28813 .24299 L .299 .25536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26839 .25105 m .28813 .24299 L .299 .25536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77529 .24672 m .75353 .25474 L .74562 .24237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77529 .24672 m .75353 .25474 L .74562 .24237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72395 .25043 m .74562 .24237 L .75353 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72395 .25043 m .74562 .24237 L .75353 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23764 .24672 m .21816 .25474 L .20675 .24237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23764 .24672 m .21816 .25474 L .20675 .24237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18735 .25043 m .20675 .24237 L .21816 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18735 .25043 m .20675 .24237 L .21816 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69422 .2461 m .67281 .25413 L .66437 .24175 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69422 .2461 m .67281 .25413 L .66437 .24175 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64303 .24981 m .66437 .24175 L .67281 .25413 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64303 .24981 m .66437 .24175 L .67281 .25413 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61311 .24548 m .59203 .25351 L .58306 .24112 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61311 .24548 m .59203 .25351 L .58306 .24112 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56206 .24919 m .58306 .24112 L .59203 .25351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56206 .24919 m .58306 .24112 L .59203 .25351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53195 .24486 m .5112 .2529 L .5017 .2405 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53195 .24486 m .5112 .2529 L .5017 .2405 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48103 .24858 m .5017 .2405 L .5112 .2529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48103 .24858 m .5017 .2405 L .5112 .2529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45073 .24424 m .43032 .25228 L .42028 .23988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45073 .24424 m .43032 .25228 L .42028 .23988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39995 .24796 m .42028 .23988 L .43032 .25228 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39995 .24796 m .42028 .23988 L .43032 .25228 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36945 .24361 m .34938 .25166 L .33881 .23925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36945 .24361 m .34938 .25166 L .33881 .23925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31882 .24734 m .33881 .23925 L .34938 .25166 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31882 .24734 m .33881 .23925 L .34938 .25166 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28813 .24299 m .26839 .25105 L .25729 .23863 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28813 .24299 m .26839 .25105 L .25729 .23863 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23764 .24672 m .25729 .23863 L .26839 .25105 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23764 .24672 m .25729 .23863 L .26839 .25105 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74562 .24237 m .72395 .25043 L .71582 .238 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74562 .24237 m .72395 .25043 L .71582 .238 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69422 .2461 m .71582 .238 L .72395 .25043 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69422 .2461 m .71582 .238 L .72395 .25043 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66437 .24175 m .64303 .24981 L .63437 .23738 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66437 .24175 m .64303 .24981 L .63437 .23738 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61311 .24548 m .63437 .23738 L .64303 .24981 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61311 .24548 m .63437 .23738 L .64303 .24981 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58306 .24112 m .56206 .24919 L .55287 .23675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58306 .24112 m .56206 .24919 L .55287 .23675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53195 .24486 m .55287 .23675 L .56206 .24919 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53195 .24486 m .55287 .23675 L .56206 .24919 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5017 .2405 m .48103 .24858 L .47131 .23612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5017 .2405 m .48103 .24858 L .47131 .23612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45073 .24424 m .47131 .23612 L .48103 .24858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45073 .24424 m .47131 .23612 L .48103 .24858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42028 .23988 m .39995 .24796 L .3897 .2355 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42028 .23988 m .39995 .24796 L .3897 .2355 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36945 .24361 m .3897 .2355 L .39995 .24796 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36945 .24361 m .3897 .2355 L .39995 .24796 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33881 .23925 m .31882 .24734 L .30804 .23487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33881 .23925 m .31882 .24734 L .30804 .23487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28813 .24299 m .30804 .23487 L .31882 .24734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28813 .24299 m .30804 .23487 L .31882 .24734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74562 .24237 m .76748 .23424 L .77529 .24672 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74562 .24237 m .76748 .23424 L .77529 .24672 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25729 .23863 m .23764 .24672 L .22632 .23424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25729 .23863 m .23764 .24672 L .22632 .23424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20675 .24237 m .22632 .23424 L .23764 .24672 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20675 .24237 m .22632 .23424 L .23764 .24672 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71582 .238 m .69422 .2461 L .68589 .23361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71582 .238 m .69422 .2461 L .68589 .23361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66437 .24175 m .68589 .23361 L .69422 .2461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66437 .24175 m .68589 .23361 L .69422 .2461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63437 .23738 m .61311 .24548 L .60424 .23298 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63437 .23738 m .61311 .24548 L .60424 .23298 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58306 .24112 m .60424 .23298 L .61311 .24548 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58306 .24112 m .60424 .23298 L .61311 .24548 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55287 .23675 m .53195 .24486 L .52254 .23236 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55287 .23675 m .53195 .24486 L .52254 .23236 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5017 .2405 m .52254 .23236 L .53195 .24486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5017 .2405 m .52254 .23236 L .53195 .24486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47131 .23612 m .45073 .24424 L .44078 .23173 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47131 .23612 m .45073 .24424 L .44078 .23173 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42028 .23988 m .44078 .23173 L .45073 .24424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42028 .23988 m .44078 .23173 L .45073 .24424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3897 .2355 m .36945 .24361 L .35898 .2311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3897 .2355 m .36945 .24361 L .35898 .2311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33881 .23925 m .35898 .2311 L .36945 .24361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33881 .23925 m .35898 .2311 L .36945 .24361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30804 .23487 m .28813 .24299 L .27711 .23047 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30804 .23487 m .28813 .24299 L .27711 .23047 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25729 .23863 m .27711 .23047 L .28813 .24299 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25729 .23863 m .27711 .23047 L .28813 .24299 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76748 .23424 m .74562 .24237 L .73761 .22984 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76748 .23424 m .74562 .24237 L .73761 .22984 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71582 .238 m .73761 .22984 L .74562 .24237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71582 .238 m .73761 .22984 L .74562 .24237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68589 .23361 m .66437 .24175 L .65582 .2292 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68589 .23361 m .66437 .24175 L .65582 .2292 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63437 .23738 m .65582 .2292 L .66437 .24175 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63437 .23738 m .65582 .2292 L .66437 .24175 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60424 .23298 m .58306 .24112 L .57397 .22857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60424 .23298 m .58306 .24112 L .57397 .22857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55287 .23675 m .57397 .22857 L .58306 .24112 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55287 .23675 m .57397 .22857 L .58306 .24112 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52254 .23236 m .5017 .2405 L .49207 .22794 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52254 .23236 m .5017 .2405 L .49207 .22794 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47131 .23612 m .49207 .22794 L .5017 .2405 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47131 .23612 m .49207 .22794 L .5017 .2405 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44078 .23173 m .42028 .23988 L .41012 .22731 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44078 .23173 m .42028 .23988 L .41012 .22731 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3897 .2355 m .41012 .22731 L .42028 .23988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3897 .2355 m .41012 .22731 L .42028 .23988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35898 .2311 m .33881 .23925 L .32811 .22668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35898 .2311 m .33881 .23925 L .32811 .22668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30804 .23487 m .32811 .22668 L .33881 .23925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30804 .23487 m .32811 .22668 L .33881 .23925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27711 .23047 m .25729 .23863 L .24605 .22604 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27711 .23047 m .25729 .23863 L .24605 .22604 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22632 .23424 m .24605 .22604 L .25729 .23863 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22632 .23424 m .24605 .22604 L .25729 .23863 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73761 .22984 m .71582 .238 L .70759 .22541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73761 .22984 m .71582 .238 L .70759 .22541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68589 .23361 m .70759 .22541 L .71582 .238 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68589 .23361 m .70759 .22541 L .71582 .238 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65582 .2292 m .63437 .23738 L .6256 .22478 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65582 .2292 m .63437 .23738 L .6256 .22478 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60424 .23298 m .6256 .22478 L .63437 .23738 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60424 .23298 m .6256 .22478 L .63437 .23738 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57397 .22857 m .55287 .23675 L .54356 .22414 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57397 .22857 m .55287 .23675 L .54356 .22414 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52254 .23236 m .54356 .22414 L .55287 .23675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52254 .23236 m .54356 .22414 L .55287 .23675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49207 .22794 m .47131 .23612 L .46146 .22351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49207 .22794 m .47131 .23612 L .46146 .22351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44078 .23173 m .46146 .22351 L .47131 .23612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44078 .23173 m .46146 .22351 L .47131 .23612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41012 .22731 m .3897 .2355 L .37931 .22287 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41012 .22731 m .3897 .2355 L .37931 .22287 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35898 .2311 m .37931 .22287 L .3897 .2355 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35898 .2311 m .37931 .22287 L .3897 .2355 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32811 .22668 m .30804 .23487 L .2971 .22224 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32811 .22668 m .30804 .23487 L .2971 .22224 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27711 .23047 m .2971 .22224 L .30804 .23487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27711 .23047 m .2971 .22224 L .30804 .23487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73761 .22984 m .75958 .2216 L .76748 .23424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73761 .22984 m .75958 .2216 L .76748 .23424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70759 .22541 m .68589 .23361 L .67744 .22096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70759 .22541 m .68589 .23361 L .67744 .22096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65582 .2292 m .67744 .22096 L .68589 .23361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65582 .2292 m .67744 .22096 L .68589 .23361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6256 .22478 m .60424 .23298 L .59525 .22033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6256 .22478 m .60424 .23298 L .59525 .22033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57397 .22857 m .59525 .22033 L .60424 .23298 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57397 .22857 m .59525 .22033 L .60424 .23298 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54356 .22414 m .52254 .23236 L .51301 .21969 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54356 .22414 m .52254 .23236 L .51301 .21969 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49207 .22794 m .51301 .21969 L .52254 .23236 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49207 .22794 m .51301 .21969 L .52254 .23236 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46146 .22351 m .44078 .23173 L .43071 .21905 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46146 .22351 m .44078 .23173 L .43071 .21905 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41012 .22731 m .43071 .21905 L .44078 .23173 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41012 .22731 m .43071 .21905 L .44078 .23173 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37931 .22287 m .35898 .2311 L .34836 .21841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37931 .22287 m .35898 .2311 L .34836 .21841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32811 .22668 m .34836 .21841 L .35898 .2311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32811 .22668 m .34836 .21841 L .35898 .2311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2971 .22224 m .27711 .23047 L .26595 .21777 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2971 .22224 m .27711 .23047 L .26595 .21777 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24605 .22604 m .26595 .21777 L .27711 .23047 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24605 .22604 m .26595 .21777 L .27711 .23047 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75958 .2216 m .73761 .22984 L .72949 .21714 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75958 .2216 m .73761 .22984 L .72949 .21714 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70759 .22541 m .72949 .21714 L .73761 .22984 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70759 .22541 m .72949 .21714 L .73761 .22984 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67744 .22096 m .65582 .2292 L .64715 .2165 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67744 .22096 m .65582 .2292 L .64715 .2165 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6256 .22478 m .64715 .2165 L .65582 .2292 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6256 .22478 m .64715 .2165 L .65582 .2292 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59525 .22033 m .57397 .22857 L .56476 .21586 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59525 .22033 m .57397 .22857 L .56476 .21586 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54356 .22414 m .56476 .21586 L .57397 .22857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54356 .22414 m .56476 .21586 L .57397 .22857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51301 .21969 m .49207 .22794 L .48232 .21522 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51301 .21969 m .49207 .22794 L .48232 .21522 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46146 .22351 m .48232 .21522 L .49207 .22794 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46146 .22351 m .48232 .21522 L .49207 .22794 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43071 .21905 m .41012 .22731 L .39982 .21458 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43071 .21905 m .41012 .22731 L .39982 .21458 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37931 .22287 m .39982 .21458 L .41012 .22731 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37931 .22287 m .39982 .21458 L .41012 .22731 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34836 .21841 m .32811 .22668 L .31727 .21393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34836 .21841 m .32811 .22668 L .31727 .21393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2971 .22224 m .31727 .21393 L .32811 .22668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2971 .22224 m .31727 .21393 L .32811 .22668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72949 .21714 m .70759 .22541 L .69925 .21265 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72949 .21714 m .70759 .22541 L .69925 .21265 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67744 .22096 m .69925 .21265 L .70759 .22541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67744 .22096 m .69925 .21265 L .70759 .22541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64715 .2165 m .6256 .22478 L .61672 .21201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64715 .2165 m .6256 .22478 L .61672 .21201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59525 .22033 m .61672 .21201 L .6256 .22478 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59525 .22033 m .61672 .21201 L .6256 .22478 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56476 .21586 m .54356 .22414 L .53413 .21137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56476 .21586 m .54356 .22414 L .53413 .21137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51301 .21969 m .53413 .21137 L .54356 .22414 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51301 .21969 m .53413 .21137 L .54356 .22414 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48232 .21522 m .46146 .22351 L .45148 .21072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48232 .21522 m .46146 .22351 L .45148 .21072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43071 .21905 m .45148 .21072 L .46146 .22351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43071 .21905 m .45148 .21072 L .46146 .22351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39982 .21458 m .37931 .22287 L .36878 .21008 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39982 .21458 m .37931 .22287 L .36878 .21008 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34836 .21841 m .36878 .21008 L .37931 .22287 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34836 .21841 m .36878 .21008 L .37931 .22287 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31727 .21393 m .2971 .22224 L .28603 .20943 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31727 .21393 m .2971 .22224 L .28603 .20943 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26595 .21777 m .28603 .20943 L .2971 .22224 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26595 .21777 m .28603 .20943 L .2971 .22224 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72949 .21714 m .75157 .20879 L .75958 .2216 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72949 .21714 m .75157 .20879 L .75958 .2216 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69925 .21265 m .67744 .22096 L .66888 .20814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69925 .21265 m .67744 .22096 L .66888 .20814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64715 .2165 m .66888 .20814 L .67744 .22096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64715 .2165 m .66888 .20814 L .67744 .22096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61672 .21201 m .59525 .22033 L .58615 .2075 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61672 .21201 m .59525 .22033 L .58615 .2075 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56476 .21586 m .58615 .2075 L .59525 .22033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56476 .21586 m .58615 .2075 L .59525 .22033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53413 .21137 m .51301 .21969 L .50335 .20685 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53413 .21137 m .51301 .21969 L .50335 .20685 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48232 .21522 m .50335 .20685 L .51301 .21969 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48232 .21522 m .50335 .20685 L .51301 .21969 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45148 .21072 m .43071 .21905 L .4205 .20621 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45148 .21072 m .43071 .21905 L .4205 .20621 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39982 .21458 m .4205 .20621 L .43071 .21905 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39982 .21458 m .4205 .20621 L .43071 .21905 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36878 .21008 m .34836 .21841 L .3376 .20556 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36878 .21008 m .34836 .21841 L .3376 .20556 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31727 .21393 m .3376 .20556 L .34836 .21841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31727 .21393 m .3376 .20556 L .34836 .21841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75157 .20879 m .72949 .21714 L .72126 .20427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75157 .20879 m .72949 .21714 L .72126 .20427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69925 .21265 m .72126 .20427 L .72949 .21714 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69925 .21265 m .72126 .20427 L .72949 .21714 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66888 .20814 m .64715 .2165 L .63837 .20362 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66888 .20814 m .64715 .2165 L .63837 .20362 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61672 .21201 m .63837 .20362 L .64715 .2165 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61672 .21201 m .63837 .20362 L .64715 .2165 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58615 .2075 m .56476 .21586 L .55543 .20297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58615 .2075 m .56476 .21586 L .55543 .20297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53413 .21137 m .55543 .20297 L .56476 .21586 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53413 .21137 m .55543 .20297 L .56476 .21586 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50335 .20685 m .48232 .21522 L .47243 .20232 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50335 .20685 m .48232 .21522 L .47243 .20232 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45148 .21072 m .47243 .20232 L .48232 .21522 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45148 .21072 m .47243 .20232 L .48232 .21522 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4205 .20621 m .39982 .21458 L .38938 .20167 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4205 .20621 m .39982 .21458 L .38938 .20167 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36878 .21008 m .38938 .20167 L .39982 .21458 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36878 .21008 m .38938 .20167 L .39982 .21458 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3376 .20556 m .31727 .21393 L .30627 .20102 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3376 .20556 m .31727 .21393 L .30627 .20102 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28603 .20943 m .30627 .20102 L .31727 .21393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28603 .20943 m .30627 .20102 L .31727 .21393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72126 .20427 m .69925 .21265 L .6908 .19972 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72126 .20427 m .69925 .21265 L .6908 .19972 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66888 .20814 m .6908 .19972 L .69925 .21265 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66888 .20814 m .6908 .19972 L .69925 .21265 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63837 .20362 m .61672 .21201 L .60771 .19907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63837 .20362 m .61672 .21201 L .60771 .19907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58615 .2075 m .60771 .19907 L .61672 .21201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58615 .2075 m .60771 .19907 L .61672 .21201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55543 .20297 m .53413 .21137 L .52457 .19842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55543 .20297 m .53413 .21137 L .52457 .19842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50335 .20685 m .52457 .19842 L .53413 .21137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50335 .20685 m .52457 .19842 L .53413 .21137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47243 .20232 m .45148 .21072 L .44137 .19777 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47243 .20232 m .45148 .21072 L .44137 .19777 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4205 .20621 m .44137 .19777 L .45148 .21072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4205 .20621 m .44137 .19777 L .45148 .21072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38938 .20167 m .36878 .21008 L .35811 .19711 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38938 .20167 m .36878 .21008 L .35811 .19711 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3376 .20556 m .35811 .19711 L .36878 .21008 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3376 .20556 m .35811 .19711 L .36878 .21008 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72126 .20427 m .74345 .19581 L .75157 .20879 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72126 .20427 m .74345 .19581 L .75157 .20879 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6908 .19972 m .66888 .20814 L .66021 .19516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6908 .19972 m .66888 .20814 L .66021 .19516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63837 .20362 m .66021 .19516 L .66888 .20814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63837 .20362 m .66021 .19516 L .66888 .20814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60771 .19907 m .58615 .2075 L .57692 .1945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60771 .19907 m .58615 .2075 L .57692 .1945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55543 .20297 m .57692 .1945 L .58615 .2075 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55543 .20297 m .57692 .1945 L .58615 .2075 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52457 .19842 m .50335 .20685 L .49357 .19385 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52457 .19842 m .50335 .20685 L .49357 .19385 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47243 .20232 m .49357 .19385 L .50335 .20685 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47243 .20232 m .49357 .19385 L .50335 .20685 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44137 .19777 m .4205 .20621 L .41016 .19319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44137 .19777 m .4205 .20621 L .41016 .19319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38938 .20167 m .41016 .19319 L .4205 .20621 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38938 .20167 m .41016 .19319 L .4205 .20621 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35811 .19711 m .3376 .20556 L .3267 .19254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35811 .19711 m .3376 .20556 L .3267 .19254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30627 .20102 m .3267 .19254 L .3376 .20556 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30627 .20102 m .3267 .19254 L .3376 .20556 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74345 .19581 m .72126 .20427 L .71292 .19122 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74345 .19581 m .72126 .20427 L .71292 .19122 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6908 .19972 m .71292 .19122 L .72126 .20427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6908 .19972 m .71292 .19122 L .72126 .20427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66021 .19516 m .63837 .20362 L .62947 .19057 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66021 .19516 m .63837 .20362 L .62947 .19057 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60771 .19907 m .62947 .19057 L .63837 .20362 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60771 .19907 m .62947 .19057 L .63837 .20362 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57692 .1945 m .55543 .20297 L .54597 .18991 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57692 .1945 m .55543 .20297 L .54597 .18991 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52457 .19842 m .54597 .18991 L .55543 .20297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52457 .19842 m .54597 .18991 L .55543 .20297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49357 .19385 m .47243 .20232 L .46242 .18925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49357 .19385 m .47243 .20232 L .46242 .18925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44137 .19777 m .46242 .18925 L .47243 .20232 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44137 .19777 m .46242 .18925 L .47243 .20232 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41016 .19319 m .38938 .20167 L .37881 .1886 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41016 .19319 m .38938 .20167 L .37881 .1886 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35811 .19711 m .37881 .1886 L .38938 .20167 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35811 .19711 m .37881 .1886 L .38938 .20167 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71292 .19122 m .6908 .19972 L .68224 .18662 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71292 .19122 m .6908 .19972 L .68224 .18662 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66021 .19516 m .68224 .18662 L .6908 .19972 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66021 .19516 m .68224 .18662 L .6908 .19972 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62947 .19057 m .60771 .19907 L .59859 .18596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62947 .19057 m .60771 .19907 L .59859 .18596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57692 .1945 m .59859 .18596 L .60771 .19907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57692 .1945 m .59859 .18596 L .60771 .19907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54597 .18991 m .52457 .19842 L .51488 .1853 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54597 .18991 m .52457 .19842 L .51488 .1853 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49357 .19385 m .51488 .1853 L .52457 .19842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49357 .19385 m .51488 .1853 L .52457 .19842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46242 .18925 m .44137 .19777 L .43112 .18464 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46242 .18925 m .44137 .19777 L .43112 .18464 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41016 .19319 m .43112 .18464 L .44137 .19777 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41016 .19319 m .43112 .18464 L .44137 .19777 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37881 .1886 m .35811 .19711 L .3473 .18398 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37881 .1886 m .35811 .19711 L .3473 .18398 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3267 .19254 m .3473 .18398 L .35811 .19711 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3267 .19254 m .3473 .18398 L .35811 .19711 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71292 .19122 m .73522 .18265 L .74345 .19581 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71292 .19122 m .73522 .18265 L .74345 .19581 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68224 .18662 m .66021 .19516 L .65142 .18199 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68224 .18662 m .66021 .19516 L .65142 .18199 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62947 .19057 m .65142 .18199 L .66021 .19516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62947 .19057 m .65142 .18199 L .66021 .19516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59859 .18596 m .57692 .1945 L .56756 .18133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59859 .18596 m .57692 .1945 L .56756 .18133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54597 .18991 m .56756 .18133 L .57692 .1945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54597 .18991 m .56756 .18133 L .57692 .1945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51488 .1853 m .49357 .19385 L .48365 .18066 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51488 .1853 m .49357 .19385 L .48365 .18066 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46242 .18925 m .48365 .18066 L .49357 .19385 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46242 .18925 m .48365 .18066 L .49357 .19385 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43112 .18464 m .41016 .19319 L .39968 .18 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43112 .18464 m .41016 .19319 L .39968 .18 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37881 .1886 m .39968 .18 L .41016 .19319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37881 .1886 m .39968 .18 L .41016 .19319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73522 .18265 m .71292 .19122 L .70446 .17801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73522 .18265 m .71292 .19122 L .70446 .17801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68224 .18662 m .70446 .17801 L .71292 .19122 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68224 .18662 m .70446 .17801 L .71292 .19122 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65142 .18199 m .62947 .19057 L .62045 .17734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65142 .18199 m .62947 .19057 L .62045 .17734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59859 .18596 m .62045 .17734 L .62947 .19057 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59859 .18596 m .62045 .17734 L .62947 .19057 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56756 .18133 m .54597 .18991 L .53639 .17668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56756 .18133 m .54597 .18991 L .53639 .17668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51488 .1853 m .53639 .17668 L .54597 .18991 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51488 .1853 m .53639 .17668 L .54597 .18991 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48365 .18066 m .46242 .18925 L .45227 .17601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48365 .18066 m .46242 .18925 L .45227 .17601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43112 .18464 m .45227 .17601 L .46242 .18925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43112 .18464 m .45227 .17601 L .46242 .18925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39968 .18 m .37881 .1886 L .36809 .17534 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39968 .18 m .37881 .1886 L .36809 .17534 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3473 .18398 m .36809 .17534 L .37881 .1886 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3473 .18398 m .36809 .17534 L .37881 .1886 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70446 .17801 m .68224 .18662 L .67356 .17334 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70446 .17801 m .68224 .18662 L .67356 .17334 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65142 .18199 m .67356 .17334 L .68224 .18662 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65142 .18199 m .67356 .17334 L .68224 .18662 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62045 .17734 m .59859 .18596 L .58934 .17267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62045 .17734 m .59859 .18596 L .58934 .17267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56756 .18133 m .58934 .17267 L .59859 .18596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56756 .18133 m .58934 .17267 L .59859 .18596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53639 .17668 m .51488 .1853 L .50507 .172 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53639 .17668 m .51488 .1853 L .50507 .172 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48365 .18066 m .50507 .172 L .51488 .1853 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48365 .18066 m .50507 .172 L .51488 .1853 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45227 .17601 m .43112 .18464 L .42073 .17133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45227 .17601 m .43112 .18464 L .42073 .17133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39968 .18 m .42073 .17133 L .43112 .18464 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39968 .18 m .42073 .17133 L .43112 .18464 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70446 .17801 m .72688 .16932 L .73522 .18265 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70446 .17801 m .72688 .16932 L .73522 .18265 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67356 .17334 m .65142 .18199 L .64251 .16865 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67356 .17334 m .65142 .18199 L .64251 .16865 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62045 .17734 m .64251 .16865 L .65142 .18199 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62045 .17734 m .64251 .16865 L .65142 .18199 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58934 .17267 m .56756 .18133 L .55808 .16798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58934 .17267 m .56756 .18133 L .55808 .16798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53639 .17668 m .55808 .16798 L .56756 .18133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53639 .17668 m .55808 .16798 L .56756 .18133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50507 .172 m .48365 .18066 L .4736 .1673 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50507 .172 m .48365 .18066 L .4736 .1673 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45227 .17601 m .4736 .1673 L .48365 .18066 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45227 .17601 m .4736 .1673 L .48365 .18066 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42073 .17133 m .39968 .18 L .38905 .16663 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42073 .17133 m .39968 .18 L .38905 .16663 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36809 .17534 m .38905 .16663 L .39968 .18 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36809 .17534 m .38905 .16663 L .39968 .18 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72688 .16932 m .70446 .17801 L .6959 .16461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72688 .16932 m .70446 .17801 L .6959 .16461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67356 .17334 m .6959 .16461 L .70446 .17801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67356 .17334 m .6959 .16461 L .70446 .17801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64251 .16865 m .62045 .17734 L .61131 .16394 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64251 .16865 m .62045 .17734 L .61131 .16394 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58934 .17267 m .61131 .16394 L .62045 .17734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58934 .17267 m .61131 .16394 L .62045 .17734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55808 .16798 m .53639 .17668 L .52667 .16326 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55808 .16798 m .53639 .17668 L .52667 .16326 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50507 .172 m .52667 .16326 L .53639 .17668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50507 .172 m .52667 .16326 L .53639 .17668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4736 .1673 m .45227 .17601 L .44198 .16258 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4736 .1673 m .45227 .17601 L .44198 .16258 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42073 .17133 m .44198 .16258 L .45227 .17601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42073 .17133 m .44198 .16258 L .45227 .17601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6959 .16461 m .67356 .17334 L .66476 .15988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6959 .16461 m .67356 .17334 L .66476 .15988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64251 .16865 m .66476 .15988 L .67356 .17334 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64251 .16865 m .66476 .15988 L .67356 .17334 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61131 .16394 m .58934 .17267 L .57997 .1592 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61131 .16394 m .58934 .17267 L .57997 .1592 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55808 .16798 m .57997 .1592 L .58934 .17267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55808 .16798 m .57997 .1592 L .58934 .17267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52667 .16326 m .50507 .172 L .49512 .15852 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52667 .16326 m .50507 .172 L .49512 .15852 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4736 .1673 m .49512 .15852 L .50507 .172 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4736 .1673 m .49512 .15852 L .50507 .172 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44198 .16258 m .42073 .17133 L .41021 .15784 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44198 .16258 m .42073 .17133 L .41021 .15784 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38905 .16663 m .41021 .15784 L .42073 .17133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38905 .16663 m .41021 .15784 L .42073 .17133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6959 .16461 m .71843 .1558 L .72688 .16932 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6959 .16461 m .71843 .1558 L .72688 .16932 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66476 .15988 m .64251 .16865 L .63348 .15512 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66476 .15988 m .64251 .16865 L .63348 .15512 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61131 .16394 m .63348 .15512 L .64251 .16865 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61131 .16394 m .63348 .15512 L .64251 .16865 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57997 .1592 m .55808 .16798 L .54847 .15444 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57997 .1592 m .55808 .16798 L .54847 .15444 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52667 .16326 m .54847 .15444 L .55808 .16798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52667 .16326 m .54847 .15444 L .55808 .16798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49512 .15852 m .4736 .1673 L .46341 .15376 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49512 .15852 m .4736 .1673 L .46341 .15376 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44198 .16258 m .46341 .15376 L .4736 .1673 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44198 .16258 m .46341 .15376 L .4736 .1673 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71843 .1558 m .6959 .16461 L .68721 .15103 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71843 .1558 m .6959 .16461 L .68721 .15103 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66476 .15988 m .68721 .15103 L .6959 .16461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66476 .15988 m .68721 .15103 L .6959 .16461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63348 .15512 m .61131 .16394 L .60205 .15035 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63348 .15512 m .61131 .16394 L .60205 .15035 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57997 .1592 m .60205 .15035 L .61131 .16394 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57997 .1592 m .60205 .15035 L .61131 .16394 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54847 .15444 m .52667 .16326 L .51683 .14966 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54847 .15444 m .52667 .16326 L .51683 .14966 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49512 .15852 m .51683 .14966 L .52667 .16326 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49512 .15852 m .51683 .14966 L .52667 .16326 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46341 .15376 m .44198 .16258 L .43155 .14898 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46341 .15376 m .44198 .16258 L .43155 .14898 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41021 .15784 m .43155 .14898 L .44198 .16258 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41021 .15784 m .43155 .14898 L .44198 .16258 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68721 .15103 m .66476 .15988 L .65585 .14623 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68721 .15103 m .66476 .15988 L .65585 .14623 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63348 .15512 m .65585 .14623 L .66476 .15988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63348 .15512 m .65585 .14623 L .66476 .15988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60205 .15035 m .57997 .1592 L .57047 .14555 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60205 .15035 m .57997 .1592 L .57047 .14555 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54847 .15444 m .57047 .14555 L .57997 .1592 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54847 .15444 m .57047 .14555 L .57997 .1592 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51683 .14966 m .49512 .15852 L .48503 .14486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51683 .14966 m .49512 .15852 L .48503 .14486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46341 .15376 m .48503 .14486 L .49512 .15852 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46341 .15376 m .48503 .14486 L .49512 .15852 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68721 .15103 m .70986 .1421 L .71843 .1558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68721 .15103 m .70986 .1421 L .71843 .1558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65585 .14623 m .63348 .15512 L .62433 .14141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65585 .14623 m .63348 .15512 L .62433 .14141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60205 .15035 m .62433 .14141 L .63348 .15512 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60205 .15035 m .62433 .14141 L .63348 .15512 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57047 .14555 m .54847 .15444 L .53873 .14072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57047 .14555 m .54847 .15444 L .53873 .14072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51683 .14966 m .53873 .14072 L .54847 .15444 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51683 .14966 m .53873 .14072 L .54847 .15444 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48503 .14486 m .46341 .15376 L .45308 .14003 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48503 .14486 m .46341 .15376 L .45308 .14003 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43155 .14898 m .45308 .14003 L .46341 .15376 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43155 .14898 m .45308 .14003 L .46341 .15376 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70986 .1421 m .68721 .15103 L .67841 .13726 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70986 .1421 m .68721 .15103 L .67841 .13726 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65585 .14623 m .67841 .13726 L .68721 .15103 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65585 .14623 m .67841 .13726 L .68721 .15103 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62433 .14141 m .60205 .15035 L .59266 .13657 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62433 .14141 m .60205 .15035 L .59266 .13657 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57047 .14555 m .59266 .13657 L .60205 .15035 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57047 .14555 m .59266 .13657 L .60205 .15035 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53873 .14072 m .51683 .14966 L .50684 .13588 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53873 .14072 m .51683 .14966 L .50684 .13588 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48503 .14486 m .50684 .13588 L .51683 .14966 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48503 .14486 m .50684 .13588 L .51683 .14966 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67841 .13726 m .65585 .14623 L .6468 .1324 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67841 .13726 m .65585 .14623 L .6468 .1324 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62433 .14141 m .6468 .1324 L .65585 .14623 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62433 .14141 m .6468 .1324 L .65585 .14623 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59266 .13657 m .57047 .14555 L .56083 .1317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59266 .13657 m .57047 .14555 L .56083 .1317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53873 .14072 m .56083 .1317 L .57047 .14555 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53873 .14072 m .56083 .1317 L .57047 .14555 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50684 .13588 m .48503 .14486 L .4748 .13101 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50684 .13588 m .48503 .14486 L .4748 .13101 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45308 .14003 m .4748 .13101 L .48503 .14486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45308 .14003 m .4748 .13101 L .48503 .14486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67841 .13726 m .70118 .12821 L .70986 .1421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67841 .13726 m .70118 .12821 L .70986 .1421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6468 .1324 m .62433 .14141 L .61505 .12751 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6468 .1324 m .62433 .14141 L .61505 .12751 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59266 .13657 m .61505 .12751 L .62433 .14141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59266 .13657 m .61505 .12751 L .62433 .14141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56083 .1317 m .53873 .14072 L .52885 .12681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56083 .1317 m .53873 .14072 L .52885 .12681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50684 .13588 m .52885 .12681 L .53873 .14072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50684 .13588 m .52885 .12681 L .53873 .14072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70118 .12821 m .67841 .13726 L .66948 .12331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70118 .12821 m .67841 .13726 L .66948 .12331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6468 .1324 m .66948 .12331 L .67841 .13726 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6468 .1324 m .66948 .12331 L .67841 .13726 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61505 .12751 m .59266 .13657 L .58313 .1226 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61505 .12751 m .59266 .13657 L .58313 .1226 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56083 .1317 m .58313 .1226 L .59266 .13657 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56083 .1317 m .58313 .1226 L .59266 .13657 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52885 .12681 m .50684 .13588 L .49672 .1219 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52885 .12681 m .50684 .13588 L .49672 .1219 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4748 .13101 m .49672 .1219 L .50684 .13588 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4748 .13101 m .49672 .1219 L .50684 .13588 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66948 .12331 m .6468 .1324 L .63764 .11837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66948 .12331 m .6468 .1324 L .63764 .11837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61505 .12751 m .63764 .11837 L .6468 .1324 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61505 .12751 m .63764 .11837 L .6468 .1324 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58313 .1226 m .56083 .1317 L .55107 .11767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58313 .1226 m .56083 .1317 L .55107 .11767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52885 .12681 m .55107 .11767 L .56083 .1317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52885 .12681 m .55107 .11767 L .56083 .1317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66948 .12331 m .69237 .11413 L .70118 .12821 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66948 .12331 m .69237 .11413 L .70118 .12821 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63764 .11837 m .61505 .12751 L .60564 .11342 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63764 .11837 m .61505 .12751 L .60564 .11342 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58313 .1226 m .60564 .11342 L .61505 .12751 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58313 .1226 m .60564 .11342 L .61505 .12751 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55107 .11767 m .52885 .12681 L .51884 .11271 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55107 .11767 m .52885 .12681 L .51884 .11271 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49672 .1219 m .51884 .11271 L .52885 .12681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49672 .1219 m .51884 .11271 L .52885 .12681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69237 .11413 m .66948 .12331 L .66043 .10915 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69237 .11413 m .66948 .12331 L .66043 .10915 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63764 .11837 m .66043 .10915 L .66948 .12331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63764 .11837 m .66043 .10915 L .66948 .12331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60564 .11342 m .58313 .1226 L .57348 .10844 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60564 .11342 m .58313 .1226 L .57348 .10844 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55107 .11767 m .57348 .10844 L .58313 .1226 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55107 .11767 m .57348 .10844 L .58313 .1226 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66043 .10915 m .63764 .11837 L .62834 .10415 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66043 .10915 m .63764 .11837 L .62834 .10415 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60564 .11342 m .62834 .10415 L .63764 .11837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60564 .11342 m .62834 .10415 L .63764 .11837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57348 .10844 m .55107 .11767 L .54116 .10344 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57348 .10844 m .55107 .11767 L .54116 .10344 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51884 .11271 m .54116 .10344 L .55107 .11767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51884 .11271 m .54116 .10344 L .55107 .11767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66043 .10915 m .68344 .09985 L .69237 .11413 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66043 .10915 m .68344 .09985 L .69237 .11413 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62834 .10415 m .60564 .11342 L .59609 .09913 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62834 .10415 m .60564 .11342 L .59609 .09913 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57348 .10844 m .59609 .09913 L .60564 .11342 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57348 .10844 m .59609 .09913 L .60564 .11342 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68344 .09985 m .66043 .10915 L .65126 .0948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68344 .09985 m .66043 .10915 L .65126 .0948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62834 .10415 m .65126 .0948 L .66043 .10915 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62834 .10415 m .65126 .0948 L .66043 .10915 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59609 .09913 m .57348 .10844 L .56369 .09408 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59609 .09913 m .57348 .10844 L .56369 .09408 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54116 .10344 m .56369 .09408 L .57348 .10844 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54116 .10344 m .56369 .09408 L .57348 .10844 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65126 .0948 m .62834 .10415 L .61892 .08973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65126 .0948 m .62834 .10415 L .61892 .08973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59609 .09913 m .61892 .08973 L .62834 .10415 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59609 .09913 m .61892 .08973 L .62834 .10415 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65126 .0948 m .67438 .08536 L .68344 .09985 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65126 .0948 m .67438 .08536 L .68344 .09985 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61892 .08973 m .59609 .09913 L .58642 .08463 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61892 .08973 m .59609 .09913 L .58642 .08463 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56369 .09408 m .58642 .08463 L .59609 .09913 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56369 .09408 m .58642 .08463 L .59609 .09913 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67438 .08536 m .65126 .0948 L .64195 .08025 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67438 .08536 m .65126 .0948 L .64195 .08025 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61892 .08973 m .64195 .08025 L .65126 .0948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61892 .08973 m .64195 .08025 L .65126 .0948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64195 .08025 m .61892 .08973 L .60936 .0751 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64195 .08025 m .61892 .08973 L .60936 .0751 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58642 .08463 m .60936 .0751 L .61892 .08973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58642 .08463 m .60936 .0751 L .61892 .08973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64195 .08025 m .6652 .07068 L .67438 .08536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64195 .08025 m .6652 .07068 L .67438 .08536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6652 .07068 m .64195 .08025 L .63251 .06548 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6652 .07068 m .64195 .08025 L .63251 .06548 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60936 .0751 m .63251 .06548 L .64195 .08025 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60936 .0751 m .63251 .06548 L .64195 .08025 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63251 .06548 m .65588 .05578 L .6652 .07068 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63251 .06548 m .65588 .05578 L .6652 .07068 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66295 .02755 m
.90359 .41965 L
s
.90359 .41965 m
.97245 .92345 L
s
.97245 .92345 m
.69507 .58075 L
s
.69507 .58075 m
.66295 .02755 L
s
.11113 .25957 m
.02755 .78701 L
s
.02755 .78701 m
.69507 .58075 L
s
.69507 .58075 m
.66295 .02755 L
s
.66295 .02755 m
.11113 .25957 L
s
P
p
p
.002 w
.11113 .25957 m
.66295 .02755 L
s
P
p
.002 w
.11347 .25859 m
.11829 .26351 L
s
P
[(0)] .10384 .24874 .97891 1 Mshowa
p
.002 w
.30906 .17634 m
.31349 .18162 L
s
P
[(10)] .30021 .16579 .83912 1 Mshowa
p
.002 w
.52243 .08663 m
.52637 .09228 L
s
P
[(20)] .51453 .07535 .69933 1 Mshowa
p
.001 w
.15128 .24269 m
.15412 .24568 L
s
P
p
.001 w
.18972 .22652 m
.19252 .22956 L
s
P
p
.001 w
.22882 .21008 m
.23158 .21316 L
s
P
p
.001 w
.2686 .19336 m
.2713 .19648 L
s
P
p
.001 w
.35024 .15903 m
.35284 .16224 L
s
P
p
.001 w
.39214 .14141 m
.39469 .14467 L
s
P
p
.001 w
.4348 .12348 m
.43729 .12678 L
s
P
p
.001 w
.47822 .10522 m
.48065 .10856 L
s
P
p
.001 w
.56744 .0677 m
.56975 .07113 L
s
P
p
.001 w
.6133 .04843 m
.61553 .0519 L
s
P
p
.001 w
.66001 .02879 m
.66217 .0323 L
s
P
[(l)] .31959 .08564 .79718 1 Mshowa
P
p
P
% End of Graphics
MathPictureEnd

:[font = text; inactive; preserveAspect; endGroup]
What is the best combination of paramters values to use for a code?  Ideally, we should like to be able to encode the greatest number of logical qubits in the smallest number of physical qubits, whilst immunizing the computation from the greatest possible number of errors. Thus, an interesting quantity to consider is the product "l * tMax / n". Let's call this the "density" of the code. Below we compute and plot the density for a range of parameter values for l and n.
:[font = subsubsection; inactive; preserveAspect; startGroup]
Try me!
:[font = input; preserveAspect; startGroup]
density3D = Table[Table[N[l * MaxErrorsCorrectable[l, n]/n], 
                        {n, 1, 30}], 
                  {l, 1, 25}];
                  
BarChart3D[density3D, 
           FaceGrids->{{-1,0,0}, {0,1,0}},
           AxesLabel->{"l", "n", "l*tMax/n"},
           AxesEdge->{{-1,-1}, {1,-1}, {-1,-1}}];
:[font = postscript; PostScript; formatAsPostScript; output; inactive; preserveAspect; pictureLeft = 34; pictureWidth = 230; pictureHeight = 249; endGroup; endGroup; endGroup]
%!
%%Creator: Mathematica
%%AspectRatio: 1.08079 
MathPictureStart
%% Graphics3D
/Courier findfont 8  scalefont  setfont
% Scaling calculations
-0.0260524 1.10184 0.0275459 1.10184 [
[(0)] .10384 .24874 .97891 1 Msboxa
[(10)] .30021 .16579 .83912 1 Msboxa
[(20)] .51453 .07535 .69933 1 Msboxa
[(l)] .31959 .08564 .79718 1 Msboxa
[(0)] .67799 .02603 -1 .41952 Msboxa
[(10)] .76525 .16834 -1 .3848 Msboxa
[(20)] .84155 .2928 -1 .35539 Msboxa
[(30)] .90883 .40255 -1 .33016 Msboxa
[(n)] .87306 .21499 -1 .36805 Msboxa
[(0)] .09676 .27537 1 -0.41829 Msboxa
[(0.5)] .06923 .44767 1 -0.38222 Msboxa
[(1)] .03783 .64425 1 -0.34052 Msboxa
[(l*tMax/n)] -0.00432 .52779 1 -0.37036 Msboxa
[ 0 0 0 0 ]
[ 1 1.08079 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
[ ] 0 setdash
0 g
p
p
.002 w
.11113 .25957 m
.66295 .02755 L
s
P
p
.002 w
.11347 .25859 m
.11829 .26351 L
s
P
[(0)] .10384 .24874 .97891 1 Mshowa
p
.002 w
.30906 .17634 m
.31349 .18162 L
s
P
[(10)] .30021 .16579 .83912 1 Mshowa
p
.002 w
.52243 .08663 m
.52637 .09228 L
s
P
[(20)] .51453 .07535 .69933 1 Mshowa
p
.001 w
.15128 .24269 m
.15412 .24568 L
s
P
p
.001 w
.18972 .22652 m
.19252 .22956 L
s
P
p
.001 w
.22882 .21008 m
.23158 .21316 L
s
P
p
.001 w
.2686 .19336 m
.2713 .19648 L
s
P
p
.001 w
.35024 .15903 m
.35284 .16224 L
s
P
p
.001 w
.39214 .14141 m
.39469 .14467 L
s
P
p
.001 w
.4348 .12348 m
.43729 .12678 L
s
P
p
.001 w
.47822 .10522 m
.48065 .10856 L
s
P
p
.001 w
.56744 .0677 m
.56975 .07113 L
s
P
p
.001 w
.6133 .04843 m
.61553 .0519 L
s
P
p
.001 w
.66001 .02879 m
.66217 .0323 L
s
P
[(l)] .31959 .08564 .79718 1 Mshowa
P
p
p
.002 w
.66295 .02755 m
.90359 .41965 L
s
P
p
.002 w
.66529 .03136 m
.65894 .03402 L
s
P
[(0)] .67799 .02603 -1 .41952 Mshowa
p
.002 w
.7524 .17329 m
.74597 .17576 L
s
P
[(10)] .76525 .16834 -1 .3848 Mshowa
p
.002 w
.82857 .29741 m
.82208 .29971 L
s
P
[(20)] .84155 .2928 -1 .35539 Mshowa
p
.002 w
.89575 .40687 m
.88921 .40903 L
s
P
[(30)] .90883 .40255 -1 .33016 Mshowa
p
.001 w
.6837 .06135 m
.67988 .06292 L
s
P
p
.001 w
.70159 .0905 m
.69776 .09205 L
s
P
p
.001 w
.71899 .11886 m
.71515 .12039 L
s
P
p
.001 w
.73592 .14644 m
.73207 .14795 L
s
P
p
.001 w
.76844 .19942 m
.76457 .20089 L
s
P
p
.001 w
.78406 .22488 m
.78019 .22632 L
s
P
p
.001 w
.79928 .24968 m
.7954 .2511 L
s
P
p
.001 w
.81411 .27385 m
.81022 .27525 L
s
P
p
.001 w
.84267 .32039 m
.83877 .32175 L
s
P
p
.001 w
.85643 .3428 m
.85252 .34415 L
s
P
p
.001 w
.86985 .36467 m
.86594 .366 L
s
P
p
.001 w
.88296 .38603 m
.87904 .38734 L
s
P
[(n)] .87306 .21499 -1 .36805 Mshowa
P
p
p
.002 w
.11113 .25957 m
.02755 .78701 L
s
P
p
.002 w
.10947 .27006 m
.11582 .2674 L
s
P
[(0)] .09676 .27537 1 -0.41829 Mshowa
p
.002 w
.0821 .44276 m
.08853 .4403 L
s
P
[(0.5)] .06923 .44767 1 -0.38222 Mshowa
p
.002 w
.05087 .63982 m
.05739 .6376 L
s
P
[(1)] .03783 .64425 1 -0.34052 Mshowa
p
.001 w
.10427 .30287 m
.10809 .3013 L
s
P
p
.001 w
.09894 .33651 m
.10277 .33496 L
s
P
p
.001 w
.09347 .37102 m
.09731 .36949 L
s
P
p
.001 w
.08786 .40642 m
.09171 .40492 L
s
P
p
.001 w
.07619 .48006 m
.08006 .47862 L
s
P
p
.001 w
.07012 .51838 m
.07399 .51696 L
s
P
p
.001 w
.06388 .55775 m
.06777 .55636 L
s
P
p
.001 w
.05746 .59821 m
.06137 .59685 L
s
P
p
.001 w
.04409 .68261 m
.04801 .68131 L
s
P
p
.001 w
.03711 .72665 m
.04104 .72538 L
s
P
p
.001 w
.02993 .77198 m
.03387 .77075 L
s
P
[(l*tMax/n)] -0.00432 .52779 1 -0.37036 Mshowa
P
0 0 m
1 0 L
1 1.08079 L
0 1.08079 L
closepath
clip
newpath
p
p
.4 g
.002 w
.10947 .27006 m
.42216 .58687 L
s
P
p
.4 g
.002 w
.0821 .44276 m
.4166 .74608 L
s
P
p
.4 g
.002 w
.05087 .63982 m
.4104 .92353 L
s
P
p
.4 g
.002 w
.11409 .26259 m
.03127 .78963 L
s
P
p
.4 g
.002 w
.22539 .37608 m
.16931 .88677 L
s
P
p
.4 g
.002 w
.32406 .4767 m
.28897 .97098 L
s
P
p
.4 g
.002 w
.41216 .56653 m
.39369 1.04467 L
s
P
.002 w
.11113 .25957 m
.02755 .78701 L
s
.02755 .78701 m
.40586 1.05324 L
s
.40586 1.05324 m
.42251 .57709 L
s
.42251 .57709 m
.11113 .25957 L
s
.66295 .02755 m
.90359 .41965 L
s
.90359 .41965 m
.97245 .92345 L
s
.97245 .92345 m
.69507 .58075 L
s
.69507 .58075 m
.66295 .02755 L
s
.11113 .25957 m
.02755 .78701 L
s
.02755 .78701 m
.69507 .58075 L
s
.69507 .58075 m
.66295 .02755 L
s
.66295 .02755 m
.11113 .25957 L
s
p
.4 g
.42459 .57641 m
.40828 1.05269 L
s
P
p
.4 g
.59733 .51988 m
.60961 1.00657 L
s
P
p
.4 g
.78293 .45914 m
.82859 .9564 L
s
P
p
.4 g
.42216 .58687 m
.90499 .42984 L
s
P
p
.4 g
.4166 .74608 m
.92778 .59657 L
s
P
p
.4 g
.4104 .92353 m
.95347 .78457 L
s
P
.42251 .57709 m
.90359 .41965 L
s
.90359 .41965 m
.97245 .92345 L
s
.97245 .92345 m
.40586 1.05324 L
s
.40586 1.05324 m
.42251 .57709 L
s
P
p
.002 w
.44334 .57162 m .42645 .57714 L .41812 .56863 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44161 .63359 m .44334 .57162 L .42645 .57714 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.42645 .57714 m .42437 .63901 L .44161 .63359 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.41812 .56863 m .42645 .57714 L .42437 .63901 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.46035 .56606 m .44334 .57162 L .43507 .56304 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45803 .671 m .46035 .56606 L .44334 .57162 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.44334 .57162 m .44042 .67638 L .45803 .671 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.41812 .56863 m .43507 .56304 L .44334 .57162 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42437 .63901 m .41585 .63065 L .41812 .56863 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.44161 .63359 m .42437 .63901 L .41585 .63065 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.41585 .63065 m .43316 .62517 L .44161 .63359 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.43507 .56304 m .44334 .57162 L .44042 .67638 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.43507 .56304 m .44334 .57162 L .44161 .63359 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.43316 .62517 m .43507 .56304 L .41812 .56863 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.41812 .56863 m .41585 .63065 L .43316 .62517 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.43507 .56304 m .41812 .56863 L .4097 .56002 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43343 .61645 m .43507 .56304 L .41812 .56863 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.41812 .56863 m .41617 .62194 L .43343 .61645 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.4097 .56002 m .41812 .56863 L .41617 .62194 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.47749 .56045 m .46035 .56606 L .45215 .55742 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47487 .72104 m .47749 .56045 L .46035 .56606 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.46035 .56606 m .45681 .72636 L .47487 .72104 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.43507 .56304 m .45215 .55742 L .46035 .56606 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44042 .67638 m .43184 .66809 L .43507 .56304 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.45803 .671 m .44042 .67638 L .43184 .66809 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.43184 .66809 m .44952 .66264 L .45803 .671 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.45215 .55742 m .46035 .56606 L .45681 .72636 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.45215 .55742 m .46035 .56606 L .45803 .671 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.44952 .66264 m .45215 .55742 L .43507 .56304 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.44161 .63359 m .43316 .62517 L .43507 .56304 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.43507 .56304 m .43184 .66809 L .44952 .66264 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.45215 .55742 m .43507 .56304 L .42672 .55438 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44943 .66641 m .45215 .55742 L .43507 .56304 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.43507 .56304 m .43172 .67184 L .44943 .66641 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.4097 .56002 m .42672 .55438 L .43507 .56304 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41617 .62194 m .40759 .61347 L .4097 .56002 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.43343 .61645 m .41617 .62194 L .40759 .61347 L p
.669 .734 .898 r
F P
s
P
p
.002 w
.40759 .61347 m .42491 .60792 L .43343 .61645 L p
.669 .734 .898 r
F P
s
P
p
.002 w
.42672 .55438 m .43507 .56304 L .43172 .67184 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.42672 .55438 m .43507 .56304 L .43343 .61645 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.49474 .55481 m .47749 .56045 L .46935 .55176 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49247 .77335 m .49474 .55481 L .47749 .56045 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.47749 .56045 m .47394 .77858 L .49247 .77335 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.45215 .55742 m .46935 .55176 L .47749 .56045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45681 .72636 m .44813 .71816 L .45215 .55742 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.47487 .72104 m .45681 .72636 L .44813 .71816 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.44813 .71816 m .46627 .71279 L .47487 .72104 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.46935 .55176 m .47749 .56045 L .47394 .77858 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.46935 .55176 m .47749 .56045 L .47487 .72104 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.42491 .60792 m .42672 .55438 L .4097 .56002 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.4097 .56002 m .40759 .61347 L .42491 .60792 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.42672 .55438 m .4097 .56002 L .40119 .55132 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42484 .60987 m .42672 .55438 L .4097 .56002 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.4097 .56002 m .40751 .61542 L .42484 .60987 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.40119 .55132 m .4097 .56002 L .40751 .61542 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.46627 .71279 m .46935 .55176 L .45215 .55742 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.45803 .671 m .44952 .66264 L .45215 .55742 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.45215 .55742 m .44813 .71816 L .46627 .71279 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.46935 .55176 m .45215 .55742 L .44386 .54869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46616 .71866 m .46935 .55176 L .45215 .55742 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.45215 .55742 m .44799 .72402 L .46616 .71866 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.42672 .55438 m .44386 .54869 L .45215 .55742 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43172 .67184 m .42303 .66347 L .42672 .55438 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.44943 .66641 m .43172 .67184 L .42303 .66347 L p
.665 .74 .906 r
F P
s
P
p
.002 w
.42303 .66347 m .44081 .65797 L .44943 .66641 L p
.665 .74 .906 r
F P
s
P
p
.002 w
.44386 .54869 m .45215 .55742 L .44799 .72402 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.44386 .54869 m .45215 .55742 L .44943 .66641 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.51212 .54913 m .49474 .55481 L .48667 .54605 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51115 .76807 m .51212 .54913 L .49474 .55481 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.49474 .55481 m .49247 .77335 L .51115 .76807 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.46935 .55176 m .48667 .54605 L .49474 .55481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48667 .54605 m .49474 .55481 L .49247 .77335 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.48667 .54605 m .49474 .55481 L .49247 .77335 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.44081 .65797 m .44386 .54869 L .42672 .55438 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.43343 .61645 m .42491 .60792 L .42672 .55438 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.42672 .55438 m .42303 .66347 L .44081 .65797 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.44386 .54869 m .42672 .55438 L .41827 .54561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44069 .66203 m .44386 .54869 L .42672 .55438 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.42672 .55438 m .4229 .66751 L .44069 .66203 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.40119 .55132 m .41827 .54561 L .42672 .55438 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40751 .61542 m .39882 .60686 L .40119 .55132 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.42484 .60987 m .40751 .61542 L .39882 .60686 L p
.669 .734 .899 r
F P
s
P
p
.002 w
.39882 .60686 m .41622 .60124 L .42484 .60987 L p
.669 .734 .899 r
F P
s
P
p
.002 w
.41827 .54561 m .42672 .55438 L .4229 .66751 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.41827 .54561 m .42672 .55438 L .42484 .60987 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.48378 .76522 m .48667 .54605 L .46935 .55176 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.47394 .77858 m .46516 .77051 L .46935 .55176 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.47487 .72104 m .46627 .71279 L .46935 .55176 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.46935 .55176 m .46516 .77051 L .48378 .76522 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.48667 .54605 m .46935 .55176 L .46112 .54297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49247 .77335 m .47394 .77858 L .46516 .77051 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.46516 .77051 m .48378 .76522 L .49247 .77335 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.48367 .77337 m .48667 .54605 L .46935 .55176 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.46935 .55176 m .46501 .77864 L .48367 .77337 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.44386 .54869 m .46112 .54297 L .46935 .55176 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44799 .72402 m .43919 .71576 L .44386 .54869 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.46616 .71866 m .44799 .72402 L .43919 .71576 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.43919 .71576 m .45743 .71033 L .46616 .71866 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.46112 .54297 m .46935 .55176 L .46501 .77864 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.46112 .54297 m .46935 .55176 L .46616 .71866 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.41622 .60124 m .41827 .54561 L .40119 .55132 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.40119 .55132 m .39882 .60686 L .41622 .60124 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.41827 .54561 m .40119 .55132 L .39258 .54252 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41615 .60335 m .41827 .54561 L .40119 .55132 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.40119 .55132 m .39873 .60896 L .41615 .60335 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.39258 .54252 m .40119 .55132 L .39873 .60896 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.52962 .54341 m .51212 .54913 L .50411 .54031 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53006 .81068 m .52962 .54341 L .51212 .54913 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.51212 .54913 m .51094 .81589 L .53006 .81068 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.48667 .54605 m .50411 .54031 L .51212 .54913 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49247 .77335 m .48378 .76522 L .48667 .54605 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.49247 .77335 m .48378 .76522 L .48667 .54605 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.51115 .76807 m .49247 .77335 L .48378 .76522 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.48378 .76522 m .50254 .75988 L .51115 .76807 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.50411 .54031 m .51212 .54913 L .51094 .81589 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.50411 .54031 m .51212 .54913 L .51115 .76807 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.45743 .71033 m .46112 .54297 L .44386 .54869 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.44943 .66641 m .44081 .65797 L .44386 .54869 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.44386 .54869 m .43919 .71576 L .45743 .71033 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.46112 .54297 m .44386 .54869 L .43547 .53987 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45729 .71667 m .46112 .54297 L .44386 .54869 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.44386 .54869 m .43901 .72208 L .45729 .71667 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.41827 .54561 m .43547 .53987 L .44386 .54869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4229 .66751 m .4141 .65905 L .41827 .54561 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.44069 .66203 m .4229 .66751 L .4141 .65905 L p
.664 .74 .906 r
F P
s
P
p
.002 w
.4141 .65905 m .43196 .6535 L .44069 .66203 L p
.664 .74 .906 r
F P
s
P
p
.002 w
.43547 .53987 m .44386 .54869 L .43901 .72208 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.43547 .53987 m .44386 .54869 L .44069 .66203 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.50254 .75988 m .50411 .54031 L .48667 .54605 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.48667 .54605 m .48378 .76522 L .50254 .75988 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.50411 .54031 m .48667 .54605 L .47851 .5372 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50248 .76805 m .50411 .54031 L .48667 .54605 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.48667 .54605 m .48367 .77337 L .50248 .76805 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.46112 .54297 m .47851 .5372 L .48667 .54605 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47851 .5372 m .48667 .54605 L .48367 .77337 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.47851 .5372 m .48667 .54605 L .48367 .77337 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.43196 .6535 m .43547 .53987 L .41827 .54561 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.42484 .60987 m .41622 .60124 L .41827 .54561 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.41827 .54561 m .4141 .65905 L .43196 .6535 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.43547 .53987 m .41827 .54561 L .40973 .53675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43183 .65788 m .43547 .53987 L .41827 .54561 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.41827 .54561 m .41394 .66343 L .43183 .65788 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.39258 .54252 m .40973 .53675 L .41827 .54561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39873 .60896 m .38994 .60031 L .39258 .54252 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.41615 .60335 m .39873 .60896 L .38994 .60031 L p
.669 .734 .899 r
F P
s
P
p
.002 w
.38994 .60031 m .40742 .59463 L .41615 .60335 L p
.669 .734 .899 r
F P
s
P
p
.002 w
.40973 .53675 m .41827 .54561 L .41394 .66343 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.40973 .53675 m .41827 .54561 L .41615 .60335 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.54724 .53764 m .52962 .54341 L .52168 .53452 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5497 .85496 m .54724 .53764 L .52962 .54341 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.52962 .54341 m .53014 .8601 L .5497 .85496 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.50411 .54031 m .52168 .53452 L .52962 .54341 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51094 .81589 m .50219 .80786 L .50411 .54031 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.53006 .81068 m .51094 .81589 L .50219 .80786 L p
.65 .758 .928 r
F P
s
P
p
.002 w
.50219 .80786 m .52138 .80258 L .53006 .81068 L p
.65 .758 .928 r
F P
s
P
p
.002 w
.52168 .53452 m .52962 .54341 L .53014 .8601 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.52168 .53452 m .52962 .54341 L .53006 .81068 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.47485 .76517 m .47851 .5372 L .46112 .54297 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.46501 .77864 m .45611 .77051 L .46112 .54297 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.46616 .71866 m .45743 .71033 L .46112 .54297 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.46112 .54297 m .45611 .77051 L .47485 .76517 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.47851 .5372 m .46112 .54297 L .4528 .53408 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47551 .72356 m .47851 .5372 L .46112 .54297 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.46112 .54297 m .45702 .72899 L .47551 .72356 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.43547 .53987 m .4528 .53408 L .46112 .54297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43901 .72208 m .4301 .71374 L .43547 .53987 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.45729 .71667 m .43901 .72208 L .4301 .71374 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.4301 .71374 m .44845 .70826 L .45729 .71667 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.4528 .53408 m .46112 .54297 L .45702 .72899 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.4528 .53408 m .46112 .54297 L .45729 .71667 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.40742 .59463 m .40973 .53675 L .39258 .54252 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.39258 .54252 m .38994 .60031 L .40742 .59463 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.40973 .53675 m .39258 .54252 L .38388 .53363 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40733 .59691 m .40973 .53675 L .39258 .54252 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.39258 .54252 m .38984 .60258 L .40733 .59691 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.38388 .53363 m .39258 .54252 L .38984 .60258 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.52138 .80258 m .52168 .53452 L .50411 .54031 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.51115 .76807 m .50254 .75988 L .50411 .54031 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.50411 .54031 m .50219 .80786 L .52138 .80258 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.52168 .53452 m .50411 .54031 L .49602 .53139 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52137 .81272 m .52168 .53452 L .50411 .54031 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.50411 .54031 m .50212 .81797 L .52137 .81272 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.47851 .5372 m .49602 .53139 L .50411 .54031 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48367 .77337 m .47485 .76517 L .47851 .5372 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.48367 .77337 m .47485 .76517 L .47851 .5372 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.50248 .76805 m .48367 .77337 L .47485 .76517 L p
.654 .753 .922 r
F P
s
P
p
.002 w
.47485 .76517 m .49373 .75978 L .50248 .76805 L p
.654 .753 .922 r
F P
s
P
p
.002 w
.49602 .53139 m .50411 .54031 L .50212 .81797 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.49602 .53139 m .50411 .54031 L .50248 .76805 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.44845 .70826 m .4528 .53408 L .43547 .53987 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.44069 .66203 m .43196 .6535 L .43547 .53987 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.43547 .53987 m .4301 .71374 L .44845 .70826 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.4528 .53408 m .43547 .53987 L .427 .53094 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44923 .67711 m .4528 .53408 L .43547 .53987 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.43547 .53987 m .43106 .68265 L .44923 .67711 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.40973 .53675 m .427 .53094 L .43547 .53987 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41394 .66343 m .40502 .65488 L .40973 .53675 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.43183 .65788 m .41394 .66343 L .40502 .65488 L p
.664 .741 .907 r
F P
s
P
p
.002 w
.40502 .65488 m .42298 .64927 L .43183 .65788 L p
.664 .741 .907 r
F P
s
P
p
.002 w
.427 .53094 m .43547 .53987 L .43106 .68265 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.427 .53094 m .43547 .53987 L .43183 .65788 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.565 .53184 m .54724 .53764 L .53938 .5287 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57014 .90102 m .565 .53184 L .54724 .53764 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.54724 .53764 m .5501 .90607 L .57014 .90102 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.52168 .53452 m .53938 .5287 L .54724 .53764 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53014 .8601 m .52133 .85218 L .52168 .53452 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.5497 .85496 m .53014 .8601 L .52133 .85218 L p
.645 .763 .935 r
F P
s
P
p
.002 w
.52133 .85218 m .54098 .84698 L .5497 .85496 L p
.645 .763 .935 r
F P
s
P
p
.002 w
.53938 .5287 m .54724 .53764 L .5501 .90607 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.53938 .5287 m .54724 .53764 L .5497 .85496 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.49373 .75978 m .49602 .53139 L .47851 .5372 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.47851 .5372 m .47485 .76517 L .49373 .75978 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.49602 .53139 m .47851 .5372 L .47025 .52825 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49364 .76861 m .49602 .53139 L .47851 .5372 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.48367 .77337 m .46501 .77864 L .45611 .77051 L p
.654 .753 .922 r
F P
s
P
p
.002 w
.45611 .77051 m .47485 .76517 L .48367 .77337 L p
.654 .753 .922 r
F P
s
P
p
.002 w
.47851 .5372 m .47471 .77397 L .49364 .76861 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.4528 .53408 m .47025 .52825 L .47851 .5372 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45702 .72899 m .44814 .72062 L .4528 .53408 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.47551 .72356 m .45702 .72899 L .44814 .72062 L p
.658 .748 .916 r
F P
s
P
p
.002 w
.44814 .72062 m .46671 .71513 L .47551 .72356 L p
.658 .748 .916 r
F P
s
P
p
.002 w
.47025 .52825 m .47851 .5372 L .47471 .77397 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.47025 .52825 m .47851 .5372 L .47551 .72356 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.42298 .64927 m .427 .53094 L .40973 .53675 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.41615 .60335 m .40742 .59463 L .40973 .53675 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.40973 .53675 m .40502 .65488 L .42298 .64927 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.427 .53094 m .40973 .53675 L .40109 .5278 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42368 .62857 m .427 .53094 L .40973 .53675 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.40973 .53675 m .40584 .63422 L .42368 .62857 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.38388 .53363 m .40109 .5278 L .40973 .53675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38984 .60258 m .38094 .59384 L .38388 .53363 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.40733 .59691 m .38984 .60258 L .38094 .59384 L p
.668 .734 .899 r
F P
s
P
p
.002 w
.38094 .59384 m .3985 .5881 L .40733 .59691 L p
.668 .734 .899 r
F P
s
P
p
.002 w
.40109 .5278 m .40973 .53675 L .40584 .63422 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.40109 .5278 m .40973 .53675 L .40733 .59691 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.54098 .84698 m .53938 .5287 L .52168 .53452 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.53006 .81068 m .52138 .80258 L .52168 .53452 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.52168 .53452 m .52133 .85218 L .54098 .84698 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.53938 .5287 m .52168 .53452 L .51365 .52554 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54104 .85921 m .53938 .5287 L .52168 .53452 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.52168 .53452 m .52132 .86438 L .54104 .85921 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.49602 .53139 m .51365 .52554 L .52168 .53452 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50212 .81797 m .49323 .80987 L .49602 .53139 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.52137 .81272 m .50212 .81797 L .49323 .80987 L p
.649 .759 .929 r
F P
s
P
p
.002 w
.49323 .80987 m .51257 .80456 L .52137 .81272 L p
.649 .759 .929 r
F P
s
P
p
.002 w
.51365 .52554 m .52168 .53452 L .52132 .86438 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.51365 .52554 m .52168 .53452 L .52137 .81272 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.46671 .71513 m .47025 .52825 L .4528 .53408 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.45729 .71667 m .44845 .70826 L .4528 .53408 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.4528 .53408 m .44814 .72062 L .46671 .71513 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.47025 .52825 m .4528 .53408 L .44439 .52509 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46657 .72252 m .47025 .52825 L .4528 .53408 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.4528 .53408 m .44796 .728 L .46657 .72252 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.427 .53094 m .44439 .52509 L .4528 .53408 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43106 .68265 m .42214 .67411 L .427 .53094 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.44923 .67711 m .43106 .68265 L .42214 .67411 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.42214 .67411 m .44038 .66851 L .44923 .67711 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.44439 .52509 m .4528 .53408 L .44796 .728 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.44439 .52509 m .4528 .53408 L .44923 .67711 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.3985 .5881 m .40109 .5278 L .38388 .53363 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.38388 .53363 m .38094 .59384 L .3985 .5881 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.40109 .5278 m .38388 .53363 L .37509 .52464 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39839 .59057 m .40109 .5278 L .38388 .53363 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.38388 .53363 m .38082 .5963 L .39839 .59057 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.37509 .52464 m .38388 .53363 L .38082 .5963 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.58288 .52599 m .565 .53184 L .5572 .52283 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58905 .83198 m .58288 .52599 L .565 .53184 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.565 .53184 m .56925 .83722 L .58905 .83198 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.53938 .5287 m .5572 .52283 L .565 .53184 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5572 .52283 m .565 .53184 L .56925 .83722 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.5572 .52283 m .565 .53184 L .57014 .90102 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.51257 .80456 m .51365 .52554 L .49602 .53139 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.50248 .76805 m .49373 .75978 L .49602 .53139 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.49602 .53139 m .49323 .80987 L .51257 .80456 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.51365 .52554 m .49602 .53139 L .48783 .52237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51252 .81553 m .51365 .52554 L .49602 .53139 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.49602 .53139 m .49312 .82082 L .51252 .81553 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.47025 .52825 m .48783 .52237 L .49602 .53139 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47471 .77397 m .46575 .7657 L .47025 .52825 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.49364 .76861 m .47471 .77397 L .46575 .7657 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.46575 .7657 m .48477 .76028 L .49364 .76861 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.48783 .52237 m .49602 .53139 L .49312 .82082 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.48783 .52237 m .49602 .53139 L .49364 .76861 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.44038 .66851 m .44439 .52509 L .427 .53094 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.43183 .65788 m .42298 .64927 L .427 .53094 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.427 .53094 m .42214 .67411 L .44038 .66851 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.44439 .52509 m .427 .53094 L .41843 .52192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44022 .67431 m .44439 .52509 L .427 .53094 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.427 .53094 m .42194 .67991 L .44022 .67431 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.40109 .5278 m .41843 .52192 L .427 .53094 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40584 .63422 m .39689 .62551 L .40109 .5278 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.42368 .62857 m .40584 .63422 L .39689 .62551 L p
.666 .738 .904 r
F P
s
P
p
.002 w
.39689 .62551 m .4148 .6198 L .42368 .62857 L p
.666 .738 .904 r
F P
s
P
p
.002 w
.41843 .52192 m .427 .53094 L .42194 .67991 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.41843 .52192 m .427 .53094 L .42368 .62857 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.56137 .89316 m .5572 .52283 L .53938 .5287 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.5501 .90607 m .54124 .89828 L .53938 .5287 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.5497 .85496 m .54098 .84698 L .53938 .5287 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.53938 .5287 m .54124 .89828 L .56137 .89316 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.5572 .52283 m .53938 .5287 L .53142 .51965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57014 .90102 m .5501 .90607 L .54124 .89828 L p
.638 .769 .942 r
F P
s
P
p
.002 w
.54124 .89828 m .56137 .89316 L .57014 .90102 L p
.638 .769 .942 r
F P
s
P
p
.002 w
.56153 .90764 m .5572 .52283 L .53938 .5287 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.53938 .5287 m .54131 .91272 L .56153 .90764 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.51365 .52554 m .53142 .51965 L .53938 .5287 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52132 .86438 m .51236 .85641 L .51365 .52554 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.54104 .85921 m .52132 .86438 L .51236 .85641 L p
.643 .765 .937 r
F P
s
P
p
.002 w
.51236 .85641 m .53218 .85117 L .54104 .85921 L p
.643 .765 .937 r
F P
s
P
p
.002 w
.53142 .51965 m .53938 .5287 L .54131 .91272 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.53142 .51965 m .53938 .5287 L .54104 .85921 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.48477 .76028 m .48783 .52237 L .47025 .52825 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.47551 .72356 m .46671 .71513 L .47025 .52825 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.47025 .52825 m .46575 .7657 L .48477 .76028 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.48783 .52237 m .47025 .52825 L .46191 .51919 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48464 .76985 m .48783 .52237 L .47025 .52825 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.47025 .52825 m .46557 .77525 L .48464 .76985 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.44439 .52509 m .46191 .51919 L .47025 .52825 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44796 .728 m .43895 .71955 L .44439 .52509 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.46657 .72252 m .44796 .728 L .43895 .71955 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.43895 .71955 m .45764 .71401 L .46657 .72252 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.46191 .51919 m .47025 .52825 L .46557 .77525 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.46191 .51919 m .47025 .52825 L .46657 .72252 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.4148 .6198 m .41843 .52192 L .40109 .5278 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.40733 .59691 m .3985 .5881 L .40109 .5278 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.40109 .5278 m .39689 .62551 L .4148 .6198 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.41843 .52192 m .40109 .5278 L .39236 .51874 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41465 .62386 m .41843 .52192 L .40109 .5278 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.40109 .5278 m .39672 .62956 L .41465 .62386 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.37509 .52464 m .39236 .51874 L .40109 .5278 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38082 .5963 m .37182 .58746 L .37509 .52464 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.39839 .59057 m .38082 .5963 L .37182 .58746 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.37182 .58746 m .38946 .58167 L .39839 .59057 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.39236 .51874 m .40109 .5278 L .39672 .62956 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.39236 .51874 m .40109 .5278 L .39839 .59057 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.60089 .5201 m .58288 .52599 L .57515 .51692 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61002 .86484 m .60089 .5201 L .58288 .52599 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.58288 .52599 m .58982 .87004 L .61002 .86484 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.5572 .52283 m .57515 .51692 L .58288 .52599 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56925 .83722 m .56065 .82913 L .5572 .52283 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.57014 .90102 m .56137 .89316 L .5572 .52283 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.58905 .83198 m .56925 .83722 L .56065 .82913 L p
.646 .762 .933 r
F P
s
P
p
.002 w
.56065 .82913 m .58053 .82382 L .58905 .83198 L p
.646 .762 .933 r
F P
s
P
p
.002 w
.57515 .51692 m .58288 .52599 L .58982 .87004 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.57515 .51692 m .58288 .52599 L .58905 .83198 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.53218 .85117 m .53142 .51965 L .51365 .52554 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.52137 .81272 m .51257 .80456 L .51365 .52554 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.51365 .52554 m .51236 .85641 L .53218 .85117 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.53142 .51965 m .51365 .52554 L .50554 .51646 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53221 .86444 m .53142 .51965 L .51365 .52554 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.51365 .52554 m .51231 .86964 L .53221 .86444 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.48783 .52237 m .50554 .51646 L .51365 .52554 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49312 .82082 m .48409 .81266 L .48783 .52237 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.51252 .81553 m .49312 .82082 L .48409 .81266 L p
.648 .76 .931 r
F P
s
P
p
.002 w
.48409 .81266 m .50358 .80731 L .51252 .81553 L p
.648 .76 .931 r
F P
s
P
p
.002 w
.50554 .51646 m .51365 .52554 L .51231 .86964 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.50554 .51646 m .51365 .52554 L .51252 .81553 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.45764 .71401 m .46191 .51919 L .44439 .52509 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.44923 .67711 m .44038 .66851 L .44439 .52509 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.44439 .52509 m .43895 .71955 L .45764 .71401 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.46191 .51919 m .44439 .52509 L .43589 .516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45746 .72203 m .46191 .51919 L .44439 .52509 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.44439 .52509 m .43873 .72756 L .45746 .72203 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.41843 .52192 m .43589 .516 L .44439 .52509 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42194 .67991 m .41289 .67129 L .41843 .52192 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.44022 .67431 m .42194 .67991 L .41289 .67129 L p
.662 .744 .911 r
F P
s
P
p
.002 w
.41289 .67129 m .43124 .66563 L .44022 .67431 L p
.662 .744 .911 r
F P
s
P
p
.002 w
.43589 .516 m .44439 .52509 L .43873 .72756 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.43589 .516 m .44439 .52509 L .44022 .67431 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.38946 .58167 m .39236 .51874 L .37509 .52464 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.37509 .52464 m .37182 .58746 L .38946 .58167 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.39236 .51874 m .37509 .52464 L .36619 .51554 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38995 .57099 m .39236 .51874 L .37509 .52464 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.37509 .52464 m .37237 .5768 L .38995 .57099 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.36619 .51554 m .37509 .52464 L .37237 .5768 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.58053 .82382 m .57515 .51692 L .5572 .52283 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.5572 .52283 m .56065 .82913 L .58053 .82382 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.57515 .51692 m .5572 .52283 L .54931 .51371 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58074 .83558 m .57515 .51692 L .5572 .52283 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.5572 .52283 m .56078 .84086 L .58074 .83558 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.53142 .51965 m .54931 .51371 L .5572 .52283 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54931 .51371 m .5572 .52283 L .56078 .84086 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.54931 .51371 m .5572 .52283 L .56153 .90764 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.50358 .80731 m .50554 .51646 L .48783 .52237 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.49364 .76861 m .48477 .76028 L .48783 .52237 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.48783 .52237 m .48409 .81266 L .50358 .80731 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.50554 .51646 m .48783 .52237 L .47956 .51326 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5035 .81922 m .50554 .51646 L .48783 .52237 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.48783 .52237 m .48394 .82454 L .5035 .81922 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.46191 .51919 m .47956 .51326 L .48783 .52237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46557 .77525 m .45648 .76692 L .46191 .51919 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.48464 .76985 m .46557 .77525 L .45648 .76692 L p
.653 .755 .925 r
F P
s
P
p
.002 w
.45648 .76692 m .47563 .76144 L .48464 .76985 L p
.653 .755 .925 r
F P
s
P
p
.002 w
.47956 .51326 m .48783 .52237 L .48394 .82454 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.47956 .51326 m .48783 .52237 L .48464 .76985 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.43124 .66563 m .43589 .516 L .41843 .52192 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.42368 .62857 m .4148 .6198 L .41843 .52192 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.41843 .52192 m .41289 .67129 L .43124 .66563 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.43589 .516 m .41843 .52192 L .40976 .5128 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43105 .67194 m .43589 .516 L .41843 .52192 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.41843 .52192 m .41266 .67759 L .43105 .67194 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.39236 .51874 m .40976 .5128 L .41843 .52192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39672 .62956 m .38765 .62077 L .39236 .51874 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.41465 .62386 m .39672 .62956 L .38765 .62077 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.38765 .62077 m .40565 .615 L .41465 .62386 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.40976 .5128 m .41843 .52192 L .41266 .67759 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.40976 .5128 m .41843 .52192 L .41465 .62386 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.61903 .51417 m .60089 .5201 L .59324 .51096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63167 .89877 m .61903 .51417 L .60089 .5201 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.60089 .5201 m .61105 .90392 L .63167 .89877 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.57515 .51692 m .59324 .51096 L .60089 .5201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58982 .87004 m .5812 .86202 L .57515 .51692 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.61002 .86484 m .58982 .87004 L .5812 .86202 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.5812 .86202 m .6015 .85676 L .61002 .86484 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.59324 .51096 m .60089 .5201 L .61105 .90392 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.59324 .51096 m .60089 .5201 L .61002 .86484 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.55261 .89975 m .54931 .51371 L .53142 .51965 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.54131 .91272 m .5323 .90489 L .53142 .51965 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.54104 .85921 m .53218 .85117 L .53142 .51965 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.53142 .51965 m .5323 .90489 L .55261 .89975 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.54931 .51371 m .53142 .51965 L .52337 .5105 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5518 .80482 m .54931 .51371 L .53142 .51965 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.53142 .51965 m .53208 .8102 L .5518 .80482 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.50554 .51646 m .52337 .5105 L .53142 .51965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52337 .5105 m .53142 .51965 L .53208 .8102 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.52337 .5105 m .53142 .51965 L .53221 .86444 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.47563 .76144 m .47956 .51326 L .46191 .51919 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.46657 .72252 m .45764 .71401 L .46191 .51919 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.46191 .51919 m .45648 .76692 L .47563 .76144 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.47956 .51326 m .46191 .51919 L .45347 .51004 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47547 .77185 m .47956 .51326 L .46191 .51919 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.46191 .51919 m .45625 .7773 L .47547 .77185 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.43589 .516 m .45347 .51004 L .46191 .51919 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43873 .72756 m .42958 .71904 L .43589 .516 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.45746 .72203 m .43873 .72756 L .42958 .71904 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.42958 .71904 m .4484 .71344 L .45746 .72203 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.45347 .51004 m .46191 .51919 L .45625 .7773 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.45347 .51004 m .46191 .51919 L .45746 .72203 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.40565 .615 m .40976 .5128 L .39236 .51874 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.39839 .59057 m .38946 .58167 L .39236 .51874 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.39236 .51874 m .38765 .62077 L .40565 .615 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.40976 .5128 m .39236 .51874 L .38353 .50958 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40547 .61941 m .40976 .5128 L .39236 .51874 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.39236 .51874 m .38745 .62518 L .40547 .61941 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.36619 .51554 m .38353 .50958 L .39236 .51874 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37237 .5768 m .3633 .56784 L .36619 .51554 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.38995 .57099 m .37237 .5768 L .3633 .56784 L p
.669 .733 .898 r
F P
s
P
p
.002 w
.3633 .56784 m .38094 .56196 L .38995 .57099 L p
.669 .733 .898 r
F P
s
P
p
.002 w
.38353 .50958 m .39236 .51874 L .38745 .62518 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.38353 .50958 m .39236 .51874 L .38995 .57099 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.6015 .85676 m .59324 .51096 L .57515 .51692 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.58905 .83198 m .58053 .82382 L .57515 .51692 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.57515 .51692 m .5812 .86202 L .6015 .85676 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.59324 .51096 m .57515 .51692 L .56734 .50774 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60182 .87018 m .59324 .51096 L .57515 .51692 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.57515 .51692 m .58144 .87541 L .60182 .87018 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.54931 .51371 m .56734 .50774 L .57515 .51692 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56078 .84086 m .55204 .83271 L .54931 .51371 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.56153 .90764 m .55261 .89975 L .54931 .51371 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.58074 .83558 m .56078 .84086 L .55204 .83271 L p
.645 .763 .935 r
F P
s
P
p
.002 w
.55204 .83271 m .57209 .82736 L .58074 .83558 L p
.645 .763 .935 r
F P
s
P
p
.002 w
.56734 .50774 m .57515 .51692 L .58144 .87541 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.56734 .50774 m .57515 .51692 L .58074 .83558 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.52319 .85635 m .52337 .5105 L .50554 .51646 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.51231 .86964 m .50321 .86162 L .50554 .51646 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.51252 .81553 m .50358 .80731 L .50554 .51646 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.50554 .51646 m .50321 .86162 L .52319 .85635 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.52337 .5105 m .50554 .51646 L .49733 .50727 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52324 .77246 m .52337 .5105 L .50554 .51646 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.50554 .51646 m .50378 .77792 L .52324 .77246 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.47956 .51326 m .49733 .50727 L .50554 .51646 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49733 .50727 m .50554 .51646 L .50378 .77792 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.49733 .50727 m .50554 .51646 L .5035 .81922 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4484 .71344 m .45347 .51004 L .43589 .516 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.44022 .67431 m .43124 .66563 L .43589 .516 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.43589 .516 m .42958 .71904 L .4484 .71344 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.45347 .51004 m .43589 .516 L .42728 .50681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44818 .72218 m .45347 .51004 L .43589 .516 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.43589 .516 m .42931 .72776 L .44818 .72218 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.40976 .5128 m .42728 .50681 L .43589 .516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41266 .67759 m .40349 .66888 L .40976 .5128 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.43105 .67194 m .41266 .67759 L .40349 .66888 L p
.661 .745 .912 r
F P
s
P
p
.002 w
.40349 .66888 m .42195 .66317 L .43105 .67194 L p
.661 .745 .912 r
F P
s
P
p
.002 w
.42728 .50681 m .43589 .516 L .42931 .72776 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.42728 .50681 m .43589 .516 L .43105 .67194 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.38094 .56196 m .38353 .50958 L .36619 .51554 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.36619 .51554 m .3633 .56784 L .38094 .56196 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.38353 .50958 m .36619 .51554 L .3572 .50635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38083 .56428 m .38353 .50958 L .36619 .51554 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.36619 .51554 m .36317 .57016 L .38083 .56428 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.3572 .50635 m .36619 .51554 L .36317 .57016 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.63731 .5082 m .61903 .51417 L .61145 .50496 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65405 .93384 m .63731 .5082 L .61903 .51417 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.61903 .51417 m .63299 .93892 L .65405 .93384 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.59324 .51096 m .61145 .50496 L .61903 .51417 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61105 .90392 m .60243 .89599 L .59324 .51096 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.63167 .89877 m .61105 .90392 L .60243 .89599 L p
.637 .771 .944 r
F P
s
P
p
.002 w
.60243 .89599 m .62315 .89078 L .63167 .89877 L p
.637 .771 .944 r
F P
s
P
p
.002 w
.61145 .50496 m .61903 .51417 L .63299 .93892 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.61145 .50496 m .61903 .51417 L .63167 .89877 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.57209 .82736 m .56734 .50774 L .54931 .51371 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.54931 .51371 m .55204 .83271 L .57209 .82736 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.56734 .50774 m .54931 .51371 L .54134 .5045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57228 .8401 m .56734 .50774 L .54931 .51371 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.54931 .51371 m .55214 .84542 L .57228 .8401 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.52337 .5105 m .54134 .5045 L .54931 .51371 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53208 .8102 m .52322 .80191 L .52337 .5105 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.53221 .86444 m .52319 .85635 L .52337 .5105 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.5518 .80482 m .53208 .8102 L .52322 .80191 L p
.648 .76 .931 r
F P
s
P
p
.002 w
.52322 .80191 m .54302 .79647 L .5518 .80482 L p
.648 .76 .931 r
F P
s
P
p
.002 w
.54134 .5045 m .54931 .51371 L .55214 .84542 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.54134 .5045 m .54931 .51371 L .5518 .80482 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.49441 .81094 m .49733 .50727 L .47956 .51326 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.48394 .82454 m .47476 .81633 L .47956 .51326 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.48464 .76985 m .47563 .76144 L .47956 .51326 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.47956 .51326 m .47476 .81633 L .49441 .81094 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.49733 .50727 m .47956 .51326 L .47119 .50403 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49511 .73835 m .49733 .50727 L .47956 .51326 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.47956 .51326 m .47591 .74391 L .49511 .73835 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.45347 .51004 m .47119 .50403 L .47956 .51326 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47119 .50403 m .47956 .51326 L .47591 .74391 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.47119 .50403 m .47956 .51326 L .47547 .77185 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.42195 .66317 m .42728 .50681 L .40976 .5128 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.41465 .62386 m .40565 .615 L .40976 .5128 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.40976 .5128 m .40349 .66888 L .42195 .66317 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.42728 .50681 m .40976 .5128 L .40099 .50357 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42171 .67006 m .42728 .50681 L .40976 .5128 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.40976 .5128 m .40321 .67576 L .42171 .67006 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.38353 .50958 m .40099 .50357 L .40976 .5128 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38745 .62518 m .37826 .61629 L .38353 .50958 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.40547 .61941 m .38745 .62518 L .37826 .61629 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.37826 .61629 m .39636 .61047 L .40547 .61941 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.40099 .50357 m .40976 .5128 L .40321 .67576 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.40099 .50357 m .40976 .5128 L .40547 .61941 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.62315 .89078 m .61145 .50496 L .59324 .51096 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.61002 .86484 m .6015 .85676 L .59324 .51096 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.59324 .51096 m .60243 .89599 L .62315 .89078 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.61145 .50496 m .59324 .51096 L .5855 .50171 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62361 .90595 m .61145 .50496 L .59324 .51096 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.59324 .51096 m .6028 .91112 L .62361 .90595 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.56734 .50774 m .5855 .50171 L .59324 .51096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58144 .87541 m .57268 .86734 L .56734 .50774 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.60182 .87018 m .58144 .87541 L .57268 .86734 L p
.64 .768 .941 r
F P
s
P
p
.002 w
.57268 .86734 m .59315 .86204 L .60182 .87018 L p
.64 .768 .941 r
F P
s
P
p
.002 w
.5855 .50171 m .59324 .51096 L .6028 .91112 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.5855 .50171 m .59324 .51096 L .60182 .87018 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.54302 .79647 m .54134 .5045 L .52337 .5105 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.52337 .5105 m .52322 .80191 L .54302 .79647 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.54134 .5045 m .52337 .5105 L .51524 .50125 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54309 .80843 m .54134 .5045 L .52337 .5105 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.52337 .5105 m .52322 .81385 L .54309 .80843 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.49733 .50727 m .51524 .50125 L .52337 .5105 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50378 .77792 m .49481 .7695 L .49733 .50727 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.5035 .81922 m .49441 .81094 L .49733 .50727 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.52324 .77246 m .50378 .77792 L .49481 .7695 L p
.651 .757 .927 r
F P
s
P
p
.002 w
.49481 .7695 m .51435 .76397 L .52324 .77246 L p
.651 .757 .927 r
F P
s
P
p
.002 w
.51524 .50125 m .52337 .5105 L .52322 .81385 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.51524 .50125 m .52337 .5105 L .52324 .77246 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.46631 .76338 m .47119 .50403 L .45347 .51004 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.45625 .7773 m .44701 .7689 L .45347 .51004 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.45746 .72203 m .4484 .71344 L .45347 .51004 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.45347 .51004 m .44701 .7689 L .46631 .76338 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.47119 .50403 m .45347 .51004 L .44494 .50078 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46746 .70236 m .47119 .50403 L .45347 .51004 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.45347 .51004 m .44853 .70801 L .46746 .70236 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.42728 .50681 m .44494 .50078 L .45347 .51004 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44494 .50078 m .45347 .51004 L .44853 .70801 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.44494 .50078 m .45347 .51004 L .44818 .72218 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.39636 .61047 m .40099 .50357 L .38353 .50958 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.38995 .57099 m .38094 .56196 L .38353 .50958 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.38353 .50958 m .37826 .61629 L .39636 .61047 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.40099 .50357 m .38353 .50958 L .3746 .50032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39615 .6153 m .40099 .50357 L .38353 .50958 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.38353 .50958 m .37802 .62112 L .39615 .6153 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.3572 .50635 m .3746 .50032 L .38353 .50958 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36317 .57016 m .35399 .5611 L .3572 .50635 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.38083 .56428 m .36317 .57016 L .35399 .5611 L p
.669 .734 .898 r
F P
s
P
p
.002 w
.35399 .5611 m .37171 .55515 L .38083 .56428 L p
.669 .734 .898 r
F P
s
P
p
.002 w
.3746 .50032 m .38353 .50958 L .37802 .62112 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.3746 .50032 m .38353 .50958 L .38083 .56428 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.65572 .50218 m .63731 .5082 L .6298 .49892 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67717 .97008 m .65572 .50218 L .63731 .5082 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.63731 .5082 m .65567 .9751 L .67717 .97008 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.61145 .50496 m .6298 .49892 L .63731 .5082 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63299 .93892 m .62437 .93108 L .61145 .50496 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.65405 .93384 m .63299 .93892 L .62437 .93108 L p
.631 .776 .95 r
F P
s
P
p
.002 w
.62437 .93108 m .64552 .92593 L .65405 .93384 L p
.631 .776 .95 r
F P
s
P
p
.002 w
.6298 .49892 m .63731 .5082 L .65567 .9751 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.6298 .49892 m .63731 .5082 L .65405 .93384 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.59315 .86204 m .5855 .50171 L .56734 .50774 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.58074 .83558 m .57209 .82736 L .56734 .50774 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.56734 .50774 m .57268 .86734 L .59315 .86204 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.5855 .50171 m .56734 .50774 L .55944 .49845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59346 .8766 m .5855 .50171 L .56734 .50774 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.56734 .50774 m .5729 .88186 L .59346 .8766 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.54134 .5045 m .55944 .49845 L .56734 .50774 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55214 .84542 m .54326 .83722 L .54134 .5045 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.57228 .8401 m .55214 .84542 L .54326 .83722 L p
.643 .765 .937 r
F P
s
P
p
.002 w
.54326 .83722 m .56348 .83183 L .57228 .8401 L p
.643 .765 .937 r
F P
s
P
p
.002 w
.55944 .49845 m .56734 .50774 L .5729 .88186 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.55944 .49845 m .56734 .50774 L .57228 .8401 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.51435 .76397 m .51524 .50125 L .49733 .50727 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.49733 .50727 m .49481 .7695 L .51435 .76397 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.51524 .50125 m .49733 .50727 L .48903 .49798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51431 .77504 m .51524 .50125 L .49733 .50727 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.49733 .50727 m .4947 .78055 L .51431 .77504 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.47119 .50403 m .48903 .49798 L .49733 .50727 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47591 .74391 m .46684 .73534 L .47119 .50403 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.47547 .77185 m .46631 .76338 L .47119 .50403 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.49511 .73835 m .47591 .74391 L .46684 .73534 L p
.654 .753 .922 r
F P
s
P
p
.002 w
.46684 .73534 m .48612 .72972 L .49511 .73835 L p
.654 .753 .922 r
F P
s
P
p
.002 w
.48903 .49798 m .49733 .50727 L .4947 .78055 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.48903 .49798 m .49733 .50727 L .49511 .73835 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.43898 .71352 m .44494 .50078 L .42728 .50681 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.42931 .72776 m .42003 .71916 L .42728 .50681 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.43105 .67194 m .42195 .66317 L .42728 .50681 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.42728 .50681 m .42003 .71916 L .43898 .71352 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.44494 .50078 m .42728 .50681 L .41859 .49752 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44818 .72218 m .42931 .72776 L .42003 .71916 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.42003 .71916 m .43898 .71352 L .44818 .72218 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.43871 .72308 m .44494 .50078 L .42728 .50681 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.42728 .50681 m .41971 .7287 L .43871 .72308 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.40099 .50357 m .41859 .49752 L .42728 .50681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40321 .67576 m .3939 .66697 L .40099 .50357 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.42171 .67006 m .40321 .67576 L .3939 .66697 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.3939 .66697 m .41248 .66121 L .42171 .67006 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.41859 .49752 m .42728 .50681 L .41971 .7287 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.41859 .49752 m .42728 .50681 L .42171 .67006 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.37171 .55515 m .3746 .50032 L .3572 .50635 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.3572 .50635 m .35399 .5611 L .37171 .55515 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.3746 .50032 m .3572 .50635 L .3481 .49705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37158 .5577 m .3746 .50032 L .3572 .50635 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.3572 .50635 m .35384 .56364 L .37158 .5577 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.3481 .49705 m .3572 .50635 L .35384 .56364 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.64552 .92593 m .6298 .49892 L .61145 .50496 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.63167 .89877 m .62315 .89078 L .61145 .50496 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.61145 .50496 m .62437 .93108 L .64552 .92593 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.6298 .49892 m .61145 .50496 L .60379 .49565 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64615 .94295 m .6298 .49892 L .61145 .50496 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.61145 .50496 m .62489 .94806 L .64615 .94295 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.5855 .50171 m .60379 .49565 L .61145 .50496 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6028 .91112 m .59403 .90314 L .5855 .50171 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.62361 .90595 m .6028 .91112 L .59403 .90314 L p
.635 .773 .946 r
F P
s
P
p
.002 w
.59403 .90314 m .61494 .89791 L .62361 .90595 L p
.635 .773 .946 r
F P
s
P
p
.002 w
.60379 .49565 m .61145 .50496 L .62489 .94806 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.60379 .49565 m .61145 .50496 L .62361 .90595 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.56348 .83183 m .55944 .49845 L .54134 .5045 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.5518 .80482 m .54302 .79647 L .54134 .5045 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.54134 .5045 m .54326 .83722 L .56348 .83183 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.55944 .49845 m .54134 .5045 L .53327 .49518 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56365 .84568 m .55944 .49845 L .54134 .5045 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.54134 .5045 m .54334 .85104 L .56365 .84568 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.51524 .50125 m .53327 .49518 L .54134 .5045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52322 .81385 m .51421 .8055 L .51524 .50125 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.54309 .80843 m .52322 .81385 L .51421 .8055 L p
.647 .761 .933 r
F P
s
P
p
.002 w
.51421 .8055 m .53418 .80002 L .54309 .80843 L p
.647 .761 .933 r
F P
s
P
p
.002 w
.53327 .49518 m .54134 .5045 L .54334 .85104 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.53327 .49518 m .54134 .5045 L .54309 .80843 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.48612 .72972 m .48903 .49798 L .47119 .50403 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.47119 .50403 m .46684 .73534 L .48612 .72972 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.48903 .49798 m .47119 .50403 L .46272 .49471 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48599 .73978 m .48903 .49798 L .47119 .50403 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.47119 .50403 m .46665 .74538 L .48599 .73978 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.44494 .50078 m .46272 .49471 L .47119 .50403 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44853 .70801 m .43938 .6993 L .44494 .50078 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.44818 .72218 m .43898 .71352 L .44494 .50078 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.46746 .70236 m .44853 .70801 L .43938 .6993 L p
.657 .749 .918 r
F P
s
P
p
.002 w
.43938 .6993 m .45838 .69358 L .46746 .70236 L p
.657 .749 .918 r
F P
s
P
p
.002 w
.46272 .49471 m .47119 .50403 L .46665 .74538 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.46272 .49471 m .47119 .50403 L .46746 .70236 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.41248 .66121 m .41859 .49752 L .40099 .50357 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.40547 .61941 m .39636 .61047 L .40099 .50357 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.40099 .50357 m .3939 .66697 L .41248 .66121 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.41859 .49752 m .40099 .50357 L .39213 .49424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4122 .66875 m .41859 .49752 L .40099 .50357 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.40099 .50357 m .39358 .6745 L .4122 .66875 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.3746 .50032 m .39213 .49424 L .40099 .50357 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37802 .62112 m .36871 .61214 L .3746 .50032 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.39615 .6153 m .37802 .62112 L .36871 .61214 L p
.665 .74 .906 r
F P
s
P
p
.002 w
.36871 .61214 m .38691 .60625 L .39615 .6153 L p
.665 .74 .906 r
F P
s
P
p
.002 w
.39213 .49424 m .40099 .50357 L .39358 .6745 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.39213 .49424 m .40099 .50357 L .39615 .6153 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.67426 .49611 m .65572 .50218 L .64829 .49283 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69114 .81783 m .67426 .49611 L .65572 .50218 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.65572 .50218 m .67044 .82326 L .69114 .81783 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.6298 .49892 m .64829 .49283 L .65572 .50218 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64829 .49283 m .65572 .50218 L .67044 .82326 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.64829 .49283 m .65572 .50218 L .67717 .97008 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.61494 .89791 m .60379 .49565 L .5855 .50171 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.60182 .87018 m .59315 .86204 L .5855 .50171 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.5855 .50171 m .59403 .90314 L .61494 .89791 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.60379 .49565 m .5855 .50171 L .57767 .49236 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6154 .91438 m .60379 .49565 L .5855 .50171 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.5855 .50171 m .59438 .91958 L .6154 .91438 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.55944 .49845 m .57767 .49236 L .5855 .50171 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5729 .88186 m .56399 .87375 L .55944 .49845 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.59346 .8766 m .5729 .88186 L .56399 .87375 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.56399 .87375 m .58465 .86842 L .59346 .8766 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.57767 .49236 m .5855 .50171 L .59438 .91958 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.57767 .49236 m .5855 .50171 L .59346 .8766 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.53418 .80002 m .53327 .49518 L .51524 .50125 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.52324 .77246 m .51435 .76397 L .51524 .50125 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.51524 .50125 m .51421 .8055 L .53418 .80002 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.53327 .49518 m .51524 .50125 L .50701 .49189 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53421 .81305 m .53327 .49518 L .51524 .50125 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.51524 .50125 m .51417 .8185 L .53421 .81305 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.48903 .49798 m .50701 .49189 L .51524 .50125 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4947 .78055 m .48559 .77206 L .48903 .49798 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.51431 .77504 m .4947 .78055 L .48559 .77206 L p
.65 .758 .928 r
F P
s
P
p
.002 w
.48559 .77206 m .50528 .76648 L .51431 .77504 L p
.65 .758 .928 r
F P
s
P
p
.002 w
.50701 .49189 m .51524 .50125 L .51417 .8185 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.50701 .49189 m .51524 .50125 L .51431 .77504 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.45838 .69358 m .46272 .49471 L .44494 .50078 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.44494 .50078 m .43938 .6993 L .45838 .69358 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.46272 .49471 m .44494 .50078 L .43631 .49142 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45819 .70248 m .46272 .49471 L .44494 .50078 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.44494 .50078 m .43913 .70817 L .45819 .70248 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.41859 .49752 m .43631 .49142 L .44494 .50078 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43631 .49142 m .44494 .50078 L .43913 .70817 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.43631 .49142 m .44494 .50078 L .43871 .72308 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.38691 .60625 m .39213 .49424 L .3746 .50032 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.38083 .56428 m .37171 .55515 L .3746 .50032 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.3746 .50032 m .36871 .61214 L .38691 .60625 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.39213 .49424 m .3746 .50032 L .36557 .49095 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38666 .61156 m .39213 .49424 L .3746 .50032 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.3746 .50032 m .36843 .61743 L .38666 .61156 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.3481 .49705 m .36557 .49095 L .3746 .50032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35384 .56364 m .34454 .55448 L .3481 .49705 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.37158 .5577 m .35384 .56364 L .34454 .55448 L p
.669 .734 .899 r
F P
s
P
p
.002 w
.34454 .55448 m .36235 .54847 L .37158 .5577 L p
.669 .734 .899 r
F P
s
P
p
.002 w
.36557 .49095 m .3746 .50032 L .36843 .61743 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.36557 .49095 m .3746 .50032 L .37158 .5577 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.66866 .96228 m .64829 .49283 L .6298 .49892 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.65567 .9751 m .64705 .96737 L .6298 .49892 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.65405 .93384 m .64552 .92593 L .6298 .49892 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.6298 .49892 m .64705 .96737 L .66866 .96228 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.64829 .49283 m .6298 .49892 L .62221 .48953 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66165 .80081 m .64829 .49283 L .6298 .49892 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.6298 .49892 m .64112 .80631 L .66165 .80081 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.60379 .49565 m .62221 .48953 L .6298 .49892 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62221 .48953 m .6298 .49892 L .64112 .80631 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.62221 .48953 m .6298 .49892 L .64615 .94295 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.58465 .86842 m .57767 .49236 L .55944 .49845 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.57228 .8401 m .56348 .83183 L .55944 .49845 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.55944 .49845 m .56399 .87375 L .58465 .86842 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.57767 .49236 m .55944 .49845 L .55145 .48906 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58495 .88427 m .57767 .49236 L .55944 .49845 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.55944 .49845 m .56418 .88956 L .58495 .88427 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.53327 .49518 m .55145 .48906 L .55944 .49845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54334 .85104 m .5343 .84278 L .53327 .49518 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.56365 .84568 m .54334 .85104 L .5343 .84278 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.5343 .84278 m .5547 .83735 L .56365 .84568 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.55145 .48906 m .55944 .49845 L .56418 .88956 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.55145 .48906 m .55944 .49845 L .56365 .84568 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.50528 .76648 m .50701 .49189 L .48903 .49798 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.49511 .73835 m .48612 .72972 L .48903 .49798 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.48903 .49798 m .48559 .77206 L .50528 .76648 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.50701 .49189 m .48903 .49798 L .48064 .48859 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50521 .77857 m .50701 .49189 L .48903 .49798 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.48903 .49798 m .48543 .78412 L .50521 .77857 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.46272 .49471 m .48064 .48859 L .48903 .49798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46665 .74538 m .45744 .73674 L .46272 .49471 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.48599 .73978 m .46665 .74538 L .45744 .73674 L p
.653 .754 .924 r
F P
s
P
p
.002 w
.45744 .73674 m .47686 .73107 L .48599 .73978 L p
.653 .754 .924 r
F P
s
P
p
.002 w
.48064 .48859 m .48903 .49798 L .48543 .78412 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.48064 .48859 m .48903 .49798 L .48599 .73978 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.42936 .71434 m .43631 .49142 L .41859 .49752 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.41971 .7287 m .41029 .72003 L .41859 .49752 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.42171 .67006 m .41248 .66121 L .41859 .49752 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.41859 .49752 m .41029 .72003 L .42936 .71434 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.43631 .49142 m .41859 .49752 L .40979 .48812 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43097 .66295 m .43631 .49142 L .41859 .49752 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.41859 .49752 m .4122 .66875 L .43097 .66295 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.39213 .49424 m .40979 .48812 L .41859 .49752 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40979 .48812 m .41859 .49752 L .4122 .66875 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.40979 .48812 m .41859 .49752 L .4122 .66875 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.36235 .54847 m .36557 .49095 L .3481 .49705 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.3481 .49705 m .34454 .55448 L .36235 .54847 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.36557 .49095 m .3481 .49705 L .3389 .48764 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36219 .55127 m .36557 .49095 L .3481 .49705 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.3481 .49705 m .34436 .55727 L .36219 .55127 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.3389 .48764 m .3481 .49705 L .34436 .55727 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.69294 .49001 m .67426 .49611 L .66691 .4867 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71354 .83819 m .69294 .49001 L .67426 .49611 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.67426 .49611 m .69249 .84361 L .71354 .83819 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.64829 .49283 m .66691 .4867 L .67426 .49611 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67044 .82326 m .66226 .81488 L .64829 .49283 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.67717 .97008 m .66866 .96228 L .64829 .49283 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.69114 .81783 m .67044 .82326 L .66226 .81488 L p
.645 .763 .935 r
F P
s
P
p
.002 w
.66226 .81488 m .68306 .80938 L .69114 .81783 L p
.645 .763 .935 r
F P
s
P
p
.002 w
.66691 .4867 m .67426 .49611 L .69249 .84361 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.66691 .4867 m .67426 .49611 L .69114 .81783 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.63748 .93501 m .62221 .48953 L .60379 .49565 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.62489 .94806 m .61611 .94018 L .60379 .49565 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.62361 .90595 m .61494 .89791 L .60379 .49565 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.60379 .49565 m .61611 .94018 L .63748 .93501 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.62221 .48953 m .60379 .49565 L .59603 .48622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63227 .78289 m .62221 .48953 L .60379 .49565 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.60379 .49565 m .6119 .78844 L .63227 .78289 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.57767 .49236 m .59603 .48622 L .60379 .49565 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59603 .48622 m .60379 .49565 L .6119 .78844 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.59603 .48622 m .60379 .49565 L .6154 .91438 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.5547 .83735 m .55145 .48906 L .53327 .49518 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.54309 .80843 m .53418 .80002 L .53327 .49518 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.53327 .49518 m .5343 .84278 L .5547 .83735 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.55145 .48906 m .53327 .49518 L .52512 .48575 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55485 .85246 m .55145 .48906 L .53327 .49518 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.53327 .49518 m .53435 .85785 L .55485 .85246 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.50701 .49189 m .52512 .48575 L .53327 .49518 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51417 .8185 m .50501 .8101 L .50701 .49189 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.53421 .81305 m .51417 .8185 L .50501 .8101 L p
.645 .763 .934 r
F P
s
P
p
.002 w
.50501 .8101 m .52515 .80458 L .53421 .81305 L p
.645 .763 .934 r
F P
s
P
p
.002 w
.52512 .48575 m .53327 .49518 L .53435 .85785 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.52512 .48575 m .53327 .49518 L .53421 .81305 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.47686 .73107 m .48064 .48859 L .46272 .49471 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.46746 .70236 m .45838 .69358 L .46272 .49471 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.46272 .49471 m .45744 .73674 L .47686 .73107 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.48064 .48859 m .46272 .49471 L .45416 .48528 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47669 .74208 m .48064 .48859 L .46272 .49471 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.46272 .49471 m .4572 .74772 L .47669 .74208 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.43631 .49142 m .45416 .48528 L .46272 .49471 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43913 .70817 m .42983 .69939 L .43631 .49142 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.43871 .72308 m .42936 .71434 L .43631 .49142 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.45819 .70248 m .43913 .70817 L .42983 .69939 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.42983 .69939 m .44897 .69362 L .45819 .70248 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.45416 .48528 m .46272 .49471 L .4572 .74772 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.45416 .48528 m .46272 .49471 L .45819 .70248 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.40283 .65981 m .40979 .48812 L .39213 .49424 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.39358 .6745 m .38414 .66564 L .39213 .49424 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.39615 .6153 m .38691 .60625 L .39213 .49424 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.39213 .49424 m .38414 .66564 L .40283 .65981 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.40979 .48812 m .39213 .49424 L .38317 .4848 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4044 .62101 m .40979 .48812 L .39213 .49424 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.39213 .49424 m .38594 .6269 L .4044 .62101 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.36557 .49095 m .38317 .4848 L .39213 .49424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36843 .61743 m .35899 .60837 L .36557 .49095 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.38666 .61156 m .36843 .61743 L .35899 .60837 L p
.664 .74 .907 r
F P
s
P
p
.002 w
.35899 .60837 m .37729 .60242 L .38666 .61156 L p
.664 .74 .907 r
F P
s
P
p
.002 w
.38317 .4848 m .39213 .49424 L .38594 .6269 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.38317 .4848 m .39213 .49424 L .38666 .61156 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.68306 .80938 m .66691 .4867 L .64829 .49283 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.64829 .49283 m .66226 .81488 L .68306 .80938 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.66691 .4867 m .64829 .49283 L .64078 .48338 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68368 .82182 m .66691 .4867 L .64829 .49283 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.64829 .49283 m .6628 .82729 L .68368 .82182 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.62221 .48953 m .64078 .48338 L .64829 .49283 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64112 .80631 m .63278 .79784 L .62221 .48953 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.64615 .94295 m .63748 .93501 L .62221 .48953 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.66165 .80081 m .64112 .80631 L .63278 .79784 L p
.646 .762 .933 r
F P
s
P
p
.002 w
.63278 .79784 m .65341 .79228 L .66165 .80081 L p
.646 .762 .933 r
F P
s
P
p
.002 w
.64078 .48338 m .64829 .49283 L .6628 .82729 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.64078 .48338 m .64829 .49283 L .66165 .80081 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.60657 .9063 m .59603 .48622 L .57767 .49236 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.59438 .91958 m .58545 .91157 L .57767 .49236 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.59346 .8766 m .58465 .86842 L .57767 .49236 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.57767 .49236 m .58545 .91157 L .60657 .9063 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.59603 .48622 m .57767 .49236 L .56975 .4829 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.603 .76396 m .59603 .48622 L .57767 .49236 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.57767 .49236 m .58282 .76957 L .603 .76396 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.55145 .48906 m .56975 .4829 L .57767 .49236 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56975 .4829 m .57767 .49236 L .58282 .76957 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.56975 .4829 m .57767 .49236 L .58495 .88427 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.52515 .80458 m .52512 .48575 L .50701 .49189 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.51431 .77504 m .50528 .76648 L .50701 .49189 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.50701 .49189 m .50501 .8101 L .52515 .80458 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.52512 .48575 m .50701 .49189 L .49869 .48243 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52515 .81883 m .52512 .48575 L .50701 .49189 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.50701 .49189 m .50492 .82432 L .52515 .81883 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.48064 .48859 m .49869 .48243 L .50701 .49189 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48543 .78412 m .47617 .77557 L .48064 .48859 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.50521 .77857 m .48543 .78412 L .47617 .77557 L p
.649 .759 .93 r
F P
s
P
p
.002 w
.47617 .77557 m .49603 .76995 L .50521 .77857 L p
.649 .759 .93 r
F P
s
P
p
.002 w
.49869 .48243 m .50701 .49189 L .50492 .82432 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.49869 .48243 m .50701 .49189 L .50521 .77857 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.44897 .69362 m .45416 .48528 L .43631 .49142 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.43631 .49142 m .42983 .69939 L .44897 .69362 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.45416 .48528 m .43631 .49142 L .42758 .48195 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44873 .70338 m .45416 .48528 L .43631 .49142 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.43631 .49142 m .42953 .70912 L .44873 .70338 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.40979 .48812 m .42758 .48195 L .43631 .49142 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4122 .66875 m .40283 .65981 L .40979 .48812 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.4122 .66875 m .40283 .65981 L .40979 .48812 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.43097 .66295 m .4122 .66875 L .40283 .65981 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.40283 .65981 m .42167 .65395 L .43097 .66295 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.42758 .48195 m .43631 .49142 L .42953 .70912 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.42758 .48195 m .43631 .49142 L .43097 .66295 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.37729 .60242 m .38317 .4848 L .36557 .49095 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.37158 .5577 m .36235 .54847 L .36557 .49095 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.36557 .49095 m .35899 .60837 L .37729 .60242 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.38317 .4848 m .36557 .49095 L .35643 .48147 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.377 .60827 m .38317 .4848 L .36557 .49095 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.36557 .49095 m .35866 .61421 L .377 .60827 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.3389 .48764 m .35643 .48147 L .36557 .49095 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34436 .55727 m .33495 .54802 L .3389 .48764 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.36219 .55127 m .34436 .55727 L .33495 .54802 L p
.668 .734 .899 r
F P
s
P
p
.002 w
.33495 .54802 m .35284 .54194 L .36219 .55127 L p
.668 .734 .899 r
F P
s
P
p
.002 w
.35643 .48147 m .36557 .49095 L .35866 .61421 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.35643 .48147 m .36557 .49095 L .36219 .55127 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.71176 .48385 m .69294 .49001 L .68567 .48052 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73649 .85907 m .71176 .48385 L .69294 .49001 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.69294 .49001 m .71509 .86446 L .73649 .85907 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.66691 .4867 m .68567 .48052 L .69294 .49001 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69249 .84361 m .68436 .83526 L .66691 .4867 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.71354 .83819 m .69249 .84361 L .68436 .83526 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.68436 .83526 m .7055 .82977 L .71354 .83819 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.68567 .48052 m .69294 .49001 L .71509 .86446 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.68567 .48052 m .69294 .49001 L .71354 .83819 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.65341 .79228 m .64078 .48338 L .62221 .48953 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.62221 .48953 m .63278 .79784 L .65341 .79228 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.64078 .48338 m .62221 .48953 L .61454 .48004 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65391 .80456 m .64078 .48338 L .62221 .48953 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.62221 .48953 m .6332 .81009 L .65391 .80456 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.59603 .48622 m .61454 .48004 L .62221 .48953 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6119 .78844 m .6034 .77988 L .59603 .48622 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.6154 .91438 m .60657 .9063 L .59603 .48622 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.63227 .78289 m .6119 .78844 L .6034 .77988 L p
.648 .76 .931 r
F P
s
P
p
.002 w
.6034 .77988 m .62385 .77426 L .63227 .78289 L p
.648 .76 .931 r
F P
s
P
p
.002 w
.61454 .48004 m .62221 .48953 L .6332 .81009 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.61454 .48004 m .62221 .48953 L .63227 .78289 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.57598 .87604 m .56975 .4829 L .55145 .48906 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.56418 .88956 m .55512 .8814 L .55145 .48906 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.56365 .84568 m .5547 .83735 L .55145 .48906 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.55145 .48906 m .55512 .8814 L .57598 .87604 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.56975 .4829 m .55145 .48906 L .54336 .47957 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57389 .74393 m .56975 .4829 L .55145 .48906 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.55145 .48906 m .55388 .74961 L .57389 .74393 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.52512 .48575 m .54336 .47957 L .55145 .48906 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54336 .47957 m .55145 .48906 L .55388 .74961 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.54336 .47957 m .55145 .48906 L .55485 .85246 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.49603 .76995 m .49869 .48243 L .48064 .48859 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.48599 .73978 m .47686 .73107 L .48064 .48859 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.48064 .48859 m .47617 .77557 L .49603 .76995 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.49869 .48243 m .48064 .48859 L .47215 .47909 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49591 .7832 m .49869 .48243 L .48064 .48859 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.48064 .48859 m .47596 .78879 L .49591 .7832 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.45416 .48528 m .47215 .47909 L .48064 .48859 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4572 .74772 m .44784 .73902 L .45416 .48528 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.47669 .74208 m .4572 .74772 L .44784 .73902 L p
.652 .756 .925 r
F P
s
P
p
.002 w
.44784 .73902 m .46741 .7333 L .47669 .74208 L p
.652 .756 .925 r
F P
s
P
p
.002 w
.47215 .47909 m .48064 .48859 L .47596 .78879 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.47215 .47909 m .48064 .48859 L .47669 .74208 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.42167 .65395 m .42758 .48195 L .40979 .48812 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.40979 .48812 m .40283 .65981 L .42167 .65395 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.42758 .48195 m .40979 .48812 L .40089 .47861 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42139 .66226 m .42758 .48195 L .40979 .48812 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.4122 .66875 m .39358 .6745 L .38414 .66564 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.38414 .66564 m .40283 .65981 L .4122 .66875 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.40979 .48812 m .4025 .66811 L .42139 .66226 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.38317 .4848 m .40089 .47861 L .40979 .48812 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38594 .6269 m .37652 .61782 L .38317 .4848 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.4044 .62101 m .38594 .6269 L .37652 .61782 L p
.663 .742 .909 r
F P
s
P
p
.002 w
.37652 .61782 m .39506 .61185 L .4044 .62101 L p
.663 .742 .909 r
F P
s
P
p
.002 w
.40089 .47861 m .40979 .48812 L .4025 .66811 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.40089 .47861 m .40979 .48812 L .4044 .62101 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.35284 .54194 m .35643 .48147 L .3389 .48764 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.3389 .48764 m .33495 .54802 L .35284 .54194 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.35643 .48147 m .3389 .48764 L .3296 .47813 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35266 .54504 m .35643 .48147 L .3389 .48764 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.3389 .48764 m .33475 .5511 L .35266 .54504 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.3296 .47813 m .3389 .48764 L .33475 .5511 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.7055 .82977 m .68567 .48052 L .66691 .4867 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.69114 .81783 m .68306 .80938 L .66691 .4867 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.66691 .4867 m .68436 .83526 L .7055 .82977 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.68567 .48052 m .66691 .4867 L .65948 .47718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70627 .84336 m .68567 .48052 L .66691 .4867 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.66691 .4867 m .68503 .84881 L .70627 .84336 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.64078 .48338 m .65948 .47718 L .66691 .4867 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6628 .82729 m .65449 .81885 L .64078 .48338 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.68368 .82182 m .6628 .82729 L .65449 .81885 L p
.643 .765 .937 r
F P
s
P
p
.002 w
.65449 .81885 m .67547 .81331 L .68368 .82182 L p
.643 .765 .937 r
F P
s
P
p
.002 w
.65948 .47718 m .66691 .4867 L .68503 .84881 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.65948 .47718 m .66691 .4867 L .68368 .82182 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.62385 .77426 m .61454 .48004 L .59603 .48622 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.59603 .48622 m .6034 .77988 L .62385 .77426 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.61454 .48004 m .59603 .48622 L .58819 .4767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62424 .78634 m .61454 .48004 L .59603 .48622 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.59603 .48622 m .6037 .79193 L .62424 .78634 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.56975 .4829 m .58819 .4767 L .59603 .48622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58282 .76957 m .57415 .76092 L .56975 .4829 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.58495 .88427 m .57598 .87604 L .56975 .4829 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.603 .76396 m .58282 .76957 L .57415 .76092 L p
.65 .758 .929 r
F P
s
P
p
.002 w
.57415 .76092 m .59443 .75524 L .603 .76396 L p
.65 .758 .929 r
F P
s
P
p
.002 w
.58819 .4767 m .59603 .48622 L .6037 .79193 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.58819 .4767 m .59603 .48622 L .603 .76396 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.54574 .84409 m .54336 .47957 L .52512 .48575 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.53435 .85785 m .52515 .84955 L .52512 .48575 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.53421 .81305 m .52515 .80458 L .52512 .48575 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.52512 .48575 m .52515 .84955 L .54574 .84409 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.54336 .47957 m .52512 .48575 L .51687 .47622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55485 .85246 m .53435 .85785 L .52515 .84955 L p
.64 .768 .941 r
F P
s
P
p
.002 w
.52515 .84955 m .54574 .84409 L .55485 .85246 L p
.64 .768 .941 r
F P
s
P
p
.002 w
.54585 .86063 m .54336 .47957 L .52512 .48575 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.53221 .86444 m .51231 .86964 L .50321 .86162 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.50321 .86162 m .52319 .85635 L .53221 .86444 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.52512 .48575 m .52515 .86605 L .54585 .86063 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.49869 .48243 m .51687 .47622 L .52512 .48575 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5035 .81922 m .48394 .82454 L .47476 .81633 L p
.647 .761 .933 r
F P
s
P
p
.002 w
.47476 .81633 m .49441 .81094 L .5035 .81922 L p
.647 .761 .933 r
F P
s
P
p
.002 w
.50492 .82432 m .49561 .81586 L .49869 .48243 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.52515 .81883 m .50492 .82432 L .49561 .81586 L p
.644 .765 .936 r
F P
s
P
p
.002 w
.49561 .81586 m .51592 .8103 L .52515 .81883 L p
.644 .765 .936 r
F P
s
P
p
.002 w
.51687 .47622 m .52512 .48575 L .52515 .86605 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.51687 .47622 m .52512 .48575 L .52515 .81883 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.46741 .7333 m .47215 .47909 L .45416 .48528 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.45819 .70248 m .44897 .69362 L .45416 .48528 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.45416 .48528 m .44784 .73902 L .46741 .7333 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.47215 .47909 m .45416 .48528 L .4455 .47574 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46719 .74539 m .47215 .47909 L .45416 .48528 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.45416 .48528 m .44754 .75108 L .46719 .74539 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.42758 .48195 m .4455 .47574 L .45416 .48528 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42953 .70912 m .42008 .70026 L .42758 .48195 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.44873 .70338 m .42953 .70912 L .42008 .70026 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.42008 .70026 m .43936 .69444 L .44873 .70338 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.4455 .47574 m .45416 .48528 L .44754 .75108 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.4455 .47574 m .45416 .48528 L .44873 .70338 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.39506 .61185 m .40089 .47861 L .38317 .4848 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.38666 .61156 m .37729 .60242 L .38317 .4848 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.38317 .4848 m .37652 .61782 L .39506 .61185 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.40089 .47861 m .38317 .4848 L .3741 .47526 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39476 .61851 m .40089 .47861 L .38317 .4848 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.38317 .4848 m .37619 .62446 L .39476 .61851 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.35643 .48147 m .3741 .47526 L .38317 .4848 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35866 .61421 m .34909 .60505 L .35643 .48147 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.377 .60827 m .35866 .61421 L .34909 .60505 L p
.664 .741 .907 r
F P
s
P
p
.002 w
.34909 .60505 m .3675 .59904 L .377 .60827 L p
.664 .741 .907 r
F P
s
P
p
.002 w
.3741 .47526 m .38317 .4848 L .37619 .62446 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.3741 .47526 m .38317 .4848 L .377 .60827 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.73072 .47765 m .71176 .48385 L .70457 .4743 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76003 .88047 m .73072 .47765 L .71176 .48385 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.71176 .48385 m .73826 .88584 L .76003 .88047 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.68567 .48052 m .70457 .4743 L .71176 .48385 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71509 .86446 m .707 .85614 L .68567 .48052 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.73649 .85907 m .71509 .86446 L .707 .85614 L p
.638 .769 .942 r
F P
s
P
p
.002 w
.707 .85614 m .72851 .85067 L .73649 .85907 L p
.638 .769 .942 r
F P
s
P
p
.002 w
.70457 .4743 m .71176 .48385 L .73826 .88584 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.70457 .4743 m .71176 .48385 L .73649 .85907 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.67547 .81331 m .65948 .47718 L .64078 .48338 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.66165 .80081 m .65341 .79228 L .64078 .48338 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.64078 .48338 m .65449 .81885 L .67547 .81331 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.65948 .47718 m .64078 .48338 L .63318 .47382 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67611 .8268 m .65948 .47718 L .64078 .48338 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.64078 .48338 m .65504 .83231 L .67611 .8268 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.61454 .48004 m .63318 .47382 L .64078 .48338 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6332 .81009 m .62472 .80156 L .61454 .48004 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.65391 .80456 m .6332 .81009 L .62472 .80156 L p
.645 .763 .935 r
F P
s
P
p
.002 w
.62472 .80156 m .64553 .79596 L .65391 .80456 L p
.645 .763 .935 r
F P
s
P
p
.002 w
.63318 .47382 m .64078 .48338 L .65504 .83231 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.63318 .47382 m .64078 .48338 L .65391 .80456 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.59443 .75524 m .58819 .4767 L .56975 .4829 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.56975 .4829 m .57415 .76092 L .59443 .75524 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.58819 .4767 m .56975 .4829 L .56174 .47334 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59469 .76705 m .58819 .4767 L .56975 .4829 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.56975 .4829 m .57434 .77271 L .59469 .76705 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.54336 .47957 m .56174 .47334 L .56975 .4829 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55388 .74961 m .54507 .74086 L .54336 .47957 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.55485 .85246 m .54574 .84409 L .54336 .47957 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.57389 .74393 m .55388 .74961 L .54507 .74086 L p
.652 .756 .926 r
F P
s
P
p
.002 w
.54507 .74086 m .56516 .73512 L .57389 .74393 L p
.652 .756 .926 r
F P
s
P
p
.002 w
.56174 .47334 m .56975 .4829 L .57434 .77271 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.56174 .47334 m .56975 .4829 L .57389 .74393 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.51592 .8103 m .51687 .47622 L .49869 .48243 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.50521 .77857 m .49603 .76995 L .49869 .48243 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.49869 .48243 m .49561 .81586 L .51592 .8103 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.51687 .47622 m .49869 .48243 L .49027 .47285 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51588 .82594 m .51687 .47622 L .49869 .48243 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.49869 .48243 m .49546 .83146 L .51588 .82594 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.47215 .47909 m .49027 .47285 L .49869 .48243 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47547 .77185 m .45625 .7773 L .44701 .7689 L p
.652 .756 .926 r
F P
s
P
p
.002 w
.44701 .7689 m .46631 .76338 L .47547 .77185 L p
.652 .756 .926 r
F P
s
P
p
.002 w
.47596 .78879 m .46654 .78017 L .47215 .47909 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.49591 .7832 m .47596 .78879 L .46654 .78017 L p
.647 .761 .932 r
F P
s
P
p
.002 w
.46654 .78017 m .48657 .77452 L .49591 .7832 L p
.647 .761 .932 r
F P
s
P
p
.002 w
.49027 .47285 m .49869 .48243 L .49546 .83146 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.49027 .47285 m .49869 .48243 L .49591 .7832 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.43936 .69444 m .4455 .47574 L .42758 .48195 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.43097 .66295 m .42167 .65395 L .42758 .48195 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.42758 .48195 m .42008 .70026 L .43936 .69444 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.4455 .47574 m .42758 .48195 L .41875 .47237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43906 .70519 m .4455 .47574 L .42758 .48195 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.42758 .48195 m .41972 .71099 L .43906 .70519 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.40089 .47861 m .41875 .47237 L .42758 .48195 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4025 .66811 m .39299 .65909 L .40089 .47861 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.42139 .66226 m .4025 .66811 L .39299 .65909 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.39299 .65909 m .41196 .65317 L .42139 .66226 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.41875 .47237 m .42758 .48195 L .41972 .71099 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.41875 .47237 m .42758 .48195 L .42139 .66226 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3675 .59904 m .3741 .47526 L .35643 .48147 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.36219 .55127 m .35284 .54194 L .35643 .48147 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.35643 .48147 m .34909 .60505 L .3675 .59904 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.3741 .47526 m .35643 .48147 L .3472 .47189 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36895 .57184 m .3741 .47526 L .35643 .48147 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.35643 .48147 m .3507 .5779 L .36895 .57184 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.3296 .47813 m .3472 .47189 L .35643 .48147 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33475 .5511 m .32521 .54175 L .3296 .47813 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.35266 .54504 m .33475 .5511 L .32521 .54175 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.32521 .54175 m .34319 .53561 L .35266 .54504 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.3472 .47189 m .35643 .48147 L .3507 .5779 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.3472 .47189 m .35643 .48147 L .35266 .54504 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.72851 .85067 m .70457 .4743 L .68567 .48052 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.71354 .83819 m .7055 .82977 L .68567 .48052 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.68567 .48052 m .707 .85614 L .72851 .85067 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.70457 .4743 m .68567 .48052 L .67832 .47093 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72945 .86545 m .70457 .4743 L .68567 .48052 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.68567 .48052 m .70784 .87088 L .72945 .86545 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.65948 .47718 m .67832 .47093 L .68567 .48052 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68503 .84881 m .67676 .8404 L .65948 .47718 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.70627 .84336 m .68503 .84881 L .67676 .8404 L p
.64 .768 .941 r
F P
s
P
p
.002 w
.67676 .8404 m .6981 .83488 L .70627 .84336 L p
.64 .768 .941 r
F P
s
P
p
.002 w
.67832 .47093 m .68567 .48052 L .70784 .87088 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.67832 .47093 m .68567 .48052 L .70627 .84336 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.64553 .79596 m .63318 .47382 L .61454 .48004 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.63227 .78289 m .62385 .77426 L .61454 .48004 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.61454 .48004 m .62472 .80156 L .64553 .79596 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.63318 .47382 m .61454 .48004 L .60677 .47044 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64604 .80931 m .63318 .47382 L .61454 .48004 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.61454 .48004 m .62514 .81488 L .64604 .80931 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.58819 .4767 m .60677 .47044 L .61454 .48004 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6037 .79193 m .59506 .78331 L .58819 .4767 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.62424 .78634 m .6037 .79193 L .59506 .78331 L p
.647 .761 .933 r
F P
s
P
p
.002 w
.59506 .78331 m .61569 .77764 L .62424 .78634 L p
.647 .761 .933 r
F P
s
P
p
.002 w
.60677 .47044 m .61454 .48004 L .62514 .81488 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.60677 .47044 m .61454 .48004 L .62424 .78634 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.56516 .73512 m .56174 .47334 L .54336 .47957 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.54336 .47957 m .54507 .74086 L .56516 .73512 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.56174 .47334 m .54336 .47957 L .53519 .46996 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56531 .7466 m .56174 .47334 L .54336 .47957 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.54336 .47957 m .54514 .75232 L .56531 .7466 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.51687 .47622 m .53519 .46996 L .54336 .47957 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53519 .46996 m .54336 .47957 L .54514 .75232 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.53519 .46996 m .54336 .47957 L .54585 .86063 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.48657 .77452 m .49027 .47285 L .47215 .47909 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.47669 .74208 m .46741 .7333 L .47215 .47909 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.47215 .47909 m .46654 .78017 L .48657 .77452 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.49027 .47285 m .47215 .47909 L .46356 .46948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48639 .7891 m .49027 .47285 L .47215 .47909 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.47215 .47909 m .46627 .79473 L .48639 .7891 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.4455 .47574 m .46356 .46948 L .47215 .47909 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43871 .72308 m .41971 .7287 L .41029 .72003 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.41029 .72003 m .42936 .71434 L .43871 .72308 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.44754 .75108 m .43802 .74231 L .4455 .47574 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.46719 .74539 m .44754 .75108 L .43802 .74231 L p
.651 .757 .927 r
F P
s
P
p
.002 w
.43802 .74231 m .45776 .73655 L .46719 .74539 L p
.651 .757 .927 r
F P
s
P
p
.002 w
.46356 .46948 m .47215 .47909 L .46627 .79473 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.46356 .46948 m .47215 .47909 L .46719 .74539 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.41196 .65317 m .41875 .47237 L .40089 .47861 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.4044 .62101 m .39506 .61185 L .40089 .47861 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.40089 .47861 m .39299 .65909 L .41196 .65317 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.41875 .47237 m .40089 .47861 L .39189 .46899 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41161 .66236 m .41875 .47237 L .40089 .47861 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.40089 .47861 m .39258 .66827 L .41161 .66236 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.3741 .47526 m .39189 .46899 L .40089 .47861 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37619 .62446 m .36663 .61528 L .3741 .47526 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.39476 .61851 m .37619 .62446 L .36663 .61528 L p
.663 .743 .91 r
F P
s
P
p
.002 w
.36663 .61528 m .38528 .60926 L .39476 .61851 L p
.663 .743 .91 r
F P
s
P
p
.002 w
.39189 .46899 m .40089 .47861 L .39258 .66827 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.39189 .46899 m .40089 .47861 L .39476 .61851 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.34319 .53561 m .3472 .47189 L .3296 .47813 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.3296 .47813 m .32521 .54175 L .34319 .53561 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.3472 .47189 m .3296 .47813 L .32018 .46851 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34404 .52196 m .3472 .47189 L .3296 .47813 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.3296 .47813 m .32615 .52812 L .34404 .52196 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.32018 .46851 m .3296 .47813 L .32615 .52812 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.74982 .47141 m .73072 .47765 L .72361 .46803 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78417 .90242 m .74982 .47141 L .73072 .47765 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.73072 .47765 m .76202 .90777 L .78417 .90242 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.70457 .4743 m .72361 .46803 L .73072 .47765 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73826 .88584 m .73022 .87755 L .70457 .4743 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.76003 .88047 m .73826 .88584 L .73022 .87755 L p
.635 .773 .946 r
F P
s
P
p
.002 w
.73022 .87755 m .7521 .87211 L .76003 .88047 L p
.635 .773 .946 r
F P
s
P
p
.002 w
.72361 .46803 m .73072 .47765 L .76202 .90777 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.72361 .46803 m .73072 .47765 L .76003 .88047 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.6981 .83488 m .67832 .47093 L .65948 .47718 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.68368 .82182 m .67547 .81331 L .65948 .47718 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.65948 .47718 m .67676 .8404 L .6981 .83488 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.67832 .47093 m .65948 .47718 L .65196 .46754 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69891 .84962 m .67832 .47093 L .65948 .47718 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.65948 .47718 m .67746 .85511 L .69891 .84962 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.63318 .47382 m .65196 .46754 L .65948 .47718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65504 .83231 m .6466 .82381 L .63318 .47382 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.67611 .8268 m .65504 .83231 L .6466 .82381 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.6466 .82381 m .66777 .81823 L .67611 .8268 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.65196 .46754 m .65948 .47718 L .67746 .85511 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.65196 .46754 m .65948 .47718 L .67611 .8268 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.61569 .77764 m .60677 .47044 L .58819 .4767 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.603 .76396 m .59443 .75524 L .58819 .4767 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.58819 .4767 m .59506 .78331 L .61569 .77764 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.60677 .47044 m .58819 .4767 L .58026 .46706 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61607 .7908 m .60677 .47044 L .58819 .4767 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.58819 .4767 m .59535 .79643 L .61607 .7908 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.56174 .47334 m .58026 .46706 L .58819 .4767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57434 .77271 m .56554 .76399 L .56174 .47334 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.59469 .76705 m .57434 .77271 L .56554 .76399 L p
.649 .76 .93 r
F P
s
P
p
.002 w
.56554 .76399 m .58598 .75826 L .59469 .76705 L p
.649 .76 .93 r
F P
s
P
p
.002 w
.58026 .46706 m .58819 .4767 L .59535 .79643 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.58026 .46706 m .58819 .4767 L .59469 .76705 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.53659 .85221 m .53519 .46996 L .51687 .47622 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.52515 .86605 m .51579 .8577 L .51687 .47622 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.52515 .81883 m .51592 .8103 L .51687 .47622 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.51687 .47622 m .51579 .8577 L .53659 .85221 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.53519 .46996 m .51687 .47622 L .50852 .46657 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53612 .72486 m .53519 .46996 L .51687 .47622 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.51687 .47622 m .51615 .73064 L .53612 .72486 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.49027 .47285 m .50852 .46657 L .51687 .47622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50852 .46657 m .51687 .47622 L .51615 .73064 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.50852 .46657 m .51687 .47622 L .51588 .82594 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.45776 .73655 m .46356 .46948 L .4455 .47574 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.44873 .70338 m .43936 .69444 L .4455 .47574 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.4455 .47574 m .43802 .74231 L .45776 .73655 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.46356 .46948 m .4455 .47574 L .43675 .46609 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45747 .74989 m .46356 .46948 L .4455 .47574 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.4455 .47574 m .43765 .75562 L .45747 .74989 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.41875 .47237 m .43675 .46609 L .4455 .47574 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41972 .71099 m .41012 .70205 L .41875 .47237 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.43906 .70519 m .41972 .71099 L .41012 .70205 L p
.655 .753 .922 r
F P
s
P
p
.002 w
.41012 .70205 m .42955 .69618 L .43906 .70519 L p
.655 .753 .922 r
F P
s
P
p
.002 w
.43675 .46609 m .4455 .47574 L .43765 .75562 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.43675 .46609 m .4455 .47574 L .43906 .70519 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.38528 .60926 m .39189 .46899 L .3741 .47526 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.377 .60827 m .3675 .59904 L .3741 .47526 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.3741 .47526 m .36663 .61528 L .38528 .60926 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.39189 .46899 m .3741 .47526 L .36493 .4656 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38493 .61664 m .39189 .46899 L .3741 .47526 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.3741 .47526 m .36624 .62265 L .38493 .61664 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.3472 .47189 m .36493 .4656 L .3741 .47526 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3507 .5779 m .34111 .56856 L .3472 .47189 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.36895 .57184 m .3507 .5779 L .34111 .56856 L p
.666 .738 .904 r
F P
s
P
p
.002 w
.34111 .56856 m .35943 .56243 L .36895 .57184 L p
.666 .738 .904 r
F P
s
P
p
.002 w
.36493 .4656 m .3741 .47526 L .36624 .62265 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.36493 .4656 m .3741 .47526 L .36895 .57184 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.7521 .87211 m .72361 .46803 L .70457 .4743 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.73649 .85907 m .72851 .85067 L .70457 .4743 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.70457 .4743 m .73022 .87755 L .7521 .87211 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.72361 .46803 m .70457 .4743 L .6973 .46463 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75323 .88812 m .72361 .46803 L .70457 .4743 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.70457 .4743 m .73123 .89352 L .75323 .88812 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.67832 .47093 m .6973 .46463 L .70457 .4743 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70784 .87088 m .69961 .86251 L .67832 .47093 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.72945 .86545 m .70784 .87088 L .69961 .86251 L p
.637 .771 .944 r
F P
s
P
p
.002 w
.69961 .86251 m .72132 .857 L .72945 .86545 L p
.637 .771 .944 r
F P
s
P
p
.002 w
.6973 .46463 m .70457 .4743 L .73123 .89352 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.6973 .46463 m .70457 .4743 L .72945 .86545 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.66777 .81823 m .65196 .46754 L .63318 .47382 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.65391 .80456 m .64553 .79596 L .63318 .47382 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.63318 .47382 m .6466 .82381 L .66777 .81823 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.65196 .46754 m .63318 .47382 L .62549 .46414 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66843 .8329 m .65196 .46754 L .63318 .47382 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.63318 .47382 m .64716 .83845 L .66843 .8329 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.60677 .47044 m .62549 .46414 L .63318 .47382 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62514 .81488 m .61652 .80629 L .60677 .47044 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.64604 .80931 m .62514 .81488 L .61652 .80629 L p
.643 .765 .937 r
F P
s
P
p
.002 w
.61652 .80629 m .63752 .80064 L .64604 .80931 L p
.643 .765 .937 r
F P
s
P
p
.002 w
.62549 .46414 m .63318 .47382 L .64716 .83845 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.62549 .46414 m .63318 .47382 L .64604 .80931 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.58598 .75826 m .58026 .46706 L .56174 .47334 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.57389 .74393 m .56516 .73512 L .56174 .47334 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.56174 .47334 m .56554 .76399 L .58598 .75826 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.58026 .46706 m .56174 .47334 L .55364 .46366 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58624 .77117 m .58026 .46706 L .56174 .47334 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.56174 .47334 m .56571 .77686 L .58624 .77117 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.53519 .46996 m .55364 .46366 L .56174 .47334 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54514 .75232 m .53619 .7435 L .53519 .46996 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.54585 .86063 m .53659 .85221 L .53519 .46996 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.56531 .7466 m .54514 .75232 L .53619 .7435 L p
.65 .758 .928 r
F P
s
P
p
.002 w
.53619 .7435 m .55644 .73771 L .56531 .7466 L p
.65 .758 .928 r
F P
s
P
p
.002 w
.55364 .46366 m .56174 .47334 L .56571 .77686 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.55364 .46366 m .56174 .47334 L .56531 .7466 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.50649 .81736 m .50852 .46657 L .49027 .47285 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.49546 .83146 m .48598 .82295 L .49027 .47285 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.49591 .7832 m .48657 .77452 L .49027 .47285 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.49027 .47285 m .48598 .82295 L .50649 .81736 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.50852 .46657 m .49027 .47285 L .48175 .46317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50716 .70169 m .50852 .46657 L .49027 .47285 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.49027 .47285 m .48739 .70755 L .50716 .70169 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.46356 .46948 m .48175 .46317 L .49027 .47285 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48175 .46317 m .49027 .47285 L .48739 .70755 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.48175 .46317 m .49027 .47285 L .48639 .7891 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.42955 .69618 m .43675 .46609 L .41875 .47237 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.42139 .66226 m .41196 .65317 L .41875 .47237 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.41875 .47237 m .41012 .70205 L .42955 .69618 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.43675 .46609 m .41875 .47237 L .40982 .46268 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42918 .70808 m .43675 .46609 L .41875 .47237 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.41875 .47237 m .40967 .71392 L .42918 .70808 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.39189 .46899 m .40982 .46268 L .41875 .47237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39258 .66827 m .38293 .65916 L .39189 .46899 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.41161 .66236 m .39258 .66827 L .38293 .65916 L p
.658 .748 .916 r
F P
s
P
p
.002 w
.38293 .65916 m .40203 .65319 L .41161 .66236 L p
.658 .748 .916 r
F P
s
P
p
.002 w
.40982 .46268 m .41875 .47237 L .40967 .71392 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.40982 .46268 m .41875 .47237 L .41161 .66236 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.35943 .56243 m .36493 .4656 L .3472 .47189 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.35266 .54504 m .34319 .53561 L .3472 .47189 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.3472 .47189 m .34111 .56856 L .35943 .56243 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.36493 .4656 m .3472 .47189 L .33785 .4622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35913 .56771 m .36493 .4656 L .3472 .47189 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.3472 .47189 m .34078 .57383 L .35913 .56771 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.32018 .46851 m .33785 .4622 L .3472 .47189 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32615 .52812 m .31655 .51862 L .32018 .46851 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.34404 .52196 m .32615 .52812 L .31655 .51862 L p
.669 .733 .898 r
F P
s
P
p
.002 w
.31655 .51862 m .33451 .51238 L .34404 .52196 L p
.669 .733 .898 r
F P
s
P
p
.002 w
.33785 .4622 m .3472 .47189 L .34078 .57383 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.33785 .4622 m .3472 .47189 L .34404 .52196 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.76906 .46512 m .74982 .47141 L .74279 .46171 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78749 .67767 m .76906 .46512 L .74982 .47141 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.74982 .47141 m .76673 .68359 L .78749 .67767 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.72361 .46803 m .74279 .46171 L .74982 .47141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74279 .46171 m .74982 .47141 L .76673 .68359 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.74279 .46171 m .74982 .47141 L .78417 .90242 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.72132 .857 m .6973 .46463 L .67832 .47093 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.70627 .84336 m .6981 .83488 L .67832 .47093 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.67832 .47093 m .69961 .86251 L .72132 .857 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.6973 .46463 m .67832 .47093 L .67088 .46122 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72231 .87305 m .6973 .46463 L .67832 .47093 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.67832 .47093 m .70048 .87851 L .72231 .87305 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.65196 .46754 m .67088 .46122 L .67832 .47093 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67746 .85511 m .66905 .84664 L .65196 .46754 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.69891 .84962 m .67746 .85511 L .66905 .84664 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.66905 .84664 m .6906 .84108 L .69891 .84962 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.67088 .46122 m .67832 .47093 L .70048 .87851 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.67088 .46122 m .67832 .47093 L .69891 .84962 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.63752 .80064 m .62549 .46414 L .60677 .47044 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.62424 .78634 m .61569 .77764 L .60677 .47044 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.60677 .47044 m .61652 .80629 L .63752 .80064 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.62549 .46414 m .60677 .47044 L .59892 .46073 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63804 .8152 m .62549 .46414 L .60677 .47044 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.60677 .47044 m .61694 .82081 L .63804 .8152 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.58026 .46706 m .59892 .46073 L .60677 .47044 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59535 .79643 m .58656 .78775 L .58026 .46706 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.61607 .7908 m .59535 .79643 L .58656 .78775 L p
.645 .763 .934 r
F P
s
P
p
.002 w
.58656 .78775 m .60738 .78204 L .61607 .7908 L p
.645 .763 .934 r
F P
s
P
p
.002 w
.59892 .46073 m .60677 .47044 L .61694 .82081 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.59892 .46073 m .60677 .47044 L .61607 .7908 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.55644 .73771 m .55364 .46366 L .53519 .46996 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.53519 .46996 m .53619 .7435 L .55644 .73771 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.55364 .46366 m .53519 .46996 L .52692 .46024 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55657 .75029 m .55364 .46366 L .53519 .46996 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.53519 .46996 m .53623 .75605 L .55657 .75029 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.50852 .46657 m .52692 .46024 L .53519 .46996 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51615 .73064 m .50704 .72172 L .50852 .46657 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.51588 .82594 m .50649 .81736 L .50852 .46657 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.53612 .72486 m .51615 .73064 L .50704 .72172 L p
.652 .756 .925 r
F P
s
P
p
.002 w
.50704 .72172 m .5271 .71587 L .53612 .72486 L p
.652 .756 .925 r
F P
s
P
p
.002 w
.52692 .46024 m .53519 .46996 L .53623 .75605 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.52692 .46024 m .53519 .46996 L .53612 .72486 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.47689 .78036 m .48175 .46317 L .46356 .46948 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.46627 .79473 m .45668 .78605 L .46356 .46948 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.46719 .74539 m .45776 .73655 L .46356 .46948 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.46356 .46948 m .45668 .78605 L .47689 .78036 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.48175 .46317 m .46356 .46948 L .45487 .45976 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47848 .67694 m .48175 .46317 L .46356 .46948 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.46356 .46948 m .45892 .68286 L .47848 .67694 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.43675 .46609 m .45487 .45976 L .46356 .46948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45487 .45976 m .46356 .46948 L .45892 .68286 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.45487 .45976 m .46356 .46948 L .45747 .74989 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.40203 .65319 m .40982 .46268 L .39189 .46899 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.39476 .61851 m .38528 .60926 L .39189 .46899 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.39189 .46899 m .38293 .65916 L .40203 .65319 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.40982 .46268 m .39189 .46899 L .38279 .45927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40162 .66339 m .40982 .46268 L .39189 .46899 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.39189 .46899 m .38245 .66935 L .40162 .66339 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.36493 .4656 m .38279 .45927 L .39189 .46899 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36624 .62265 m .35654 .61338 L .36493 .4656 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.38493 .61664 m .36624 .62265 L .35654 .61338 L p
.662 .744 .911 r
F P
s
P
p
.002 w
.35654 .61338 m .37531 .6073 L .38493 .61664 L p
.662 .744 .911 r
F P
s
P
p
.002 w
.38279 .45927 m .39189 .46899 L .38245 .66935 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.38279 .45927 m .39189 .46899 L .38493 .61664 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.33451 .51238 m .33785 .4622 L .32018 .46851 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.32018 .46851 m .31655 .51862 L .33451 .51238 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.33785 .4622 m .32018 .46851 L .31066 .45878 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33432 .51522 m .33785 .4622 L .32018 .46851 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.32018 .46851 m .31634 .52145 L .33432 .51522 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.31066 .45878 m .32018 .46851 L .31634 .52145 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.77629 .8941 m .74279 .46171 L .72361 .46803 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.76202 .90777 m .75403 .89952 L .72361 .46803 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.76003 .88047 m .7521 .87211 L .72361 .46803 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.72361 .46803 m .75403 .89952 L .77629 .8941 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.74279 .46171 m .72361 .46803 L .71642 .45829 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75892 .66991 m .74279 .46171 L .72361 .46803 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.72361 .46803 m .73826 .67586 L .75892 .66991 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.6973 .46463 m .71642 .45829 L .72361 .46803 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71642 .45829 m .72361 .46803 L .73826 .67586 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.71642 .45829 m .72361 .46803 L .75323 .88812 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.6906 .84108 m .67088 .46122 L .65196 .46754 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.67611 .8268 m .66777 .81823 L .65196 .46754 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.65196 .46754 m .66905 .84664 L .6906 .84108 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.67088 .46122 m .65196 .46754 L .64435 .4578 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69143 .85713 m .67088 .46122 L .65196 .46754 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.65196 .46754 m .66977 .86265 L .69143 .85713 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.62549 .46414 m .64435 .4578 L .65196 .46754 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64716 .83845 m .63856 .82989 L .62549 .46414 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.66843 .8329 m .64716 .83845 L .63856 .82989 L p
.64 .768 .941 r
F P
s
P
p
.002 w
.63856 .82989 m .65994 .82427 L .66843 .8329 L p
.64 .768 .941 r
F P
s
P
p
.002 w
.64435 .4578 m .65196 .46754 L .66977 .86265 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.64435 .4578 m .65196 .46754 L .66843 .8329 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.60738 .78204 m .59892 .46073 L .58026 .46706 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.59469 .76705 m .58598 .75826 L .58026 .46706 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.58026 .46706 m .58656 .78775 L .60738 .78204 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.59892 .46073 m .58026 .46706 L .57224 .45731 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60776 .79643 m .59892 .46073 L .58026 .46706 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.58026 .46706 m .58685 .8021 L .60776 .79643 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.55364 .46366 m .57224 .45731 L .58026 .46706 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56571 .77686 m .55675 .76808 L .55364 .46366 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.58624 .77117 m .56571 .77686 L .55675 .76808 L p
.647 .761 .932 r
F P
s
P
p
.002 w
.55675 .76808 m .57738 .76231 L .58624 .77117 L p
.647 .761 .932 r
F P
s
P
p
.002 w
.57224 .45731 m .58026 .46706 L .58685 .8021 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.57224 .45731 m .58026 .46706 L .58624 .77117 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.5271 .71587 m .52692 .46024 L .50852 .46657 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.50852 .46657 m .50704 .72172 L .5271 .71587 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.52692 .46024 m .50852 .46657 L .50008 .45682 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52711 .72804 m .52692 .46024 L .50852 .46657 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.50852 .46657 m .50697 .73387 L .52711 .72804 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.48175 .46317 m .50008 .45682 L .50852 .46657 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48739 .70755 m .47815 .69852 L .48175 .46317 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.48639 .7891 m .47689 .78036 L .48175 .46317 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.50716 .70169 m .48739 .70755 L .47815 .69852 L p
.654 .753 .923 r
F P
s
P
p
.002 w
.47815 .69852 m .498 .69259 L .50716 .70169 L p
.654 .753 .923 r
F P
s
P
p
.002 w
.50008 .45682 m .50852 .46657 L .50697 .73387 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.50008 .45682 m .50852 .46657 L .50716 .70169 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.44787 .74098 m .45487 .45976 L .43675 .46609 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.43765 .75562 m .42797 .74679 L .43675 .46609 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.43906 .70519 m .42955 .69618 L .43675 .46609 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.43675 .46609 m .42797 .74679 L .44787 .74098 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.45487 .45976 m .43675 .46609 L .42789 .45633 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45013 .65041 m .45487 .45976 L .43675 .46609 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.43675 .46609 m .43079 .65641 L .45013 .65041 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.40982 .46268 m .42789 .45633 L .43675 .46609 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42789 .45633 m .43675 .46609 L .43079 .65641 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.42789 .45633 m .43675 .46609 L .42918 .70808 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.37531 .6073 m .38279 .45927 L .36493 .4656 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.36895 .57184 m .35943 .56243 L .36493 .4656 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.36493 .4656 m .35654 .61338 L .37531 .6073 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.38279 .45927 m .36493 .4656 L .35565 .45584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37489 .61552 m .38279 .45927 L .36493 .4656 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.36493 .4656 m .35607 .62159 L .37489 .61552 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.33785 .4622 m .35565 .45584 L .36493 .4656 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34078 .57383 m .33106 .56439 L .33785 .4622 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.35913 .56771 m .34078 .57383 L .33106 .56439 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.33106 .56439 m .34948 .5582 L .35913 .56771 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.35565 .45584 m .36493 .4656 L .35607 .62159 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.35565 .45584 m .36493 .4656 L .35913 .56771 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.78845 .45878 m .76906 .46512 L .76212 .45534 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80956 .68382 m .78845 .45878 L .76906 .46512 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.76906 .46512 m .78854 .68975 L .80956 .68382 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.74279 .46171 m .76212 .45534 L .76906 .46512 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76673 .68359 m .75928 .67447 L .74279 .46171 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.78417 .90242 m .77629 .8941 L .74279 .46171 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.78749 .67767 m .76673 .68359 L .75928 .67447 L p
.656 .751 .919 r
F P
s
P
p
.002 w
.75928 .67447 m .78014 .66848 L .78749 .67767 L p
.656 .751 .919 r
F P
s
P
p
.002 w
.76212 .45534 m .76906 .46512 L .78854 .68975 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.76212 .45534 m .76906 .46512 L .78749 .67767 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.74515 .87971 m .71642 .45829 L .6973 .46463 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.73123 .89352 m .72305 .88519 L .6973 .46463 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.72945 .86545 m .72132 .857 L .6973 .46463 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.6973 .46463 m .72305 .88519 L .74515 .87971 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.71642 .45829 m .6973 .46463 L .68994 .45485 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7303 .66181 m .71642 .45829 L .6973 .46463 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.6973 .46463 m .70974 .66779 L .7303 .66181 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.67088 .46122 m .68994 .45485 L .6973 .46463 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68994 .45485 m .6973 .46463 L .70974 .66779 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.68994 .45485 m .6973 .46463 L .72231 .87305 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.65994 .82427 m .64435 .4578 L .62549 .46414 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.64604 .80931 m .63752 .80064 L .62549 .46414 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.62549 .46414 m .63856 .82989 L .65994 .82427 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.64435 .4578 m .62549 .46414 L .61771 .45436 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66062 .84029 m .64435 .4578 L .62549 .46414 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.62549 .46414 m .63913 .84587 L .66062 .84029 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.59892 .46073 m .61771 .45436 L .62549 .46414 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61694 .82081 m .60817 .81216 L .59892 .46073 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.63804 .8152 m .61694 .82081 L .60817 .81216 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.60817 .81216 m .62937 .80648 L .63804 .8152 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.61771 .45436 m .62549 .46414 L .63913 .84587 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.61771 .45436 m .62549 .46414 L .63804 .8152 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.57738 .76231 m .57224 .45731 L .55364 .46366 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.56531 .7466 m .55644 .73771 L .55364 .46366 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.55364 .46366 m .55675 .76808 L .57738 .76231 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.57224 .45731 m .55364 .46366 L .54545 .45387 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57762 .77646 m .57224 .45731 L .55364 .46366 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.55364 .46366 m .5569 .7822 L .57762 .77646 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.52692 .46024 m .54545 .45387 L .55364 .46366 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53623 .75605 m .52713 .74717 L .52692 .46024 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.55657 .75029 m .53623 .75605 L .52713 .74717 L p
.649 .759 .93 r
F P
s
P
p
.002 w
.52713 .74717 m .54756 .74133 L .55657 .75029 L p
.649 .759 .93 r
F P
s
P
p
.002 w
.54545 .45387 m .55364 .46366 L .5569 .7822 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.54545 .45387 m .55364 .46366 L .55657 .75029 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.498 .69259 m .50008 .45682 L .48175 .46317 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.48175 .46317 m .47815 .69852 L .498 .69259 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.50008 .45682 m .48175 .46317 L .47314 .45338 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4979 .70426 m .50008 .45682 L .48175 .46317 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.48175 .46317 m .47797 .71016 L .4979 .70426 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.45487 .45976 m .47314 .45338 L .48175 .46317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45892 .68286 m .44955 .67373 L .45487 .45976 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.45747 .74989 m .44787 .74098 L .45487 .45976 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.47848 .67694 m .45892 .68286 L .44955 .67373 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.44955 .67373 m .46919 .66773 L .47848 .67694 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.47314 .45338 m .48175 .46317 L .47797 .71016 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.47314 .45338 m .48175 .46317 L .47848 .67694 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.4195 .699 m .42789 .45633 L .40982 .46268 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.40967 .71392 m .39992 .70491 L .40982 .46268 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.41161 .66236 m .40203 .65319 L .40982 .46268 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.40982 .46268 m .39992 .70491 L .4195 .699 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.42789 .45633 m .40982 .46268 L .40079 .45288 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42918 .70808 m .40967 .71392 L .39992 .70491 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.39992 .70491 m .4195 .699 L .42918 .70808 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.41905 .71224 m .42789 .45633 L .40982 .46268 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.40982 .46268 m .39938 .71813 L .41905 .71224 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.38279 .45927 m .40079 .45288 L .40982 .46268 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38245 .66935 m .37263 .66017 L .38279 .45927 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.40162 .66339 m .38245 .66935 L .37263 .66017 L p
.657 .749 .918 r
F P
s
P
p
.002 w
.37263 .66017 m .39188 .65414 L .40162 .66339 L p
.657 .749 .918 r
F P
s
P
p
.002 w
.40079 .45288 m .40982 .46268 L .39938 .71813 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.40079 .45288 m .40982 .46268 L .40162 .66339 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.34948 .5582 m .35565 .45584 L .33785 .4622 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.34404 .52196 m .33451 .51238 L .33785 .4622 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.33785 .4622 m .33106 .56439 L .34948 .5582 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.35565 .45584 m .33785 .4622 L .3284 .45239 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34912 .56411 m .35565 .45584 L .33785 .4622 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.33785 .4622 m .33066 .57029 L .34912 .56411 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.31066 .45878 m .3284 .45239 L .33785 .4622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31634 .52145 m .30662 .51185 L .31066 .45878 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.33432 .51522 m .31634 .52145 L .30662 .51185 L p
.669 .733 .898 r
F P
s
P
p
.002 w
.30662 .51185 m .32467 .50555 L .33432 .51522 L p
.669 .733 .898 r
F P
s
P
p
.002 w
.3284 .45239 m .33785 .4622 L .33066 .57029 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.3284 .45239 m .33785 .4622 L .33432 .51522 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.78014 .66848 m .76212 .45534 L .74279 .46171 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.74279 .46171 m .75928 .67447 L .78014 .66848 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.76212 .45534 m .74279 .46171 L .73569 .4519 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78081 .67641 m .76212 .45534 L .74279 .46171 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.74279 .46171 m .75989 .68238 L .78081 .67641 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.71642 .45829 m .73569 .4519 L .74279 .46171 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73826 .67586 m .73063 .66669 L .71642 .45829 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.75323 .88812 m .74515 .87971 L .71642 .45829 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.75892 .66991 m .73826 .67586 L .73063 .66669 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.73063 .66669 m .75139 .66066 L .75892 .66991 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.73569 .4519 m .74279 .46171 L .75989 .68238 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.73569 .4519 m .74279 .46171 L .75892 .66991 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.71404 .86456 m .68994 .45485 L .67088 .46122 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.70048 .87851 m .6921 .87009 L .67088 .46122 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.69891 .84962 m .6906 .84108 L .67088 .46122 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.67088 .46122 m .6921 .87009 L .71404 .86456 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.68994 .45485 m .67088 .46122 L .66335 .4514 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70162 .65334 m .68994 .45485 L .67088 .46122 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.67088 .46122 m .68116 .65936 L .70162 .65334 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.64435 .4578 m .66335 .4514 L .67088 .46122 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66335 .4514 m .67088 .46122 L .68116 .65936 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.66335 .4514 m .67088 .46122 L .69143 .85713 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.62937 .80648 m .61771 .45436 L .59892 .46073 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.61607 .7908 m .60738 .78204 L .59892 .46073 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.59892 .46073 m .60817 .81216 L .62937 .80648 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.61771 .45436 m .59892 .46073 L .59097 .45091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6299 .82241 m .61771 .45436 L .59892 .46073 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.59892 .46073 m .60859 .82806 L .6299 .82241 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.57224 .45731 m .59097 .45091 L .59892 .46073 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58685 .8021 m .5779 .79335 L .57224 .45731 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.60776 .79643 m .58685 .8021 L .5779 .79335 L p
.644 .765 .936 r
F P
s
P
p
.002 w
.5779 .79335 m .59891 .78761 L .60776 .79643 L p
.644 .765 .936 r
F P
s
P
p
.002 w
.59097 .45091 m .59892 .46073 L .60859 .82806 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.59097 .45091 m .59892 .46073 L .60776 .79643 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.54756 .74133 m .54545 .45387 L .52692 .46024 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.53612 .72486 m .5271 .71587 L .52692 .46024 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.52692 .46024 m .52713 .74717 L .54756 .74133 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.54545 .45387 m .52692 .46024 L .51855 .45041 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54766 .75517 m .54545 .45387 L .52692 .46024 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.52692 .46024 m .52714 .76098 L .54766 .75517 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.50008 .45682 m .51855 .45041 L .52692 .46024 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50697 .73387 m .49772 .72488 L .50008 .45682 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.52711 .72804 m .50697 .73387 L .49772 .72488 L p
.651 .757 .927 r
F P
s
P
p
.002 w
.49772 .72488 m .51794 .71897 L .52711 .72804 L p
.651 .757 .927 r
F P
s
P
p
.002 w
.51855 .45041 m .52692 .46024 L .52714 .76098 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.51855 .45041 m .52692 .46024 L .52711 .72804 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.46919 .66773 m .47314 .45338 L .45487 .45976 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.45487 .45976 m .44955 .67373 L .46919 .66773 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.47314 .45338 m .45487 .45976 L .44609 .44992 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46898 .67876 m .47314 .45338 L .45487 .45976 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.45487 .45976 m .44927 .68473 L .46898 .67876 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.42789 .45633 m .44609 .44992 L .45487 .45976 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43079 .65641 m .42129 .64716 L .42789 .45633 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.42918 .70808 m .4195 .699 L .42789 .45633 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.45013 .65041 m .43079 .65641 L .42129 .64716 L p
.658 .748 .916 r
F P
s
P
p
.002 w
.42129 .64716 m .44071 .64109 L .45013 .65041 L p
.658 .748 .916 r
F P
s
P
p
.002 w
.44609 .44992 m .45487 .45976 L .44927 .68473 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.44609 .44992 m .45487 .45976 L .45013 .65041 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.39188 .65414 m .40079 .45288 L .38279 .45927 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.38493 .61664 m .37531 .6073 L .38279 .45927 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.38279 .45927 m .37263 .66017 L .39188 .65414 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.40079 .45288 m .38279 .45927 L .37358 .44943 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39138 .66553 m .40079 .45288 L .38279 .45927 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.38279 .45927 m .37206 .67154 L .39138 .66553 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.35565 .45584 m .37358 .44943 L .38279 .45927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35607 .62159 m .34622 .61223 L .35565 .45584 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.37489 .61552 m .35607 .62159 L .34622 .61223 L p
.661 .744 .912 r
F P
s
P
p
.002 w
.34622 .61223 m .36512 .60609 L .37489 .61552 L p
.661 .744 .912 r
F P
s
P
p
.002 w
.37358 .44943 m .38279 .45927 L .37206 .67154 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.37358 .44943 m .38279 .45927 L .37489 .61552 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.80798 .45239 m .78845 .45878 L .7816 .44893 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83199 .69006 m .80798 .45239 L .78845 .45878 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.78845 .45878 m .8107 .69602 L .83199 .69006 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.76212 .45534 m .7816 .44893 L .78845 .45878 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78854 .68975 m .78117 .6806 L .76212 .45534 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.80956 .68382 m .78854 .68975 L .78117 .6806 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.78117 .6806 m .80229 .67459 L .80956 .68382 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.7816 .44893 m .78845 .45878 L .8107 .69602 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.7816 .44893 m .78845 .45878 L .80956 .68382 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.32467 .50555 m .3284 .45239 L .31066 .45878 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.31066 .45878 m .30662 .51185 L .32467 .50555 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.3284 .45239 m .31066 .45878 L .30103 .44893 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32445 .50873 m .3284 .45239 L .31066 .45878 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.31066 .45878 m .30638 .51503 L .32445 .50873 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.30103 .44893 m .31066 .45878 L .30638 .51503 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.75139 .66066 m .73569 .4519 L .71642 .45829 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.71642 .45829 m .73063 .66669 L .75139 .66066 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.73569 .4519 m .71642 .45829 L .70915 .44844 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75199 .6687 m .73569 .4519 L .71642 .45829 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.71642 .45829 m .73118 .6747 L .75199 .6687 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.68994 .45485 m .70915 .44844 L .71642 .45829 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70974 .66779 m .70192 .65857 L .68994 .45485 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.72231 .87305 m .71404 .86456 L .68994 .45485 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.7303 .66181 m .70974 .66779 L .70192 .65857 L p
.657 .75 .918 r
F P
s
P
p
.002 w
.70192 .65857 m .72258 .65251 L .7303 .66181 L p
.657 .75 .918 r
F P
s
P
p
.002 w
.70915 .44844 m .71642 .45829 L .73118 .6747 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.70915 .44844 m .71642 .45829 L .7303 .66181 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.68298 .84855 m .66335 .4514 L .64435 .4578 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.66977 .86265 m .66121 .85414 L .64435 .4578 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.66843 .8329 m .65994 .82427 L .64435 .4578 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.64435 .4578 m .66121 .85414 L .68298 .84855 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.66335 .4514 m .64435 .4578 L .63665 .44794 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67289 .64446 m .66335 .4514 L .64435 .4578 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.64435 .4578 m .65255 .65052 L .67289 .64446 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.61771 .45436 m .63665 .44794 L .64435 .4578 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63665 .44794 m .64435 .4578 L .65255 .65052 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.63665 .44794 m .64435 .4578 L .66062 .84029 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.59891 .78761 m .59097 .45091 L .57224 .45731 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.58624 .77117 m .57738 .76231 L .57224 .45731 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.57224 .45731 m .5779 .79335 L .59891 .78761 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.59097 .45091 m .57224 .45731 L .56412 .44744 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59929 .8034 m .59097 .45091 L .57224 .45731 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.57224 .45731 m .57817 .80911 L .59929 .8034 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.54545 .45387 m .56412 .44744 L .57224 .45731 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5569 .7822 m .54779 .77335 L .54545 .45387 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.57762 .77646 m .5569 .7822 L .54779 .77335 L p
.646 .763 .934 r
F P
s
P
p
.002 w
.54779 .77335 m .56861 .76754 L .57762 .77646 L p
.646 .763 .934 r
F P
s
P
p
.002 w
.56412 .44744 m .57224 .45731 L .57817 .80911 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.56412 .44744 m .57224 .45731 L .57762 .77646 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.51794 .71897 m .51855 .45041 L .50008 .45682 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.50716 .70169 m .498 .69259 L .50008 .45682 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.50008 .45682 m .49772 .72488 L .51794 .71897 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.51855 .45041 m .50008 .45682 L .49154 .44695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51791 .73241 m .51855 .45041 L .50008 .45682 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.50008 .45682 m .4976 .73829 L .51791 .73241 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.47314 .45338 m .49154 .44695 L .50008 .45682 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47797 .71016 m .46857 .70106 L .47314 .45338 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.4979 .70426 m .47797 .71016 L .46857 .70106 L p
.653 .755 .924 r
F P
s
P
p
.002 w
.46857 .70106 m .48859 .69508 L .4979 .70426 L p
.653 .755 .924 r
F P
s
P
p
.002 w
.49154 .44695 m .50008 .45682 L .4976 .73829 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.49154 .44695 m .50008 .45682 L .4979 .70426 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.44071 .64109 m .44609 .44992 L .42789 .45633 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.42789 .45633 m .42129 .64716 L .44071 .64109 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.44609 .44992 m .42789 .45633 L .41892 .44645 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44042 .65134 m .44609 .44992 L .42789 .45633 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.42789 .45633 m .42094 .65739 L .44042 .65134 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.40079 .45288 m .41892 .44645 L .42789 .45633 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41892 .44645 m .42789 .45633 L .42094 .65739 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.41892 .44645 m .42789 .45633 L .41905 .71224 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.36512 .60609 m .37358 .44943 L .35565 .45584 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.35913 .56771 m .34948 .5582 L .35565 .45584 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.35565 .45584 m .34622 .61223 L .36512 .60609 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.37358 .44943 m .35565 .45584 L .34626 .44595 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36462 .61531 m .37358 .44943 L .35565 .45584 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.35565 .45584 m .34566 .62143 L .36462 .61531 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.3284 .45239 m .34626 .44595 L .35565 .45584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33066 .57029 m .3208 .56076 L .3284 .45239 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.34912 .56411 m .33066 .57029 L .3208 .56076 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.3208 .56076 m .33933 .5545 L .34912 .56411 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.34626 .44595 m .35565 .45584 L .34566 .62143 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.34626 .44595 m .35565 .45584 L .34912 .56411 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.80229 .67459 m .7816 .44893 L .76212 .45534 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.78749 .67767 m .78014 .66848 L .76212 .45534 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.76212 .45534 m .78117 .6806 L .80229 .67459 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.7816 .44893 m .76212 .45534 L .7551 .44546 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80306 .68302 m .7816 .44893 L .76212 .45534 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.76212 .45534 m .78188 .68902 L .80306 .68302 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.73569 .4519 m .7551 .44546 L .76212 .45534 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75989 .68238 m .75233 .67318 L .73569 .4519 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.78081 .67641 m .75989 .68238 L .75233 .67318 L p
.656 .752 .921 r
F P
s
P
p
.002 w
.75233 .67318 m .77335 .66713 L .78081 .67641 L p
.656 .752 .921 r
F P
s
P
p
.002 w
.7551 .44546 m .76212 .45534 L .78188 .68902 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.7551 .44546 m .76212 .45534 L .78081 .67641 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.72258 .65251 m .70915 .44844 L .68994 .45485 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.68994 .45485 m .70192 .65857 L .72258 .65251 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.70915 .44844 m .68994 .45485 L .6825 .44496 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72312 .66064 m .70915 .44844 L .68994 .45485 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.68994 .45485 m .7024 .66668 L .72312 .66064 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.66335 .4514 m .6825 .44496 L .68994 .45485 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68116 .65936 m .67317 .65008 L .66335 .4514 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.69143 .85713 m .68298 .84855 L .66335 .4514 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.70162 .65334 m .68116 .65936 L .67317 .65008 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.67317 .65008 m .69372 .64398 L .70162 .65334 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.6825 .44496 m .68994 .45485 L .7024 .66668 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.6825 .44496 m .68994 .45485 L .70162 .65334 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.65198 .83161 m .63665 .44794 L .61771 .45436 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.63913 .84587 m .63039 .83726 L .61771 .45436 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.63804 .8152 m .62937 .80648 L .61771 .45436 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.61771 .45436 m .63039 .83726 L .65198 .83161 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.63665 .44794 m .61771 .45436 L .60985 .44446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64413 .63513 m .63665 .44794 L .61771 .45436 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.61771 .45436 m .6239 .64123 L .64413 .63513 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.59097 .45091 m .60985 .44446 L .61771 .45436 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60985 .44446 m .61771 .45436 L .6239 .64123 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.60985 .44446 m .61771 .45436 L .6299 .82241 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.56861 .76754 m .56412 .44744 L .54545 .45387 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.55657 .75029 m .54756 .74133 L .54545 .45387 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.54545 .45387 m .54779 .77335 L .56861 .76754 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.56412 .44744 m .54545 .45387 L .53716 .44396 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56883 .78313 m .56412 .44744 L .54545 .45387 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.54545 .45387 m .54791 .7889 L .56883 .78313 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.51855 .45041 m .53716 .44396 L .54545 .45387 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52714 .76098 m .51787 .75203 L .51855 .45041 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.54766 .75517 m .52714 .76098 L .51787 .75203 L p
.648 .761 .932 r
F P
s
P
p
.002 w
.51787 .75203 m .53849 .74614 L .54766 .75517 L p
.648 .761 .932 r
F P
s
P
p
.002 w
.53716 .44396 m .54545 .45387 L .54791 .7889 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.53716 .44396 m .54545 .45387 L .54766 .75517 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.48859 .69508 m .49154 .44695 L .47314 .45338 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.47848 .67694 m .46919 .66773 L .47314 .45338 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.47314 .45338 m .46857 .70106 L .48859 .69508 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.49154 .44695 m .47314 .45338 L .46442 .44347 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48843 .708 m .49154 .44695 L .47314 .45338 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.47314 .45338 m .46833 .71395 L .48843 .708 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.44609 .44992 m .46442 .44347 L .47314 .45338 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44927 .68473 m .43974 .67552 L .44609 .44992 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.46898 .67876 m .44927 .68473 L .43974 .67552 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.43974 .67552 m .45954 .66947 L .46898 .67876 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.46442 .44347 m .47314 .45338 L .46833 .71395 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.46442 .44347 m .47314 .45338 L .46898 .67876 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.4092 .70308 m .41892 .44645 L .40079 .45288 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.39938 .71813 m .38945 .70905 L .40079 .45288 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.40162 .66339 m .39188 .65414 L .40079 .45288 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.40079 .45288 m .38945 .70905 L .4092 .70308 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.41892 .44645 m .40079 .45288 L .39165 .44297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41228 .62175 m .41892 .44645 L .40079 .45288 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.40079 .45288 m .39304 .62788 L .41228 .62175 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.37358 .44943 m .39165 .44297 L .40079 .45288 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39165 .44297 m .40079 .45288 L .39304 .62788 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.39165 .44297 m .40079 .45288 L .39138 .66553 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.82767 .44595 m .80798 .45239 L .80122 .44247 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.85478 .69641 m .82767 .44595 L .80798 .45239 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.80798 .45239 m .83323 .70239 L .85478 .69641 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.7816 .44893 m .80122 .44247 L .80798 .45239 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.8107 .69602 m .80341 .68683 L .7816 .44893 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.83199 .69006 m .8107 .69602 L .80341 .68683 L p
.654 .753 .923 r
F P
s
P
p
.002 w
.80341 .68683 m .8248 .6808 L .83199 .69006 L p
.654 .753 .923 r
F P
s
P
p
.002 w
.80122 .44247 m .80798 .45239 L .83323 .70239 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.80122 .44247 m .80798 .45239 L .83199 .69006 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.33933 .5545 m .34626 .44595 L .3284 .45239 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.33432 .51522 m .32467 .50555 L .3284 .45239 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.3284 .45239 m .3208 .56076 L .33933 .5545 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.34626 .44595 m .3284 .45239 L .31883 .44247 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33891 .56114 m .34626 .44595 L .3284 .45239 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.3284 .45239 m .32034 .56739 L .33891 .56114 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.30103 .44893 m .31883 .44247 L .3284 .45239 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30638 .51503 m .29652 .50532 L .30103 .44893 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.32445 .50873 m .30638 .51503 L .29652 .50532 L p
.669 .734 .899 r
F P
s
P
p
.002 w
.29652 .50532 m .31467 .49895 L .32445 .50873 L p
.669 .734 .899 r
F P
s
P
p
.002 w
.31883 .44247 m .3284 .45239 L .32034 .56739 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.31883 .44247 m .3284 .45239 L .32445 .50873 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.77335 .66713 m .7551 .44546 L .73569 .4519 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.75892 .66991 m .75139 .66066 L .73569 .4519 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.73569 .4519 m .75233 .67318 L .77335 .66713 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.7551 .44546 m .73569 .4519 L .7285 .44197 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77406 .6757 m .7551 .44546 L .73569 .4519 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.73569 .4519 m .75297 .68173 L .77406 .6757 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.70915 .44844 m .7285 .44197 L .73569 .4519 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73118 .6747 m .72343 .66544 L .70915 .44844 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.75199 .6687 m .73118 .6747 L .72343 .66544 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.72343 .66544 m .74435 .65936 L .75199 .6687 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.7285 .44197 m .73569 .4519 L .75297 .68173 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.7285 .44197 m .73569 .4519 L .75199 .6687 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.69372 .64398 m .6825 .44496 L .66335 .4514 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.66335 .4514 m .67317 .65008 L .69372 .64398 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.6825 .44496 m .66335 .4514 L .65574 .44147 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69418 .6522 m .6825 .44496 L .66335 .4514 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.66335 .4514 m .67357 .65828 L .69418 .6522 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.63665 .44794 m .65574 .44147 L .66335 .4514 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65255 .65052 m .64437 .64118 L .63665 .44794 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.66062 .84029 m .65198 .83161 L .63665 .44794 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.67289 .64446 m .65255 .65052 L .64437 .64118 L p
.658 .748 .917 r
F P
s
P
p
.002 w
.64437 .64118 m .66481 .63505 L .67289 .64446 L p
.658 .748 .917 r
F P
s
P
p
.002 w
.65574 .44147 m .66335 .4514 L .67357 .65828 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.65574 .44147 m .66335 .4514 L .67289 .64446 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.62107 .81364 m .60985 .44446 L .59097 .45091 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.60859 .82806 m .59966 .81935 L .59097 .45091 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.60776 .79643 m .59891 .78761 L .59097 .45091 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.59097 .45091 m .59966 .81935 L .62107 .81364 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.60985 .44446 m .59097 .45091 L .58293 .44097 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61535 .6253 m .60985 .44446 L .59097 .45091 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.59097 .45091 m .59523 .63143 L .61535 .6253 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.56412 .44744 m .58293 .44097 L .59097 .45091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58293 .44097 m .59097 .45091 L .59523 .63143 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.58293 .44097 m .59097 .45091 L .59929 .8034 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.53849 .74614 m .53716 .44396 L .51855 .45041 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.52711 .72804 m .51794 .71897 L .51855 .45041 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.51855 .45041 m .51787 .75203 L .53849 .74614 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.53716 .44396 m .51855 .45041 L .51009 .44047 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53855 .76144 m .53716 .44396 L .51855 .45041 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.51855 .45041 m .51784 .76729 L .53855 .76144 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.49154 .44695 m .51009 .44047 L .51855 .45041 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4976 .73829 m .48818 .72922 L .49154 .44695 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.51791 .73241 m .4976 .73829 L .48818 .72922 L p
.65 .758 .929 r
F P
s
P
p
.002 w
.48818 .72922 m .50859 .72327 L .51791 .73241 L p
.65 .758 .929 r
F P
s
P
p
.002 w
.51009 .44047 m .51855 .45041 L .51784 .76729 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.51009 .44047 m .51855 .45041 L .51791 .73241 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.45954 .66947 m .46442 .44347 L .44609 .44992 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.45013 .65041 m .44071 .64109 L .44609 .44992 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.44609 .44992 m .43974 .67552 L .45954 .66947 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.46442 .44347 m .44609 .44992 L .4372 .43997 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45927 .68173 m .46442 .44347 L .44609 .44992 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.44609 .44992 m .4394 .68776 L .45927 .68173 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.41892 .44645 m .4372 .43997 L .44609 .44992 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42094 .65739 m .41129 .64806 L .41892 .44645 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.41905 .71224 m .4092 .70308 L .41892 .44645 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.44042 .65134 m .42094 .65739 L .41129 .64806 L p
.657 .749 .918 r
F P
s
P
p
.002 w
.41129 .64806 m .43085 .64193 L .44042 .65134 L p
.657 .749 .918 r
F P
s
P
p
.002 w
.4372 .43997 m .44609 .44992 L .4394 .68776 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4372 .43997 m .44609 .44992 L .44042 .65134 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.38148 .6562 m .39165 .44297 L .37358 .44943 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.37206 .67154 m .36208 .66228 L .37358 .44943 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.37489 .61552 m .36512 .60609 L .37358 .44943 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.37358 .44943 m .36208 .66228 L .38148 .6562 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.39165 .44297 m .37358 .44943 L .36426 .43947 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38465 .5897 m .39165 .44297 L .37358 .44943 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.37358 .44943 m .36567 .59591 L .38465 .5897 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.34626 .44595 m .36426 .43947 L .37358 .44943 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36426 .43947 m .37358 .44943 L .36567 .59591 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.36426 .43947 m .37358 .44943 L .36462 .61531 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.8248 .6808 m .80122 .44247 L .7816 .44893 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.80956 .68382 m .80229 .67459 L .7816 .44893 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.7816 .44893 m .80341 .68683 L .8248 .6808 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.80122 .44247 m .7816 .44893 L .77467 .43897 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82568 .68975 m .80122 .44247 L .7816 .44893 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.7816 .44893 m .80423 .69576 L .82568 .68975 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.7551 .44546 m .77467 .43897 L .7816 .44893 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78188 .68902 m .77439 .67978 L .7551 .44546 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.80306 .68302 m .78188 .68902 L .77439 .67978 L p
.654 .753 .922 r
F P
s
P
p
.002 w
.77439 .67978 m .79568 .67371 L .80306 .68302 L p
.654 .753 .922 r
F P
s
P
p
.002 w
.77467 .43897 m .7816 .44893 L .80423 .69576 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.77467 .43897 m .7816 .44893 L .80306 .68302 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.31467 .49895 m .31883 .44247 L .30103 .44893 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.30103 .44893 m .29652 .50532 L .31467 .49895 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.31883 .44247 m .30103 .44893 L .29129 .43897 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3144 .50255 m .31883 .44247 L .30103 .44893 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.30103 .44893 m .29624 .50892 L .3144 .50255 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.29129 .43897 m .30103 .44893 L .29624 .50892 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.74435 .65936 m .7285 .44197 L .70915 .44844 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.7303 .66181 m .72258 .65251 L .70915 .44844 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.70915 .44844 m .72343 .66544 L .74435 .65936 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.7285 .44197 m .70915 .44844 L .70179 .43847 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74498 .66807 m .7285 .44197 L .70915 .44844 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.70915 .44844 m .724 .67413 L .74498 .66807 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.6825 .44496 m .70179 .43847 L .70915 .44844 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7024 .66668 m .69447 .65737 L .6825 .44496 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.72312 .66064 m .7024 .66668 L .69447 .65737 L p
.656 .751 .919 r
F P
s
P
p
.002 w
.69447 .65737 m .71528 .65125 L .72312 .66064 L p
.656 .751 .919 r
F P
s
P
p
.002 w
.70179 .43847 m .70915 .44844 L .724 .67413 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.70179 .43847 m .70915 .44844 L .72312 .66064 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.66481 .63505 m .65574 .44147 L .63665 .44794 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.63665 .44794 m .64437 .64118 L .66481 .63505 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.65574 .44147 m .63665 .44794 L .62887 .43797 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6652 .64335 m .65574 .44147 L .63665 .44794 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.63665 .44794 m .6447 .64947 L .6652 .64335 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.60985 .44446 m .62887 .43797 L .63665 .44794 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6239 .64123 m .61554 .63183 L .60985 .44446 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.6299 .82241 m .62107 .81364 L .60985 .44446 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.64413 .63513 m .6239 .64123 L .61554 .63183 L p
.659 .748 .916 r
F P
s
P
p
.002 w
.61554 .63183 m .63587 .62566 L .64413 .63513 L p
.659 .748 .916 r
F P
s
P
p
.002 w
.62887 .43797 m .63665 .44794 L .6447 .64947 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.62887 .43797 m .63665 .44794 L .64413 .63513 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.59029 .79452 m .58293 .44097 L .56412 .44744 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.57817 .80911 m .56907 .80031 L .56412 .44744 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.57762 .77646 m .56861 .76754 L .56412 .44744 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.56412 .44744 m .56907 .80031 L .59029 .79452 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.58293 .44097 m .56412 .44744 L .55591 .43746 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58655 .6149 m .58293 .44097 L .56412 .44744 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.56412 .44744 m .56655 .62108 L .58655 .6149 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.53716 .44396 m .55591 .43746 L .56412 .44744 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55591 .43746 m .56412 .44744 L .56655 .62108 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.55591 .43746 m .56412 .44744 L .56883 .78313 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.50859 .72327 m .51009 .44047 L .49154 .44695 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.4979 .70426 m .48859 .69508 L .49154 .44695 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.49154 .44695 m .48818 .72922 L .50859 .72327 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.51009 .44047 m .49154 .44695 L .4829 .43696 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50851 .73818 m .51009 .44047 L .49154 .44695 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.49154 .44695 m .488 .74409 L .50851 .73818 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.46442 .44347 m .4829 .43696 L .49154 .44695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46833 .71395 m .45877 .70477 L .46442 .44347 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.48843 .708 m .46833 .71395 L .45877 .70477 L p
.652 .756 .926 r
F P
s
P
p
.002 w
.45877 .70477 m .47896 .69875 L .48843 .708 L p
.652 .756 .926 r
F P
s
P
p
.002 w
.4829 .43696 m .49154 .44695 L .488 .74409 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.4829 .43696 m .49154 .44695 L .48843 .708 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.43085 .64193 m .4372 .43997 L .41892 .44645 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.41892 .44645 m .41129 .64806 L .43085 .64193 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.4372 .43997 m .41892 .44645 L .40986 .43646 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43049 .65337 m .4372 .43997 L .41892 .44645 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.41892 .44645 m .41086 .65948 L .43049 .65337 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.39165 .44297 m .40986 .43646 L .41892 .44645 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39304 .62788 m .38328 .61843 L .39165 .44297 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.39138 .66553 m .38148 .6562 L .39165 .44297 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.41228 .62175 m .39304 .62788 L .38328 .61843 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.38328 .61843 m .4026 .61222 L .41228 .62175 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.40986 .43646 m .41892 .44645 L .41086 .65948 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.40986 .43646 m .41892 .44645 L .41228 .62175 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.8475 .43947 m .82767 .44595 L .82099 .43596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.87795 .70286 m .8475 .43947 L .82767 .44595 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.82767 .44595 m .85613 .70886 L .87795 .70286 L p
.573 .367 .563 r
F P
s
P
p
.002 w
.80122 .44247 m .82099 .43596 L .82767 .44595 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83323 .70239 m .82602 .69317 L .80122 .44247 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.85478 .69641 m .83323 .70239 L .82602 .69317 L p
.653 .755 .924 r
F P
s
P
p
.002 w
.82602 .69317 m .84768 .68711 L .85478 .69641 L p
.653 .755 .924 r
F P
s
P
p
.002 w
.82099 .43596 m .82767 .44595 L .85613 .70886 L p
.976 .674 .353 r
F P
s
P
p
.002 w
.82099 .43596 m .82767 .44595 L .85478 .69641 L p
.976 .674 .353 r
F P
s
P
p
.002 w
.35469 .60579 m .36426 .43947 L .34626 .44595 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.34566 .62143 m .33566 .61199 L .34626 .44595 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.34912 .56411 m .33933 .5545 L .34626 .44595 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.34626 .44595 m .33566 .61199 L .35469 .60579 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.36426 .43947 m .34626 .44595 L .33676 .43596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35762 .55485 m .36426 .43947 L .34626 .44595 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.34626 .44595 m .33891 .56114 L .35762 .55485 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.31883 .44247 m .33676 .43596 L .34626 .44595 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33676 .43596 m .34626 .44595 L .33891 .56114 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.33676 .43596 m .34626 .44595 L .33891 .56114 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.79568 .67371 m .77467 .43897 L .7551 .44546 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.78081 .67641 m .77335 .66713 L .7551 .44546 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.7551 .44546 m .77439 .67978 L .79568 .67371 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.77467 .43897 m .7551 .44546 L .748 .43545 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7965 .68283 m .77467 .43897 L .7551 .44546 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.7551 .44546 m .77514 .68887 L .7965 .68283 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.7285 .44197 m .748 .43545 L .7551 .44546 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75297 .68173 m .7453 .67244 L .7285 .44197 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.77406 .6757 m .75297 .68173 L .7453 .67244 L p
.655 .753 .922 r
F P
s
P
p
.002 w
.7453 .67244 m .76649 .66633 L .77406 .6757 L p
.655 .753 .922 r
F P
s
P
p
.002 w
.748 .43545 m .7551 .44546 L .77514 .68887 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.748 .43545 m .7551 .44546 L .77406 .6757 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.71528 .65125 m .70179 .43847 L .6825 .44496 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.70162 .65334 m .69372 .64398 L .6825 .44496 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.6825 .44496 m .69447 .65737 L .71528 .65125 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.70179 .43847 m .6825 .44496 L .67497 .43495 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71584 .66008 m .70179 .43847 L .6825 .44496 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.6825 .44496 m .69497 .66618 L .71584 .66008 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.65574 .44147 m .67497 .43495 L .6825 .44496 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67357 .65828 m .66546 .64891 L .65574 .44147 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.69418 .6522 m .67357 .65828 L .66546 .64891 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.66546 .64891 m .68616 .64276 L .69418 .6522 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.67497 .43495 m .6825 .44496 L .69497 .66618 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.67497 .43495 m .6825 .44496 L .69418 .6522 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.63587 .62566 m .62887 .43797 L .60985 .44446 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.60985 .44446 m .61554 .63183 L .63587 .62566 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.62887 .43797 m .60985 .44446 L .60189 .43445 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63618 .63403 m .62887 .43797 L .60985 .44446 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.60985 .44446 m .6158 .64018 L .63618 .63403 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.58293 .44097 m .60189 .43445 L .60985 .44446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59523 .63143 m .5867 .62197 L .58293 .44097 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.59929 .8034 m .59029 .79452 L .58293 .44097 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.61535 .6253 m .59523 .63143 L .5867 .62197 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.5867 .62197 m .6069 .61576 L .61535 .6253 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.60189 .43445 m .60985 .44446 L .6158 .64018 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.60189 .43445 m .60985 .44446 L .61535 .6253 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.55965 .77414 m .55591 .43746 L .53716 .44396 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.54791 .7889 m .53864 .78 L .53716 .44396 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.54766 .75517 m .53849 .74614 L .53716 .44396 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.53716 .44396 m .53864 .78 L .55965 .77414 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.55591 .43746 m .53716 .44396 L .52877 .43394 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55776 .60387 m .55591 .43746 L .53716 .44396 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.53716 .44396 m .53789 .61009 L .55776 .60387 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.51009 .44047 m .52877 .43394 L .53716 .44396 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52877 .43394 m .53716 .44396 L .53789 .61009 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.52877 .43394 m .53716 .44396 L .53855 .76144 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.47896 .69875 m .4829 .43696 L .46442 .44347 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.46898 .67876 m .45954 .66947 L .46442 .44347 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.46442 .44347 m .45877 .70477 L .47896 .69875 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.4829 .43696 m .46442 .44347 L .45561 .43344 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47875 .71313 m .4829 .43696 L .46442 .44347 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.46442 .44347 m .45846 .71913 L .47875 .71313 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.4372 .43997 m .45561 .43344 L .46442 .44347 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4394 .68776 m .4297 .67847 L .4372 .43997 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.45927 .68173 m .4394 .68776 L .4297 .67847 L p
.654 .753 .923 r
F P
s
P
p
.002 w
.4297 .67847 m .44967 .67236 L .45927 .68173 L p
.654 .753 .923 r
F P
s
P
p
.002 w
.45561 .43344 m .46442 .44347 L .45846 .71913 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.45561 .43344 m .46442 .44347 L .45927 .68173 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.4026 .61222 m .40986 .43646 L .39165 .44297 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.39165 .44297 m .38328 .61843 L .4026 .61222 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.40986 .43646 m .39165 .44297 L .3824 .43294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40217 .62264 m .40986 .43646 L .39165 .44297 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.39165 .44297 m .38279 .62882 L .40217 .62264 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.36426 .43947 m .3824 .43294 L .39165 .44297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36567 .59591 m .35581 .58633 L .36426 .43947 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.36462 .61531 m .35469 .60579 L .36426 .43947 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.38465 .5897 m .36567 .59591 L .35581 .58633 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.35581 .58633 m .37487 .58005 L .38465 .5897 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.3824 .43294 m .39165 .44297 L .38279 .62882 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3824 .43294 m .39165 .44297 L .38465 .5897 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.84768 .68711 m .82099 .43596 L .80122 .44247 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.83199 .69006 m .8248 .6808 L .80122 .44247 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.80122 .44247 m .82602 .69317 L .84768 .68711 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.82099 .43596 m .80122 .44247 L .79438 .43243 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84868 .69658 m .82099 .43596 L .80122 .44247 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.80122 .44247 m .82695 .70262 L .84868 .69658 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.77467 .43897 m .79438 .43243 L .80122 .44247 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80423 .69576 m .79682 .68649 L .77467 .43897 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.82568 .68975 m .80423 .69576 L .79682 .68649 L p
.653 .754 .924 r
F P
s
P
p
.002 w
.79682 .68649 m .81838 .6804 L .82568 .68975 L p
.653 .754 .924 r
F P
s
P
p
.002 w
.79438 .43243 m .80122 .44247 L .82695 .70262 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.79438 .43243 m .80122 .44247 L .82568 .68975 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.32897 .55144 m .33676 .43596 L .31883 .44247 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.32034 .56739 m .31033 .55776 L .31883 .44247 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.32445 .50873 m .31467 .49895 L .31883 .44247 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.31883 .44247 m .31033 .55776 L .32897 .55144 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.33676 .43596 m .31883 .44247 L .30915 .43243 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31033 .55776 m .32897 .55144 L .33891 .56114 L p
.665 .74 .906 r
F P
s
P
p
.002 w
.32846 .55897 m .33676 .43596 L .31883 .44247 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.33891 .56114 m .32034 .56739 L .31033 .55776 L p
.665 .74 .906 r
F P
s
P
p
.002 w
.31883 .44247 m .30977 .56528 L .32846 .55897 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.29129 .43897 m .30915 .43243 L .31883 .44247 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29624 .50892 m .28625 .49911 L .29129 .43897 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.3144 .50255 m .29624 .50892 L .28625 .49911 L p
.668 .734 .899 r
F P
s
P
p
.002 w
.28625 .49911 m .30449 .49266 L .3144 .50255 L p
.668 .734 .899 r
F P
s
P
p
.002 w
.30915 .43243 m .31883 .44247 L .30977 .56528 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.30915 .43243 m .31883 .44247 L .3144 .50255 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.76649 .66633 m .748 .43545 L .7285 .44197 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.75199 .6687 m .74435 .65936 L .7285 .44197 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.7285 .44197 m .7453 .67244 L .76649 .66633 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.748 .43545 m .7285 .44197 L .72123 .43193 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76723 .67562 m .748 .43545 L .7285 .44197 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.7285 .44197 m .74598 .6817 L .76723 .67562 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.70179 .43847 m .72123 .43193 L .7285 .44197 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.724 .67413 m .71614 .66478 L .70179 .43847 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.74498 .66807 m .724 .67413 L .71614 .66478 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.71614 .66478 m .73722 .65864 L .74498 .66807 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.72123 .43193 m .7285 .44197 L .74598 .6817 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.72123 .43193 m .7285 .44197 L .74498 .66807 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.68616 .64276 m .67497 .43495 L .65574 .44147 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.67289 .64446 m .66481 .63505 L .65574 .44147 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.65574 .44147 m .66546 .64891 L .68616 .64276 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.67497 .43495 m .65574 .44147 L .64804 .43142 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68664 .65171 m .67497 .43495 L .65574 .44147 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.65574 .44147 m .66588 .65785 L .68664 .65171 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.62887 .43797 m .64804 .43142 L .65574 .44147 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6447 .64947 m .6364 .64004 L .62887 .43797 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.6652 .64335 m .6447 .64947 L .6364 .64004 L p
.657 .749 .918 r
F P
s
P
p
.002 w
.6364 .64004 m .657 .63384 L .6652 .64335 L p
.657 .749 .918 r
F P
s
P
p
.002 w
.64804 .43142 m .65574 .44147 L .66588 .65785 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.64804 .43142 m .65574 .44147 L .6652 .64335 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.6069 .61576 m .60189 .43445 L .58293 .44097 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.58293 .44097 m .5867 .62197 L .6069 .61576 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.60189 .43445 m .58293 .44097 L .5748 .43092 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60714 .62418 m .60189 .43445 L .58293 .44097 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.58293 .44097 m .58687 .63038 L .60714 .62418 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.55591 .43746 m .5748 .43092 L .58293 .44097 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56655 .62108 m .55785 .61155 L .55591 .43746 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.56883 .78313 m .55965 .77414 L .55591 .43746 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.58655 .6149 m .56655 .62108 L .55785 .61155 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.55785 .61155 m .57793 .6053 L .58655 .6149 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.5748 .43092 m .58293 .44097 L .58687 .63038 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.5748 .43092 m .58293 .44097 L .58655 .6149 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.52922 .75235 m .52877 .43394 L .51009 .44047 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.51784 .76729 m .5084 .75827 L .51009 .44047 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.51791 .73241 m .50859 .72327 L .51009 .44047 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.51009 .44047 m .5084 .75827 L .52922 .75235 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.52877 .43394 m .51009 .44047 L .50152 .43041 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53855 .76144 m .51784 .76729 L .5084 .75827 L p
.646 .762 .934 r
F P
s
P
p
.002 w
.5084 .75827 m .52922 .75235 L .53855 .76144 L p
.646 .762 .934 r
F P
s
P
p
.002 w
.52924 .76934 m .52877 .43394 L .51009 .44047 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.51009 .44047 m .50831 .77522 L .52924 .76934 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.4829 .43696 m .50152 .43041 L .51009 .44047 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.488 .74409 m .47842 .73497 L .4829 .43696 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.50851 .73818 m .488 .74409 L .47842 .73497 L p
.648 .76 .931 r
F P
s
P
p
.002 w
.47842 .73497 m .49902 .72897 L .50851 .73818 L p
.648 .76 .931 r
F P
s
P
p
.002 w
.50152 .43041 m .51009 .44047 L .50831 .77522 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.50152 .43041 m .51009 .44047 L .50851 .73818 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.44967 .67236 m .45561 .43344 L .4372 .43997 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.44042 .65134 m .43085 .64193 L .4372 .43997 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4372 .43997 m .4297 .67847 L .44967 .67236 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.45561 .43344 m .4372 .43997 L .4282 .4299 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44932 .68608 m .45561 .43344 L .4372 .43997 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.4372 .43997 m .42927 .69215 L .44932 .68608 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.40986 .43646 m .4282 .4299 L .4372 .43997 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41086 .65948 m .40104 .65007 L .40986 .43646 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.43049 .65337 m .41086 .65948 L .40104 .65007 L p
.656 .751 .919 r
F P
s
P
p
.002 w
.40104 .65007 m .42076 .64388 L .43049 .65337 L p
.656 .751 .919 r
F P
s
P
p
.002 w
.4282 .4299 m .4372 .43997 L .42927 .69215 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4282 .4299 m .4372 .43997 L .43049 .65337 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.86748 .43294 m .8475 .43947 L .84092 .4294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.86748 .43294 m .8475 .43947 L .84092 .4294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82099 .43596 m .84092 .4294 L .8475 .43947 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84092 .4294 m .8475 .43947 L .87795 .70286 L p
.976 .671 .339 r
F P
s
P
p
.002 w
.37487 .58005 m .3824 .43294 L .36426 .43947 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.36426 .43947 m .35581 .58633 L .37487 .58005 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.3824 .43294 m .36426 .43947 L .35484 .4294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37441 .58918 m .3824 .43294 L .36426 .43947 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.36426 .43947 m .35529 .59546 L .37441 .58918 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.33676 .43596 m .35484 .4294 L .36426 .43947 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33891 .56114 m .32897 .55144 L .33676 .43596 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.33891 .56114 m .32897 .55144 L .33676 .43596 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.35762 .55485 m .33891 .56114 L .32897 .55144 L p
.665 .74 .906 r
F P
s
P
p
.002 w
.32897 .55144 m .34776 .54506 L .35762 .55485 L p
.665 .74 .906 r
F P
s
P
p
.002 w
.35484 .4294 m .36426 .43947 L .35529 .59546 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.35484 .4294 m .36426 .43947 L .35762 .55485 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.81838 .6804 m .79438 .43243 L .77467 .43897 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.80306 .68302 m .79568 .67371 L .77467 .43897 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.77467 .43897 m .79682 .68649 L .81838 .6804 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.79438 .43243 m .77467 .43897 L .76765 .42889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81932 .69007 m .79438 .43243 L .77467 .43897 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.77467 .43897 m .79769 .69614 L .81932 .69007 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.748 .43545 m .76765 .42889 L .77467 .43897 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77514 .68887 m .76755 .67955 L .748 .43545 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.7965 .68283 m .77514 .68887 L .76755 .67955 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.76755 .67955 m .78901 .67342 L .7965 .68283 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.76765 .42889 m .77467 .43897 L .79769 .69614 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.76765 .42889 m .77467 .43897 L .7965 .68283 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.30449 .49266 m .30915 .43243 L .29129 .43897 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.29129 .43897 m .28625 .49911 L .30449 .49266 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.30915 .43243 m .29129 .43897 L .28143 .42889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30417 .49677 m .30915 .43243 L .29129 .43897 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.29129 .43897 m .28591 .50321 L .30417 .49677 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.28143 .42889 m .29129 .43897 L .28591 .50321 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.73722 .65864 m .72123 .43193 L .70179 .43847 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.72312 .66064 m .71528 .65125 L .70179 .43847 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.70179 .43847 m .71614 .66478 L .73722 .65864 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.72123 .43193 m .70179 .43847 L .69434 .42838 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73789 .6681 m .72123 .43193 L .70179 .43847 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.70179 .43847 m .71674 .67422 L .73789 .6681 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.67497 .43495 m .69434 .42838 L .70179 .43847 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69497 .66618 m .68692 .65678 L .67497 .43495 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.71584 .66008 m .69497 .66618 L .68692 .65678 L p
.656 .752 .92 r
F P
s
P
p
.002 w
.68692 .65678 m .70789 .6506 L .71584 .66008 L p
.656 .752 .92 r
F P
s
P
p
.002 w
.69434 .42838 m .70179 .43847 L .71674 .67422 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.69434 .42838 m .70179 .43847 L .71584 .66008 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.657 .63384 m .64804 .43142 L .62887 .43797 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.64413 .63513 m .63587 .62566 L .62887 .43797 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.62887 .43797 m .6364 .64004 L .657 .63384 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.64804 .43142 m .62887 .43797 L .62099 .42788 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6574 .64292 m .64804 .43142 L .62887 .43797 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.62887 .43797 m .63674 .64909 L .6574 .64292 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.60189 .43445 m .62099 .42788 L .62887 .43797 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6158 .64018 m .60732 .63069 L .60189 .43445 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.63618 .63403 m .6158 .64018 L .60732 .63069 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.60732 .63069 m .62779 .62446 L .63618 .63403 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.62099 .42788 m .62887 .43797 L .63674 .64909 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.62099 .42788 m .62887 .43797 L .63618 .63403 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.57793 .6053 m .5748 .43092 L .55591 .43746 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.55591 .43746 m .55785 .61155 L .57793 .6053 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.5748 .43092 m .55591 .43746 L .5476 .42737 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57808 .61375 m .5748 .43092 L .55591 .43746 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.55591 .43746 m .55794 .61999 L .57808 .61375 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.52877 .43394 m .5476 .42737 L .55591 .43746 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53789 .61009 m .529 .6005 L .52877 .43394 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.53855 .76144 m .52922 .75235 L .52877 .43394 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.55776 .60387 m .53789 .61009 L .529 .6005 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.529 .6005 m .54896 .5942 L .55776 .60387 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.5476 .42737 m .55591 .43746 L .55794 .61999 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.5476 .42737 m .55591 .43746 L .55776 .60387 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.49902 .72897 m .50152 .43041 L .4829 .43696 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.48843 .708 m .47896 .69875 L .4829 .43696 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.4829 .43696 m .47842 .73497 L .49902 .72897 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.50152 .43041 m .4829 .43696 L .47416 .42686 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49888 .74559 m .50152 .43041 L .4829 .43696 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.4829 .43696 m .47817 .75155 L .49888 .74559 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.45561 .43344 m .47416 .42686 L .4829 .43696 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45846 .71913 m .44873 .70989 L .45561 .43344 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.47875 .71313 m .45846 .71913 L .44873 .70989 L p
.65 .758 .928 r
F P
s
P
p
.002 w
.44873 .70989 m .46911 .70381 L .47875 .71313 L p
.65 .758 .928 r
F P
s
P
p
.002 w
.47416 .42686 m .4829 .43696 L .47817 .75155 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.47416 .42686 m .4829 .43696 L .47875 .71313 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.42076 .64388 m .4282 .4299 L .40986 .43646 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.41228 .62175 m .4026 .61222 L .40986 .43646 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.40986 .43646 m .40104 .65007 L .42076 .64388 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.4282 .4299 m .40986 .43646 L .40068 .42635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42032 .65674 m .4282 .4299 L .40986 .43646 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.40986 .43646 m .40051 .6629 L .42032 .65674 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.3824 .43294 m .40068 .42635 L .40986 .43646 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38279 .62882 m .37287 .61928 L .3824 .43294 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.40217 .62264 m .38279 .62882 L .37287 .61928 L p
.659 .748 .916 r
F P
s
P
p
.002 w
.37287 .61928 m .39233 .61302 L .40217 .62264 L p
.659 .748 .916 r
F P
s
P
p
.002 w
.40068 .42635 m .40986 .43646 L .40051 .6629 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.40068 .42635 m .40986 .43646 L .40217 .62264 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.87094 .69353 m .84092 .4294 L .82099 .43596 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.85613 .70886 m .849 .69961 L .82099 .43596 L p
.976 .674 .353 r
F P
s
P
p
.002 w
.85478 .69641 m .84768 .68711 L .82099 .43596 L p
.976 .674 .353 r
F P
s
P
p
.002 w
.82099 .43596 m .849 .69961 L .87094 .69353 L p
.571 .364 .561 r
F P
s
P
p
.002 w
.84092 .4294 m .82099 .43596 L .81424 .42584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84092 .4294 m .82099 .43596 L .81424 .42584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79438 .43243 m .81424 .42584 L .82099 .43596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81424 .42584 m .82099 .43596 L .84868 .69658 L p
.976 .674 .353 r
F P
s
P
p
.002 w
.34776 .54506 m .35484 .4294 L .33676 .43596 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.33676 .43596 m .32897 .55144 L .34776 .54506 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.35484 .4294 m .33676 .43596 L .32716 .42584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3473 .55261 m .35484 .4294 L .33676 .43596 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.33676 .43596 m .32846 .55897 L .3473 .55261 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.30915 .43243 m .32716 .42584 L .33676 .43596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32716 .42584 m .33676 .43596 L .32846 .55897 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.32716 .42584 m .33676 .43596 L .32846 .55897 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.78901 .67342 m .76765 .42889 L .748 .43545 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.77406 .6757 m .76649 .66633 L .748 .43545 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.748 .43545 m .76755 .67955 L .78901 .67342 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.76765 .42889 m .748 .43545 L .74082 .42533 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78987 .68331 m .76765 .42889 L .748 .43545 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.748 .43545 m .76834 .68941 L .78987 .68331 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.72123 .43193 m .74082 .42533 L .748 .43545 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74598 .6817 m .73819 .67232 L .72123 .43193 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.76723 .67562 m .74598 .6817 L .73819 .67232 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.73819 .67232 m .75955 .66616 L .76723 .67562 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.74082 .42533 m .748 .43545 L .76834 .68941 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.74082 .42533 m .748 .43545 L .76723 .67562 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.70789 .6506 m .69434 .42838 L .67497 .43495 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.69418 .6522 m .68616 .64276 L .67497 .43495 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.67497 .43495 m .68692 .65678 L .70789 .6506 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.69434 .42838 m .67497 .43495 L .66735 .42482 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70848 .66023 m .69434 .42838 L .67497 .43495 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.67497 .43495 m .68744 .66639 L .70848 .66023 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.64804 .43142 m .66735 .42482 L .67497 .43495 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66588 .65785 m .65764 .64839 L .64804 .43142 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.68664 .65171 m .66588 .65785 L .65764 .64839 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.65764 .64839 m .6785 .64217 L .68664 .65171 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.66735 .42482 m .67497 .43495 L .68744 .66639 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.66735 .42482 m .67497 .43495 L .68664 .65171 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.62779 .62446 m .62099 .42788 L .60189 .43445 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.61535 .6253 m .6069 .61576 L .60189 .43445 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.60189 .43445 m .60732 .63069 L .62779 .62446 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.62099 .42788 m .60189 .43445 L .59384 .42432 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62811 .63364 m .62099 .42788 L .60189 .43445 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.60189 .43445 m .60757 .63985 L .62811 .63364 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.5748 .43092 m .59384 .42432 L .60189 .43445 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58687 .63038 m .57821 .62083 L .5748 .43092 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.60714 .62418 m .58687 .63038 L .57821 .62083 L p
.659 .748 .916 r
F P
s
P
p
.002 w
.57821 .62083 m .59857 .61455 L .60714 .62418 L p
.659 .748 .916 r
F P
s
P
p
.002 w
.59384 .42432 m .60189 .43445 L .60757 .63985 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.59384 .42432 m .60189 .43445 L .60714 .62418 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.54896 .5942 m .5476 .42737 L .52877 .43394 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.52877 .43394 m .529 .6005 L .54896 .5942 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.5476 .42737 m .52877 .43394 L .52028 .42381 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54903 .60265 m .5476 .42737 L .52877 .43394 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.52877 .43394 m .52902 .60893 L .54903 .60265 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.50152 .43041 m .52028 .42381 L .52877 .43394 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52028 .42381 m .52877 .43394 L .52902 .60893 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.52028 .42381 m .52877 .43394 L .52924 .76934 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.46911 .70381 m .47416 .42686 L .45561 .43344 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.45927 .68173 m .44967 .67236 L .45561 .43344 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.45561 .43344 m .44873 .70989 L .46911 .70381 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.47416 .42686 m .45561 .43344 L .44669 .4233 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46881 .71993 m .47416 .42686 L .45561 .43344 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.45561 .43344 m .44833 .72596 L .46881 .71993 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.4282 .4299 m .44669 .4233 L .45561 .43344 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42927 .69215 m .41941 .68279 L .4282 .4299 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.44932 .68608 m .42927 .69215 L .41941 .68279 L p
.653 .755 .925 r
F P
s
P
p
.002 w
.41941 .68279 m .43955 .67664 L .44932 .68608 L p
.653 .755 .925 r
F P
s
P
p
.002 w
.44669 .4233 m .45561 .43344 L .44833 .72596 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.44669 .4233 m .45561 .43344 L .44932 .68608 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.88762 .42635 m .86748 .43294 L .86099 .42279 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.88762 .42635 m .86748 .43294 L .86099 .42279 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84092 .4294 m .86099 .42279 L .86748 .43294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84092 .4294 m .86099 .42279 L .86748 .43294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39233 .61302 m .40068 .42635 L .3824 .43294 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.38465 .5897 m .37487 .58005 L .3824 .43294 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3824 .43294 m .37287 .61928 L .39233 .61302 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.40068 .42635 m .3824 .43294 L .37305 .42279 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39181 .62478 m .40068 .42635 L .3824 .43294 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.3824 .43294 m .37227 .63103 L .39181 .62478 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.35484 .4294 m .37305 .42279 L .3824 .43294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35529 .59546 m .34527 .58579 L .35484 .4294 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.37441 .58918 m .35529 .59546 L .34527 .58579 L p
.661 .744 .912 r
F P
s
P
p
.002 w
.34527 .58579 m .36447 .57944 L .37441 .58918 L p
.661 .744 .912 r
F P
s
P
p
.002 w
.37305 .42279 m .3824 .43294 L .37227 .63103 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.37305 .42279 m .3824 .43294 L .37441 .58918 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.84147 .6872 m .81424 .42584 L .79438 .43243 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.82695 .70262 m .81963 .69331 L .79438 .43243 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.82568 .68975 m .81838 .6804 L .79438 .43243 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.79438 .43243 m .81963 .69331 L .84147 .6872 L p
.57 .362 .559 r
F P
s
P
p
.002 w
.81424 .42584 m .79438 .43243 L .78745 .42227 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81424 .42584 m .79438 .43243 L .78745 .42227 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76765 .42889 m .78745 .42227 L .79438 .43243 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78745 .42227 m .79438 .43243 L .81932 .69007 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.31838 .54917 m .32716 .42584 L .30915 .43243 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.30977 .56528 m .29961 .55555 L .30915 .43243 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.3144 .50255 m .30449 .49266 L .30915 .43243 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.30915 .43243 m .29961 .55555 L .31838 .54917 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.32716 .42584 m .30915 .43243 L .29936 .42227 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32099 .51243 m .32716 .42584 L .30915 .43243 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.30915 .43243 m .30245 .51888 L .32099 .51243 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.28143 .42889 m .29936 .42227 L .30915 .43243 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28591 .50321 m .27579 .49328 L .28143 .42889 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.30417 .49677 m .28591 .50321 L .27579 .49328 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.27579 .49328 m .29412 .48676 L .30417 .49677 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.29936 .42227 m .30915 .43243 L .30245 .51888 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.29936 .42227 m .30915 .43243 L .30417 .49677 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.75955 .66616 m .74082 .42533 L .72123 .43193 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.74498 .66807 m .73722 .65864 L .72123 .43193 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.72123 .43193 m .73819 .67232 L .75955 .66616 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.74082 .42533 m .72123 .43193 L .71387 .42176 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76033 .67626 m .74082 .42533 L .72123 .43193 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.72123 .43193 m .7389 .6824 L .76033 .67626 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.69434 .42838 m .71387 .42176 L .72123 .43193 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71674 .67422 m .70876 .66478 L .69434 .42838 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.73789 .6681 m .71674 .67422 L .70876 .66478 L p
.654 .753 .922 r
F P
s
P
p
.002 w
.70876 .66478 m .73001 .65858 L .73789 .6681 L p
.654 .753 .922 r
F P
s
P
p
.002 w
.71387 .42176 m .72123 .43193 L .7389 .6824 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.71387 .42176 m .72123 .43193 L .73789 .6681 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.6785 .64217 m .66735 .42482 L .64804 .43142 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.6652 .64335 m .657 .63384 L .64804 .43142 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.64804 .43142 m .65764 .64839 L .6785 .64217 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.66735 .42482 m .64804 .43142 L .64024 .42125 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67901 .65197 m .66735 .42482 L .64804 .43142 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.64804 .43142 m .65807 .65817 L .67901 .65197 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.62099 .42788 m .64024 .42125 L .64804 .43142 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63674 .64909 m .62831 .63957 L .62099 .42788 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.6574 .64292 m .63674 .64909 L .62831 .63957 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.62831 .63957 m .64907 .63332 L .6574 .64292 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.64024 .42125 m .64804 .43142 L .65807 .65817 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.64024 .42125 m .64804 .43142 L .6574 .64292 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.59857 .61455 m .59384 .42432 L .5748 .43092 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.58655 .6149 m .57793 .6053 L .5748 .43092 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.5748 .43092 m .57821 .62083 L .59857 .61455 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.59384 .42432 m .5748 .43092 L .56657 .42074 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5988 .62382 m .59384 .42432 L .5748 .43092 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.5748 .43092 m .57837 .63008 L .5988 .62382 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.5476 .42737 m .56657 .42074 L .5748 .43092 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55794 .61999 m .5491 .61037 L .5476 .42737 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.57808 .61375 m .55794 .61999 L .5491 .61037 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.5491 .61037 m .56933 .60405 L .57808 .61375 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.56657 .42074 m .5748 .43092 L .57837 .63008 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.56657 .42074 m .5748 .43092 L .57808 .61375 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.51973 .76019 m .52028 .42381 L .50152 .43041 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.50831 .77522 m .4987 .76615 L .50152 .43041 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.50851 .73818 m .49902 .72897 L .50152 .43041 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.50152 .43041 m .4987 .76615 L .51973 .76019 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.52028 .42381 m .50152 .43041 L .49286 .42023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52001 .59079 m .52028 .42381 L .50152 .43041 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.50152 .43041 m .50012 .59712 L .52001 .59079 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.47416 .42686 m .49286 .42023 L .50152 .43041 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49286 .42023 m .50152 .43041 L .50012 .59712 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.49286 .42023 m .50152 .43041 L .49888 .74559 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.43955 .67664 m .44669 .4233 L .4282 .4299 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.43049 .65337 m .42076 .64388 L .4282 .4299 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4282 .4299 m .41941 .68279 L .43955 .67664 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.44669 .4233 m .4282 .4299 L .4191 .41972 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43911 .69208 m .44669 .4233 L .4282 .4299 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.4282 .4299 m .41888 .69819 L .43911 .69208 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.40068 .42635 m .4191 .41972 L .4282 .4299 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40051 .6629 m .39053 .6534 L .40068 .42635 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.42032 .65674 m .40051 .6629 L .39053 .6534 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.39053 .6534 m .41042 .64717 L .42032 .65674 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.4191 .41972 m .4282 .4299 L .41888 .69819 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4191 .41972 m .4282 .4299 L .42032 .65674 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.86099 .42279 m .84092 .4294 L .83425 .4192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.86099 .42279 m .84092 .4294 L .83425 .4192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.87795 .70286 m .87094 .69353 L .84092 .4294 L p
.976 .671 .339 r
F P
s
P
p
.002 w
.81424 .42584 m .83425 .4192 L .84092 .4294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81424 .42584 m .83425 .4192 L .84092 .4294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36447 .57944 m .37305 .42279 L .35484 .4294 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.35762 .55485 m .34776 .54506 L .35484 .4294 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.35484 .4294 m .34527 .58579 L .36447 .57944 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.37305 .42279 m .35484 .4294 L .3453 .4192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3639 .58982 m .37305 .42279 L .35484 .4294 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.35484 .4294 m .34464 .59615 L .3639 .58982 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.32716 .42584 m .3453 .4192 L .35484 .4294 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32846 .55897 m .31838 .54917 L .32716 .42584 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.32846 .55897 m .31838 .54917 L .32716 .42584 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.3473 .55261 m .32846 .55897 L .31838 .54917 L p
.664 .741 .907 r
F P
s
P
p
.002 w
.31838 .54917 m .33729 .54273 L .3473 .55261 L p
.664 .741 .907 r
F P
s
P
p
.002 w
.3453 .4192 m .35484 .4294 L .34464 .59615 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.3453 .4192 m .35484 .4294 L .3473 .55261 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.81191 .68063 m .78745 .42227 L .76765 .42889 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.79769 .69614 m .79017 .68678 L .76765 .42889 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.7965 .68283 m .78901 .67342 L .76765 .42889 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.76765 .42889 m .79017 .68678 L .81191 .68063 L p
.568 .359 .557 r
F P
s
P
p
.002 w
.78745 .42227 m .76765 .42889 L .76055 .41869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78745 .42227 m .76765 .42889 L .76055 .41869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74082 .42533 m .76055 .41869 L .76765 .42889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76055 .41869 m .76765 .42889 L .78987 .68331 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.29412 .48676 m .29936 .42227 L .28143 .42889 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.28143 .42889 m .27579 .49328 L .29412 .48676 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.29936 .42227 m .28143 .42889 L .27145 .41869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29564 .46803 m .29936 .42227 L .28143 .42889 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.28143 .42889 m .27743 .47458 L .29564 .46803 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.27145 .41869 m .28143 .42889 L .27743 .47458 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.73001 .65858 m .71387 .42176 L .69434 .42838 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.71584 .66008 m .70789 .6506 L .69434 .42838 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.69434 .42838 m .70876 .66478 L .73001 .65858 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.71387 .42176 m .69434 .42838 L .68681 .41818 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73071 .6689 m .71387 .42176 L .69434 .42838 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.69434 .42838 m .70938 .67508 L .73071 .6689 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.66735 .42482 m .68681 .41818 L .69434 .42838 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68744 .66639 m .67926 .65689 L .66735 .42482 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.70848 .66023 m .68744 .66639 L .67926 .65689 L p
.655 .753 .922 r
F P
s
P
p
.002 w
.67926 .65689 m .7004 .65066 L .70848 .66023 L p
.655 .753 .922 r
F P
s
P
p
.002 w
.68681 .41818 m .69434 .42838 L .70938 .67508 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.68681 .41818 m .69434 .42838 L .70848 .66023 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.64907 .63332 m .64024 .42125 L .62099 .42788 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.63618 .63403 m .62779 .62446 L .62099 .42788 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.62099 .42788 m .62831 .63957 L .64907 .63332 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.64024 .42125 m .62099 .42788 L .61302 .41766 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64948 .64328 m .64024 .42125 L .62099 .42788 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.62099 .42788 m .62866 .64951 L .64948 .64328 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.59384 .42432 m .61302 .41766 L .62099 .42788 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60757 .63985 m .59896 .63027 L .59384 .42432 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.62811 .63364 m .60757 .63985 L .59896 .63027 L p
.657 .75 .918 r
F P
s
P
p
.002 w
.59896 .63027 m .61959 .62398 L .62811 .63364 L p
.657 .75 .918 r
F P
s
P
p
.002 w
.61302 .41766 m .62099 .42788 L .62866 .64951 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.61302 .41766 m .62099 .42788 L .62811 .63364 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.56933 .60405 m .56657 .42074 L .5476 .42737 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.55776 .60387 m .54896 .5942 L .5476 .42737 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.5476 .42737 m .5491 .61037 L .56933 .60405 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.56657 .42074 m .5476 .42737 L .53919 .41715 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56947 .61339 m .56657 .42074 L .5476 .42737 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.5476 .42737 m .54917 .61969 L .56947 .61339 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.52028 .42381 m .53919 .41715 L .5476 .42737 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52902 .60893 m .51999 .59925 L .52028 .42381 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.52924 .76934 m .51973 .76019 L .52028 .42381 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.54903 .60265 m .52902 .60893 L .51999 .59925 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.51999 .59925 m .5401 .59289 L .54903 .60265 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.53919 .41715 m .5476 .42737 L .54917 .61969 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.53919 .41715 m .5476 .42737 L .54903 .60265 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.48921 .73633 m .49286 .42023 L .47416 .42686 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.47817 .75155 m .4684 .74236 L .47416 .42686 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.47875 .71313 m .46911 .70381 L .47416 .42686 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.47416 .42686 m .4684 .74236 L .48921 .73633 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.49286 .42023 m .47416 .42686 L .46532 .41664 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49103 .57807 m .49286 .42023 L .47416 .42686 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.47416 .42686 m .47128 .58444 L .49103 .57807 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.44669 .4233 m .46532 .41664 L .47416 .42686 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46532 .41664 m .47416 .42686 L .47128 .58444 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.46532 .41664 m .47416 .42686 L .46881 .71993 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.86099 .42279 m .88122 .41612 L .88762 .42635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.86099 .42279 m .88122 .41612 L .88762 .42635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41042 .64717 m .4191 .41972 L .40068 .42635 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.40217 .62264 m .39233 .61302 L .40068 .42635 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.40068 .42635 m .39053 .6534 L .41042 .64717 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.4191 .41972 m .40068 .42635 L .3914 .41612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40986 .66172 m .4191 .41972 L .40068 .42635 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.39138 .66553 m .37206 .67154 L .36208 .66228 L p
.656 .751 .919 r
F P
s
P
p
.002 w
.36208 .66228 m .38148 .6562 L .39138 .66553 L p
.656 .751 .919 r
F P
s
P
p
.002 w
.40068 .42635 m .38988 .66792 L .40986 .66172 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.37305 .42279 m .3914 .41612 L .40068 .42635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36462 .61531 m .34566 .62143 L .33566 .61199 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.33566 .61199 m .35469 .60579 L .36462 .61531 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.37227 .63103 m .36218 .6214 L .37305 .42279 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.39181 .62478 m .37227 .63103 L .36218 .6214 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.36218 .6214 m .3818 .61508 L .39181 .62478 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.3914 .41612 m .40068 .42635 L .38988 .66792 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.3914 .41612 m .40068 .42635 L .39181 .62478 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.83425 .4192 m .81424 .42584 L .80741 .41561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83425 .4192 m .81424 .42584 L .80741 .41561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84868 .69658 m .84147 .6872 L .81424 .42584 L p
.976 .674 .353 r
F P
s
P
p
.002 w
.78745 .42227 m .80741 .41561 L .81424 .42584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78745 .42227 m .80741 .41561 L .81424 .42584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33729 .54273 m .3453 .4192 L .32716 .42584 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.32716 .42584 m .31838 .54917 L .33729 .54273 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.3453 .4192 m .32716 .42584 L .31743 .41561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33673 .55135 m .3453 .4192 L .32716 .42584 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.32846 .55897 m .30977 .56528 L .29961 .55555 L p
.664 .741 .907 r
F P
s
P
p
.002 w
.29961 .55555 m .31838 .54917 L .32846 .55897 L p
.664 .741 .907 r
F P
s
P
p
.002 w
.32716 .42584 m .31776 .55778 L .33673 .55135 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.29936 .42227 m .31743 .41561 L .32716 .42584 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30245 .51888 m .29232 .50893 L .29936 .42227 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.32099 .51243 m .30245 .51888 L .29232 .50893 L p
.667 .737 .902 r
F P
s
P
p
.002 w
.29232 .50893 m .31093 .5024 L .32099 .51243 L p
.667 .737 .902 r
F P
s
P
p
.002 w
.31743 .41561 m .32716 .42584 L .31776 .55778 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.31743 .41561 m .32716 .42584 L .32099 .51243 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.78226 .67381 m .76055 .41869 L .74082 .42533 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.76834 .68941 m .76062 .68 L .74082 .42533 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.76723 .67562 m .75955 .66616 L .74082 .42533 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.74082 .42533 m .76062 .68 L .78226 .67381 L p
.566 .357 .555 r
F P
s
P
p
.002 w
.76055 .41869 m .74082 .42533 L .73354 .41509 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76055 .41869 m .74082 .42533 L .73354 .41509 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71387 .42176 m .73354 .41509 L .74082 .42533 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73354 .41509 m .74082 .42533 L .76033 .67626 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.7004 .65066 m .68681 .41818 L .66735 .42482 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.68664 .65171 m .6785 .64217 L .66735 .42482 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.66735 .42482 m .67926 .65689 L .7004 .65066 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.68681 .41818 m .66735 .42482 L .65964 .41458 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70102 .66119 m .68681 .41818 L .66735 .42482 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.66735 .42482 m .6798 .66741 L .70102 .66119 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.64024 .42125 m .65964 .41458 L .66735 .42482 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65807 .65817 m .6497 .64861 L .64024 .42125 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.67901 .65197 m .65807 .65817 L .6497 .64861 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.6497 .64861 m .67074 .64234 L .67901 .65197 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.65964 .41458 m .66735 .42482 L .6798 .66741 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.65964 .41458 m .66735 .42482 L .67901 .65197 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.61959 .62398 m .61302 .41766 L .59384 .42432 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.60714 .62418 m .59857 .61455 L .59384 .42432 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.59384 .42432 m .59896 .63027 L .61959 .62398 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.61302 .41766 m .59384 .42432 L .58569 .41406 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61991 .63409 m .61302 .41766 L .59384 .42432 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.59384 .42432 m .59921 .64036 L .61991 .63409 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.56657 .42074 m .58569 .41406 L .59384 .42432 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57837 .63008 m .56958 .62043 L .56657 .42074 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.5988 .62382 m .57837 .63008 L .56958 .62043 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.56958 .62043 m .59009 .6141 L .5988 .62382 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.58569 .41406 m .59384 .42432 L .59921 .64036 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.58569 .41406 m .59384 .42432 L .5988 .62382 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.5401 .59289 m .53919 .41715 L .52028 .42381 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.52028 .42381 m .51999 .59925 L .5401 .59289 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.53919 .41715 m .52028 .42381 L .5117 .41355 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54015 .60226 m .53919 .41715 L .52028 .42381 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.52028 .42381 m .51998 .6086 L .54015 .60226 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.49286 .42023 m .5117 .41355 L .52028 .42381 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50012 .59712 m .49093 .58737 L .49286 .42023 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.49888 .74559 m .48921 .73633 L .49286 .42023 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.52001 .59079 m .50012 .59712 L .49093 .58737 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.49093 .58737 m .5109 .58096 L .52001 .59079 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.5117 .41355 m .52028 .42381 L .51998 .6086 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.5117 .41355 m .52028 .42381 L .52001 .59079 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.45899 .71054 m .46532 .41664 L .44669 .4233 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.44833 .72596 m .43842 .71666 L .44669 .4233 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.44932 .68608 m .43955 .67664 L .44669 .4233 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.44669 .4233 m .43842 .71666 L .45899 .71054 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.46532 .41664 m .44669 .4233 L .43766 .41303 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46214 .56435 m .46532 .41664 L .44669 .4233 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.44669 .4233 m .44253 .57076 L .46214 .56435 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.4191 .41972 m .43766 .41303 L .44669 .4233 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43766 .41303 m .44669 .4233 L .44253 .57076 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.43766 .41303 m .44669 .4233 L .43911 .69208 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.88122 .41612 m .86099 .42279 L .85442 .41251 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.88122 .41612 m .86099 .42279 L .85442 .41251 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83425 .4192 m .85442 .41251 L .86099 .42279 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83425 .4192 m .85442 .41251 L .86099 .42279 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3818 .61508 m .3914 .41612 L .37305 .42279 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.37441 .58918 m .36447 .57944 L .37305 .42279 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.37305 .42279 m .36218 .6214 L .3818 .61508 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.3914 .41612 m .37305 .42279 L .36358 .41251 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38116 .62847 m .3914 .41612 L .37305 .42279 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.37305 .42279 m .36144 .63476 L .38116 .62847 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.3453 .4192 m .36358 .41251 L .37305 .42279 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34464 .59615 m .33446 .58639 L .3453 .4192 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.3639 .58982 m .34464 .59615 L .33446 .58639 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.33446 .58639 m .35381 .57998 L .3639 .58982 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.36358 .41251 m .37305 .42279 L .36144 .63476 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.36358 .41251 m .37305 .42279 L .3639 .58982 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.80741 .41561 m .78745 .42227 L .78044 .412 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80741 .41561 m .78745 .42227 L .78044 .412 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81932 .69007 m .81191 .68063 L .78745 .42227 L p
.976 .676 .366 r
F P
s
P
p
.002 w
.76055 .41869 m .78044 .412 L .78745 .42227 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76055 .41869 m .78044 .412 L .78745 .42227 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31093 .5024 m .31743 .41561 L .29936 .42227 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.30417 .49677 m .29412 .48676 L .29936 .42227 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.29936 .42227 m .29232 .50893 L .31093 .5024 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.31743 .41561 m .29936 .42227 L .28945 .412 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31045 .50881 m .31743 .41561 L .29936 .42227 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.29936 .42227 m .2918 .51533 L .31045 .50881 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.27145 .41869 m .28945 .412 L .29936 .42227 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27743 .47458 m .26726 .46449 L .27145 .41869 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.29564 .46803 m .27743 .47458 L .26726 .46449 L p
.669 .733 .897 r
F P
s
P
p
.002 w
.26726 .46449 m .28555 .45786 L .29564 .46803 L p
.669 .733 .897 r
F P
s
P
p
.002 w
.28945 .412 m .29936 .42227 L .2918 .51533 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.28945 .412 m .29936 .42227 L .29564 .46803 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.75252 .66671 m .73354 .41509 L .71387 .42176 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.7389 .6824 m .73099 .67293 L .71387 .42176 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.73789 .6681 m .73001 .65858 L .71387 .42176 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.71387 .42176 m .73099 .67293 L .75252 .66671 L p
.564 .354 .554 r
F P
s
P
p
.002 w
.73354 .41509 m .71387 .42176 L .70642 .41148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73354 .41509 m .71387 .42176 L .70642 .41148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68681 .41818 m .70642 .41148 L .71387 .42176 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70642 .41148 m .71387 .42176 L .73071 .6689 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.67074 .64234 m .65964 .41458 L .64024 .42125 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.6574 .64292 m .64907 .63332 L .64024 .42125 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.64024 .42125 m .6497 .64861 L .67074 .64234 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.65964 .41458 m .64024 .42125 L .63235 .41096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67126 .6531 m .65964 .41458 L .64024 .42125 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.64024 .42125 m .65015 .65935 L .67126 .6531 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.61302 .41766 m .63235 .41096 L .64024 .42125 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62866 .64951 m .6201 .6399 L .61302 .41766 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.64948 .64328 m .62866 .64951 L .6201 .6399 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.6201 .6399 m .64102 .63358 L .64948 .64328 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.63235 .41096 m .64024 .42125 L .65015 .65935 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.63235 .41096 m .64024 .42125 L .64948 .64328 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.59009 .6141 m .58569 .41406 L .56657 .42074 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.57808 .61375 m .56933 .60405 L .56657 .42074 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.56657 .42074 m .56958 .62043 L .59009 .6141 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.58569 .41406 m .56657 .42074 L .55824 .41045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59032 .62435 m .58569 .41406 L .56657 .42074 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.56657 .42074 m .56973 .63066 L .59032 .62435 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.53919 .41715 m .55824 .41045 L .56657 .42074 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54917 .61969 m .54019 .60998 L .53919 .41715 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.56947 .61339 m .54917 .61969 L .54019 .60998 L p
.658 .748 .916 r
F P
s
P
p
.002 w
.54019 .60998 m .56058 .6036 L .56947 .61339 L p
.658 .748 .916 r
F P
s
P
p
.002 w
.55824 .41045 m .56657 .42074 L .56973 .63066 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.55824 .41045 m .56657 .42074 L .56947 .61339 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.5109 .58096 m .5117 .41355 L .49286 .42023 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.49286 .42023 m .49093 .58737 L .5109 .58096 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.5117 .41355 m .49286 .42023 L .48409 .40993 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51085 .59034 m .5117 .41355 L .49286 .42023 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.49286 .42023 m .49082 .59672 L .51085 .59034 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.46532 .41664 m .48409 .40993 L .49286 .42023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47128 .58444 m .46192 .57462 L .46532 .41664 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.46881 .71993 m .45899 .71054 L .46532 .41664 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.49103 .57807 m .47128 .58444 L .46192 .57462 L p
.661 .744 .912 r
F P
s
P
p
.002 w
.46192 .57462 m .48175 .56817 L .49103 .57807 L p
.661 .744 .912 r
F P
s
P
p
.002 w
.48409 .40993 m .49286 .42023 L .49082 .59672 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.48409 .40993 m .49286 .42023 L .49103 .57807 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.42916 .68256 m .43766 .41303 L .4191 .41972 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.41888 .69819 m .40883 .68876 L .4191 .41972 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.42032 .65674 m .41042 .64717 L .4191 .41972 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.4191 .41972 m .40883 .68876 L .42916 .68256 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.43766 .41303 m .4191 .41972 L .40989 .40941 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43336 .54946 m .43766 .41303 L .4191 .41972 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.4191 .41972 m .4139 .55593 L .43336 .54946 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.3914 .41612 m .40989 .40941 L .4191 .41972 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40989 .40941 m .4191 .41972 L .4139 .55593 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.40989 .40941 m .4191 .41972 L .40986 .66172 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.85442 .41251 m .83425 .4192 L .82751 .40889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.85442 .41251 m .83425 .4192 L .82751 .40889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80741 .41561 m .82751 .40889 L .83425 .4192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80741 .41561 m .82751 .40889 L .83425 .4192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35381 .57998 m .36358 .41251 L .3453 .4192 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.3473 .55261 m .33729 .54273 L .3453 .4192 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.3453 .4192 m .33446 .58639 L .35381 .57998 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.36358 .41251 m .3453 .4192 L .33565 .40889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35311 .59186 m .36358 .41251 L .3453 .4192 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.3453 .4192 m .33369 .59825 L .35311 .59186 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.31743 .41561 m .33565 .40889 L .3453 .4192 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31776 .55778 m .30752 .54787 L .31743 .41561 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.33673 .55135 m .31776 .55778 L .30752 .54787 L p
.663 .742 .908 r
F P
s
P
p
.002 w
.30752 .54787 m .32656 .54137 L .33673 .55135 L p
.663 .742 .908 r
F P
s
P
p
.002 w
.33565 .40889 m .3453 .4192 L .33369 .59825 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.33565 .40889 m .3453 .4192 L .33673 .55135 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.78044 .412 m .76055 .41869 L .75337 .40837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78044 .412 m .76055 .41869 L .75337 .40837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78987 .68331 m .78226 .67381 L .76055 .41869 L p
.976 .679 .378 r
F P
s
P
p
.002 w
.73354 .41509 m .75337 .40837 L .76055 .41869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73354 .41509 m .75337 .40837 L .76055 .41869 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28555 .45786 m .28945 .412 L .27145 .41869 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.27145 .41869 m .26726 .46449 L .28555 .45786 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.28945 .412 m .27145 .41869 L .26136 .40837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28524 .46146 m .28945 .412 L .27145 .41869 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.27145 .41869 m .26693 .46807 L .28524 .46146 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.26136 .40837 m .27145 .41869 L .26693 .46807 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.72271 .6593 m .70642 .41148 L .68681 .41818 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.70938 .67508 m .70128 .66555 L .68681 .41818 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.70848 .66023 m .7004 .65066 L .68681 .41818 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.68681 .41818 m .70128 .66555 L .72271 .6593 L p
.563 .351 .552 r
F P
s
P
p
.002 w
.70642 .41148 m .68681 .41818 L .67919 .40785 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70642 .41148 m .68681 .41818 L .67919 .40785 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65964 .41458 m .67919 .40785 L .68681 .41818 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67919 .40785 m .68681 .41818 L .70102 .66119 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.64102 .63358 m .63235 .41096 L .61302 .41766 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.62811 .63364 m .61959 .62398 L .61302 .41766 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.61302 .41766 m .6201 .6399 L .64102 .63358 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.63235 .41096 m .61302 .41766 L .60496 .40733 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64144 .64456 m .63235 .41096 L .61302 .41766 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.61302 .41766 m .62045 .65085 L .64144 .64456 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.58569 .41406 m .60496 .40733 L .61302 .41766 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59921 .64036 m .59045 .63069 L .58569 .41406 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.61991 .63409 m .59921 .64036 L .59045 .63069 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.59045 .63069 m .61126 .62433 L .61991 .63409 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.60496 .40733 m .61302 .41766 L .62045 .65085 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.60496 .40733 m .61302 .41766 L .61991 .63409 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.56058 .6036 m .55824 .41045 L .53919 .41715 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.54903 .60265 m .5401 .59289 L .53919 .41715 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.53919 .41715 m .54019 .60998 L .56058 .6036 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.55824 .41045 m .53919 .41715 L .53068 .40681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5607 .61397 m .55824 .41045 L .53919 .41715 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.53919 .41715 m .54024 .62033 L .5607 .61397 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.5117 .41355 m .53068 .40681 L .53919 .41715 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51998 .6086 m .51081 .59883 L .5117 .41355 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.54015 .60226 m .51998 .6086 L .51081 .59883 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.51081 .59883 m .53107 .59241 L .54015 .60226 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.53068 .40681 m .53919 .41715 L .54024 .62033 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.53068 .40681 m .53919 .41715 L .54015 .60226 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.48175 .56817 m .48409 .40993 L .46532 .41664 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.46532 .41664 m .46192 .57462 L .48175 .56817 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.48409 .40993 m .46532 .41664 L .45636 .40629 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48161 .5775 m .48409 .40993 L .46532 .41664 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.46532 .41664 m .46172 .58393 L .48161 .5775 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.43766 .41303 m .45636 .40629 L .46532 .41664 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44253 .57076 m .433 .56087 L .43766 .41303 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.43911 .69208 m .42916 .68256 L .43766 .41303 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.46214 .56435 m .44253 .57076 L .433 .56087 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.433 .56087 m .45269 .55437 L .46214 .56435 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.45636 .40629 m .46532 .41664 L .46172 .58393 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.45636 .40629 m .46532 .41664 L .46214 .56435 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.85442 .41251 m .87475 .40577 L .88122 .41612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.85442 .41251 m .87475 .40577 L .88122 .41612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39978 .65207 m .40989 .40941 L .3914 .41612 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.38988 .66792 m .37971 .65836 L .3914 .41612 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.39181 .62478 m .3818 .61508 L .3914 .41612 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.3914 .41612 m .37971 .65836 L .39978 .65207 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.40989 .40941 m .3914 .41612 L .382 .40577 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40473 .53322 m .40989 .40941 L .3914 .41612 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.3914 .41612 m .38543 .53974 L .40473 .53322 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.36358 .41251 m .382 .40577 L .3914 .41612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.382 .40577 m .3914 .41612 L .38543 .53974 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.382 .40577 m .3914 .41612 L .38116 .62847 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.82751 .40889 m .80741 .41561 L .80049 .40525 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82751 .40889 m .80741 .41561 L .80049 .40525 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78044 .412 m .80049 .40525 L .80741 .41561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78044 .412 m .80049 .40525 L .80741 .41561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32656 .54137 m .33565 .40889 L .31743 .41561 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.32099 .51243 m .31093 .5024 L .31743 .41561 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.31743 .41561 m .30752 .54787 L .32656 .54137 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.33565 .40889 m .31743 .41561 L .3076 .40525 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32588 .55132 m .33565 .40889 L .31743 .41561 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.31743 .41561 m .30677 .55781 L .32588 .55132 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.28945 .412 m .3076 .40525 L .31743 .41561 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2918 .51533 m .28151 .50528 L .28945 .412 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.31045 .50881 m .2918 .51533 L .28151 .50528 L p
.666 .737 .903 r
F P
s
P
p
.002 w
.28151 .50528 m .30023 .49868 L .31045 .50881 L p
.666 .737 .903 r
F P
s
P
p
.002 w
.3076 .40525 m .31743 .41561 L .30677 .55781 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.3076 .40525 m .31743 .41561 L .31045 .50881 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.75337 .40837 m .73354 .41509 L .72619 .40473 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75337 .40837 m .73354 .41509 L .72619 .40473 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76033 .67626 m .75252 .66671 L .73354 .41509 L p
.976 .681 .389 r
F P
s
P
p
.002 w
.70642 .41148 m .72619 .40473 L .73354 .41509 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70642 .41148 m .72619 .40473 L .73354 .41509 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69282 .65153 m .67919 .40785 L .65964 .41458 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.6798 .66741 m .67149 .65783 L .65964 .41458 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.67901 .65197 m .67074 .64234 L .65964 .41458 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.65964 .41458 m .67149 .65783 L .69282 .65153 L p
.561 .349 .55 r
F P
s
P
p
.002 w
.67919 .40785 m .65964 .41458 L .65184 .40421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67919 .40785 m .65964 .41458 L .65184 .40421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63235 .41096 m .65184 .40421 L .65964 .41458 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65184 .40421 m .65964 .41458 L .67126 .6531 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.61126 .62433 m .60496 .40733 L .58569 .41406 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.5988 .62382 m .59009 .6141 L .58569 .41406 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.58569 .41406 m .59045 .63069 L .61126 .62433 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.60496 .40733 m .58569 .41406 L .57744 .40369 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61158 .63552 m .60496 .40733 L .58569 .41406 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.58569 .41406 m .5907 .64185 L .61158 .63552 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.55824 .41045 m .57744 .40369 L .58569 .41406 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56973 .63066 m .56079 .62092 L .55824 .41045 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.59032 .62435 m .56973 .63066 L .56079 .62092 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.56079 .62092 m .58147 .61453 L .59032 .62435 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.57744 .40369 m .58569 .41406 L .5907 .64185 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.57744 .40369 m .58569 .41406 L .59032 .62435 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.53107 .59241 m .53068 .40681 L .5117 .41355 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.52001 .59079 m .5109 .58096 L .5117 .41355 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.5117 .41355 m .51081 .59883 L .53107 .59241 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.53068 .40681 m .5117 .41355 L .50301 .40317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5311 .60287 m .53068 .40681 L .5117 .41355 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.5117 .41355 m .51076 .60927 L .5311 .60287 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.48409 .40993 m .50301 .40317 L .5117 .41355 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49082 .59672 m .48148 .58688 L .48409 .40993 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.51085 .59034 m .49082 .59672 L .48148 .58688 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.48148 .58688 m .5016 .58041 L .51085 .59034 L p
.66 .746 .914 r
F P
s
P
p
.002 w
.50301 .40317 m .5117 .41355 L .51076 .60927 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.50301 .40317 m .5117 .41355 L .51085 .59034 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.45269 .55437 m .45636 .40629 L .43766 .41303 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.43766 .41303 m .433 .56087 L .45269 .55437 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.45636 .40629 m .43766 .41303 L .42853 .40264 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45246 .56358 m .45636 .40629 L .43766 .41303 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.43766 .41303 m .43271 .57006 L .45246 .56358 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.40989 .40941 m .42853 .40264 L .43766 .41303 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4139 .55593 m .4042 .54596 L .40989 .40941 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.40986 .66172 m .39978 .65207 L .40989 .40941 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.43336 .54946 m .4139 .55593 L .4042 .54596 L p
.663 .742 .909 r
F P
s
P
p
.002 w
.4042 .54596 m .42374 .53942 L .43336 .54946 L p
.663 .742 .909 r
F P
s
P
p
.002 w
.42853 .40264 m .43766 .41303 L .43271 .57006 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.42853 .40264 m .43766 .41303 L .43336 .54946 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.87475 .40577 m .85442 .41251 L .84778 .40212 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.87475 .40577 m .85442 .41251 L .84778 .40212 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82751 .40889 m .84778 .40212 L .85442 .41251 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82751 .40889 m .84778 .40212 L .85442 .41251 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37097 .61869 m .382 .40577 L .36358 .41251 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.36144 .63476 m .35118 .62506 L .36358 .41251 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3639 .58982 m .35381 .57998 L .36358 .41251 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.36358 .41251 m .35118 .62506 L .37097 .61869 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.382 .40577 m .36358 .41251 L .354 .40212 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38116 .62847 m .36144 .63476 L .35118 .62506 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.35118 .62506 m .37097 .61869 L .38116 .62847 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.37017 .63405 m .382 .40577 L .36358 .41251 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.36358 .41251 m .35028 .6404 L .37017 .63405 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.33565 .40889 m .354 .40212 L .36358 .41251 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33369 .59825 m .32334 .5884 L .33565 .40889 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.35311 .59186 m .33369 .59825 L .32334 .5884 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.32334 .5884 m .34284 .58193 L .35311 .59186 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.354 .40212 m .36358 .41251 L .35028 .6404 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.354 .40212 m .36358 .41251 L .35311 .59186 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.80049 .40525 m .78044 .412 L .77335 .4016 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80049 .40525 m .78044 .412 L .77335 .4016 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75337 .40837 m .77335 .4016 L .78044 .412 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75337 .40837 m .77335 .4016 L .78044 .412 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30023 .49868 m .3076 .40525 L .28945 .412 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.29564 .46803 m .28555 .45786 L .28945 .412 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.28945 .412 m .28151 .50528 L .30023 .49868 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.3076 .40525 m .28945 .412 L .27943 .4016 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29964 .50613 m .3076 .40525 L .28945 .412 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.28945 .412 m .28087 .51272 L .29964 .50613 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.26136 .40837 m .27943 .4016 L .28945 .412 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26693 .46807 m .25663 .45787 L .26136 .40837 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.28524 .46146 m .26693 .46807 L .25663 .45787 L p
.669 .733 .898 r
F P
s
P
p
.002 w
.25663 .45787 m .27501 .45117 L .28524 .46146 L p
.669 .733 .898 r
F P
s
P
p
.002 w
.27943 .4016 m .28945 .412 L .28087 .51272 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.27943 .4016 m .28945 .412 L .28524 .46146 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.72619 .40473 m .70642 .41148 L .69889 .40107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72619 .40473 m .70642 .41148 L .69889 .40107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73071 .6689 m .72271 .6593 L .70642 .41148 L p
.975 .682 .4 r
F P
s
P
p
.002 w
.67919 .40785 m .69889 .40107 L .70642 .41148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67919 .40785 m .69889 .40107 L .70642 .41148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66286 .64337 m .65184 .40421 L .63235 .41096 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.65015 .65935 m .64165 .64971 L .63235 .41096 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.64948 .64328 m .64102 .63358 L .63235 .41096 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.63235 .41096 m .64165 .64971 L .66286 .64337 L p
.559 .346 .548 r
F P
s
P
p
.002 w
.65184 .40421 m .63235 .41096 L .62437 .40055 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65184 .40421 m .63235 .41096 L .62437 .40055 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60496 .40733 m .62437 .40055 L .63235 .41096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62437 .40055 m .63235 .41096 L .64144 .64456 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.58147 .61453 m .57744 .40369 L .55824 .41045 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.56947 .61339 m .56058 .6036 L .55824 .41045 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.55824 .41045 m .56079 .62092 L .58147 .61453 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.57744 .40369 m .55824 .41045 L .54982 .40003 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58169 .62592 m .57744 .40369 L .55824 .41045 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.55824 .41045 m .56092 .63229 L .58169 .62592 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.53068 .40681 m .54982 .40003 L .55824 .41045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54024 .62033 m .53111 .61053 L .53068 .40681 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.5607 .61397 m .54024 .62033 L .53111 .61053 L p
.657 .749 .918 r
F P
s
P
p
.002 w
.53111 .61053 m .55167 .60409 L .5607 .61397 L p
.657 .749 .918 r
F P
s
P
p
.002 w
.54982 .40003 m .55824 .41045 L .56092 .63229 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.54982 .40003 m .55824 .41045 L .5607 .61397 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.5016 .58041 m .50301 .40317 L .48409 .40993 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.49103 .57807 m .48175 .56817 L .48409 .40993 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.48409 .40993 m .48148 .58688 L .5016 .58041 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.50301 .40317 m .48409 .40993 L .47522 .3995 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50152 .59093 m .50301 .40317 L .48409 .40993 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.48409 .40993 m .48132 .59738 L .50152 .59093 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.45636 .40629 m .47522 .3995 L .48409 .40993 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46172 .58393 m .4522 .57401 L .45636 .40629 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.48161 .5775 m .46172 .58393 L .4522 .57401 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.4522 .57401 m .47218 .5675 L .48161 .5775 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.47522 .3995 m .48409 .40993 L .48132 .59738 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.47522 .3995 m .48409 .40993 L .48161 .5775 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.42374 .53942 m .42853 .40264 L .40989 .40941 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.40989 .40941 m .4042 .54596 L .42374 .53942 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.42853 .40264 m .40989 .40941 L .40057 .39898 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42343 .54842 m .42853 .40264 L .40989 .40941 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.40989 .40941 m .40383 .55495 L .42343 .54842 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.382 .40577 m .40057 .39898 L .40989 .40941 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38543 .53974 m .37558 .52969 L .382 .40577 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.38116 .62847 m .37097 .61869 L .382 .40577 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.40473 .53322 m .38543 .53974 L .37558 .52969 L p
.664 .741 .907 r
F P
s
P
p
.002 w
.37558 .52969 m .39496 .5231 L .40473 .53322 L p
.664 .741 .907 r
F P
s
P
p
.002 w
.40057 .39898 m .40989 .40941 L .40383 .55495 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.40057 .39898 m .40989 .40941 L .40473 .53322 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.84778 .40212 m .82751 .40889 L .82069 .39845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84778 .40212 m .82751 .40889 L .82069 .39845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80049 .40525 m .82069 .39845 L .82751 .40889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80049 .40525 m .82069 .39845 L .82751 .40889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34284 .58193 m .354 .40212 L .33565 .40889 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.33673 .55135 m .32656 .54137 L .33565 .40889 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.33565 .40889 m .32334 .5884 L .34284 .58193 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.354 .40212 m .33565 .40889 L .32588 .39845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34199 .59568 m .354 .40212 L .33565 .40889 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.33565 .40889 m .3224 .60212 L .34199 .59568 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.3076 .40525 m .32588 .39845 L .33565 .40889 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30677 .55781 m .29636 .54781 L .3076 .40525 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.32588 .55132 m .30677 .55781 L .29636 .54781 L p
.663 .743 .91 r
F P
s
P
p
.002 w
.29636 .54781 m .31554 .54124 L .32588 .55132 L p
.663 .743 .91 r
F P
s
P
p
.002 w
.32588 .39845 m .33565 .40889 L .3224 .60212 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.32588 .39845 m .33565 .40889 L .32588 .55132 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.77335 .4016 m .75337 .40837 L .7461 .39793 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77335 .4016 m .75337 .40837 L .7461 .39793 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72619 .40473 m .7461 .39793 L .75337 .40837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72619 .40473 m .7461 .39793 L .75337 .40837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27501 .45117 m .27943 .4016 L .26136 .40837 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.26136 .40837 m .25663 .45787 L .27501 .45117 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.27943 .4016 m .26136 .40837 L .25114 .39793 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27463 .45538 m .27943 .4016 L .26136 .40837 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.26136 .40837 m .25623 .46207 L .27463 .45538 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.25114 .39793 m .26136 .40837 L .25623 .46207 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.69889 .40107 m .67919 .40785 L .67147 .3974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69889 .40107 m .67919 .40785 L .67147 .3974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70102 .66119 m .69282 .65153 L .67919 .40785 L p
.974 .684 .411 r
F P
s
P
p
.002 w
.65184 .40421 m .67147 .3974 L .67919 .40785 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65184 .40421 m .67147 .3974 L .67919 .40785 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63284 .63478 m .62437 .40055 L .60496 .40733 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.62045 .65085 m .61175 .64115 L .60496 .40733 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.61991 .63409 m .61126 .62433 L .60496 .40733 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.60496 .40733 m .61175 .64115 L .63284 .63478 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.62437 .40055 m .60496 .40733 L .59679 .39688 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64144 .64456 m .62045 .65085 L .61175 .64115 L p
.655 .753 .922 r
F P
s
P
p
.002 w
.61175 .64115 m .63284 .63478 L .64144 .64456 L p
.655 .753 .922 r
F P
s
P
p
.002 w
.63328 .64693 m .62437 .40055 L .60496 .40733 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.60496 .40733 m .6121 .65328 L .63328 .64693 L p
.557 .343 .545 r
F P
s
P
p
.002 w
.57744 .40369 m .59679 .39688 L .60496 .40733 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5907 .64185 m .5818 .63209 L .57744 .40369 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.61158 .63552 m .5907 .64185 L .5818 .63209 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.5818 .63209 m .60278 .62568 L .61158 .63552 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.59679 .39688 m .60496 .40733 L .6121 .65328 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.59679 .39688 m .60496 .40733 L .61158 .63552 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.55167 .60409 m .54982 .40003 L .53068 .40681 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.54015 .60226 m .53107 .59241 L .53068 .40681 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.53068 .40681 m .53111 .61053 L .55167 .60409 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.54982 .40003 m .53068 .40681 L .52207 .39635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55177 .61567 m .54982 .40003 L .53068 .40681 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.53068 .40681 m .53114 .62208 L .55177 .61567 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.50301 .40317 m .52207 .39635 L .53068 .40681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51076 .60927 m .50145 .5994 L .50301 .40317 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.5311 .60287 m .51076 .60927 L .50145 .5994 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.50145 .5994 m .52187 .59292 L .5311 .60287 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.52207 .39635 m .53068 .40681 L .53114 .62208 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.52207 .39635 m .53068 .40681 L .5311 .60287 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.47218 .5675 m .47522 .3995 L .45636 .40629 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.46214 .56435 m .45269 .55437 L .45636 .40629 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.45636 .40629 m .4522 .57401 L .47218 .5675 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.47522 .3995 m .45636 .40629 L .44731 .39582 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47199 .57802 m .47522 .3995 L .45636 .40629 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.45636 .40629 m .45194 .58451 L .47199 .57802 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.42853 .40264 m .44731 .39582 L .45636 .40629 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43271 .57006 m .42302 .56007 L .42853 .40264 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.45246 .56358 m .43271 .57006 L .42302 .56007 L p
.661 .744 .912 r
F P
s
P
p
.002 w
.42302 .56007 m .44285 .55351 L .45246 .56358 L p
.661 .744 .912 r
F P
s
P
p
.002 w
.44731 .39582 m .45636 .40629 L .45194 .58451 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.44731 .39582 m .45636 .40629 L .45246 .56358 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.84778 .40212 m .8682 .3953 L .87475 .40577 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84778 .40212 m .8682 .3953 L .87475 .40577 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39496 .5231 m .40057 .39898 L .382 .40577 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.382 .40577 m .37558 .52969 L .39496 .5231 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.40057 .39898 m .382 .40577 L .3725 .3953 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39457 .53177 m .40057 .39898 L .382 .40577 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.382 .40577 m .37513 .53835 L .39457 .53177 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.354 .40212 m .3725 .3953 L .382 .40577 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3725 .3953 m .382 .40577 L .37513 .53835 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.3725 .3953 m .382 .40577 L .37017 .63405 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.82069 .39845 m .80049 .40525 L .79349 .39477 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82069 .39845 m .80049 .40525 L .79349 .39477 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77335 .4016 m .79349 .39477 L .80049 .40525 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77335 .4016 m .79349 .39477 L .80049 .40525 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31554 .54124 m .32588 .39845 L .3076 .40525 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.31045 .50881 m .30023 .49868 L .3076 .40525 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.3076 .40525 m .29636 .54781 L .31554 .54124 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.32588 .39845 m .3076 .40525 L .29764 .39477 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31471 .55284 m .32588 .39845 L .3076 .40525 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.3076 .40525 m .29545 .55939 L .31471 .55284 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.27943 .4016 m .29764 .39477 L .3076 .40525 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28087 .51272 m .27043 .50256 L .27943 .4016 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.29964 .50613 m .28087 .51272 L .27043 .50256 L p
.666 .738 .904 r
F P
s
P
p
.002 w
.27043 .50256 m .28927 .49589 L .29964 .50613 L p
.666 .738 .904 r
F P
s
P
p
.002 w
.29764 .39477 m .3076 .40525 L .29545 .55939 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.29764 .39477 m .3076 .40525 L .29964 .50613 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.7461 .39793 m .72619 .40473 L .71874 .39424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7461 .39793 m .72619 .40473 L .71874 .39424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69889 .40107 m .71874 .39424 L .72619 .40473 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69889 .40107 m .71874 .39424 L .72619 .40473 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67147 .3974 m .65184 .40421 L .64394 .39372 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67147 .3974 m .65184 .40421 L .64394 .39372 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67126 .6531 m .66286 .64337 L .65184 .40421 L p
.972 .685 .421 r
F P
s
P
p
.002 w
.62437 .40055 m .64394 .39372 L .65184 .40421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62437 .40055 m .64394 .39372 L .65184 .40421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60278 .62568 m .59679 .39688 L .57744 .40369 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.59032 .62435 m .58147 .61453 L .57744 .40369 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.57744 .40369 m .5818 .63209 L .60278 .62568 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.59679 .39688 m .57744 .40369 L .5691 .39319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60311 .63812 m .59679 .39688 L .57744 .40369 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.57744 .40369 m .58204 .64451 L .60311 .63812 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.54982 .40003 m .5691 .39319 L .57744 .40369 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56092 .63229 m .55183 .62247 L .54982 .40003 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.58169 .62592 m .56092 .63229 L .55183 .62247 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.55183 .62247 m .57269 .61601 L .58169 .62592 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.5691 .39319 m .57744 .40369 L .58204 .64451 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.5691 .39319 m .57744 .40369 L .58169 .62592 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.52187 .59292 m .52207 .39635 L .50301 .40317 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.51085 .59034 m .5016 .58041 L .50301 .40317 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.50301 .40317 m .50145 .5994 L .52187 .59292 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.52207 .39635 m .50301 .40317 L .49421 .39266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52186 .60467 m .52207 .39635 L .50301 .40317 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.50301 .40317 m .50136 .61113 L .52186 .60467 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.47522 .3995 m .49421 .39266 L .50301 .40317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48132 .59738 m .47182 .58744 L .47522 .3995 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.50152 .59093 m .48132 .59738 L .47182 .58744 L p
.659 .748 .916 r
F P
s
P
p
.002 w
.47182 .58744 m .49211 .58091 L .50152 .59093 L p
.659 .748 .916 r
F P
s
P
p
.002 w
.49421 .39266 m .50301 .40317 L .50136 .61113 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.49421 .39266 m .50301 .40317 L .50152 .59093 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.44285 .55351 m .44731 .39582 L .42853 .40264 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.43336 .54946 m .42374 .53942 L .42853 .40264 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.42853 .40264 m .42302 .56007 L .44285 .55351 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.44731 .39582 m .42853 .40264 L .41928 .39213 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44256 .56397 m .44731 .39582 L .42853 .40264 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.42853 .40264 m .42265 .57051 L .44256 .56397 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.40057 .39898 m .41928 .39213 L .42853 .40264 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40383 .55495 m .39397 .54488 L .40057 .39898 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.42343 .54842 m .40383 .55495 L .39397 .54488 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.39397 .54488 m .41366 .53827 L .42343 .54842 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.41928 .39213 m .42853 .40264 L .42265 .57051 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.41928 .39213 m .42853 .40264 L .42343 .54842 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.8682 .3953 m .84778 .40212 L .84105 .3916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.8682 .3953 m .84778 .40212 L .84105 .3916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82069 .39845 m .84105 .3916 L .84778 .40212 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82069 .39845 m .84105 .3916 L .84778 .40212 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3598 .62419 m .3725 .3953 L .354 .40212 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.35028 .6404 m .33982 .63062 L .354 .40212 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.35311 .59186 m .34284 .58193 L .354 .40212 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.354 .40212 m .33982 .63062 L .3598 .62419 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.3725 .3953 m .354 .40212 L .3443 .3916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36595 .51335 m .3725 .3953 L .354 .40212 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.354 .40212 m .34668 .51999 L .36595 .51335 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.32588 .39845 m .3443 .3916 L .354 .40212 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3443 .3916 m .354 .40212 L .34668 .51999 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3443 .3916 m .354 .40212 L .34199 .59568 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.79349 .39477 m .77335 .4016 L .76618 .39107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79349 .39477 m .77335 .4016 L .76618 .39107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7461 .39793 m .76618 .39107 L .77335 .4016 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7461 .39793 m .76618 .39107 L .77335 .4016 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28927 .49589 m .29764 .39477 L .27943 .4016 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.28524 .46146 m .27501 .45117 L .27943 .4016 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.27943 .4016 m .27043 .50256 L .28927 .49589 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.29764 .39477 m .27943 .4016 L .26928 .39107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28855 .50465 m .29764 .39477 L .27943 .4016 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.27943 .4016 m .26965 .51131 L .28855 .50465 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.25114 .39793 m .26928 .39107 L .27943 .4016 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25623 .46207 m .24578 .45175 L .25114 .39793 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.27463 .45538 m .25623 .46207 L .24578 .45175 L p
.669 .733 .898 r
F P
s
P
p
.002 w
.24578 .45175 m .26426 .44498 L .27463 .45538 L p
.669 .733 .898 r
F P
s
P
p
.002 w
.26928 .39107 m .27943 .4016 L .26965 .51131 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.26928 .39107 m .27943 .4016 L .27463 .45538 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.71874 .39424 m .69889 .40107 L .69126 .39054 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71874 .39424 m .69889 .40107 L .69126 .39054 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67147 .3974 m .69126 .39054 L .69889 .40107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67147 .3974 m .69126 .39054 L .69889 .40107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64394 .39372 m .62437 .40055 L .6163 .39001 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64394 .39372 m .62437 .40055 L .6163 .39001 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64144 .64456 m .63284 .63478 L .62437 .40055 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.59679 .39688 m .6163 .39001 L .62437 .40055 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6163 .39001 m .62437 .40055 L .63328 .64693 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.57269 .61601 m .5691 .39319 L .54982 .40003 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.5607 .61397 m .55167 .60409 L .54982 .40003 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.54982 .40003 m .55183 .62247 L .57269 .61601 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.5691 .39319 m .54982 .40003 L .54129 .38948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5729 .62875 m .5691 .39319 L .54982 .40003 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.54982 .40003 m .55195 .63518 L .5729 .62875 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.52207 .39635 m .54129 .38948 L .54982 .40003 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53114 .62208 m .52185 .61219 L .52207 .39635 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.55177 .61567 m .53114 .62208 L .52185 .61219 L p
.656 .751 .919 r
F P
s
P
p
.002 w
.52185 .61219 m .54258 .60569 L .55177 .61567 L p
.656 .751 .919 r
F P
s
P
p
.002 w
.54129 .38948 m .54982 .40003 L .55195 .63518 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.54129 .38948 m .54982 .40003 L .55177 .61567 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.49211 .58091 m .49421 .39266 L .47522 .3995 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.48161 .5775 m .47218 .5675 L .47522 .3995 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.47522 .3995 m .47182 .58744 L .49211 .58091 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.49421 .39266 m .47522 .3995 L .46623 .38895 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49198 .59279 m .49421 .39266 L .47522 .3995 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.47522 .3995 m .47161 .5993 L .49198 .59279 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.44731 .39582 m .46623 .38895 L .47522 .3995 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45194 .58451 m .44226 .5745 L .44731 .39582 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.47199 .57802 m .45194 .58451 L .44226 .5745 L p
.66 .747 .914 r
F P
s
P
p
.002 w
.44226 .5745 m .46241 .56793 L .47199 .57802 L p
.66 .747 .914 r
F P
s
P
p
.002 w
.46623 .38895 m .47522 .3995 L .47161 .5993 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.46623 .38895 m .47522 .3995 L .47199 .57802 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.41366 .53827 m .41928 .39213 L .40057 .39898 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.40473 .53322 m .39496 .5231 L .40057 .39898 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.40057 .39898 m .39397 .54488 L .41366 .53827 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.41928 .39213 m .40057 .39898 L .39114 .38842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41326 .54857 m .41928 .39213 L .40057 .39898 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.40057 .39898 m .39351 .55516 L .41326 .54857 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.3725 .3953 m .39114 .38842 L .40057 .39898 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37513 .53835 m .36512 .5282 L .3725 .3953 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.37017 .63405 m .3598 .62419 L .3725 .3953 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.39457 .53177 m .37513 .53835 L .36512 .5282 L p
.663 .742 .908 r
F P
s
P
p
.002 w
.36512 .5282 m .38464 .52154 L .39457 .53177 L p
.663 .742 .908 r
F P
s
P
p
.002 w
.39114 .38842 m .40057 .39898 L .39351 .55516 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.39114 .38842 m .40057 .39898 L .39457 .53177 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.84105 .3916 m .82069 .39845 L .81379 .38789 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84105 .3916 m .82069 .39845 L .81379 .38789 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79349 .39477 m .81379 .38789 L .82069 .39845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79349 .39477 m .81379 .38789 L .82069 .39845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33153 .58566 m .3443 .3916 L .32588 .39845 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.3224 .60212 m .31186 .59219 L .32588 .39845 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.32588 .55132 m .31554 .54124 L .32588 .39845 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.32588 .39845 m .31186 .59219 L .33153 .58566 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.3443 .3916 m .32588 .39845 L .31599 .38789 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33764 .49279 m .3443 .3916 L .32588 .39845 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.32588 .39845 m .31857 .49949 L .33764 .49279 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.29764 .39477 m .31599 .38789 L .32588 .39845 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31599 .38789 m .32588 .39845 L .31857 .49949 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.31599 .38789 m .32588 .39845 L .31471 .55284 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.76618 .39107 m .7461 .39793 L .73875 .38736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76618 .39107 m .7461 .39793 L .73875 .38736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71874 .39424 m .73875 .38736 L .7461 .39793 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71874 .39424 m .73875 .38736 L .7461 .39793 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26426 .44498 m .26928 .39107 L .25114 .39793 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.25114 .39793 m .24578 .45175 L .26426 .44498 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.26928 .39107 m .25114 .39793 L .2408 .38736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26379 .44998 m .26928 .39107 L .25114 .39793 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.25114 .39793 m .24529 .45675 L .26379 .44998 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.2408 .38736 m .25114 .39793 L .24529 .45675 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.69126 .39054 m .67147 .3974 L .66366 .38683 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69126 .39054 m .67147 .3974 L .66366 .38683 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64394 .39372 m .66366 .38683 L .67147 .3974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64394 .39372 m .66366 .38683 L .67147 .3974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62454 .63706 m .6163 .39001 L .59679 .39688 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.6121 .65328 m .60325 .64349 L .59679 .39688 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.61158 .63552 m .60278 .62568 L .59679 .39688 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.59679 .39688 m .60325 .64349 L .62454 .63706 L p
.555 .34 .543 r
F P
s
P
p
.002 w
.6163 .39001 m .59679 .39688 L .58853 .38629 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6163 .39001 m .59679 .39688 L .58853 .38629 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5691 .39319 m .58853 .38629 L .59679 .39688 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58853 .38629 m .59679 .39688 L .60311 .63812 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.54258 .60569 m .54129 .38948 L .52207 .39635 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.5311 .60287 m .52187 .59292 L .52207 .39635 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.52207 .39635 m .52185 .61219 L .54258 .60569 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.54129 .38948 m .52207 .39635 L .51336 .38576 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54266 .61871 m .54129 .38948 L .52207 .39635 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.52207 .39635 m .52184 .62518 L .54266 .61871 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.49421 .39266 m .51336 .38576 L .52207 .39635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50136 .61113 m .49188 .60117 L .49421 .39266 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.52186 .60467 m .50136 .61113 L .49188 .60117 L p
.657 .75 .918 r
F P
s
P
p
.002 w
.49188 .60117 m .51248 .59462 L .52186 .60467 L p
.657 .75 .918 r
F P
s
P
p
.002 w
.51336 .38576 m .52207 .39635 L .52184 .62518 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.51336 .38576 m .52207 .39635 L .52186 .60467 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.46241 .56793 m .46623 .38895 L .44731 .39582 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.45246 .56358 m .44285 .55351 L .44731 .39582 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.44731 .39582 m .44226 .5745 L .46241 .56793 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.46623 .38895 m .44731 .39582 L .43814 .38523 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46215 .57989 m .46623 .38895 L .44731 .39582 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.44731 .39582 m .44192 .58645 L .46215 .57989 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.41928 .39213 m .43814 .38523 L .44731 .39582 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42265 .57051 m .4128 .56042 L .41928 .39213 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.44256 .56397 m .42265 .57051 L .4128 .56042 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.4128 .56042 m .4328 .5538 L .44256 .56397 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.43814 .38523 m .44731 .39582 L .44192 .58645 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.43814 .38523 m .44731 .39582 L .44256 .56397 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.84105 .3916 m .86157 .3847 L .8682 .3953 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84105 .3916 m .86157 .3847 L .8682 .3953 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38464 .52154 m .39114 .38842 L .3725 .3953 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.3725 .3953 m .36512 .5282 L .38464 .52154 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.39114 .38842 m .3725 .3953 L .36287 .3847 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38415 .53154 m .39114 .38842 L .3725 .3953 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.3725 .3953 m .36457 .53819 L .38415 .53154 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.3443 .3916 m .36287 .3847 L .3725 .3953 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34668 .51999 m .33653 .50976 L .3443 .3916 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.34199 .59568 m .33153 .58566 L .3443 .3916 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.36595 .51335 m .34668 .51999 L .33653 .50976 L p
.664 .74 .906 r
F P
s
P
p
.002 w
.33653 .50976 m .35587 .50303 L .36595 .51335 L p
.664 .74 .906 r
F P
s
P
p
.002 w
.36287 .3847 m .3725 .3953 L .36457 .53819 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.36287 .3847 m .3725 .3953 L .36595 .51335 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.81379 .38789 m .79349 .39477 L .78641 .38416 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81379 .38789 m .79349 .39477 L .78641 .38416 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76618 .39107 m .78641 .38416 L .79349 .39477 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76618 .39107 m .78641 .38416 L .79349 .39477 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30419 .54266 m .31599 .38789 L .29764 .39477 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.29545 .55939 m .28485 .54929 L .29764 .39477 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.29964 .50613 m .28927 .49589 L .29764 .39477 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.29764 .39477 m .28485 .54929 L .30419 .54266 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.31599 .38789 m .29764 .39477 L .28756 .38416 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30976 .46961 m .31599 .38789 L .29764 .39477 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.29764 .39477 m .29089 .47637 L .30976 .46961 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.26928 .39107 m .28756 .38416 L .29764 .39477 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28756 .38416 m .29764 .39477 L .29089 .47637 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.28756 .38416 m .29764 .39477 L .28855 .50465 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.73875 .38736 m .71874 .39424 L .7112 .38363 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73875 .38736 m .71874 .39424 L .7112 .38363 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69126 .39054 m .7112 .38363 L .71874 .39424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69126 .39054 m .7112 .38363 L .71874 .39424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66366 .38683 m .64394 .39372 L .63595 .3831 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66366 .38683 m .64394 .39372 L .63595 .3831 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6163 .39001 m .63595 .3831 L .64394 .39372 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6163 .39001 m .63595 .3831 L .64394 .39372 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59416 .62819 m .58853 .38629 L .5691 .39319 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.58204 .64451 m .57299 .63466 L .5691 .39319 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.58169 .62592 m .57269 .61601 L .5691 .39319 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.5691 .39319 m .57299 .63466 L .59416 .62819 L p
.553 .337 .541 r
F P
s
P
p
.002 w
.58853 .38629 m .5691 .39319 L .56065 .38256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58853 .38629 m .5691 .39319 L .56065 .38256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54129 .38948 m .56065 .38256 L .5691 .39319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56065 .38256 m .5691 .39319 L .5729 .62875 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.51248 .59462 m .51336 .38576 L .49421 .39266 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.50152 .59093 m .49211 .58091 L .49421 .39266 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.49421 .39266 m .49188 .60117 L .51248 .59462 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.51336 .38576 m .49421 .39266 L .48531 .38203 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51243 .60791 m .51336 .38576 L .49421 .39266 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.49421 .39266 m .49174 .61442 L .51243 .60791 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.46623 .38895 m .48531 .38203 L .49421 .39266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47161 .5993 m .46195 .58927 L .46623 .38895 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.49198 .59279 m .47161 .5993 L .46195 .58927 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.46195 .58927 m .48241 .58268 L .49198 .59279 L p
.658 .749 .917 r
F P
s
P
p
.002 w
.48531 .38203 m .49421 .39266 L .49174 .61442 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.48531 .38203 m .49421 .39266 L .49198 .59279 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.4328 .5538 m .43814 .38523 L .41928 .39213 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.42343 .54842 m .41366 .53827 L .41928 .39213 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.41928 .39213 m .4128 .56042 L .4328 .5538 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.43814 .38523 m .41928 .39213 L .40992 .38149 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43242 .56578 m .43814 .38523 L .41928 .39213 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.41928 .39213 m .41234 .57238 L .43242 .56578 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.39114 .38842 m .40992 .38149 L .41928 .39213 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39351 .55516 m .38349 .54499 L .39114 .38842 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.41326 .54857 m .39351 .55516 L .38349 .54499 L p
.661 .744 .911 r
F P
s
P
p
.002 w
.38349 .54499 m .40333 .53832 L .41326 .54857 L p
.661 .744 .911 r
F P
s
P
p
.002 w
.40992 .38149 m .41928 .39213 L .41234 .57238 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.40992 .38149 m .41928 .39213 L .41326 .54857 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.86157 .3847 m .84105 .3916 L .83424 .38096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.86157 .3847 m .84105 .3916 L .83424 .38096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81379 .38789 m .83424 .38096 L .84105 .3916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81379 .38789 m .83424 .38096 L .84105 .3916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35587 .50303 m .36287 .3847 L .3443 .3916 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.3443 .3916 m .33653 .50976 L .35587 .50303 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.36287 .3847 m .3443 .3916 L .33449 .38096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3553 .51256 m .36287 .3847 L .3443 .3916 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.3443 .3916 m .3359 .51927 L .3553 .51256 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.31599 .38789 m .33449 .38096 L .3443 .3916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31857 .49949 m .30827 .48916 L .31599 .38789 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.31471 .55284 m .30419 .54266 L .31599 .38789 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.33764 .49279 m .31857 .49949 L .30827 .48916 L p
.666 .738 .904 r
F P
s
P
p
.002 w
.30827 .48916 m .32742 .48239 L .33764 .49279 L p
.666 .738 .904 r
F P
s
P
p
.002 w
.33449 .38096 m .3443 .3916 L .3359 .51927 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.33449 .38096 m .3443 .3916 L .33764 .49279 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.78641 .38416 m .76618 .39107 L .75892 .38042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78641 .38416 m .76618 .39107 L .75892 .38042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73875 .38736 m .75892 .38042 L .76618 .39107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73875 .38736 m .75892 .38042 L .76618 .39107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27801 .49431 m .28756 .38416 L .26928 .39107 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.26965 .51131 m .25903 .50105 L .26928 .39107 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.27463 .45538 m .26426 .44498 L .26928 .39107 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.26928 .39107 m .25903 .50105 L .27801 .49431 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.28756 .38416 m .26928 .39107 L .25901 .38042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28244 .44316 m .28756 .38416 L .26928 .39107 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.26928 .39107 m .26379 .44998 L .28244 .44316 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.2408 .38736 m .25901 .38042 L .26928 .39107 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25901 .38042 m .26928 .39107 L .26379 .44998 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.25901 .38042 m .26928 .39107 L .26379 .44998 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.7112 .38363 m .69126 .39054 L .68354 .37988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7112 .38363 m .69126 .39054 L .68354 .37988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66366 .38683 m .68354 .37988 L .69126 .39054 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66366 .38683 m .68354 .37988 L .69126 .39054 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63595 .3831 m .6163 .39001 L .60812 .37935 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63595 .3831 m .6163 .39001 L .60812 .37935 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63328 .64693 m .62454 .63706 L .6163 .39001 L p
.971 .686 .43 r
F P
s
P
p
.002 w
.58853 .38629 m .60812 .37935 L .6163 .39001 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58853 .38629 m .60812 .37935 L .6163 .39001 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56375 .61875 m .56065 .38256 L .54129 .38948 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.55195 .63518 m .5427 .62526 L .54129 .38948 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.55177 .61567 m .54258 .60569 L .54129 .38948 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.54129 .38948 m .5427 .62526 L .56375 .61875 L p
.551 .334 .539 r
F P
s
P
p
.002 w
.56065 .38256 m .54129 .38948 L .53266 .37881 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56065 .38256 m .54129 .38948 L .53266 .37881 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51336 .38576 m .53266 .37881 L .54129 .38948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53266 .37881 m .54129 .38948 L .54266 .61871 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.48241 .58268 m .48531 .38203 L .46623 .38895 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.47199 .57802 m .46241 .56793 L .46623 .38895 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.46623 .38895 m .46195 .58927 L .48241 .58268 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.48531 .38203 m .46623 .38895 L .45715 .37827 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48221 .5962 m .48531 .38203 L .46623 .38895 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.46623 .38895 m .46166 .60276 L .48221 .5962 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.43814 .38523 m .45715 .37827 L .46623 .38895 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44192 .58645 m .43208 .57634 L .43814 .38523 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.46215 .57989 m .44192 .58645 L .43208 .57634 L p
.659 .748 .916 r
F P
s
P
p
.002 w
.43208 .57634 m .4524 .5697 L .46215 .57989 L p
.659 .748 .916 r
F P
s
P
p
.002 w
.45715 .37827 m .46623 .38895 L .46166 .60276 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.45715 .37827 m .46623 .38895 L .46215 .57989 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.40333 .53832 m .40992 .38149 L .39114 .38842 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.39457 .53177 m .38464 .52154 L .39114 .38842 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.39114 .38842 m .38349 .54499 L .40333 .53832 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.40992 .38149 m .39114 .38842 L .38159 .37774 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40283 .55021 m .40992 .38149 L .39114 .38842 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.39114 .38842 m .38291 .55686 L .40283 .55021 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.36287 .3847 m .38159 .37774 L .39114 .38842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36457 .53819 m .35439 .52794 L .36287 .3847 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.38415 .53154 m .36457 .53819 L .35439 .52794 L p
.663 .743 .91 r
F P
s
P
p
.002 w
.35439 .52794 m .37406 .52121 L .38415 .53154 L p
.663 .743 .91 r
F P
s
P
p
.002 w
.38159 .37774 m .39114 .38842 L .38291 .55686 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.38159 .37774 m .39114 .38842 L .38415 .53154 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.83424 .38096 m .81379 .38789 L .8068 .3772 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83424 .38096 m .81379 .38789 L .8068 .3772 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78641 .38416 m .8068 .3772 L .81379 .38789 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78641 .38416 m .8068 .3772 L .81379 .38789 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32742 .48239 m .33449 .38096 L .31599 .38789 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.31599 .38789 m .30827 .48916 L .32742 .48239 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.33449 .38096 m .31599 .38789 L .30599 .3772 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32681 .49118 m .33449 .38096 L .31599 .38789 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.31599 .38789 m .3076 .49794 L .32681 .49118 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.28756 .38416 m .30599 .3772 L .31599 .38789 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29089 .47637 m .28047 .46595 L .28756 .38416 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.28855 .50465 m .27801 .49431 L .28756 .38416 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.30976 .46961 m .29089 .47637 L .28047 .46595 L p
.667 .736 .902 r
F P
s
P
p
.002 w
.28047 .46595 m .29942 .45911 L .30976 .46961 L p
.667 .736 .902 r
F P
s
P
p
.002 w
.30599 .3772 m .31599 .38789 L .3076 .49794 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.30599 .3772 m .31599 .38789 L .30976 .46961 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.75892 .38042 m .73875 .38736 L .7313 .37666 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75892 .38042 m .73875 .38736 L .7313 .37666 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7112 .38363 m .7313 .37666 L .73875 .38736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7112 .38363 m .7313 .37666 L .73875 .38736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25326 .43947 m .25901 .38042 L .2408 .38736 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.24529 .45675 m .23469 .44632 L .2408 .38736 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.2408 .38736 m .23469 .44632 L .25326 .43947 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.25901 .38042 m .2408 .38736 L .23034 .37666 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23469 .44632 m .25326 .43947 L .26379 .44998 L p
.669 .734 .899 r
F P
s
P
p
.002 w
.25268 .44551 m .25901 .38042 L .2408 .38736 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.26379 .44998 m .24529 .45675 L .23469 .44632 L p
.669 .734 .899 r
F P
s
P
p
.002 w
.2408 .38736 m .23406 .45235 L .25268 .44551 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.23034 .37666 m .2408 .38736 L .23406 .45235 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.68354 .37988 m .66366 .38683 L .65576 .37612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68354 .37988 m .66366 .38683 L .65576 .37612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63595 .3831 m .65576 .37612 L .66366 .38683 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63595 .3831 m .65576 .37612 L .66366 .38683 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60812 .37935 m .58853 .38629 L .58017 .37558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60812 .37935 m .58853 .38629 L .58017 .37558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60311 .63812 m .59416 .62819 L .58853 .38629 L p
.969 .686 .439 r
F P
s
P
p
.002 w
.56065 .38256 m .58017 .37558 L .58853 .38629 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56065 .38256 m .58017 .37558 L .58853 .38629 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53331 .60865 m .53266 .37881 L .51336 .38576 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.52184 .62518 m .51239 .6152 L .51336 .38576 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.52186 .60467 m .51248 .59462 L .51336 .38576 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.51336 .38576 m .51239 .6152 L .53331 .60865 L p
.548 .331 .536 r
F P
s
P
p
.002 w
.53266 .37881 m .51336 .38576 L .50454 .37505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53266 .37881 m .51336 .38576 L .50454 .37505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48531 .38203 m .50454 .37505 L .51336 .38576 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50454 .37505 m .51336 .38576 L .51243 .60791 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.4524 .5697 m .45715 .37827 L .43814 .38523 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.44256 .56397 m .4328 .5538 L .43814 .38523 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.43814 .38523 m .43208 .57634 L .4524 .5697 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.45715 .37827 m .43814 .38523 L .42886 .37451 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45206 .58342 m .45715 .37827 L .43814 .38523 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.43814 .38523 m .43165 .59004 L .45206 .58342 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.40992 .38149 m .42886 .37451 L .43814 .38523 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41234 .57238 m .40232 .5622 L .40992 .38149 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.43242 .56578 m .41234 .57238 L .40232 .5622 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.40232 .5622 m .42248 .55551 L .43242 .56578 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.42886 .37451 m .43814 .38523 L .43165 .59004 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.42886 .37451 m .43814 .38523 L .43242 .56578 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.83424 .38096 m .85486 .37397 L .86157 .3847 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83424 .38096 m .85486 .37397 L .86157 .3847 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37406 .52121 m .38159 .37774 L .36287 .3847 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.36595 .51335 m .35587 .50303 L .36287 .3847 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.36287 .3847 m .35439 .52794 L .37406 .52121 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.38159 .37774 m .36287 .3847 L .35313 .37397 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37344 .53287 m .38159 .37774 L .36287 .3847 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.36287 .3847 m .3537 .53958 L .37344 .53287 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.33449 .38096 m .35313 .37397 L .36287 .3847 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3359 .51927 m .32557 .50893 L .33449 .38096 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3553 .51256 m .3359 .51927 L .32557 .50893 L p
.664 .741 .908 r
F P
s
P
p
.002 w
.32557 .50893 m .34506 .50214 L .3553 .51256 L p
.664 .741 .908 r
F P
s
P
p
.002 w
.35313 .37397 m .36287 .3847 L .3537 .53958 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.35313 .37397 m .36287 .3847 L .3553 .51256 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.8068 .3772 m .78641 .38416 L .77924 .37343 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.8068 .3772 m .78641 .38416 L .77924 .37343 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75892 .38042 m .77924 .37343 L .78641 .38416 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75892 .38042 m .77924 .37343 L .78641 .38416 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29942 .45911 m .30599 .3772 L .28756 .38416 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.28756 .38416 m .28047 .46595 L .29942 .45911 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.30599 .3772 m .28756 .38416 L .27736 .37343 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2988 .4668 m .30599 .3772 L .28756 .38416 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.28756 .38416 m .2798 .47363 L .2988 .4668 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.25901 .38042 m .27736 .37343 L .28756 .38416 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26379 .44998 m .25326 .43947 L .25901 .38042 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.26379 .44998 m .25326 .43947 L .25901 .38042 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.28244 .44316 m .26379 .44998 L .25326 .43947 L p
.669 .734 .899 r
F P
s
P
p
.002 w
.25326 .43947 m .27199 .43256 L .28244 .44316 L p
.669 .734 .899 r
F P
s
P
p
.002 w
.27736 .37343 m .28756 .38416 L .2798 .47363 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.27736 .37343 m .28756 .38416 L .28244 .44316 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.7313 .37666 m .7112 .38363 L .70358 .37289 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7313 .37666 m .7112 .38363 L .70358 .37289 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68354 .37988 m .70358 .37289 L .7112 .38363 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68354 .37988 m .70358 .37289 L .7112 .38363 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65576 .37612 m .63595 .3831 L .62786 .37235 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65576 .37612 m .63595 .3831 L .62786 .37235 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60812 .37935 m .62786 .37235 L .63595 .3831 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60812 .37935 m .62786 .37235 L .63595 .3831 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58017 .37558 m .56065 .38256 L .55211 .3718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58017 .37558 m .56065 .38256 L .55211 .3718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5729 .62875 m .56375 .61875 L .56065 .38256 L p
.967 .687 .448 r
F P
s
P
p
.002 w
.53266 .37881 m .55211 .3718 L .56065 .38256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53266 .37881 m .55211 .3718 L .56065 .38256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50288 .59777 m .50454 .37505 L .48531 .38203 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.49174 .61442 m .48209 .60438 L .48531 .38203 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.49198 .59279 m .48241 .58268 L .48531 .38203 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.48531 .38203 m .48209 .60438 L .50288 .59777 L p
.546 .327 .534 r
F P
s
P
p
.002 w
.50454 .37505 m .48531 .38203 L .4763 .37126 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50454 .37505 m .48531 .38203 L .4763 .37126 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45715 .37827 m .4763 .37126 L .48531 .38203 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4763 .37126 m .48531 .38203 L .48221 .5962 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.42248 .55551 m .42886 .37451 L .40992 .38149 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.41326 .54857 m .40333 .53832 L .40992 .38149 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.40992 .38149 m .40232 .5622 L .42248 .55551 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.42886 .37451 m .40992 .38149 L .40045 .37072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.422 .56936 m .42886 .37451 L .40992 .38149 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.40992 .38149 m .40174 .57603 L .422 .56936 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.38159 .37774 m .40045 .37072 L .40992 .38149 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38291 .55686 m .37272 .5466 L .38159 .37774 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.40283 .55021 m .38291 .55686 L .37272 .5466 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.37272 .5466 m .39272 .53986 L .40283 .55021 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.40045 .37072 m .40992 .38149 L .40174 .57603 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.40045 .37072 m .40992 .38149 L .40283 .55021 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.85486 .37397 m .83424 .38096 L .82735 .37018 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.85486 .37397 m .83424 .38096 L .82735 .37018 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.8068 .3772 m .82735 .37018 L .83424 .38096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.8068 .3772 m .82735 .37018 L .83424 .38096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34506 .50214 m .35313 .37397 L .33449 .38096 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.33764 .49279 m .32742 .48239 L .33449 .38096 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.33449 .38096 m .32557 .50893 L .34506 .50214 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.35313 .37397 m .33449 .38096 L .32456 .37018 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34435 .51337 m .35313 .37397 L .33449 .38096 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.33449 .38096 m .32479 .52014 L .34435 .51337 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.30599 .3772 m .32456 .37018 L .33449 .38096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3076 .49794 m .29714 .48751 L .30599 .3772 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.32681 .49118 m .3076 .49794 L .29714 .48751 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.29714 .48751 m .31643 .48066 L .32681 .49118 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.32456 .37018 m .33449 .38096 L .32479 .52014 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.32456 .37018 m .33449 .38096 L .32681 .49118 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.77924 .37343 m .75892 .38042 L .75156 .36964 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77924 .37343 m .75892 .38042 L .75156 .36964 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7313 .37666 m .75156 .36964 L .75892 .38042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7313 .37666 m .75156 .36964 L .75892 .38042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27199 .43256 m .27736 .37343 L .25901 .38042 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.25901 .38042 m .25326 .43947 L .27199 .43256 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.27736 .37343 m .25901 .38042 L .24862 .36964 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27144 .43861 m .27736 .37343 L .25901 .38042 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.25901 .38042 m .25268 .44551 L .27144 .43861 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.23034 .37666 m .24862 .36964 L .25901 .38042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24862 .36964 m .25901 .38042 L .25268 .44551 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.24862 .36964 m .25901 .38042 L .25268 .44551 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.70358 .37289 m .68354 .37988 L .67573 .3691 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70358 .37289 m .68354 .37988 L .67573 .3691 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65576 .37612 m .67573 .3691 L .68354 .37988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65576 .37612 m .67573 .3691 L .68354 .37988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62786 .37235 m .60812 .37935 L .59985 .36855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62786 .37235 m .60812 .37935 L .59985 .36855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58017 .37558 m .59985 .36855 L .60812 .37935 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58017 .37558 m .59985 .36855 L .60812 .37935 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55211 .3718 m .53266 .37881 L .52392 .36801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55211 .3718 m .53266 .37881 L .52392 .36801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54266 .61871 m .53331 .60865 L .53266 .37881 L p
.965 .687 .456 r
F P
s
P
p
.002 w
.50454 .37505 m .52392 .36801 L .53266 .37881 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50454 .37505 m .52392 .36801 L .53266 .37881 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47247 .586 m .4763 .37126 L .45715 .37827 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.46166 .60276 m .45183 .59264 L .45715 .37827 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.46215 .57989 m .4524 .5697 L .45715 .37827 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.45715 .37827 m .45183 .59264 L .47247 .586 L p
.544 .324 .532 r
F P
s
P
p
.002 w
.4763 .37126 m .45715 .37827 L .44795 .36747 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4763 .37126 m .45715 .37827 L .44795 .36747 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42886 .37451 m .44795 .36747 L .45715 .37827 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44795 .36747 m .45715 .37827 L .45206 .58342 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.39272 .53986 m .40045 .37072 L .38159 .37774 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.38415 .53154 m .37406 .52121 L .38159 .37774 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.38159 .37774 m .37272 .5466 L .39272 .53986 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.40045 .37072 m .38159 .37774 L .37193 .36692 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39209 .55375 m .40045 .37072 L .38159 .37774 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.38159 .37774 m .372 .56046 L .39209 .55375 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.35313 .37397 m .37193 .36692 L .38159 .37774 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3537 .53958 m .34335 .52924 L .35313 .37397 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.37344 .53287 m .3537 .53958 L .34335 .52924 L p
.662 .744 .911 r
F P
s
P
p
.002 w
.34335 .52924 m .36318 .52244 L .37344 .53287 L p
.662 .744 .911 r
F P
s
P
p
.002 w
.37193 .36692 m .38159 .37774 L .372 .56046 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.37193 .36692 m .38159 .37774 L .37344 .53287 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.82735 .37018 m .8068 .3772 L .79973 .36638 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82735 .37018 m .8068 .3772 L .79973 .36638 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77924 .37343 m .79973 .36638 L .8068 .3772 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77924 .37343 m .79973 .36638 L .8068 .3772 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31643 .48066 m .32456 .37018 L .30599 .3772 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.30976 .46961 m .29942 .45911 L .30599 .3772 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.30599 .3772 m .29714 .48751 L .31643 .48066 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.32456 .37018 m .30599 .3772 L .29586 .36638 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31565 .49115 m .32456 .37018 L .30599 .3772 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.30599 .3772 m .2963 .49799 L .31565 .49115 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.27736 .37343 m .29586 .36638 L .30599 .3772 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2798 .47363 m .26922 .4631 L .27736 .37343 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.2988 .4668 m .2798 .47363 L .26922 .4631 L p
.667 .737 .903 r
F P
s
P
p
.002 w
.26922 .4631 m .28829 .45618 L .2988 .4668 L p
.667 .737 .903 r
F P
s
P
p
.002 w
.29586 .36638 m .30599 .3772 L .2963 .49799 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.29586 .36638 m .30599 .3772 L .2988 .4668 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.75156 .36964 m .7313 .37666 L .72377 .36583 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75156 .36964 m .7313 .37666 L .72377 .36583 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70358 .37289 m .72377 .36583 L .7313 .37666 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70358 .37289 m .72377 .36583 L .7313 .37666 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24199 .43488 m .24862 .36964 L .23034 .37666 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.23406 .45235 m .22331 .4418 L .23034 .37666 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.23034 .37666 m .22331 .4418 L .24199 .43488 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.24862 .36964 m .23034 .37666 L .21975 .36583 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24498 .4055 m .24862 .36964 L .23034 .37666 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.23034 .37666 m .22647 .41247 L .24498 .4055 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.21975 .36583 m .23034 .37666 L .22647 .41247 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.67573 .3691 m .65576 .37612 L .64776 .36529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67573 .3691 m .65576 .37612 L .64776 .36529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62786 .37235 m .64776 .36529 L .65576 .37612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62786 .37235 m .64776 .36529 L .65576 .37612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59985 .36855 m .58017 .37558 L .57171 .36474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59985 .36855 m .58017 .37558 L .57171 .36474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55211 .3718 m .57171 .36474 L .58017 .37558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55211 .3718 m .57171 .36474 L .58017 .37558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52392 .36801 m .50454 .37505 L .49561 .3642 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52392 .36801 m .50454 .37505 L .49561 .3642 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51243 .60791 m .50288 .59777 L .50454 .37505 L p
.963 .688 .463 r
F P
s
P
p
.002 w
.4763 .37126 m .49561 .3642 L .50454 .37505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4763 .37126 m .49561 .3642 L .50454 .37505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44213 .57314 m .44795 .36747 L .42886 .37451 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.43165 .59004 m .42163 .57984 L .42886 .37451 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.43242 .56578 m .42248 .55551 L .42886 .37451 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.42886 .37451 m .42163 .57984 L .44213 .57314 L p
.541 .321 .529 r
F P
s
P
p
.002 w
.44795 .36747 m .42886 .37451 L .41947 .36365 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44795 .36747 m .42886 .37451 L .41947 .36365 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40045 .37072 m .41947 .36365 L .42886 .37451 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41947 .36365 m .42886 .37451 L .422 .56936 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.82735 .37018 m .84807 .36311 L .85486 .37397 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82735 .37018 m .84807 .36311 L .85486 .37397 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36318 .52244 m .37193 .36692 L .35313 .37397 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.3553 .51256 m .34506 .50214 L .35313 .37397 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.35313 .37397 m .34335 .52924 L .36318 .52244 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.37193 .36692 m .35313 .37397 L .34328 .36311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3624 .53621 m .37193 .36692 L .35313 .37397 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.35313 .37397 m .34248 .54298 L .3624 .53621 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.32456 .37018 m .34328 .36311 L .35313 .37397 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32479 .52014 m .31429 .5097 L .32456 .37018 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.34435 .51337 m .32479 .52014 L .31429 .5097 L p
.663 .742 .909 r
F P
s
P
p
.002 w
.31429 .5097 m .33393 .50284 L .34435 .51337 L p
.663 .742 .909 r
F P
s
P
p
.002 w
.34328 .36311 m .35313 .37397 L .34248 .54298 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.34328 .36311 m .35313 .37397 L .34435 .51337 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.79973 .36638 m .77924 .37343 L .77198 .36256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79973 .36638 m .77924 .37343 L .77198 .36256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75156 .36964 m .77198 .36256 L .77924 .37343 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75156 .36964 m .77198 .36256 L .77924 .37343 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28829 .45618 m .29586 .36638 L .27736 .37343 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.28244 .44316 m .27199 .43256 L .27736 .37343 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.27736 .37343 m .26922 .4631 L .28829 .45618 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.29586 .36638 m .27736 .37343 L .26704 .36256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28751 .46549 m .29586 .36638 L .27736 .37343 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.27736 .37343 m .26837 .47239 L .28751 .46549 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.24862 .36964 m .26704 .36256 L .27736 .37343 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25268 .44551 m .24199 .43488 L .24862 .36964 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.25268 .44551 m .24199 .43488 L .24862 .36964 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.27144 .43861 m .25268 .44551 L .24199 .43488 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.24199 .43488 m .26083 .42789 L .27144 .43861 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.26704 .36256 m .27736 .37343 L .26837 .47239 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.26704 .36256 m .27736 .37343 L .27144 .43861 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.72377 .36583 m .70358 .37289 L .69585 .36201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72377 .36583 m .70358 .37289 L .69585 .36201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67573 .3691 m .69585 .36201 L .70358 .37289 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67573 .3691 m .69585 .36201 L .70358 .37289 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64776 .36529 m .62786 .37235 L .61968 .36146 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64776 .36529 m .62786 .37235 L .61968 .36146 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59985 .36855 m .61968 .36146 L .62786 .37235 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59985 .36855 m .61968 .36146 L .62786 .37235 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57171 .36474 m .55211 .3718 L .54345 .36092 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57171 .36474 m .55211 .3718 L .54345 .36092 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52392 .36801 m .54345 .36092 L .55211 .3718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52392 .36801 m .54345 .36092 L .55211 .3718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49561 .3642 m .4763 .37126 L .46718 .36037 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49561 .3642 m .4763 .37126 L .46718 .36037 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48221 .5962 m .47247 .586 L .4763 .37126 L p
.961 .688 .471 r
F P
s
P
p
.002 w
.44795 .36747 m .46718 .36037 L .4763 .37126 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44795 .36747 m .46718 .36037 L .4763 .37126 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41188 .55901 m .41947 .36365 L .40045 .37072 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.40174 .57603 m .39154 .56575 L .40045 .37072 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.40283 .55021 m .39272 .53986 L .40045 .37072 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.40045 .37072 m .39154 .56575 L .41188 .55901 L p
.539 .317 .526 r
F P
s
P
p
.002 w
.41947 .36365 m .40045 .37072 L .39087 .35982 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41947 .36365 m .40045 .37072 L .39087 .35982 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37193 .36692 m .39087 .35982 L .40045 .37072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39087 .35982 m .40045 .37072 L .39209 .55375 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.84807 .36311 m .82735 .37018 L .82038 .35927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84807 .36311 m .82735 .37018 L .82038 .35927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79973 .36638 m .82038 .35927 L .82735 .37018 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79973 .36638 m .82038 .35927 L .82735 .37018 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33393 .50284 m .34328 .36311 L .32456 .37018 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.32681 .49118 m .31643 .48066 L .32456 .37018 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.32456 .37018 m .31429 .5097 L .33393 .50284 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.34328 .36311 m .32456 .37018 L .3145 .35927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33303 .51627 m .34328 .36311 L .32456 .37018 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.32456 .37018 m .3133 .5231 L .33303 .51627 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.29586 .36638 m .3145 .35927 L .32456 .37018 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2963 .49799 m .28566 .48745 L .29586 .36638 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.31565 .49115 m .2963 .49799 L .28566 .48745 L p
.664 .74 .907 r
F P
s
P
p
.002 w
.28566 .48745 m .30509 .48053 L .31565 .49115 L p
.664 .74 .907 r
F P
s
P
p
.002 w
.3145 .35927 m .32456 .37018 L .3133 .5231 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3145 .35927 m .32456 .37018 L .31565 .49115 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.77198 .36256 m .75156 .36964 L .74412 .35872 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77198 .36256 m .75156 .36964 L .74412 .35872 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72377 .36583 m .74412 .35872 L .75156 .36964 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72377 .36583 m .74412 .35872 L .75156 .36964 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26083 .42789 m .26704 .36256 L .24862 .36964 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.24862 .36964 m .24199 .43488 L .26083 .42789 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.26704 .36256 m .24862 .36964 L .23809 .35872 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26012 .43535 m .26704 .36256 L .24862 .36964 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.22331 .4418 m .24199 .43488 L .25268 .44551 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.24862 .36964 m .24124 .44232 L .26012 .43535 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.21975 .36583 m .23809 .35872 L .24862 .36964 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22647 .41247 m .21572 .40172 L .21975 .36583 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.24498 .4055 m .22647 .41247 L .21572 .40172 L p
.67 .731 .896 r
F P
s
P
p
.002 w
.21572 .40172 m .23429 .39466 L .24498 .4055 L p
.67 .731 .896 r
F P
s
P
p
.002 w
.23809 .35872 m .24862 .36964 L .24124 .44232 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.23809 .35872 m .24862 .36964 L .24498 .4055 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.69585 .36201 m .67573 .3691 L .66782 .35817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69585 .36201 m .67573 .3691 L .66782 .35817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64776 .36529 m .66782 .35817 L .67573 .3691 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64776 .36529 m .66782 .35817 L .67573 .3691 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61968 .36146 m .59985 .36855 L .59147 .35762 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61968 .36146 m .59985 .36855 L .59147 .35762 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57171 .36474 m .59147 .35762 L .59985 .36855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57171 .36474 m .59147 .35762 L .59985 .36855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54345 .36092 m .52392 .36801 L .51508 .35707 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54345 .36092 m .52392 .36801 L .51508 .35707 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49561 .3642 m .51508 .35707 L .52392 .36801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49561 .3642 m .51508 .35707 L .52392 .36801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46718 .36037 m .44795 .36747 L .43863 .35652 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46718 .36037 m .44795 .36747 L .43863 .35652 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45206 .58342 m .44213 .57314 L .44795 .36747 L p
.959 .688 .478 r
F P
s
P
p
.002 w
.41947 .36365 m .43863 .35652 L .44795 .36747 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41947 .36365 m .43863 .35652 L .44795 .36747 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3818 .5433 m .39087 .35982 L .37193 .36692 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.372 .56046 m .36162 .55011 L .37193 .36692 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.37344 .53287 m .36318 .52244 L .37193 .36692 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.37193 .36692 m .36162 .55011 L .3818 .5433 L p
.536 .314 .524 r
F P
s
P
p
.002 w
.39087 .35982 m .37193 .36692 L .36214 .35597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39087 .35982 m .37193 .36692 L .36214 .35597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34328 .36311 m .36214 .35597 L .37193 .36692 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36214 .35597 m .37193 .36692 L .3624 .53621 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.82038 .35927 m .79973 .36638 L .79257 .35542 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82038 .35927 m .79973 .36638 L .79257 .35542 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77198 .36256 m .79257 .35542 L .79973 .36638 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77198 .36256 m .79257 .35542 L .79973 .36638 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30509 .48053 m .3145 .35927 L .29586 .36638 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.2988 .4668 m .28829 .45618 L .29586 .36638 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.29586 .36638 m .28566 .48745 L .30509 .48053 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.3145 .35927 m .29586 .36638 L .28561 .35542 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30411 .49326 m .3145 .35927 L .29586 .36638 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.29586 .36638 m .28459 .50016 L .30411 .49326 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.26704 .36256 m .28561 .35542 L .29586 .36638 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26837 .47239 m .25761 .46175 L .26704 .36256 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.28751 .46549 m .26837 .47239 L .25761 .46175 L p
.666 .738 .904 r
F P
s
P
p
.002 w
.25761 .46175 m .27683 .45477 L .28751 .46549 L p
.666 .738 .904 r
F P
s
P
p
.002 w
.28561 .35542 m .29586 .36638 L .28459 .50016 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.28561 .35542 m .29586 .36638 L .28751 .46549 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.74412 .35872 m .72377 .36583 L .71614 .35487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74412 .35872 m .72377 .36583 L .71614 .35487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69585 .36201 m .71614 .35487 L .72377 .36583 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69585 .36201 m .71614 .35487 L .72377 .36583 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23429 .39466 m .23809 .35872 L .21975 .36583 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.21975 .36583 m .21572 .40172 L .23429 .39466 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.23809 .35872 m .21975 .36583 L .20903 .35487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23381 .39922 m .23809 .35872 L .21975 .36583 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.21975 .36583 m .21521 .40627 L .23381 .39922 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.20903 .35487 m .21975 .36583 L .21521 .40627 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.66782 .35817 m .64776 .36529 L .63967 .35432 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66782 .35817 m .64776 .36529 L .63967 .35432 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61968 .36146 m .63967 .35432 L .64776 .36529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61968 .36146 m .63967 .35432 L .64776 .36529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59147 .35762 m .57171 .36474 L .56314 .35377 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59147 .35762 m .57171 .36474 L .56314 .35377 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54345 .36092 m .56314 .35377 L .57171 .36474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54345 .36092 m .56314 .35377 L .57171 .36474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51508 .35707 m .49561 .3642 L .48658 .35322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51508 .35707 m .49561 .3642 L .48658 .35322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46718 .36037 m .48658 .35322 L .49561 .3642 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46718 .36037 m .48658 .35322 L .49561 .3642 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43863 .35652 m .41947 .36365 L .40996 .35266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43863 .35652 m .41947 .36365 L .40996 .35266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.422 .56936 m .41188 .55901 L .41947 .36365 L p
.957 .688 .484 r
F P
s
P
p
.002 w
.39087 .35982 m .40996 .35266 L .41947 .36365 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39087 .35982 m .40996 .35266 L .41947 .36365 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82038 .35927 m .84119 .35211 L .84807 .36311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82038 .35927 m .84119 .35211 L .84807 .36311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35194 .52568 m .36214 .35597 L .34328 .36311 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.34248 .54298 m .33194 .53254 L .34328 .36311 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.34435 .51337 m .33393 .50284 L .34328 .36311 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.34328 .36311 m .33194 .53254 L .35194 .52568 L p
.534 .31 .521 r
F P
s
P
p
.002 w
.36214 .35597 m .34328 .36311 L .3333 .35211 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36214 .35597 m .34328 .36311 L .3333 .35211 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3145 .35927 m .3333 .35211 L .34328 .36311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3333 .35211 m .34328 .36311 L .33303 .51627 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.79257 .35542 m .77198 .36256 L .76464 .35156 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79257 .35542 m .77198 .36256 L .76464 .35156 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74412 .35872 m .76464 .35156 L .77198 .36256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74412 .35872 m .76464 .35156 L .77198 .36256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27683 .45477 m .28561 .35542 L .26704 .36256 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.27144 .43861 m .26083 .42789 L .26704 .36256 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.26704 .36256 m .25761 .46175 L .27683 .45477 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.28561 .35542 m .26704 .36256 L .25659 .35156 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27581 .46626 m .28561 .35542 L .26704 .36256 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.26704 .36256 m .25652 .47322 L .27581 .46626 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.23809 .35872 m .25659 .35156 L .26704 .36256 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24124 .44232 m .23039 .43157 L .23809 .35872 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.26012 .43535 m .24124 .44232 L .23039 .43157 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.23039 .43157 m .24935 .42451 L .26012 .43535 L p
.668 .735 .9 r
F P
s
P
p
.002 w
.25659 .35156 m .26704 .36256 L .25652 .47322 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.25659 .35156 m .26704 .36256 L .26012 .43535 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.71614 .35487 m .69585 .36201 L .68804 .351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71614 .35487 m .69585 .36201 L .68804 .351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66782 .35817 m .68804 .351 L .69585 .36201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66782 .35817 m .68804 .351 L .69585 .36201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63967 .35432 m .61968 .36146 L .61139 .35045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63967 .35432 m .61968 .36146 L .61139 .35045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59147 .35762 m .61139 .35045 L .61968 .36146 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59147 .35762 m .61139 .35045 L .61968 .36146 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56314 .35377 m .54345 .36092 L .5347 .3499 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56314 .35377 m .54345 .36092 L .5347 .3499 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51508 .35707 m .5347 .3499 L .54345 .36092 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51508 .35707 m .5347 .3499 L .54345 .36092 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48658 .35322 m .46718 .36037 L .45795 .34934 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48658 .35322 m .46718 .36037 L .45795 .34934 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43863 .35652 m .45795 .34934 L .46718 .36037 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43863 .35652 m .45795 .34934 L .46718 .36037 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40996 .35266 m .39087 .35982 L .38116 .34879 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40996 .35266 m .39087 .35982 L .38116 .34879 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39209 .55375 m .3818 .5433 L .39087 .35982 L p
.955 .688 .491 r
F P
s
P
p
.002 w
.36214 .35597 m .38116 .34879 L .39087 .35982 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36214 .35597 m .38116 .34879 L .39087 .35982 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84119 .35211 m .82038 .35927 L .81332 .34823 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84119 .35211 m .82038 .35927 L .81332 .34823 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79257 .35542 m .81332 .34823 L .82038 .35927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79257 .35542 m .81332 .34823 L .82038 .35927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32242 .50564 m .3333 .35211 L .3145 .35927 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.3133 .5231 m .30261 .51256 L .3145 .35927 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.31565 .49115 m .30509 .48053 L .3145 .35927 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.3145 .35927 m .30261 .51256 L .32242 .50564 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.3333 .35211 m .3145 .35927 L .30433 .34823 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33303 .51627 m .3133 .5231 L .30261 .51256 L p
.662 .744 .911 r
F P
s
P
p
.002 w
.30261 .51256 m .32242 .50564 L .33303 .51627 L p
.662 .744 .911 r
F P
s
P
p
.002 w
.32126 .52196 m .3333 .35211 L .3145 .35927 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.3145 .35927 m .30134 .52885 L .32126 .52196 L p
.531 .306 .518 r
F P
s
P
p
.002 w
.28561 .35542 m .30433 .34823 L .3145 .35927 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28459 .50016 m .27375 .48952 L .28561 .35542 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.30411 .49326 m .28459 .50016 L .27375 .48952 L p
.663 .742 .908 r
F P
s
P
p
.002 w
.27375 .48952 m .29335 .48253 L .30411 .49326 L p
.663 .742 .908 r
F P
s
P
p
.002 w
.30433 .34823 m .3145 .35927 L .30134 .52885 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.30433 .34823 m .3145 .35927 L .30411 .49326 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.76464 .35156 m .74412 .35872 L .73659 .34767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76464 .35156 m .74412 .35872 L .73659 .34767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71614 .35487 m .73659 .34767 L .74412 .35872 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71614 .35487 m .73659 .34767 L .74412 .35872 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24935 .42451 m .25659 .35156 L .23809 .35872 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.24498 .4055 m .23429 .39466 L .23809 .35872 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.23809 .35872 m .23039 .43157 L .24935 .42451 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.25659 .35156 m .23809 .35872 L .22744 .34767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24841 .4339 m .25659 .35156 L .23809 .35872 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.23809 .35872 m .22939 .44095 L .24841 .4339 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.20903 .35487 m .22744 .34767 L .23809 .35872 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21521 .40627 m .20429 .3954 L .20903 .35487 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.23381 .39922 m .21521 .40627 L .20429 .3954 L p
.67 .732 .896 r
F P
s
P
p
.002 w
.20429 .3954 m .22297 .38826 L .23381 .39922 L p
.67 .732 .896 r
F P
s
P
p
.002 w
.22744 .34767 m .23809 .35872 L .22939 .44095 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.22744 .34767 m .23809 .35872 L .23381 .39922 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.68804 .351 m .66782 .35817 L .65981 .34712 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68804 .351 m .66782 .35817 L .65981 .34712 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63967 .35432 m .65981 .34712 L .66782 .35817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63967 .35432 m .65981 .34712 L .66782 .35817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61139 .35045 m .59147 .35762 L .58299 .34656 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61139 .35045 m .59147 .35762 L .58299 .34656 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56314 .35377 m .58299 .34656 L .59147 .35762 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56314 .35377 m .58299 .34656 L .59147 .35762 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5347 .3499 m .51508 .35707 L .50612 .34601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5347 .3499 m .51508 .35707 L .50612 .34601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48658 .35322 m .50612 .34601 L .51508 .35707 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48658 .35322 m .50612 .34601 L .51508 .35707 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45795 .34934 m .43863 .35652 L .42921 .34545 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45795 .34934 m .43863 .35652 L .42921 .34545 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40996 .35266 m .42921 .34545 L .43863 .35652 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40996 .35266 m .42921 .34545 L .43863 .35652 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38116 .34879 m .36214 .35597 L .35224 .34489 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38116 .34879 m .36214 .35597 L .35224 .34489 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3624 .53621 m .35194 .52568 L .36214 .35597 L p
.952 .687 .497 r
F P
s
P
p
.002 w
.3333 .35211 m .35224 .34489 L .36214 .35597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3333 .35211 m .35224 .34489 L .36214 .35597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81332 .34823 m .79257 .35542 L .78532 .34433 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81332 .34823 m .79257 .35542 L .78532 .34433 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76464 .35156 m .78532 .34433 L .79257 .35542 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76464 .35156 m .78532 .34433 L .79257 .35542 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29335 .48253 m .30433 .34823 L .28561 .35542 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.28751 .46549 m .27683 .45477 L .28561 .35542 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.28561 .35542 m .27375 .48952 L .29335 .48253 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.30433 .34823 m .28561 .35542 L .27523 .34433 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29207 .49829 m .30433 .34823 L .28561 .35542 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.28855 .50465 m .26965 .51131 L .25903 .50105 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.25903 .50105 m .27801 .49431 L .28855 .50465 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.28561 .35542 m .27236 .50524 L .29207 .49829 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.25659 .35156 m .27523 .34433 L .28561 .35542 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25268 .44551 m .23406 .45235 L .22331 .4418 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.25652 .47322 m .24558 .46248 L .25659 .35156 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.27581 .46626 m .25652 .47322 L .24558 .46248 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.24558 .46248 m .26494 .45542 L .27581 .46626 L p
.665 .739 .905 r
F P
s
P
p
.002 w
.27523 .34433 m .28561 .35542 L .27236 .50524 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.27523 .34433 m .28561 .35542 L .27581 .46626 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.73659 .34767 m .71614 .35487 L .70842 .34378 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73659 .34767 m .71614 .35487 L .70842 .34378 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68804 .351 m .70842 .34378 L .71614 .35487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68804 .351 m .70842 .34378 L .71614 .35487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22297 .38826 m .22744 .34767 L .20903 .35487 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.20903 .35487 m .20429 .3954 L .22297 .38826 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.22744 .34767 m .20903 .35487 L .19817 .34378 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22232 .39416 m .22744 .34767 L .20903 .35487 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.20903 .35487 m .20361 .40129 L .22232 .39416 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.19817 .34378 m .20903 .35487 L .20361 .40129 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.65981 .34712 m .63967 .35432 L .63147 .34322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65981 .34712 m .63967 .35432 L .63147 .34322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61139 .35045 m .63147 .34322 L .63967 .35432 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61139 .35045 m .63147 .34322 L .63967 .35432 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58299 .34656 m .56314 .35377 L .55447 .34266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58299 .34656 m .56314 .35377 L .55447 .34266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5347 .3499 m .55447 .34266 L .56314 .35377 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5347 .3499 m .55447 .34266 L .56314 .35377 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50612 .34601 m .48658 .35322 L .47743 .3421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50612 .34601 m .48658 .35322 L .47743 .3421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45795 .34934 m .47743 .3421 L .48658 .35322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45795 .34934 m .47743 .3421 L .48658 .35322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42921 .34545 m .40996 .35266 L .40034 .34154 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42921 .34545 m .40996 .35266 L .40034 .34154 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38116 .34879 m .40034 .34154 L .40996 .35266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38116 .34879 m .40034 .34154 L .40996 .35266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81332 .34823 m .83423 .34098 L .84119 .35211 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81332 .34823 m .83423 .34098 L .84119 .35211 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35224 .34489 m .3333 .35211 L .3232 .34098 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35224 .34489 m .3333 .35211 L .3232 .34098 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33303 .51627 m .32242 .50564 L .3333 .35211 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.30433 .34823 m .3232 .34098 L .3333 .35211 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3232 .34098 m .3333 .35211 L .32126 .52196 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.78532 .34433 m .76464 .35156 L .7572 .34042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78532 .34433 m .76464 .35156 L .7572 .34042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73659 .34767 m .7572 .34042 L .76464 .35156 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73659 .34767 m .7572 .34042 L .76464 .35156 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26494 .45542 m .27523 .34433 L .25659 .35156 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.26012 .43535 m .24935 .42451 L .25659 .35156 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.25659 .35156 m .24558 .46248 L .26494 .45542 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.27523 .34433 m .25659 .35156 L .24601 .34042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2636 .46995 m .27523 .34433 L .25659 .35156 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.25659 .35156 m .24414 .47699 L .2636 .46995 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.22744 .34767 m .24601 .34042 L .25659 .35156 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22939 .44095 m .21836 .43009 L .22744 .34767 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.24841 .4339 m .22939 .44095 L .21836 .43009 L p
.667 .736 .902 r
F P
s
P
p
.002 w
.21836 .43009 m .23746 .42295 L .24841 .4339 L p
.667 .736 .902 r
F P
s
P
p
.002 w
.24601 .34042 m .25659 .35156 L .24414 .47699 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.24601 .34042 m .25659 .35156 L .24841 .4339 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.70842 .34378 m .68804 .351 L .68013 .33986 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70842 .34378 m .68804 .351 L .68013 .33986 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65981 .34712 m .68013 .33986 L .68804 .351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65981 .34712 m .68013 .33986 L .68804 .351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63147 .34322 m .61139 .35045 L .603 .3393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63147 .34322 m .61139 .35045 L .603 .3393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58299 .34656 m .603 .3393 L .61139 .35045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58299 .34656 m .603 .3393 L .61139 .35045 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55447 .34266 m .5347 .3499 L .52583 .33874 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55447 .34266 m .5347 .3499 L .52583 .33874 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50612 .34601 m .52583 .33874 L .5347 .3499 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50612 .34601 m .52583 .33874 L .5347 .3499 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47743 .3421 m .45795 .34934 L .44861 .33818 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47743 .3421 m .45795 .34934 L .44861 .33818 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42921 .34545 m .44861 .33818 L .45795 .34934 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42921 .34545 m .44861 .33818 L .45795 .34934 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40034 .34154 m .38116 .34879 L .37134 .33761 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40034 .34154 m .38116 .34879 L .37134 .33761 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35224 .34489 m .37134 .33761 L .38116 .34879 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35224 .34489 m .37134 .33761 L .38116 .34879 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83423 .34098 m .81332 .34823 L .80617 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.83423 .34098 m .81332 .34823 L .80617 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78532 .34433 m .80617 .33705 L .81332 .34823 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78532 .34433 m .80617 .33705 L .81332 .34823 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31044 .51124 m .3232 .34098 L .30433 .34823 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.30134 .52885 m .29044 .51822 L .30433 .34823 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.30411 .49326 m .29335 .48253 L .30433 .34823 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.30433 .34823 m .29044 .51822 L .31044 .51124 L p
.528 .302 .515 r
F P
s
P
p
.002 w
.3232 .34098 m .30433 .34823 L .29402 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3232 .34098 m .30433 .34823 L .29402 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27523 .34433 m .29402 .33705 L .30433 .34823 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29402 .33705 m .30433 .34823 L .29207 .49829 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.7572 .34042 m .73659 .34767 L .72896 .33649 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7572 .34042 m .73659 .34767 L .72896 .33649 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70842 .34378 m .72896 .33649 L .73659 .34767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70842 .34378 m .72896 .33649 L .73659 .34767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23746 .42295 m .24601 .34042 L .22744 .34767 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.23381 .39922 m .22297 .38826 L .22744 .34767 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.22744 .34767 m .21836 .43009 L .23746 .42295 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.24601 .34042 m .22744 .34767 L .21666 .33649 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23619 .43516 m .24601 .34042 L .22744 .34767 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.22744 .34767 m .21702 .44227 L .23619 .43516 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.19817 .34378 m .21666 .33649 L .22744 .34767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20361 .40129 m .19253 .39029 L .19817 .34378 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.22232 .39416 m .20361 .40129 L .19253 .39029 L p
.669 .733 .897 r
F P
s
P
p
.002 w
.19253 .39029 m .21132 .38307 L .22232 .39416 L p
.669 .733 .897 r
F P
s
P
p
.002 w
.21666 .33649 m .22744 .34767 L .21702 .44227 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.21666 .33649 m .22744 .34767 L .22232 .39416 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.68013 .33986 m .65981 .34712 L .65171 .33593 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68013 .33986 m .65981 .34712 L .65171 .33593 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63147 .34322 m .65171 .33593 L .65981 .34712 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63147 .34322 m .65171 .33593 L .65981 .34712 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.603 .3393 m .58299 .34656 L .57441 .33536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.603 .3393 m .58299 .34656 L .57441 .33536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55447 .34266 m .57441 .33536 L .58299 .34656 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55447 .34266 m .57441 .33536 L .58299 .34656 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52583 .33874 m .50612 .34601 L .49706 .3348 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52583 .33874 m .50612 .34601 L .49706 .3348 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47743 .3421 m .49706 .3348 L .50612 .34601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47743 .3421 m .49706 .3348 L .50612 .34601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44861 .33818 m .42921 .34545 L .41966 .33424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44861 .33818 m .42921 .34545 L .41966 .33424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40034 .34154 m .41966 .33424 L .42921 .34545 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40034 .34154 m .41966 .33424 L .42921 .34545 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37134 .33761 m .35224 .34489 L .34222 .33367 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37134 .33761 m .35224 .34489 L .34222 .33367 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3232 .34098 m .34222 .33367 L .35224 .34489 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3232 .34098 m .34222 .33367 L .35224 .34489 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80617 .33705 m .78532 .34433 L .77798 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80617 .33705 m .78532 .34433 L .77798 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7572 .34042 m .77798 .33311 L .78532 .34433 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7572 .34042 m .77798 .33311 L .78532 .34433 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28111 .48746 m .29402 .33705 L .27523 .34433 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.27236 .50524 m .26132 .49451 L .27523 .34433 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.27581 .46626 m .26494 .45542 L .27523 .34433 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.27523 .34433 m .26132 .49451 L .28111 .48746 L p
.526 .298 .512 r
F P
s
P
p
.002 w
.29402 .33705 m .27523 .34433 L .26472 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29402 .33705 m .27523 .34433 L .26472 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24601 .34042 m .26472 .33311 L .27523 .34433 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26472 .33311 m .27523 .34433 L .2636 .46995 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.72896 .33649 m .70842 .34378 L .7006 .33254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72896 .33649 m .70842 .34378 L .7006 .33254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68013 .33986 m .7006 .33254 L .70842 .34378 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68013 .33986 m .7006 .33254 L .70842 .34378 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21132 .38307 m .21666 .33649 L .19817 .34378 L p
.52 .29 .506 r
F P
s
P
p
.002 w
.19817 .34378 m .19253 .39029 L .21132 .38307 L p
.52 .29 .506 r
F P
s
P
p
.002 w
.21666 .33649 m .19817 .34378 L .18718 .33254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21041 .39099 m .21666 .33649 L .19817 .34378 L p
.52 .29 .506 r
F P
s
P
p
.002 w
.19817 .34378 m .19157 .3982 L .21041 .39099 L p
.52 .29 .506 r
F P
s
P
p
.002 w
.18718 .33254 m .19817 .34378 L .19157 .3982 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.65171 .33593 m .63147 .34322 L .62317 .33198 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65171 .33593 m .63147 .34322 L .62317 .33198 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.603 .3393 m .62317 .33198 L .63147 .34322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.603 .3393 m .62317 .33198 L .63147 .34322 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57441 .33536 m .55447 .34266 L .54569 .33141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57441 .33536 m .55447 .34266 L .54569 .33141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52583 .33874 m .54569 .33141 L .55447 .34266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52583 .33874 m .54569 .33141 L .55447 .34266 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49706 .3348 m .47743 .3421 L .46816 .33084 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49706 .3348 m .47743 .3421 L .46816 .33084 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44861 .33818 m .46816 .33084 L .47743 .3421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44861 .33818 m .46816 .33084 L .47743 .3421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41966 .33424 m .40034 .34154 L .39059 .33028 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41966 .33424 m .40034 .34154 L .39059 .33028 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37134 .33761 m .39059 .33028 L .40034 .34154 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37134 .33761 m .39059 .33028 L .40034 .34154 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80617 .33705 m .82718 .32971 L .83423 .34098 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80617 .33705 m .82718 .32971 L .83423 .34098 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34222 .33367 m .3232 .34098 L .31297 .32971 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34222 .33367 m .3232 .34098 L .31297 .32971 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32126 .52196 m .31044 .51124 L .3232 .34098 L p
.95 .687 .503 r
F P
s
P
p
.002 w
.29402 .33705 m .31297 .32971 L .3232 .34098 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29402 .33705 m .31297 .32971 L .3232 .34098 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77798 .33311 m .7572 .34042 L .74967 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77798 .33311 m .7572 .34042 L .74967 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72896 .33649 m .74967 .32914 L .7572 .34042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72896 .33649 m .74967 .32914 L .7572 .34042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25252 .45902 m .26472 .33311 L .24601 .34042 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.24414 .47699 m .23298 .46614 L .24601 .34042 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.24841 .4339 m .23746 .42295 L .24601 .34042 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.24601 .34042 m .23298 .46614 L .25252 .45902 L p
.523 .294 .509 r
F P
s
P
p
.002 w
.26472 .33311 m .24601 .34042 L .2353 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26472 .33311 m .24601 .34042 L .2353 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21666 .33649 m .2353 .32914 L .24601 .34042 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2353 .32914 m .24601 .34042 L .23619 .43516 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.7006 .33254 m .68013 .33986 L .67211 .32858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7006 .33254 m .68013 .33986 L .67211 .32858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65171 .33593 m .67211 .32858 L .68013 .33986 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65171 .33593 m .67211 .32858 L .68013 .33986 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62317 .33198 m .603 .3393 L .59451 .32801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62317 .33198 m .603 .3393 L .59451 .32801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57441 .33536 m .59451 .32801 L .603 .3393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57441 .33536 m .59451 .32801 L .603 .3393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54569 .33141 m .52583 .33874 L .51685 .32744 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54569 .33141 m .52583 .33874 L .51685 .32744 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49706 .3348 m .51685 .32744 L .52583 .33874 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49706 .3348 m .51685 .32744 L .52583 .33874 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46816 .33084 m .44861 .33818 L .43914 .32687 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46816 .33084 m .44861 .33818 L .43914 .32687 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41966 .33424 m .43914 .32687 L .44861 .33818 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41966 .33424 m .43914 .32687 L .44861 .33818 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39059 .33028 m .37134 .33761 L .36139 .3263 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39059 .33028 m .37134 .33761 L .36139 .3263 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34222 .33367 m .36139 .3263 L .37134 .33761 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34222 .33367 m .36139 .3263 L .37134 .33761 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82718 .32971 m .80617 .33705 L .79893 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82718 .32971 m .80617 .33705 L .79893 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77798 .33311 m .79893 .32573 L .80617 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77798 .33311 m .79893 .32573 L .80617 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31297 .32971 m .29402 .33705 L .28359 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31297 .32971 m .29402 .33705 L .28359 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29207 .49829 m .28111 .48746 L .29402 .33705 L p
.948 .687 .508 r
F P
s
P
p
.002 w
.26472 .33311 m .28359 .32573 L .29402 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26472 .33311 m .28359 .32573 L .29402 .33705 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74967 .32914 m .72896 .33649 L .72124 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74967 .32914 m .72896 .33649 L .72124 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7006 .33254 m .72124 .32516 L .72896 .33649 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7006 .33254 m .72124 .32516 L .72896 .33649 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22232 .39416 m .21132 .38307 L .21666 .33649 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.22504 .42409 m .2353 .32914 L .21666 .33649 L p
.52 .29 .506 r
F P
s
P
p
.002 w
.21702 .44227 m .20579 .4313 L .21666 .33649 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.21666 .33649 m .20579 .4313 L .22504 .42409 L p
.52 .29 .506 r
F P
s
P
p
.002 w
.2353 .32914 m .21666 .33649 L .20574 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2353 .32914 m .21666 .33649 L .20574 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18718 .33254 m .20574 .32516 L .21666 .33649 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20574 .32516 m .21666 .33649 L .21041 .39099 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.67211 .32858 m .65171 .33593 L .6435 .32459 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67211 .32858 m .65171 .33593 L .6435 .32459 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62317 .33198 m .6435 .32459 L .65171 .33593 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62317 .33198 m .6435 .32459 L .65171 .33593 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59451 .32801 m .57441 .33536 L .56572 .32402 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59451 .32801 m .57441 .33536 L .56572 .32402 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54569 .33141 m .56572 .32402 L .57441 .33536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54569 .33141 m .56572 .32402 L .57441 .33536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51685 .32744 m .49706 .3348 L .48788 .32345 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51685 .32744 m .49706 .3348 L .48788 .32345 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46816 .33084 m .48788 .32345 L .49706 .3348 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46816 .33084 m .48788 .32345 L .49706 .3348 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43914 .32687 m .41966 .33424 L .41 .32288 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43914 .32687 m .41966 .33424 L .41 .32288 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39059 .33028 m .41 .32288 L .41966 .33424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39059 .33028 m .41 .32288 L .41966 .33424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36139 .3263 m .34222 .33367 L .33207 .32231 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36139 .3263 m .34222 .33367 L .33207 .32231 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31297 .32971 m .33207 .32231 L .34222 .33367 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31297 .32971 m .33207 .32231 L .34222 .33367 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79893 .32573 m .77798 .33311 L .77055 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79893 .32573 m .77798 .33311 L .77055 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74967 .32914 m .77055 .32174 L .77798 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74967 .32914 m .77055 .32174 L .77798 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28359 .32573 m .26472 .33311 L .25408 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28359 .32573 m .26472 .33311 L .25408 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2636 .46995 m .25252 .45902 L .26472 .33311 L p
.945 .686 .513 r
F P
s
P
p
.002 w
.2353 .32914 m .25408 .32174 L .26472 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2353 .32914 m .25408 .32174 L .26472 .33311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72124 .32516 m .7006 .33254 L .69268 .32117 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72124 .32516 m .7006 .33254 L .69268 .32117 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67211 .32858 m .69268 .32117 L .7006 .33254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67211 .32858 m .69268 .32117 L .7006 .33254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19923 .37978 m .20574 .32516 L .18718 .33254 L p
.517 .285 .503 r
F P
s
P
p
.002 w
.19157 .3982 m .18032 .38709 L .18718 .33254 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.18718 .33254 m .18032 .38709 L .19923 .37978 L p
.517 .285 .503 r
F P
s
P
p
.002 w
.20574 .32516 m .18718 .33254 L .17605 .32117 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21041 .39099 m .19157 .3982 L .18032 .38709 L p
.669 .733 .898 r
F P
s
P
p
.002 w
.18032 .38709 m .19923 .37978 L .21041 .39099 L p
.669 .733 .898 r
F P
s
P
p
.002 w
.19789 .39098 m .20574 .32516 L .18718 .33254 L p
.517 .285 .503 r
F P
s
P
p
.002 w
.18718 .33254 m .17891 .39826 L .19789 .39098 L p
.517 .285 .503 r
F P
s
P
p
.002 w
.17605 .32117 m .18718 .33254 L .17891 .39826 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.6435 .32459 m .62317 .33198 L .61477 .32059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6435 .32459 m .62317 .33198 L .61477 .32059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59451 .32801 m .61477 .32059 L .62317 .33198 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59451 .32801 m .61477 .32059 L .62317 .33198 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56572 .32402 m .54569 .33141 L .5368 .32002 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56572 .32402 m .54569 .33141 L .5368 .32002 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51685 .32744 m .5368 .32002 L .54569 .33141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51685 .32744 m .5368 .32002 L .54569 .33141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48788 .32345 m .46816 .33084 L .45879 .31945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48788 .32345 m .46816 .33084 L .45879 .31945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43914 .32687 m .45879 .31945 L .46816 .33084 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43914 .32687 m .45879 .31945 L .46816 .33084 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41 .32288 m .39059 .33028 L .38072 .31887 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41 .32288 m .39059 .33028 L .38072 .31887 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36139 .3263 m .38072 .31887 L .39059 .33028 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36139 .3263 m .38072 .31887 L .39059 .33028 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79893 .32573 m .82005 .3183 L .82718 .32971 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79893 .32573 m .82005 .3183 L .82718 .32971 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33207 .32231 m .31297 .32971 L .30261 .3183 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33207 .32231 m .31297 .32971 L .30261 .3183 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28359 .32573 m .30261 .3183 L .31297 .32971 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28359 .32573 m .30261 .3183 L .31297 .32971 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77055 .32174 m .74967 .32914 L .74205 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77055 .32174 m .74967 .32914 L .74205 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72124 .32516 m .74205 .31773 L .74967 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72124 .32516 m .74205 .31773 L .74967 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25408 .32174 m .2353 .32914 L .22445 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25408 .32174 m .2353 .32914 L .22445 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23619 .43516 m .22504 .42409 L .2353 .32914 L p
.943 .686 .518 r
F P
s
P
p
.002 w
.20574 .32516 m .22445 .31773 L .2353 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20574 .32516 m .22445 .31773 L .2353 .32914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69268 .32117 m .67211 .32858 L .664 .31715 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69268 .32117 m .67211 .32858 L .664 .31715 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6435 .32459 m .664 .31715 L .67211 .32858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6435 .32459 m .664 .31715 L .67211 .32858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61477 .32059 m .59451 .32801 L .5859 .31658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61477 .32059 m .59451 .32801 L .5859 .31658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56572 .32402 m .5859 .31658 L .59451 .32801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56572 .32402 m .5859 .31658 L .59451 .32801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5368 .32002 m .51685 .32744 L .50776 .316 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5368 .32002 m .51685 .32744 L .50776 .316 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48788 .32345 m .50776 .316 L .51685 .32744 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48788 .32345 m .50776 .316 L .51685 .32744 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45879 .31945 m .43914 .32687 L .42956 .31543 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45879 .31945 m .43914 .32687 L .42956 .31543 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41 .32288 m .42956 .31543 L .43914 .32687 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41 .32288 m .42956 .31543 L .43914 .32687 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38072 .31887 m .36139 .3263 L .35132 .31485 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38072 .31887 m .36139 .3263 L .35132 .31485 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33207 .32231 m .35132 .31485 L .36139 .3263 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33207 .32231 m .35132 .31485 L .36139 .3263 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82005 .3183 m .79893 .32573 L .7916 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.82005 .3183 m .79893 .32573 L .7916 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77055 .32174 m .7916 .31427 L .79893 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77055 .32174 m .7916 .31427 L .79893 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30261 .3183 m .28359 .32573 L .27303 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30261 .3183 m .28359 .32573 L .27303 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25408 .32174 m .27303 .31427 L .28359 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25408 .32174 m .27303 .31427 L .28359 .32573 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74205 .31773 m .72124 .32516 L .71342 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74205 .31773 m .72124 .32516 L .71342 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69268 .32117 m .71342 .3137 L .72124 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69268 .32117 m .71342 .3137 L .72124 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22445 .31773 m .20574 .32516 L .19468 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22445 .31773 m .20574 .32516 L .19468 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21041 .39099 m .19923 .37978 L .20574 .32516 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.17605 .32117 m .19468 .3137 L .20574 .32516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19468 .3137 m .20574 .32516 L .19789 .39098 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.664 .31715 m .6435 .32459 L .63519 .31312 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.664 .31715 m .6435 .32459 L .63519 .31312 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61477 .32059 m .63519 .31312 L .6435 .32459 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61477 .32059 m .63519 .31312 L .6435 .32459 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5859 .31658 m .56572 .32402 L .55692 .31254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5859 .31658 m .56572 .32402 L .55692 .31254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5368 .32002 m .55692 .31254 L .56572 .32402 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5368 .32002 m .55692 .31254 L .56572 .32402 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50776 .316 m .48788 .32345 L .47859 .31196 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50776 .316 m .48788 .32345 L .47859 .31196 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45879 .31945 m .47859 .31196 L .48788 .32345 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45879 .31945 m .47859 .31196 L .48788 .32345 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42956 .31543 m .41 .32288 L .40021 .31139 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42956 .31543 m .41 .32288 L .40021 .31139 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38072 .31887 m .40021 .31139 L .41 .32288 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38072 .31887 m .40021 .31139 L .41 .32288 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35132 .31485 m .33207 .32231 L .32179 .31081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35132 .31485 m .33207 .32231 L .32179 .31081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30261 .3183 m .32179 .31081 L .33207 .32231 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30261 .3183 m .32179 .31081 L .33207 .32231 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7916 .31427 m .77055 .32174 L .76303 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7916 .31427 m .77055 .32174 L .76303 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74205 .31773 m .76303 .31023 L .77055 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74205 .31773 m .76303 .31023 L .77055 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27303 .31427 m .25408 .32174 L .24331 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27303 .31427 m .25408 .32174 L .24331 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22445 .31773 m .24331 .31023 L .25408 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22445 .31773 m .24331 .31023 L .25408 .32174 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71342 .3137 m .69268 .32117 L .68467 .30965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71342 .3137 m .69268 .32117 L .68467 .30965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.664 .31715 m .68467 .30965 L .69268 .32117 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.664 .31715 m .68467 .30965 L .69268 .32117 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18651 .37965 m .19468 .3137 L .17605 .32117 L p
.513 .281 .499 r
F P
s
P
p
.002 w
.17891 .39826 m .16746 .38703 L .17605 .32117 L p
.938 .685 .528 r
F P
s
P
p
.002 w
.17605 .32117 m .16746 .38703 L .18651 .37965 L p
.513 .281 .499 r
F P
s
P
p
.002 w
.19468 .3137 m .17605 .32117 L .16479 .30965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19468 .3137 m .17605 .32117 L .16479 .30965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63519 .31312 m .61477 .32059 L .60626 .30907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63519 .31312 m .61477 .32059 L .60626 .30907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5859 .31658 m .60626 .30907 L .61477 .32059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5859 .31658 m .60626 .30907 L .61477 .32059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55692 .31254 m .5368 .32002 L .5278 .30849 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55692 .31254 m .5368 .32002 L .5278 .30849 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50776 .316 m .5278 .30849 L .5368 .32002 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50776 .316 m .5278 .30849 L .5368 .32002 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47859 .31196 m .45879 .31945 L .44929 .30791 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47859 .31196 m .45879 .31945 L .44929 .30791 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42956 .31543 m .44929 .30791 L .45879 .31945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42956 .31543 m .44929 .30791 L .45879 .31945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40021 .31139 m .38072 .31887 L .37073 .30733 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40021 .31139 m .38072 .31887 L .37073 .30733 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35132 .31485 m .37073 .30733 L .38072 .31887 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35132 .31485 m .37073 .30733 L .38072 .31887 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7916 .31427 m .81282 .30675 L .82005 .3183 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7916 .31427 m .81282 .30675 L .82005 .3183 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32179 .31081 m .30261 .3183 L .29212 .30675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32179 .31081 m .30261 .3183 L .29212 .30675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27303 .31427 m .29212 .30675 L .30261 .3183 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27303 .31427 m .29212 .30675 L .30261 .3183 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76303 .31023 m .74205 .31773 L .73433 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76303 .31023 m .74205 .31773 L .73433 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71342 .3137 m .73433 .30616 L .74205 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71342 .3137 m .73433 .30616 L .74205 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24331 .31023 m .22445 .31773 L .21347 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24331 .31023 m .22445 .31773 L .21347 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19468 .3137 m .21347 .30616 L .22445 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19468 .3137 m .21347 .30616 L .22445 .31773 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68467 .30965 m .664 .31715 L .65579 .30558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68467 .30965 m .664 .31715 L .65579 .30558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63519 .31312 m .65579 .30558 L .664 .31715 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63519 .31312 m .65579 .30558 L .664 .31715 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60626 .30907 m .5859 .31658 L .5772 .305 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60626 .30907 m .5859 .31658 L .5772 .305 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55692 .31254 m .5772 .305 L .5859 .31658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55692 .31254 m .5772 .305 L .5859 .31658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5278 .30849 m .50776 .316 L .49855 .30442 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5278 .30849 m .50776 .316 L .49855 .30442 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47859 .31196 m .49855 .30442 L .50776 .316 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47859 .31196 m .49855 .30442 L .50776 .316 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44929 .30791 m .42956 .31543 L .41986 .30383 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44929 .30791 m .42956 .31543 L .41986 .30383 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40021 .31139 m .41986 .30383 L .42956 .31543 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40021 .31139 m .41986 .30383 L .42956 .31543 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37073 .30733 m .35132 .31485 L .34112 .30325 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37073 .30733 m .35132 .31485 L .34112 .30325 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32179 .31081 m .34112 .30325 L .35132 .31485 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32179 .31081 m .34112 .30325 L .35132 .31485 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81282 .30675 m .7916 .31427 L .78418 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81282 .30675 m .7916 .31427 L .78418 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76303 .31023 m .78418 .30267 L .7916 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76303 .31023 m .78418 .30267 L .7916 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29212 .30675 m .27303 .31427 L .26233 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29212 .30675 m .27303 .31427 L .26233 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24331 .31023 m .26233 .30267 L .27303 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24331 .31023 m .26233 .30267 L .27303 .31427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73433 .30616 m .71342 .3137 L .7055 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73433 .30616 m .71342 .3137 L .7055 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68467 .30965 m .7055 .30208 L .71342 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68467 .30965 m .7055 .30208 L .71342 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21347 .30616 m .19468 .3137 L .18349 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21347 .30616 m .19468 .3137 L .18349 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19789 .39098 m .18651 .37965 L .19468 .3137 L p
.94 .685 .523 r
F P
s
P
p
.002 w
.16479 .30965 m .18349 .30208 L .19468 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16479 .30965 m .18349 .30208 L .19468 .3137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65579 .30558 m .63519 .31312 L .62678 .3015 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65579 .30558 m .63519 .31312 L .62678 .3015 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60626 .30907 m .62678 .3015 L .63519 .31312 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60626 .30907 m .62678 .3015 L .63519 .31312 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5772 .305 m .55692 .31254 L .548 .30091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5772 .305 m .55692 .31254 L .548 .30091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5278 .30849 m .548 .30091 L .55692 .31254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5278 .30849 m .548 .30091 L .55692 .31254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56153 .90764 m .54131 .91272 L .5323 .90489 L p
.637 .771 .944 r
F P
s
P
p
.002 w
.49855 .30442 m .47859 .31196 L .46918 .30033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49855 .30442 m .47859 .31196 L .46918 .30033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44929 .30791 m .46918 .30033 L .47859 .31196 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44929 .30791 m .46918 .30033 L .47859 .31196 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41986 .30383 m .40021 .31139 L .3903 .29974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41986 .30383 m .40021 .31139 L .3903 .29974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37073 .30733 m .3903 .29974 L .40021 .31139 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37073 .30733 m .3903 .29974 L .40021 .31139 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34112 .30325 m .32179 .31081 L .31138 .29916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34112 .30325 m .32179 .31081 L .31138 .29916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29212 .30675 m .31138 .29916 L .32179 .31081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29212 .30675 m .31138 .29916 L .32179 .31081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78418 .30267 m .76303 .31023 L .75541 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78418 .30267 m .76303 .31023 L .75541 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73433 .30616 m .75541 .29857 L .76303 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73433 .30616 m .75541 .29857 L .76303 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26233 .30267 m .24331 .31023 L .2324 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26233 .30267 m .24331 .31023 L .2324 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21347 .30616 m .2324 .29857 L .24331 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21347 .30616 m .2324 .29857 L .24331 .31023 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45747 .74989 m .43765 .75562 L .42797 .74679 L p
.65 .758 .929 r
F P
s
P
p
.002 w
.7055 .30208 m .68467 .30965 L .67655 .29798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7055 .30208 m .68467 .30965 L .67655 .29798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65579 .30558 m .67655 .29798 L .68467 .30965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65579 .30558 m .67655 .29798 L .68467 .30965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18349 .30208 m .16479 .30965 L .15338 .29798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18349 .30208 m .16479 .30965 L .15338 .29798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41905 .71224 m .39938 .71813 L .38945 .70905 L p
.652 .756 .925 r
F P
s
P
p
.002 w
.62678 .3015 m .60626 .30907 L .59764 .2974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62678 .3015 m .60626 .30907 L .59764 .2974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5772 .305 m .59764 .2974 L .60626 .30907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5772 .305 m .59764 .2974 L .60626 .30907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.548 .30091 m .5278 .30849 L .51868 .29681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.548 .30091 m .5278 .30849 L .51868 .29681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49855 .30442 m .51868 .29681 L .5278 .30849 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49855 .30442 m .51868 .29681 L .5278 .30849 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46918 .30033 m .44929 .30791 L .43967 .29622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46918 .30033 m .44929 .30791 L .43967 .29622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41986 .30383 m .43967 .29622 L .44929 .30791 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41986 .30383 m .43967 .29622 L .44929 .30791 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31471 .55284 m .29545 .55939 L .28485 .54929 L p
.662 .744 .911 r
F P
s
P
p
.002 w
.3903 .29974 m .37073 .30733 L .36061 .29563 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3903 .29974 m .37073 .30733 L .36061 .29563 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34112 .30325 m .36061 .29563 L .37073 .30733 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34112 .30325 m .36061 .29563 L .37073 .30733 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78418 .30267 m .80551 .29505 L .81282 .30675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78418 .30267 m .80551 .29505 L .81282 .30675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31138 .29916 m .29212 .30675 L .2815 .29505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31138 .29916 m .29212 .30675 L .2815 .29505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26233 .30267 m .2815 .29505 L .29212 .30675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26233 .30267 m .2815 .29505 L .29212 .30675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75541 .29857 m .73433 .30616 L .72651 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75541 .29857 m .73433 .30616 L .72651 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7055 .30208 m .72651 .29446 L .73433 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7055 .30208 m .72651 .29446 L .73433 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2324 .29857 m .21347 .30616 L .20234 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2324 .29857 m .21347 .30616 L .20234 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18349 .30208 m .20234 .29446 L .21347 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18349 .30208 m .20234 .29446 L .21347 .30616 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67655 .29798 m .65579 .30558 L .64747 .29387 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67655 .29798 m .65579 .30558 L .64747 .29387 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62678 .3015 m .64747 .29387 L .65579 .30558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62678 .3015 m .64747 .29387 L .65579 .30558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59764 .2974 m .5772 .305 L .56837 .29328 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59764 .2974 m .5772 .305 L .56837 .29328 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.548 .30091 m .56837 .29328 L .5772 .305 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.548 .30091 m .56837 .29328 L .5772 .305 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5323 .90489 m .55261 .89975 L .56153 .90764 L p
.637 .771 .944 r
F P
s
P
p
.002 w
.51868 .29681 m .49855 .30442 L .48923 .29269 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51868 .29681 m .49855 .30442 L .48923 .29269 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46918 .30033 m .48923 .29269 L .49855 .30442 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46918 .30033 m .48923 .29269 L .49855 .30442 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43967 .29622 m .41986 .30383 L .41004 .2921 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43967 .29622 m .41986 .30383 L .41004 .2921 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3903 .29974 m .41004 .2921 L .41986 .30383 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3903 .29974 m .41004 .2921 L .41986 .30383 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36061 .29563 m .34112 .30325 L .33079 .29151 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36061 .29563 m .34112 .30325 L .33079 .29151 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31138 .29916 m .33079 .29151 L .34112 .30325 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31138 .29916 m .33079 .29151 L .34112 .30325 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80551 .29505 m .78418 .30267 L .77666 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.80551 .29505 m .78418 .30267 L .77666 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75541 .29857 m .77666 .29091 L .78418 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75541 .29857 m .77666 .29091 L .78418 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2815 .29505 m .26233 .30267 L .2515 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2815 .29505 m .26233 .30267 L .2515 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2324 .29857 m .2515 .29091 L .26233 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2324 .29857 m .2515 .29091 L .26233 .30267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42797 .74679 m .44787 .74098 L .45747 .74989 L p
.65 .758 .929 r
F P
s
P
p
.002 w
.72651 .29446 m .7055 .30208 L .69749 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72651 .29446 m .7055 .30208 L .69749 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67655 .29798 m .69749 .29032 L .7055 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67655 .29798 m .69749 .29032 L .7055 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20234 .29446 m .18349 .30208 L .17215 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20234 .29446 m .18349 .30208 L .17215 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.15338 .29798 m .17215 .29032 L .18349 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.15338 .29798 m .17215 .29032 L .18349 .30208 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38945 .70905 m .4092 .70308 L .41905 .71224 L p
.652 .756 .925 r
F P
s
P
p
.002 w
.64747 .29387 m .62678 .3015 L .61826 .28973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64747 .29387 m .62678 .3015 L .61826 .28973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59764 .2974 m .61826 .28973 L .62678 .3015 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59764 .2974 m .61826 .28973 L .62678 .3015 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56837 .29328 m .548 .30091 L .53898 .28914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56837 .29328 m .548 .30091 L .53898 .28914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51868 .29681 m .53898 .28914 L .548 .30091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51868 .29681 m .53898 .28914 L .548 .30091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48923 .29269 m .46918 .30033 L .45965 .28855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48923 .29269 m .46918 .30033 L .45965 .28855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43967 .29622 m .45965 .28855 L .46918 .30033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43967 .29622 m .45965 .28855 L .46918 .30033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28485 .54929 m .30419 .54266 L .31471 .55284 L p
.662 .744 .911 r
F P
s
P
p
.002 w
.41004 .2921 m .3903 .29974 L .38027 .28795 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41004 .2921 m .3903 .29974 L .38027 .28795 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36061 .29563 m .38027 .28795 L .3903 .29974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36061 .29563 m .38027 .28795 L .3903 .29974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33079 .29151 m .31138 .29916 L .30084 .28736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33079 .29151 m .31138 .29916 L .30084 .28736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2815 .29505 m .30084 .28736 L .31138 .29916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2815 .29505 m .30084 .28736 L .31138 .29916 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77666 .29091 m .75541 .29857 L .74769 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77666 .29091 m .75541 .29857 L .74769 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72651 .29446 m .74769 .28677 L .75541 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72651 .29446 m .74769 .28677 L .75541 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2515 .29091 m .2324 .29857 L .22136 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2515 .29091 m .2324 .29857 L .22136 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20234 .29446 m .22136 .28677 L .2324 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20234 .29446 m .22136 .28677 L .2324 .29857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69749 .29032 m .67655 .29798 L .66833 .28617 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69749 .29032 m .67655 .29798 L .66833 .28617 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64747 .29387 m .66833 .28617 L .67655 .29798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64747 .29387 m .66833 .28617 L .67655 .29798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17215 .29032 m .15338 .29798 L .14182 .28617 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17215 .29032 m .15338 .29798 L .14182 .28617 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61826 .28973 m .59764 .2974 L .58892 .28558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61826 .28973 m .59764 .2974 L .58892 .28558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56837 .29328 m .58892 .28558 L .59764 .2974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56837 .29328 m .58892 .28558 L .59764 .2974 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53898 .28914 m .51868 .29681 L .50945 .28498 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53898 .28914 m .51868 .29681 L .50945 .28498 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48923 .29269 m .50945 .28498 L .51868 .29681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48923 .29269 m .50945 .28498 L .51868 .29681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45965 .28855 m .43967 .29622 L .42993 .28439 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45965 .28855 m .43967 .29622 L .42993 .28439 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41004 .2921 m .42993 .28439 L .43967 .29622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41004 .2921 m .42993 .28439 L .43967 .29622 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38027 .28795 m .36061 .29563 L .35037 .28379 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38027 .28795 m .36061 .29563 L .35037 .28379 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33079 .29151 m .35037 .28379 L .36061 .29563 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33079 .29151 m .35037 .28379 L .36061 .29563 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77666 .29091 m .7981 .28319 L .80551 .29505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77666 .29091 m .7981 .28319 L .80551 .29505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30084 .28736 m .2815 .29505 L .27075 .28319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30084 .28736 m .2815 .29505 L .27075 .28319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2515 .29091 m .27075 .28319 L .2815 .29505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2515 .29091 m .27075 .28319 L .2815 .29505 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74769 .28677 m .72651 .29446 L .7186 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74769 .28677 m .72651 .29446 L .7186 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69749 .29032 m .7186 .2826 L .72651 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69749 .29032 m .7186 .2826 L .72651 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22136 .28677 m .20234 .29446 L .19108 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22136 .28677 m .20234 .29446 L .19108 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17215 .29032 m .19108 .2826 L .20234 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17215 .29032 m .19108 .2826 L .20234 .29446 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66833 .28617 m .64747 .29387 L .63904 .282 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66833 .28617 m .64747 .29387 L .63904 .282 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61826 .28973 m .63904 .282 L .64747 .29387 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61826 .28973 m .63904 .282 L .64747 .29387 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58892 .28558 m .56837 .29328 L .55944 .2814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58892 .28558 m .56837 .29328 L .55944 .2814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53898 .28914 m .55944 .2814 L .56837 .29328 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53898 .28914 m .55944 .2814 L .56837 .29328 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50945 .28498 m .48923 .29269 L .47979 .28081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50945 .28498 m .48923 .29269 L .47979 .28081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45965 .28855 m .47979 .28081 L .48923 .29269 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45965 .28855 m .47979 .28081 L .48923 .29269 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48639 .7891 m .46627 .79473 L .45668 .78605 L p
.646 .763 .934 r
F P
s
P
p
.002 w
.42993 .28439 m .41004 .2921 L .40009 .28021 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42993 .28439 m .41004 .2921 L .40009 .28021 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38027 .28795 m .40009 .28021 L .41004 .2921 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38027 .28795 m .40009 .28021 L .41004 .2921 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35037 .28379 m .33079 .29151 L .32033 .27961 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35037 .28379 m .33079 .29151 L .32033 .27961 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30084 .28736 m .32033 .27961 L .33079 .29151 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30084 .28736 m .32033 .27961 L .33079 .29151 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27075 .28319 m .2515 .29091 L .24052 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27075 .28319 m .2515 .29091 L .24052 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22136 .28677 m .24052 .27901 L .2515 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22136 .28677 m .24052 .27901 L .2515 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7981 .28319 m .77666 .29091 L .76905 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7981 .28319 m .77666 .29091 L .76905 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74769 .28677 m .76905 .27901 L .77666 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74769 .28677 m .76905 .27901 L .77666 .29091 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7186 .2826 m .69749 .29032 L .68937 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7186 .2826 m .69749 .29032 L .68937 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66833 .28617 m .68937 .27841 L .69749 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66833 .28617 m .68937 .27841 L .69749 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19108 .2826 m .17215 .29032 L .16067 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19108 .2826 m .17215 .29032 L .16067 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.14182 .28617 m .16067 .27841 L .17215 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.14182 .28617 m .16067 .27841 L .17215 .29032 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63904 .282 m .61826 .28973 L .60963 .27781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63904 .282 m .61826 .28973 L .60963 .27781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58892 .28558 m .60963 .27781 L .61826 .28973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58892 .28558 m .60963 .27781 L .61826 .28973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55944 .2814 m .53898 .28914 L .52984 .27721 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55944 .2814 m .53898 .28914 L .52984 .27721 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50945 .28498 m .52984 .27721 L .53898 .28914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50945 .28498 m .52984 .27721 L .53898 .28914 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47979 .28081 m .45965 .28855 L .45 .27661 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47979 .28081 m .45965 .28855 L .45 .27661 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42993 .28439 m .45 .27661 L .45965 .28855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42993 .28439 m .45 .27661 L .45965 .28855 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19789 .39098 m .17891 .39826 L .16746 .38703 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.40009 .28021 m .38027 .28795 L .3701 .27601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40009 .28021 m .38027 .28795 L .3701 .27601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35037 .28379 m .3701 .27601 L .38027 .28795 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35037 .28379 m .3701 .27601 L .38027 .28795 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32033 .27961 m .30084 .28736 L .29016 .27541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32033 .27961 m .30084 .28736 L .29016 .27541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27075 .28319 m .29016 .27541 L .30084 .28736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27075 .28319 m .29016 .27541 L .30084 .28736 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34199 .59568 m .3224 .60212 L .31186 .59219 L p
.658 .748 .916 r
F P
s
P
p
.002 w
.76905 .27901 m .74769 .28677 L .73988 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76905 .27901 m .74769 .28677 L .73988 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7186 .2826 m .73988 .27481 L .74769 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.7186 .2826 m .73988 .27481 L .74769 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24052 .27901 m .22136 .28677 L .21017 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24052 .27901 m .22136 .28677 L .21017 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19108 .2826 m .21017 .27481 L .22136 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19108 .2826 m .21017 .27481 L .22136 .28677 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68937 .27841 m .66833 .28617 L .66 .27421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68937 .27841 m .66833 .28617 L .66 .27421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63904 .282 m .66 .27421 L .66833 .28617 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63904 .282 m .66 .27421 L .66833 .28617 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16067 .27841 m .14182 .28617 L .13012 .27421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16067 .27841 m .14182 .28617 L .13012 .27421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.87795 .70286 m .85613 .70886 L .849 .69961 L p
.652 .756 .926 r
F P
s
P
p
.002 w
.60963 .27781 m .58892 .28558 L .58008 .27361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60963 .27781 m .58892 .28558 L .58008 .27361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55944 .2814 m .58008 .27361 L .58892 .28558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55944 .2814 m .58008 .27361 L .58892 .28558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52984 .27721 m .50945 .28498 L .5001 .273 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52984 .27721 m .50945 .28498 L .5001 .273 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47979 .28081 m .5001 .273 L .50945 .28498 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47979 .28081 m .5001 .273 L .50945 .28498 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45668 .78605 m .47689 .78036 L .48639 .7891 L p
.646 .763 .934 r
F P
s
P
p
.002 w
.23619 .43516 m .21702 .44227 L .20579 .4313 L p
.666 .737 .903 r
F P
s
P
p
.002 w
.45 .27661 m .42993 .28439 L .42007 .2724 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45 .27661 m .42993 .28439 L .42007 .2724 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40009 .28021 m .42007 .2724 L .42993 .28439 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40009 .28021 m .42007 .2724 L .42993 .28439 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3701 .27601 m .35037 .28379 L .33999 .2718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3701 .27601 m .35037 .28379 L .33999 .2718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32033 .27961 m .33999 .2718 L .35037 .28379 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32033 .27961 m .33999 .2718 L .35037 .28379 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.84868 .69658 m .82695 .70262 L .81963 .69331 L p
.652 .756 .926 r
F P
s
P
p
.002 w
.76905 .27901 m .79059 .27119 L .7981 .28319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76905 .27901 m .79059 .27119 L .7981 .28319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29016 .27541 m .27075 .28319 L .25985 .27119 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29016 .27541 m .27075 .28319 L .25985 .27119 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24052 .27901 m .25985 .27119 L .27075 .28319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24052 .27901 m .25985 .27119 L .27075 .28319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73988 .27481 m .7186 .2826 L .71058 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73988 .27481 m .7186 .2826 L .71058 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68937 .27841 m .71058 .27059 L .7186 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68937 .27841 m .71058 .27059 L .7186 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21017 .27481 m .19108 .2826 L .17967 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21017 .27481 m .19108 .2826 L .17967 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16067 .27841 m .17967 .27059 L .19108 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16067 .27841 m .17967 .27059 L .19108 .2826 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66 .27421 m .63904 .282 L .63051 .26998 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66 .27421 m .63904 .282 L .63051 .26998 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60963 .27781 m .63051 .26998 L .63904 .282 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60963 .27781 m .63051 .26998 L .63904 .282 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58008 .27361 m .55944 .2814 L .55039 .26938 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58008 .27361 m .55944 .2814 L .55039 .26938 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52984 .27721 m .55039 .26938 L .55944 .2814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52984 .27721 m .55039 .26938 L .55944 .2814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81932 .69007 m .79769 .69614 L .79017 .68678 L p
.652 .756 .925 r
F P
s
P
p
.002 w
.5001 .273 m .47979 .28081 L .47023 .26877 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5001 .273 m .47979 .28081 L .47023 .26877 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45 .27661 m .47023 .26877 L .47979 .28081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45 .27661 m .47023 .26877 L .47979 .28081 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16746 .38703 m .18651 .37965 L .19789 .39098 L p
.668 .734 .9 r
F P
s
P
p
.002 w
.42007 .2724 m .40009 .28021 L .39001 .26817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42007 .2724 m .40009 .28021 L .39001 .26817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3701 .27601 m .39001 .26817 L .40009 .28021 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3701 .27601 m .39001 .26817 L .40009 .28021 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33999 .2718 m .32033 .27961 L .30974 .26756 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33999 .2718 m .32033 .27961 L .30974 .26756 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29016 .27541 m .30974 .26756 L .32033 .27961 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29016 .27541 m .30974 .26756 L .32033 .27961 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31186 .59219 m .33153 .58566 L .34199 .59568 L p
.658 .748 .916 r
F P
s
P
p
.002 w
.78987 .68331 m .76834 .68941 L .76062 .68 L p
.653 .755 .925 r
F P
s
P
p
.002 w
.79059 .27119 m .76905 .27901 L .76134 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79059 .27119 m .76905 .27901 L .76134 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73988 .27481 m .76134 .26695 L .76905 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73988 .27481 m .76134 .26695 L .76905 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25985 .27119 m .24052 .27901 L .22941 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25985 .27119 m .24052 .27901 L .22941 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21017 .27481 m .22941 .26695 L .24052 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21017 .27481 m .22941 .26695 L .24052 .27901 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71058 .27059 m .68937 .27841 L .68114 .26635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71058 .27059 m .68937 .27841 L .68114 .26635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66 .27421 m .68114 .26635 L .68937 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66 .27421 m .68114 .26635 L .68937 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17967 .27059 m .16067 .27841 L .14904 .26635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17967 .27059 m .16067 .27841 L .14904 .26635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.13012 .27421 m .14904 .26635 L .16067 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.13012 .27421 m .14904 .26635 L .16067 .27841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.849 .69961 m .87094 .69353 L .87795 .70286 L p
.652 .756 .926 r
F P
s
P
p
.002 w
.63051 .26998 m .60963 .27781 L .60088 .26574 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63051 .26998 m .60963 .27781 L .60088 .26574 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58008 .27361 m .60088 .26574 L .60963 .27781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58008 .27361 m .60088 .26574 L .60963 .27781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55039 .26938 m .52984 .27721 L .52058 .26513 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55039 .26938 m .52984 .27721 L .52058 .26513 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5001 .273 m .52058 .26513 L .52984 .27721 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5001 .273 m .52058 .26513 L .52984 .27721 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76033 .67626 m .7389 .6824 L .73099 .67293 L p
.653 .755 .924 r
F P
s
P
p
.002 w
.51588 .82594 m .49546 .83146 L .48598 .82295 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.20579 .4313 m .22504 .42409 L .23619 .43516 L p
.666 .737 .903 r
F P
s
P
p
.002 w
.47023 .26877 m .45 .27661 L .44022 .26453 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47023 .26877 m .45 .27661 L .44022 .26453 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42007 .2724 m .44022 .26453 L .45 .27661 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42007 .2724 m .44022 .26453 L .45 .27661 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40986 .66172 m .38988 .66792 L .37971 .65836 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.39001 .26817 m .3701 .27601 L .35981 .26392 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39001 .26817 m .3701 .27601 L .35981 .26392 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33999 .2718 m .35981 .26392 L .3701 .27601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33999 .2718 m .35981 .26392 L .3701 .27601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.81963 .69331 m .84147 .6872 L .84868 .69658 L p
.652 .756 .926 r
F P
s
P
p
.002 w
.73071 .6689 m .70938 .67508 L .70128 .66555 L p
.653 .754 .924 r
F P
s
P
p
.002 w
.30974 .26756 m .29016 .27541 L .27935 .26331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30974 .26756 m .29016 .27541 L .27935 .26331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25985 .27119 m .27935 .26331 L .29016 .27541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25985 .27119 m .27935 .26331 L .29016 .27541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76134 .26695 m .73988 .27481 L .73196 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76134 .26695 m .73988 .27481 L .73196 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71058 .27059 m .73196 .2627 L .73988 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71058 .27059 m .73196 .2627 L .73988 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22941 .26695 m .21017 .27481 L .19883 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22941 .26695 m .21017 .27481 L .19883 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17967 .27059 m .19883 .2627 L .21017 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.17967 .27059 m .19883 .2627 L .21017 .27481 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68114 .26635 m .66 .27421 L .65157 .26209 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68114 .26635 m .66 .27421 L .65157 .26209 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63051 .26998 m .65157 .26209 L .66 .27421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63051 .26998 m .65157 .26209 L .66 .27421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70102 .66119 m .6798 .66741 L .67149 .65783 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.60088 .26574 m .58008 .27361 L .57112 .26148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60088 .26574 m .58008 .27361 L .57112 .26148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55039 .26938 m .57112 .26148 L .58008 .27361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55039 .26938 m .57112 .26148 L .58008 .27361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.79017 .68678 m .81191 .68063 L .81932 .69007 L p
.652 .756 .925 r
F P
s
P
p
.002 w
.52058 .26513 m .5001 .273 L .49063 .26087 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52058 .26513 m .5001 .273 L .49063 .26087 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47023 .26877 m .49063 .26087 L .5001 .273 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47023 .26877 m .49063 .26087 L .5001 .273 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67126 .6531 m .65015 .65935 L .64165 .64971 L p
.654 .753 .923 r
F P
s
P
p
.002 w
.44022 .26453 m .42007 .2724 L .41008 .26026 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44022 .26453 m .42007 .2724 L .41008 .26026 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39001 .26817 m .41008 .26026 L .42007 .2724 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39001 .26817 m .41008 .26026 L .42007 .2724 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35981 .26392 m .33999 .2718 L .32947 .25965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35981 .26392 m .33999 .2718 L .32947 .25965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30974 .26756 m .32947 .25965 L .33999 .2718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30974 .26756 m .32947 .25965 L .33999 .2718 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58495 .88427 m .56418 .88956 L .55512 .8814 L p
.636 .771 .945 r
F P
s
P
p
.002 w
.76062 .68 m .78226 .67381 L .78987 .68331 L p
.653 .755 .925 r
F P
s
P
p
.002 w
.76134 .26695 m .78299 .25903 L .79059 .27119 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76134 .26695 m .78299 .25903 L .79059 .27119 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27935 .26331 m .25985 .27119 L .24882 .25903 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27935 .26331 m .25985 .27119 L .24882 .25903 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22941 .26695 m .24882 .25903 L .25985 .27119 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22941 .26695 m .24882 .25903 L .25985 .27119 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19883 .2627 m .17967 .27059 L .16811 .25842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19883 .2627 m .17967 .27059 L .16811 .25842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.14904 .26635 m .16811 .25842 L .17967 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.14904 .26635 m .16811 .25842 L .17967 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73196 .2627 m .71058 .27059 L .70245 .25842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73196 .2627 m .71058 .27059 L .70245 .25842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68114 .26635 m .70245 .25842 L .71058 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68114 .26635 m .70245 .25842 L .71058 .27059 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65157 .26209 m .63051 .26998 L .62187 .25781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65157 .26209 m .63051 .26998 L .62187 .25781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60088 .26574 m .62187 .25781 L .63051 .26998 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60088 .26574 m .62187 .25781 L .63051 .26998 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57112 .26148 m .55039 .26938 L .54123 .2572 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57112 .26148 m .55039 .26938 L .54123 .2572 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52058 .26513 m .54123 .2572 L .55039 .26938 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52058 .26513 m .54123 .2572 L .55039 .26938 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73099 .67293 m .75252 .66671 L .76033 .67626 L p
.653 .755 .924 r
F P
s
P
p
.002 w
.37017 .63405 m .35028 .6404 L .33982 .63062 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.48598 .82295 m .50649 .81736 L .51588 .82594 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.49063 .26087 m .47023 .26877 L .46054 .25658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49063 .26087 m .47023 .26877 L .46054 .25658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44022 .26453 m .46054 .25658 L .47023 .26877 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44022 .26453 m .46054 .25658 L .47023 .26877 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37971 .65836 m .39978 .65207 L .40986 .66172 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.41008 .26026 m .39001 .26817 L .3798 .25597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41008 .26026 m .39001 .26817 L .3798 .25597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35981 .26392 m .3798 .25597 L .39001 .26817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35981 .26392 m .3798 .25597 L .39001 .26817 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2636 .46995 m .24414 .47699 L .23298 .46614 L p
.664 .741 .907 r
F P
s
P
p
.002 w
.70128 .66555 m .72271 .6593 L .73071 .6689 L p
.653 .754 .924 r
F P
s
P
p
.002 w
.32947 .25965 m .30974 .26756 L .299 .25536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32947 .25965 m .30974 .26756 L .299 .25536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27935 .26331 m .299 .25536 L .30974 .26756 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27935 .26331 m .299 .25536 L .30974 .26756 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78299 .25903 m .76134 .26695 L .75353 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.78299 .25903 m .76134 .26695 L .75353 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73196 .2627 m .75353 .25474 L .76134 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73196 .2627 m .75353 .25474 L .76134 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24882 .25903 m .22941 .26695 L .21816 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24882 .25903 m .22941 .26695 L .21816 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19883 .2627 m .21816 .25474 L .22941 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.19883 .2627 m .21816 .25474 L .22941 .26695 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70245 .25842 m .68114 .26635 L .67281 .25413 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70245 .25842 m .68114 .26635 L .67281 .25413 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65157 .26209 m .67281 .25413 L .68114 .26635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65157 .26209 m .67281 .25413 L .68114 .26635 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67149 .65783 m .69282 .65153 L .70102 .66119 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.62187 .25781 m .60088 .26574 L .59203 .25351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62187 .25781 m .60088 .26574 L .59203 .25351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57112 .26148 m .59203 .25351 L .60088 .26574 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57112 .26148 m .59203 .25351 L .60088 .26574 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54123 .2572 m .52058 .26513 L .5112 .2529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54123 .2572 m .52058 .26513 L .5112 .2529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49063 .26087 m .5112 .2529 L .52058 .26513 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49063 .26087 m .5112 .2529 L .52058 .26513 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64165 .64971 m .66286 .64337 L .67126 .6531 L p
.654 .753 .923 r
F P
s
P
p
.002 w
.46054 .25658 m .44022 .26453 L .43032 .25228 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46054 .25658 m .44022 .26453 L .43032 .25228 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41008 .26026 m .43032 .25228 L .44022 .26453 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41008 .26026 m .43032 .25228 L .44022 .26453 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3798 .25597 m .35981 .26392 L .34938 .25166 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3798 .25597 m .35981 .26392 L .34938 .25166 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32947 .25965 m .34938 .25166 L .35981 .26392 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32947 .25965 m .34938 .25166 L .35981 .26392 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43911 .69208 m .41888 .69819 L .40883 .68876 L p
.651 .757 .927 r
F P
s
P
p
.002 w
.3624 .53621 m .34248 .54298 L .33194 .53254 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.55512 .8814 m .57598 .87604 L .58495 .88427 L p
.636 .771 .945 r
F P
s
P
p
.002 w
.299 .25536 m .27935 .26331 L .26839 .25105 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.299 .25536 m .27935 .26331 L .26839 .25105 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24882 .25903 m .26839 .25105 L .27935 .26331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24882 .25903 m .26839 .25105 L .27935 .26331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75353 .25474 m .73196 .2627 L .72395 .25043 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75353 .25474 m .73196 .2627 L .72395 .25043 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70245 .25842 m .72395 .25043 L .73196 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70245 .25842 m .72395 .25043 L .73196 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21816 .25474 m .19883 .2627 L .18735 .25043 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21816 .25474 m .19883 .2627 L .18735 .25043 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16811 .25842 m .18735 .25043 L .19883 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.16811 .25842 m .18735 .25043 L .19883 .2627 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54585 .86063 m .52515 .86605 L .51579 .8577 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.67281 .25413 m .65157 .26209 L .64303 .24981 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67281 .25413 m .65157 .26209 L .64303 .24981 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62187 .25781 m .64303 .24981 L .65157 .26209 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62187 .25781 m .64303 .24981 L .65157 .26209 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59203 .25351 m .57112 .26148 L .56206 .24919 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59203 .25351 m .57112 .26148 L .56206 .24919 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54123 .2572 m .56206 .24919 L .57112 .26148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54123 .2572 m .56206 .24919 L .57112 .26148 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33982 .63062 m .3598 .62419 L .37017 .63405 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.5112 .2529 m .49063 .26087 L .48103 .24858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5112 .2529 m .49063 .26087 L .48103 .24858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46054 .25658 m .48103 .24858 L .49063 .26087 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46054 .25658 m .48103 .24858 L .49063 .26087 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43032 .25228 m .41008 .26026 L .39995 .24796 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43032 .25228 m .41008 .26026 L .39995 .24796 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3798 .25597 m .39995 .24796 L .41008 .26026 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3798 .25597 m .39995 .24796 L .41008 .26026 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23298 .46614 m .25252 .45902 L .2636 .46995 L p
.664 .741 .907 r
F P
s
P
p
.002 w
.6154 .91438 m .59438 .91958 L .58545 .91157 L p
.632 .775 .949 r
F P
s
P
p
.002 w
.34938 .25166 m .32947 .25965 L .31882 .24734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34938 .25166 m .32947 .25965 L .31882 .24734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.299 .25536 m .31882 .24734 L .32947 .25965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.299 .25536 m .31882 .24734 L .32947 .25965 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75353 .25474 m .77529 .24672 L .78299 .25903 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75353 .25474 m .77529 .24672 L .78299 .25903 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26839 .25105 m .24882 .25903 L .23764 .24672 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26839 .25105 m .24882 .25903 L .23764 .24672 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21816 .25474 m .23764 .24672 L .24882 .25903 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.21816 .25474 m .23764 .24672 L .24882 .25903 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39209 .55375 m .372 .56046 L .36162 .55011 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.72395 .25043 m .70245 .25842 L .69422 .2461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72395 .25043 m .70245 .25842 L .69422 .2461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67281 .25413 m .69422 .2461 L .70245 .25842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67281 .25413 m .69422 .2461 L .70245 .25842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64303 .24981 m .62187 .25781 L .61311 .24548 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64303 .24981 m .62187 .25781 L .61311 .24548 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59203 .25351 m .61311 .24548 L .62187 .25781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59203 .25351 m .61311 .24548 L .62187 .25781 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56206 .24919 m .54123 .2572 L .53195 .24486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56206 .24919 m .54123 .2572 L .53195 .24486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5112 .2529 m .53195 .24486 L .54123 .2572 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5112 .2529 m .53195 .24486 L .54123 .2572 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48103 .24858 m .46054 .25658 L .45073 .24424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48103 .24858 m .46054 .25658 L .45073 .24424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43032 .25228 m .45073 .24424 L .46054 .25658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43032 .25228 m .45073 .24424 L .46054 .25658 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29207 .49829 m .27236 .50524 L .26132 .49451 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.39995 .24796 m .3798 .25597 L .36945 .24361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39995 .24796 m .3798 .25597 L .36945 .24361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34938 .25166 m .36945 .24361 L .3798 .25597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34938 .25166 m .36945 .24361 L .3798 .25597 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.40883 .68876 m .42916 .68256 L .43911 .69208 L p
.651 .757 .927 r
F P
s
P
p
.002 w
.33194 .53254 m .35194 .52568 L .3624 .53621 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.422 .56936 m .40174 .57603 L .39154 .56575 L p
.658 .748 .916 r
F P
s
P
p
.002 w
.31882 .24734 m .299 .25536 L .28813 .24299 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31882 .24734 m .299 .25536 L .28813 .24299 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26839 .25105 m .28813 .24299 L .299 .25536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26839 .25105 m .28813 .24299 L .299 .25536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77529 .24672 m .75353 .25474 L .74562 .24237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.77529 .24672 m .75353 .25474 L .74562 .24237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72395 .25043 m .74562 .24237 L .75353 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72395 .25043 m .74562 .24237 L .75353 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23764 .24672 m .21816 .25474 L .20675 .24237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23764 .24672 m .21816 .25474 L .20675 .24237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18735 .25043 m .20675 .24237 L .21816 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.18735 .25043 m .20675 .24237 L .21816 .25474 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51579 .8577 m .53659 .85221 L .54585 .86063 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.69422 .2461 m .67281 .25413 L .66437 .24175 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69422 .2461 m .67281 .25413 L .66437 .24175 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64303 .24981 m .66437 .24175 L .67281 .25413 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64303 .24981 m .66437 .24175 L .67281 .25413 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61311 .24548 m .59203 .25351 L .58306 .24112 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61311 .24548 m .59203 .25351 L .58306 .24112 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56206 .24919 m .58306 .24112 L .59203 .25351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56206 .24919 m .58306 .24112 L .59203 .25351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45206 .58342 m .43165 .59004 L .42163 .57984 L p
.657 .749 .918 r
F P
s
P
p
.002 w
.46881 .71993 m .44833 .72596 L .43842 .71666 L p
.649 .759 .93 r
F P
s
P
p
.002 w
.53195 .24486 m .5112 .2529 L .5017 .2405 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53195 .24486 m .5112 .2529 L .5017 .2405 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48103 .24858 m .5017 .2405 L .5112 .2529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48103 .24858 m .5017 .2405 L .5112 .2529 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45073 .24424 m .43032 .25228 L .42028 .23988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45073 .24424 m .43032 .25228 L .42028 .23988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39995 .24796 m .42028 .23988 L .43032 .25228 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39995 .24796 m .42028 .23988 L .43032 .25228 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58545 .91157 m .60657 .9063 L .6154 .91438 L p
.632 .775 .949 r
F P
s
P
p
.002 w
.63328 .64693 m .6121 .65328 L .60325 .64349 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.36945 .24361 m .34938 .25166 L .33881 .23925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36945 .24361 m .34938 .25166 L .33881 .23925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31882 .24734 m .33881 .23925 L .34938 .25166 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31882 .24734 m .33881 .23925 L .34938 .25166 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48221 .5962 m .46166 .60276 L .45183 .59264 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.56883 .78313 m .54791 .7889 L .53864 .78 L p
.644 .764 .936 r
F P
s
P
p
.002 w
.28813 .24299 m .26839 .25105 L .25729 .23863 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28813 .24299 m .26839 .25105 L .25729 .23863 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23764 .24672 m .25729 .23863 L .26839 .25105 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.23764 .24672 m .25729 .23863 L .26839 .25105 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36162 .55011 m .3818 .5433 L .39209 .55375 L p
.659 .747 .915 r
F P
s
P
p
.002 w
.60311 .63812 m .58204 .64451 L .57299 .63466 L p
.654 .753 .923 r
F P
s
P
p
.002 w
.51243 .60791 m .49174 .61442 L .48209 .60438 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.74562 .24237 m .72395 .25043 L .71582 .238 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74562 .24237 m .72395 .25043 L .71582 .238 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69422 .2461 m .71582 .238 L .72395 .25043 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69422 .2461 m .71582 .238 L .72395 .25043 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5729 .62875 m .55195 .63518 L .5427 .62526 L p
.655 .753 .922 r
F P
s
P
p
.002 w
.54266 .61871 m .52184 .62518 L .51239 .6152 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.66437 .24175 m .64303 .24981 L .63437 .23738 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66437 .24175 m .64303 .24981 L .63437 .23738 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61311 .24548 m .63437 .23738 L .64303 .24981 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61311 .24548 m .63437 .23738 L .64303 .24981 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64615 .94295 m .62489 .94806 L .61611 .94018 L p
.629 .778 .952 r
F P
s
P
p
.002 w
.58306 .24112 m .56206 .24919 L .55287 .23675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58306 .24112 m .56206 .24919 L .55287 .23675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53195 .24486 m .55287 .23675 L .56206 .24919 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53195 .24486 m .55287 .23675 L .56206 .24919 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5017 .2405 m .48103 .24858 L .47131 .23612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5017 .2405 m .48103 .24858 L .47131 .23612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45073 .24424 m .47131 .23612 L .48103 .24858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45073 .24424 m .47131 .23612 L .48103 .24858 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26132 .49451 m .28111 .48746 L .29207 .49829 L p
.662 .743 .91 r
F P
s
P
p
.002 w
.42028 .23988 m .39995 .24796 L .3897 .2355 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42028 .23988 m .39995 .24796 L .3897 .2355 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36945 .24361 m .3897 .2355 L .39995 .24796 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36945 .24361 m .3897 .2355 L .39995 .24796 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39154 .56575 m .41188 .55901 L .422 .56936 L p
.658 .748 .916 r
F P
s
P
p
.002 w
.33881 .23925 m .31882 .24734 L .30804 .23487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33881 .23925 m .31882 .24734 L .30804 .23487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28813 .24299 m .30804 .23487 L .31882 .24734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28813 .24299 m .30804 .23487 L .31882 .24734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32126 .52196 m .30134 .52885 L .29044 .51822 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.74562 .24237 m .76748 .23424 L .77529 .24672 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74562 .24237 m .76748 .23424 L .77529 .24672 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25729 .23863 m .23764 .24672 L .22632 .23424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25729 .23863 m .23764 .24672 L .22632 .23424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20675 .24237 m .22632 .23424 L .23764 .24672 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.20675 .24237 m .22632 .23424 L .23764 .24672 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71582 .238 m .69422 .2461 L .68589 .23361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71582 .238 m .69422 .2461 L .68589 .23361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66437 .24175 m .68589 .23361 L .69422 .2461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66437 .24175 m .68589 .23361 L .69422 .2461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63437 .23738 m .61311 .24548 L .60424 .23298 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63437 .23738 m .61311 .24548 L .60424 .23298 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58306 .24112 m .60424 .23298 L .61311 .24548 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58306 .24112 m .60424 .23298 L .61311 .24548 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59929 .8034 m .57817 .80911 L .56907 .80031 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.42163 .57984 m .44213 .57314 L .45206 .58342 L p
.657 .749 .918 r
F P
s
P
p
.002 w
.43842 .71666 m .45899 .71054 L .46881 .71993 L p
.649 .759 .93 r
F P
s
P
p
.002 w
.55287 .23675 m .53195 .24486 L .52254 .23236 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55287 .23675 m .53195 .24486 L .52254 .23236 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5017 .2405 m .52254 .23236 L .53195 .24486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5017 .2405 m .52254 .23236 L .53195 .24486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47131 .23612 m .45073 .24424 L .44078 .23173 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47131 .23612 m .45073 .24424 L .44078 .23173 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42028 .23988 m .44078 .23173 L .45073 .24424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42028 .23988 m .44078 .23173 L .45073 .24424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60325 .64349 m .62454 .63706 L .63328 .64693 L p
.654 .754 .923 r
F P
s
P
p
.002 w
.3897 .2355 m .36945 .24361 L .35898 .2311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3897 .2355 m .36945 .24361 L .35898 .2311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33881 .23925 m .35898 .2311 L .36945 .24361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.33881 .23925 m .35898 .2311 L .36945 .24361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49888 .74559 m .47817 .75155 L .4684 .74236 L p
.646 .762 .933 r
F P
s
P
p
.002 w
.45183 .59264 m .47247 .586 L .48221 .5962 L p
.657 .75 .919 r
F P
s
P
p
.002 w
.53864 .78 m .55965 .77414 L .56883 .78313 L p
.644 .764 .936 r
F P
s
P
p
.002 w
.30804 .23487 m .28813 .24299 L .27711 .23047 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30804 .23487 m .28813 .24299 L .27711 .23047 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25729 .23863 m .27711 .23047 L .28813 .24299 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.25729 .23863 m .27711 .23047 L .28813 .24299 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57299 .63466 m .59416 .62819 L .60311 .63812 L p
.654 .753 .923 r
F P
s
P
p
.002 w
.48209 .60438 m .50288 .59777 L .51243 .60791 L p
.656 .751 .92 r
F P
s
P
p
.002 w
.76748 .23424 m .74562 .24237 L .73761 .22984 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.76748 .23424 m .74562 .24237 L .73761 .22984 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71582 .238 m .73761 .22984 L .74562 .24237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71582 .238 m .73761 .22984 L .74562 .24237 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.5427 .62526 m .56375 .61875 L .5729 .62875 L p
.655 .753 .922 r
F P
s
P
p
.002 w
.51239 .6152 m .53331 .60865 L .54266 .61871 L p
.655 .752 .921 r
F P
s
P
p
.002 w
.68589 .23361 m .66437 .24175 L .65582 .2292 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68589 .23361 m .66437 .24175 L .65582 .2292 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63437 .23738 m .65582 .2292 L .66437 .24175 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63437 .23738 m .65582 .2292 L .66437 .24175 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61611 .94018 m .63748 .93501 L .64615 .94295 L p
.629 .778 .952 r
F P
s
P
p
.002 w
.60424 .23298 m .58306 .24112 L .57397 .22857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60424 .23298 m .58306 .24112 L .57397 .22857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55287 .23675 m .57397 .22857 L .58306 .24112 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55287 .23675 m .57397 .22857 L .58306 .24112 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52254 .23236 m .5017 .2405 L .49207 .22794 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52254 .23236 m .5017 .2405 L .49207 .22794 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47131 .23612 m .49207 .22794 L .5017 .2405 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47131 .23612 m .49207 .22794 L .5017 .2405 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67717 .97008 m .65567 .9751 L .64705 .96737 L p
.625 .78 .956 r
F P
s
P
p
.002 w
.6299 .82241 m .60859 .82806 L .59966 .81935 L p
.64 .768 .941 r
F P
s
P
p
.002 w
.44078 .23173 m .42028 .23988 L .41012 .22731 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44078 .23173 m .42028 .23988 L .41012 .22731 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3897 .2355 m .41012 .22731 L .42028 .23988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3897 .2355 m .41012 .22731 L .42028 .23988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35898 .2311 m .33881 .23925 L .32811 .22668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35898 .2311 m .33881 .23925 L .32811 .22668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30804 .23487 m .32811 .22668 L .33881 .23925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30804 .23487 m .32811 .22668 L .33881 .23925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.29044 .51822 m .31044 .51124 L .32126 .52196 L p
.66 .745 .913 r
F P
s
P
p
.002 w
.27711 .23047 m .25729 .23863 L .24605 .22604 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27711 .23047 m .25729 .23863 L .24605 .22604 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22632 .23424 m .24605 .22604 L .25729 .23863 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.22632 .23424 m .24605 .22604 L .25729 .23863 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73761 .22984 m .71582 .238 L .70759 .22541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73761 .22984 m .71582 .238 L .70759 .22541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68589 .23361 m .70759 .22541 L .71582 .238 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68589 .23361 m .70759 .22541 L .71582 .238 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65582 .2292 m .63437 .23738 L .6256 .22478 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65582 .2292 m .63437 .23738 L .6256 .22478 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60424 .23298 m .6256 .22478 L .63437 .23738 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60424 .23298 m .6256 .22478 L .63437 .23738 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56907 .80031 m .59029 .79452 L .59929 .8034 L p
.642 .766 .939 r
F P
s
P
p
.002 w
.57397 .22857 m .55287 .23675 L .54356 .22414 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57397 .22857 m .55287 .23675 L .54356 .22414 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52254 .23236 m .54356 .22414 L .55287 .23675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52254 .23236 m .54356 .22414 L .55287 .23675 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49207 .22794 m .47131 .23612 L .46146 .22351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49207 .22794 m .47131 .23612 L .46146 .22351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44078 .23173 m .46146 .22351 L .47131 .23612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44078 .23173 m .46146 .22351 L .47131 .23612 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41012 .22731 m .3897 .2355 L .37931 .22287 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41012 .22731 m .3897 .2355 L .37931 .22287 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35898 .2311 m .37931 .22287 L .3897 .2355 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35898 .2311 m .37931 .22287 L .3897 .2355 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4684 .74236 m .48921 .73633 L .49888 .74559 L p
.646 .762 .933 r
F P
s
P
p
.002 w
.66062 .84029 m .63913 .84587 L .63039 .83726 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.52924 .76934 m .50831 .77522 L .4987 .76615 L p
.644 .764 .936 r
F P
s
P
p
.002 w
.32811 .22668 m .30804 .23487 L .2971 .22224 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32811 .22668 m .30804 .23487 L .2971 .22224 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27711 .23047 m .2971 .22224 L .30804 .23487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.27711 .23047 m .2971 .22224 L .30804 .23487 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73761 .22984 m .75958 .2216 L .76748 .23424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73761 .22984 m .75958 .2216 L .76748 .23424 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70759 .22541 m .68589 .23361 L .67744 .22096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70759 .22541 m .68589 .23361 L .67744 .22096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65582 .2292 m .67744 .22096 L .68589 .23361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65582 .2292 m .67744 .22096 L .68589 .23361 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6256 .22478 m .60424 .23298 L .59525 .22033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6256 .22478 m .60424 .23298 L .59525 .22033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57397 .22857 m .59525 .22033 L .60424 .23298 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57397 .22857 m .59525 .22033 L .60424 .23298 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54356 .22414 m .52254 .23236 L .51301 .21969 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54356 .22414 m .52254 .23236 L .51301 .21969 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49207 .22794 m .51301 .21969 L .52254 .23236 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49207 .22794 m .51301 .21969 L .52254 .23236 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64705 .96737 m .66866 .96228 L .67717 .97008 L p
.625 .78 .956 r
F P
s
P
p
.002 w
.59966 .81935 m .62107 .81364 L .6299 .82241 L p
.64 .768 .941 r
F P
s
P
p
.002 w
.46146 .22351 m .44078 .23173 L .43071 .21905 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46146 .22351 m .44078 .23173 L .43071 .21905 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41012 .22731 m .43071 .21905 L .44078 .23173 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41012 .22731 m .43071 .21905 L .44078 .23173 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69143 .85713 m .66977 .86265 L .66121 .85414 L p
.636 .771 .945 r
F P
s
P
p
.002 w
.37931 .22287 m .35898 .2311 L .34836 .21841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37931 .22287 m .35898 .2311 L .34836 .21841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32811 .22668 m .34836 .21841 L .35898 .2311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.32811 .22668 m .34836 .21841 L .35898 .2311 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2971 .22224 m .27711 .23047 L .26595 .21777 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2971 .22224 m .27711 .23047 L .26595 .21777 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24605 .22604 m .26595 .21777 L .27711 .23047 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.24605 .22604 m .26595 .21777 L .27711 .23047 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75958 .2216 m .73761 .22984 L .72949 .21714 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75958 .2216 m .73761 .22984 L .72949 .21714 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70759 .22541 m .72949 .21714 L .73761 .22984 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70759 .22541 m .72949 .21714 L .73761 .22984 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67744 .22096 m .65582 .2292 L .64715 .2165 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67744 .22096 m .65582 .2292 L .64715 .2165 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6256 .22478 m .64715 .2165 L .65582 .2292 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6256 .22478 m .64715 .2165 L .65582 .2292 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59525 .22033 m .57397 .22857 L .56476 .21586 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59525 .22033 m .57397 .22857 L .56476 .21586 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54356 .22414 m .56476 .21586 L .57397 .22857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54356 .22414 m .56476 .21586 L .57397 .22857 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72231 .87305 m .70048 .87851 L .6921 .87009 L p
.634 .773 .947 r
F P
s
P
p
.002 w
.51301 .21969 m .49207 .22794 L .48232 .21522 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51301 .21969 m .49207 .22794 L .48232 .21522 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46146 .22351 m .48232 .21522 L .49207 .22794 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46146 .22351 m .48232 .21522 L .49207 .22794 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43071 .21905 m .41012 .22731 L .39982 .21458 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43071 .21905 m .41012 .22731 L .39982 .21458 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37931 .22287 m .39982 .21458 L .41012 .22731 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37931 .22287 m .39982 .21458 L .41012 .22731 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63039 .83726 m .65198 .83161 L .66062 .84029 L p
.638 .77 .943 r
F P
s
P
p
.002 w
.4987 .76615 m .51973 .76019 L .52924 .76934 L p
.644 .764 .936 r
F P
s
P
p
.002 w
.34836 .21841 m .32811 .22668 L .31727 .21393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34836 .21841 m .32811 .22668 L .31727 .21393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2971 .22224 m .31727 .21393 L .32811 .22668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.2971 .22224 m .31727 .21393 L .32811 .22668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75323 .88812 m .73123 .89352 L .72305 .88519 L p
.633 .774 .948 r
F P
s
P
p
.002 w
.72949 .21714 m .70759 .22541 L .69925 .21265 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72949 .21714 m .70759 .22541 L .69925 .21265 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67744 .22096 m .69925 .21265 L .70759 .22541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67744 .22096 m .69925 .21265 L .70759 .22541 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64715 .2165 m .6256 .22478 L .61672 .21201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64715 .2165 m .6256 .22478 L .61672 .21201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59525 .22033 m .61672 .21201 L .6256 .22478 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59525 .22033 m .61672 .21201 L .6256 .22478 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56476 .21586 m .54356 .22414 L .53413 .21137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56476 .21586 m .54356 .22414 L .53413 .21137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51301 .21969 m .53413 .21137 L .54356 .22414 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51301 .21969 m .53413 .21137 L .54356 .22414 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48232 .21522 m .46146 .22351 L .45148 .21072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48232 .21522 m .46146 .22351 L .45148 .21072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43071 .21905 m .45148 .21072 L .46146 .22351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43071 .21905 m .45148 .21072 L .46146 .22351 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66121 .85414 m .68298 .84855 L .69143 .85713 L p
.636 .771 .945 r
F P
s
P
p
.002 w
.78417 .90242 m .76202 .90777 L .75403 .89952 L p
.631 .776 .95 r
F P
s
P
p
.002 w
.39982 .21458 m .37931 .22287 L .36878 .21008 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39982 .21458 m .37931 .22287 L .36878 .21008 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34836 .21841 m .36878 .21008 L .37931 .22287 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.34836 .21841 m .36878 .21008 L .37931 .22287 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31727 .21393 m .2971 .22224 L .28603 .20943 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31727 .21393 m .2971 .22224 L .28603 .20943 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26595 .21777 m .28603 .20943 L .2971 .22224 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.26595 .21777 m .28603 .20943 L .2971 .22224 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72949 .21714 m .75157 .20879 L .75958 .2216 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72949 .21714 m .75157 .20879 L .75958 .2216 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69925 .21265 m .67744 .22096 L .66888 .20814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69925 .21265 m .67744 .22096 L .66888 .20814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64715 .2165 m .66888 .20814 L .67744 .22096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64715 .2165 m .66888 .20814 L .67744 .22096 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61672 .21201 m .59525 .22033 L .58615 .2075 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61672 .21201 m .59525 .22033 L .58615 .2075 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56476 .21586 m .58615 .2075 L .59525 .22033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56476 .21586 m .58615 .2075 L .59525 .22033 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6921 .87009 m .71404 .86456 L .72231 .87305 L p
.634 .773 .947 r
F P
s
P
p
.002 w
.53413 .21137 m .51301 .21969 L .50335 .20685 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53413 .21137 m .51301 .21969 L .50335 .20685 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48232 .21522 m .50335 .20685 L .51301 .21969 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48232 .21522 m .50335 .20685 L .51301 .21969 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45148 .21072 m .43071 .21905 L .4205 .20621 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45148 .21072 m .43071 .21905 L .4205 .20621 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39982 .21458 m .4205 .20621 L .43071 .21905 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39982 .21458 m .4205 .20621 L .43071 .21905 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36878 .21008 m .34836 .21841 L .3376 .20556 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36878 .21008 m .34836 .21841 L .3376 .20556 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31727 .21393 m .3376 .20556 L .34836 .21841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.31727 .21393 m .3376 .20556 L .34836 .21841 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72305 .88519 m .74515 .87971 L .75323 .88812 L p
.633 .774 .948 r
F P
s
P
p
.002 w
.75157 .20879 m .72949 .21714 L .72126 .20427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75157 .20879 m .72949 .21714 L .72126 .20427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69925 .21265 m .72126 .20427 L .72949 .21714 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69925 .21265 m .72126 .20427 L .72949 .21714 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66888 .20814 m .64715 .2165 L .63837 .20362 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66888 .20814 m .64715 .2165 L .63837 .20362 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61672 .21201 m .63837 .20362 L .64715 .2165 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61672 .21201 m .63837 .20362 L .64715 .2165 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58615 .2075 m .56476 .21586 L .55543 .20297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58615 .2075 m .56476 .21586 L .55543 .20297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53413 .21137 m .55543 .20297 L .56476 .21586 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53413 .21137 m .55543 .20297 L .56476 .21586 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50335 .20685 m .48232 .21522 L .47243 .20232 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50335 .20685 m .48232 .21522 L .47243 .20232 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45148 .21072 m .47243 .20232 L .48232 .21522 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45148 .21072 m .47243 .20232 L .48232 .21522 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.75403 .89952 m .77629 .8941 L .78417 .90242 L p
.631 .776 .95 r
F P
s
P
p
.002 w
.4205 .20621 m .39982 .21458 L .38938 .20167 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4205 .20621 m .39982 .21458 L .38938 .20167 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36878 .21008 m .38938 .20167 L .39982 .21458 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36878 .21008 m .38938 .20167 L .39982 .21458 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3376 .20556 m .31727 .21393 L .30627 .20102 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3376 .20556 m .31727 .21393 L .30627 .20102 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28603 .20943 m .30627 .20102 L .31727 .21393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.28603 .20943 m .30627 .20102 L .31727 .21393 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72126 .20427 m .69925 .21265 L .6908 .19972 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72126 .20427 m .69925 .21265 L .6908 .19972 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66888 .20814 m .6908 .19972 L .69925 .21265 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66888 .20814 m .6908 .19972 L .69925 .21265 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63837 .20362 m .61672 .21201 L .60771 .19907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63837 .20362 m .61672 .21201 L .60771 .19907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58615 .2075 m .60771 .19907 L .61672 .21201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58615 .2075 m .60771 .19907 L .61672 .21201 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55543 .20297 m .53413 .21137 L .52457 .19842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55543 .20297 m .53413 .21137 L .52457 .19842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50335 .20685 m .52457 .19842 L .53413 .21137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50335 .20685 m .52457 .19842 L .53413 .21137 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47243 .20232 m .45148 .21072 L .44137 .19777 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47243 .20232 m .45148 .21072 L .44137 .19777 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4205 .20621 m .44137 .19777 L .45148 .21072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4205 .20621 m .44137 .19777 L .45148 .21072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38938 .20167 m .36878 .21008 L .35811 .19711 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38938 .20167 m .36878 .21008 L .35811 .19711 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3376 .20556 m .35811 .19711 L .36878 .21008 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3376 .20556 m .35811 .19711 L .36878 .21008 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72126 .20427 m .74345 .19581 L .75157 .20879 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72126 .20427 m .74345 .19581 L .75157 .20879 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6908 .19972 m .66888 .20814 L .66021 .19516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6908 .19972 m .66888 .20814 L .66021 .19516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63837 .20362 m .66021 .19516 L .66888 .20814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63837 .20362 m .66021 .19516 L .66888 .20814 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60771 .19907 m .58615 .2075 L .57692 .1945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60771 .19907 m .58615 .2075 L .57692 .1945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55543 .20297 m .57692 .1945 L .58615 .2075 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55543 .20297 m .57692 .1945 L .58615 .2075 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52457 .19842 m .50335 .20685 L .49357 .19385 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52457 .19842 m .50335 .20685 L .49357 .19385 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47243 .20232 m .49357 .19385 L .50335 .20685 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.47243 .20232 m .49357 .19385 L .50335 .20685 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44137 .19777 m .4205 .20621 L .41016 .19319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44137 .19777 m .4205 .20621 L .41016 .19319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38938 .20167 m .41016 .19319 L .4205 .20621 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38938 .20167 m .41016 .19319 L .4205 .20621 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35811 .19711 m .3376 .20556 L .3267 .19254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35811 .19711 m .3376 .20556 L .3267 .19254 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30627 .20102 m .3267 .19254 L .3376 .20556 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.30627 .20102 m .3267 .19254 L .3376 .20556 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74345 .19581 m .72126 .20427 L .71292 .19122 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.74345 .19581 m .72126 .20427 L .71292 .19122 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6908 .19972 m .71292 .19122 L .72126 .20427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6908 .19972 m .71292 .19122 L .72126 .20427 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66021 .19516 m .63837 .20362 L .62947 .19057 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66021 .19516 m .63837 .20362 L .62947 .19057 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60771 .19907 m .62947 .19057 L .63837 .20362 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60771 .19907 m .62947 .19057 L .63837 .20362 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57692 .1945 m .55543 .20297 L .54597 .18991 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57692 .1945 m .55543 .20297 L .54597 .18991 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52457 .19842 m .54597 .18991 L .55543 .20297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52457 .19842 m .54597 .18991 L .55543 .20297 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49357 .19385 m .47243 .20232 L .46242 .18925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49357 .19385 m .47243 .20232 L .46242 .18925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44137 .19777 m .46242 .18925 L .47243 .20232 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44137 .19777 m .46242 .18925 L .47243 .20232 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41016 .19319 m .38938 .20167 L .37881 .1886 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41016 .19319 m .38938 .20167 L .37881 .1886 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35811 .19711 m .37881 .1886 L .38938 .20167 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.35811 .19711 m .37881 .1886 L .38938 .20167 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71292 .19122 m .6908 .19972 L .68224 .18662 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71292 .19122 m .6908 .19972 L .68224 .18662 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66021 .19516 m .68224 .18662 L .6908 .19972 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66021 .19516 m .68224 .18662 L .6908 .19972 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62947 .19057 m .60771 .19907 L .59859 .18596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62947 .19057 m .60771 .19907 L .59859 .18596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57692 .1945 m .59859 .18596 L .60771 .19907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57692 .1945 m .59859 .18596 L .60771 .19907 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54597 .18991 m .52457 .19842 L .51488 .1853 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54597 .18991 m .52457 .19842 L .51488 .1853 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49357 .19385 m .51488 .1853 L .52457 .19842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49357 .19385 m .51488 .1853 L .52457 .19842 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46242 .18925 m .44137 .19777 L .43112 .18464 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46242 .18925 m .44137 .19777 L .43112 .18464 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41016 .19319 m .43112 .18464 L .44137 .19777 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41016 .19319 m .43112 .18464 L .44137 .19777 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37881 .1886 m .35811 .19711 L .3473 .18398 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37881 .1886 m .35811 .19711 L .3473 .18398 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3267 .19254 m .3473 .18398 L .35811 .19711 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3267 .19254 m .3473 .18398 L .35811 .19711 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71292 .19122 m .73522 .18265 L .74345 .19581 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71292 .19122 m .73522 .18265 L .74345 .19581 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68224 .18662 m .66021 .19516 L .65142 .18199 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68224 .18662 m .66021 .19516 L .65142 .18199 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62947 .19057 m .65142 .18199 L .66021 .19516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62947 .19057 m .65142 .18199 L .66021 .19516 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59859 .18596 m .57692 .1945 L .56756 .18133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59859 .18596 m .57692 .1945 L .56756 .18133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54597 .18991 m .56756 .18133 L .57692 .1945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54597 .18991 m .56756 .18133 L .57692 .1945 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51488 .1853 m .49357 .19385 L .48365 .18066 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51488 .1853 m .49357 .19385 L .48365 .18066 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46242 .18925 m .48365 .18066 L .49357 .19385 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46242 .18925 m .48365 .18066 L .49357 .19385 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43112 .18464 m .41016 .19319 L .39968 .18 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43112 .18464 m .41016 .19319 L .39968 .18 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37881 .1886 m .39968 .18 L .41016 .19319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.37881 .1886 m .39968 .18 L .41016 .19319 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73522 .18265 m .71292 .19122 L .70446 .17801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.73522 .18265 m .71292 .19122 L .70446 .17801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68224 .18662 m .70446 .17801 L .71292 .19122 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68224 .18662 m .70446 .17801 L .71292 .19122 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65142 .18199 m .62947 .19057 L .62045 .17734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65142 .18199 m .62947 .19057 L .62045 .17734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59859 .18596 m .62045 .17734 L .62947 .19057 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59859 .18596 m .62045 .17734 L .62947 .19057 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56756 .18133 m .54597 .18991 L .53639 .17668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56756 .18133 m .54597 .18991 L .53639 .17668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51488 .1853 m .53639 .17668 L .54597 .18991 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51488 .1853 m .53639 .17668 L .54597 .18991 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48365 .18066 m .46242 .18925 L .45227 .17601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48365 .18066 m .46242 .18925 L .45227 .17601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43112 .18464 m .45227 .17601 L .46242 .18925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43112 .18464 m .45227 .17601 L .46242 .18925 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39968 .18 m .37881 .1886 L .36809 .17534 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39968 .18 m .37881 .1886 L .36809 .17534 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3473 .18398 m .36809 .17534 L .37881 .1886 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.3473 .18398 m .36809 .17534 L .37881 .1886 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70446 .17801 m .68224 .18662 L .67356 .17334 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70446 .17801 m .68224 .18662 L .67356 .17334 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65142 .18199 m .67356 .17334 L .68224 .18662 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65142 .18199 m .67356 .17334 L .68224 .18662 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62045 .17734 m .59859 .18596 L .58934 .17267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62045 .17734 m .59859 .18596 L .58934 .17267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56756 .18133 m .58934 .17267 L .59859 .18596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56756 .18133 m .58934 .17267 L .59859 .18596 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53639 .17668 m .51488 .1853 L .50507 .172 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53639 .17668 m .51488 .1853 L .50507 .172 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48365 .18066 m .50507 .172 L .51488 .1853 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48365 .18066 m .50507 .172 L .51488 .1853 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45227 .17601 m .43112 .18464 L .42073 .17133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45227 .17601 m .43112 .18464 L .42073 .17133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39968 .18 m .42073 .17133 L .43112 .18464 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.39968 .18 m .42073 .17133 L .43112 .18464 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70446 .17801 m .72688 .16932 L .73522 .18265 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70446 .17801 m .72688 .16932 L .73522 .18265 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67356 .17334 m .65142 .18199 L .64251 .16865 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67356 .17334 m .65142 .18199 L .64251 .16865 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62045 .17734 m .64251 .16865 L .65142 .18199 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62045 .17734 m .64251 .16865 L .65142 .18199 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58934 .17267 m .56756 .18133 L .55808 .16798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58934 .17267 m .56756 .18133 L .55808 .16798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53639 .17668 m .55808 .16798 L .56756 .18133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53639 .17668 m .55808 .16798 L .56756 .18133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50507 .172 m .48365 .18066 L .4736 .1673 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50507 .172 m .48365 .18066 L .4736 .1673 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45227 .17601 m .4736 .1673 L .48365 .18066 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45227 .17601 m .4736 .1673 L .48365 .18066 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42073 .17133 m .39968 .18 L .38905 .16663 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42073 .17133 m .39968 .18 L .38905 .16663 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36809 .17534 m .38905 .16663 L .39968 .18 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.36809 .17534 m .38905 .16663 L .39968 .18 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72688 .16932 m .70446 .17801 L .6959 .16461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.72688 .16932 m .70446 .17801 L .6959 .16461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67356 .17334 m .6959 .16461 L .70446 .17801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67356 .17334 m .6959 .16461 L .70446 .17801 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64251 .16865 m .62045 .17734 L .61131 .16394 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64251 .16865 m .62045 .17734 L .61131 .16394 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58934 .17267 m .61131 .16394 L .62045 .17734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58934 .17267 m .61131 .16394 L .62045 .17734 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55808 .16798 m .53639 .17668 L .52667 .16326 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55808 .16798 m .53639 .17668 L .52667 .16326 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50507 .172 m .52667 .16326 L .53639 .17668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50507 .172 m .52667 .16326 L .53639 .17668 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4736 .1673 m .45227 .17601 L .44198 .16258 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4736 .1673 m .45227 .17601 L .44198 .16258 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42073 .17133 m .44198 .16258 L .45227 .17601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.42073 .17133 m .44198 .16258 L .45227 .17601 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6959 .16461 m .67356 .17334 L .66476 .15988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6959 .16461 m .67356 .17334 L .66476 .15988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64251 .16865 m .66476 .15988 L .67356 .17334 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64251 .16865 m .66476 .15988 L .67356 .17334 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61131 .16394 m .58934 .17267 L .57997 .1592 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61131 .16394 m .58934 .17267 L .57997 .1592 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55808 .16798 m .57997 .1592 L .58934 .17267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55808 .16798 m .57997 .1592 L .58934 .17267 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52667 .16326 m .50507 .172 L .49512 .15852 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52667 .16326 m .50507 .172 L .49512 .15852 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4736 .1673 m .49512 .15852 L .50507 .172 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4736 .1673 m .49512 .15852 L .50507 .172 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44198 .16258 m .42073 .17133 L .41021 .15784 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44198 .16258 m .42073 .17133 L .41021 .15784 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38905 .16663 m .41021 .15784 L .42073 .17133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.38905 .16663 m .41021 .15784 L .42073 .17133 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6959 .16461 m .71843 .1558 L .72688 .16932 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6959 .16461 m .71843 .1558 L .72688 .16932 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66476 .15988 m .64251 .16865 L .63348 .15512 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66476 .15988 m .64251 .16865 L .63348 .15512 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61131 .16394 m .63348 .15512 L .64251 .16865 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61131 .16394 m .63348 .15512 L .64251 .16865 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57997 .1592 m .55808 .16798 L .54847 .15444 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57997 .1592 m .55808 .16798 L .54847 .15444 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52667 .16326 m .54847 .15444 L .55808 .16798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52667 .16326 m .54847 .15444 L .55808 .16798 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49512 .15852 m .4736 .1673 L .46341 .15376 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49512 .15852 m .4736 .1673 L .46341 .15376 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44198 .16258 m .46341 .15376 L .4736 .1673 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.44198 .16258 m .46341 .15376 L .4736 .1673 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71843 .1558 m .6959 .16461 L .68721 .15103 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.71843 .1558 m .6959 .16461 L .68721 .15103 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66476 .15988 m .68721 .15103 L .6959 .16461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66476 .15988 m .68721 .15103 L .6959 .16461 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63348 .15512 m .61131 .16394 L .60205 .15035 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63348 .15512 m .61131 .16394 L .60205 .15035 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57997 .1592 m .60205 .15035 L .61131 .16394 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57997 .1592 m .60205 .15035 L .61131 .16394 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54847 .15444 m .52667 .16326 L .51683 .14966 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54847 .15444 m .52667 .16326 L .51683 .14966 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49512 .15852 m .51683 .14966 L .52667 .16326 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49512 .15852 m .51683 .14966 L .52667 .16326 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46341 .15376 m .44198 .16258 L .43155 .14898 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46341 .15376 m .44198 .16258 L .43155 .14898 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41021 .15784 m .43155 .14898 L .44198 .16258 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.41021 .15784 m .43155 .14898 L .44198 .16258 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68721 .15103 m .66476 .15988 L .65585 .14623 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68721 .15103 m .66476 .15988 L .65585 .14623 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63348 .15512 m .65585 .14623 L .66476 .15988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63348 .15512 m .65585 .14623 L .66476 .15988 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60205 .15035 m .57997 .1592 L .57047 .14555 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60205 .15035 m .57997 .1592 L .57047 .14555 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54847 .15444 m .57047 .14555 L .57997 .1592 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54847 .15444 m .57047 .14555 L .57997 .1592 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51683 .14966 m .49512 .15852 L .48503 .14486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51683 .14966 m .49512 .15852 L .48503 .14486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46341 .15376 m .48503 .14486 L .49512 .15852 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.46341 .15376 m .48503 .14486 L .49512 .15852 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68721 .15103 m .70986 .1421 L .71843 .1558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68721 .15103 m .70986 .1421 L .71843 .1558 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65585 .14623 m .63348 .15512 L .62433 .14141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65585 .14623 m .63348 .15512 L .62433 .14141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60205 .15035 m .62433 .14141 L .63348 .15512 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60205 .15035 m .62433 .14141 L .63348 .15512 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57047 .14555 m .54847 .15444 L .53873 .14072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57047 .14555 m .54847 .15444 L .53873 .14072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51683 .14966 m .53873 .14072 L .54847 .15444 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51683 .14966 m .53873 .14072 L .54847 .15444 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48503 .14486 m .46341 .15376 L .45308 .14003 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48503 .14486 m .46341 .15376 L .45308 .14003 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43155 .14898 m .45308 .14003 L .46341 .15376 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.43155 .14898 m .45308 .14003 L .46341 .15376 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70986 .1421 m .68721 .15103 L .67841 .13726 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70986 .1421 m .68721 .15103 L .67841 .13726 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65585 .14623 m .67841 .13726 L .68721 .15103 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65585 .14623 m .67841 .13726 L .68721 .15103 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62433 .14141 m .60205 .15035 L .59266 .13657 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62433 .14141 m .60205 .15035 L .59266 .13657 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57047 .14555 m .59266 .13657 L .60205 .15035 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57047 .14555 m .59266 .13657 L .60205 .15035 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53873 .14072 m .51683 .14966 L .50684 .13588 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53873 .14072 m .51683 .14966 L .50684 .13588 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48503 .14486 m .50684 .13588 L .51683 .14966 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.48503 .14486 m .50684 .13588 L .51683 .14966 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67841 .13726 m .65585 .14623 L .6468 .1324 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67841 .13726 m .65585 .14623 L .6468 .1324 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62433 .14141 m .6468 .1324 L .65585 .14623 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62433 .14141 m .6468 .1324 L .65585 .14623 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59266 .13657 m .57047 .14555 L .56083 .1317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59266 .13657 m .57047 .14555 L .56083 .1317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53873 .14072 m .56083 .1317 L .57047 .14555 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.53873 .14072 m .56083 .1317 L .57047 .14555 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50684 .13588 m .48503 .14486 L .4748 .13101 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50684 .13588 m .48503 .14486 L .4748 .13101 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45308 .14003 m .4748 .13101 L .48503 .14486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.45308 .14003 m .4748 .13101 L .48503 .14486 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67841 .13726 m .70118 .12821 L .70986 .1421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67841 .13726 m .70118 .12821 L .70986 .1421 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6468 .1324 m .62433 .14141 L .61505 .12751 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6468 .1324 m .62433 .14141 L .61505 .12751 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59266 .13657 m .61505 .12751 L .62433 .14141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59266 .13657 m .61505 .12751 L .62433 .14141 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56083 .1317 m .53873 .14072 L .52885 .12681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56083 .1317 m .53873 .14072 L .52885 .12681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50684 .13588 m .52885 .12681 L .53873 .14072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.50684 .13588 m .52885 .12681 L .53873 .14072 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70118 .12821 m .67841 .13726 L .66948 .12331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.70118 .12821 m .67841 .13726 L .66948 .12331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6468 .1324 m .66948 .12331 L .67841 .13726 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6468 .1324 m .66948 .12331 L .67841 .13726 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61505 .12751 m .59266 .13657 L .58313 .1226 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61505 .12751 m .59266 .13657 L .58313 .1226 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56083 .1317 m .58313 .1226 L .59266 .13657 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56083 .1317 m .58313 .1226 L .59266 .13657 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52885 .12681 m .50684 .13588 L .49672 .1219 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52885 .12681 m .50684 .13588 L .49672 .1219 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4748 .13101 m .49672 .1219 L .50684 .13588 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.4748 .13101 m .49672 .1219 L .50684 .13588 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66948 .12331 m .6468 .1324 L .63764 .11837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66948 .12331 m .6468 .1324 L .63764 .11837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61505 .12751 m .63764 .11837 L .6468 .1324 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61505 .12751 m .63764 .11837 L .6468 .1324 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58313 .1226 m .56083 .1317 L .55107 .11767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58313 .1226 m .56083 .1317 L .55107 .11767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52885 .12681 m .55107 .11767 L .56083 .1317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.52885 .12681 m .55107 .11767 L .56083 .1317 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66948 .12331 m .69237 .11413 L .70118 .12821 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66948 .12331 m .69237 .11413 L .70118 .12821 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63764 .11837 m .61505 .12751 L .60564 .11342 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63764 .11837 m .61505 .12751 L .60564 .11342 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58313 .1226 m .60564 .11342 L .61505 .12751 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58313 .1226 m .60564 .11342 L .61505 .12751 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55107 .11767 m .52885 .12681 L .51884 .11271 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55107 .11767 m .52885 .12681 L .51884 .11271 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49672 .1219 m .51884 .11271 L .52885 .12681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.49672 .1219 m .51884 .11271 L .52885 .12681 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69237 .11413 m .66948 .12331 L .66043 .10915 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.69237 .11413 m .66948 .12331 L .66043 .10915 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63764 .11837 m .66043 .10915 L .66948 .12331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63764 .11837 m .66043 .10915 L .66948 .12331 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60564 .11342 m .58313 .1226 L .57348 .10844 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60564 .11342 m .58313 .1226 L .57348 .10844 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55107 .11767 m .57348 .10844 L .58313 .1226 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.55107 .11767 m .57348 .10844 L .58313 .1226 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66043 .10915 m .63764 .11837 L .62834 .10415 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66043 .10915 m .63764 .11837 L .62834 .10415 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60564 .11342 m .62834 .10415 L .63764 .11837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60564 .11342 m .62834 .10415 L .63764 .11837 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57348 .10844 m .55107 .11767 L .54116 .10344 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57348 .10844 m .55107 .11767 L .54116 .10344 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51884 .11271 m .54116 .10344 L .55107 .11767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.51884 .11271 m .54116 .10344 L .55107 .11767 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66043 .10915 m .68344 .09985 L .69237 .11413 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66043 .10915 m .68344 .09985 L .69237 .11413 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62834 .10415 m .60564 .11342 L .59609 .09913 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62834 .10415 m .60564 .11342 L .59609 .09913 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57348 .10844 m .59609 .09913 L .60564 .11342 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.57348 .10844 m .59609 .09913 L .60564 .11342 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68344 .09985 m .66043 .10915 L .65126 .0948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.68344 .09985 m .66043 .10915 L .65126 .0948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62834 .10415 m .65126 .0948 L .66043 .10915 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.62834 .10415 m .65126 .0948 L .66043 .10915 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59609 .09913 m .57348 .10844 L .56369 .09408 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59609 .09913 m .57348 .10844 L .56369 .09408 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54116 .10344 m .56369 .09408 L .57348 .10844 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.54116 .10344 m .56369 .09408 L .57348 .10844 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65126 .0948 m .62834 .10415 L .61892 .08973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65126 .0948 m .62834 .10415 L .61892 .08973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59609 .09913 m .61892 .08973 L .62834 .10415 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.59609 .09913 m .61892 .08973 L .62834 .10415 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65126 .0948 m .67438 .08536 L .68344 .09985 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.65126 .0948 m .67438 .08536 L .68344 .09985 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61892 .08973 m .59609 .09913 L .58642 .08463 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61892 .08973 m .59609 .09913 L .58642 .08463 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56369 .09408 m .58642 .08463 L .59609 .09913 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.56369 .09408 m .58642 .08463 L .59609 .09913 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67438 .08536 m .65126 .0948 L .64195 .08025 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.67438 .08536 m .65126 .0948 L .64195 .08025 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61892 .08973 m .64195 .08025 L .65126 .0948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.61892 .08973 m .64195 .08025 L .65126 .0948 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64195 .08025 m .61892 .08973 L .60936 .0751 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64195 .08025 m .61892 .08973 L .60936 .0751 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58642 .08463 m .60936 .0751 L .61892 .08973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.58642 .08463 m .60936 .0751 L .61892 .08973 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64195 .08025 m .6652 .07068 L .67438 .08536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.64195 .08025 m .6652 .07068 L .67438 .08536 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6652 .07068 m .64195 .08025 L .63251 .06548 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.6652 .07068 m .64195 .08025 L .63251 .06548 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60936 .0751 m .63251 .06548 L .64195 .08025 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.60936 .0751 m .63251 .06548 L .64195 .08025 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63251 .06548 m .65588 .05578 L .6652 .07068 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.63251 .06548 m .65588 .05578 L .6652 .07068 L p
.672 .728 .891 r
F P
s
P
p
.002 w
.66295 .02755 m
.90359 .41965 L
s
.90359 .41965 m
.97245 .92345 L
s
.97245 .92345 m
.69507 .58075 L
s
.69507 .58075 m
.66295 .02755 L
s
.11113 .25957 m
.02755 .78701 L
s
.02755 .78701 m
.69507 .58075 L
s
.69507 .58075 m
.66295 .02755 L
s
.66295 .02755 m
.11113 .25957 L
s
P
p
p
.002 w
.11113 .25957 m
.66295 .02755 L
s
P
p
.002 w
.11347 .25859 m
.11829 .26351 L
s
P
[(0)] .10384 .24874 .97891 1 Mshowa
p
.002 w
.30906 .17634 m
.31349 .18162 L
s
P
[(10)] .30021 .16579 .83912 1 Mshowa
p
.002 w
.52243 .08663 m
.52637 .09228 L
s
P
[(20)] .51453 .07535 .69933 1 Mshowa
p
.001 w
.15128 .24269 m
.15412 .24568 L
s
P
p
.001 w
.18972 .22652 m
.19252 .22956 L
s
P
p
.001 w
.22882 .21008 m
.23158 .21316 L
s
P
p
.001 w
.2686 .19336 m
.2713 .19648 L
s
P
p
.001 w
.35024 .15903 m
.35284 .16224 L
s
P
p
.001 w
.39214 .14141 m
.39469 .14467 L
s
P
p
.001 w
.4348 .12348 m
.43729 .12678 L
s
P
p
.001 w
.47822 .10522 m
.48065 .10856 L
s
P
p
.001 w
.56744 .0677 m
.56975 .07113 L
s
P
p
.001 w
.6133 .04843 m
.61553 .0519 L
s
P
p
.001 w
.66001 .02879 m
.66217 .0323 L
s
P
[(l)] .31959 .08564 .79718 1 Mshowa
P
p
P
% End of Graphics
MathPictureEnd

:[font = section; inactive; Cclosed; preserveAspect; startGroup]
Code for quantum error correction in here ...
:[font = subsubsection; inactive; preserveAspect; startGroup]
Controlled-NOT Gate
:[font = input; initialization; preserveAspect]
*)
aOP = {{0,1},   (* annihilation operator on a single bit *)
       {0,0}};
       
annihilationOP[i_, m_]:=
	Apply[Direct,
	      ReplacePart[Table[IdentityMatrix[2], {m}], aOP, i]
	     ]
(*
:[font = input; initialization; preserveAspect]
*)
cOP = {{0,0},     (* creation operator that acts on a single bit *)
       {1,0}};

creationOP[i_, m_]:=
	Apply[Direct,
	      ReplacePart[Table[IdentityMatrix[2], {m}], cOP, i]
	     ]
	
(*
:[font = input; initialization; preserveAspect; endGroup]
*)
CNGate[i_, j_, m_]:=
	(creationOP[i,m] .
	 annihilationOP[i,m] . (annihilationOP[j,m] + creationOP[j,m]) +
	 annihilationOP[i,m] . creationOP[i,m]
	)
(*
:[font = subsection; inactive; preserveAspect; startGroup]
State Vectors
:[font = subsubsection; inactive; preserveAspect; startGroup]
Converting Kets to Column Vectors
:[font = input; initialization; preserveAspect; endGroup]
*)
KetToColumnVector[ket[0]]:={{1},{0}}      (* spin up   = 0 *)
KetToColumnVector[ket[1]]:={{0},{1}}      (* spin down = 1 *)
KetToColumnVector[ket[bits__]]:=
	Apply[Direct, Map[KetToColumnVector[ket[#]]&, {bits}]]

KetToColumnVector[a_ ket_ket]:=
	a KetToColumnVector[ket]
	
KetToColumnVector[Plus[ket_, kets___]]:=
	Apply[Plus, Map[KetToColumnVector, {ket, kets}]]
	
KetToColumnVector[superposition_]:=
	KetToColumnVector[ Expand[superposition] ] 
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Converting Bras to Row Vectors
:[font = input; initialization; preserveAspect; endGroup]
*)
BraToRowVector[bra[0]]:={{1,0}}
BraToRowVector[bra[1]]:={{0,1}}
BraToRowVector[w_. bra[bits__]]:=
	w * Apply[Direct, Map[BraToRowVector[bra[#]]&, {bits}]]
BraToRowVector[w_. bra[bits__] + bras_.]:=
	BraToRowVector[w * bra[bits]] + BraToRowVector[bras]
BraToRowVector[superposition_]:=
	BraToRowVector[Expand[superposition]]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Converting Column Vectors to Kets
:[font = input; initialization; preserveAspect; endGroup]
*)
ColumnVectorToKet[amplitudes_]:=
	Apply[Plus,
		  MapThread[(#1[[1]] #2)&,
		            {amplitudes,
		             EigenKets[ Length[amplitudes] ]
		            }
		           ]
		 ]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Converting Row Vectors To Bras
:[font = input; initialization; preserveAspect; endGroup]
*)
RowVectorToBra[{{wi__}}]:=
	Module[{eigenBras},
		eigenBras = EigenKets[Length[{wi}]] /. ket->bra;
		Apply[Plus, MapThread[(#1 #2)&, {{wi}, eigenBras}]]
	]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Converting Between Bras and Kets
:[font = input; initialization; preserveAspect; endGroup]
*)
KetToBra[ket_]:=
	RowVectorToBra[Conjugate[Transpose[KetToColumnVector[ket]]]]

BraToKet[bra_]:=
	ColumnVectorToKet[Conjugate[Transpose[BraToRowVector[bra]]]]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Average Value of an Observable
:[font = input; initialization; preserveAspect; endGroup]
*)
ExpectationValue[w_. ket[bits__] + kets_., observable_]:=
	(If[!HermitianQ[observable], 
		(Message[ExpectationValue::notHermitian]; Abort[]),
		If[Length[observable] != 2^Length[{bits}],
		   (Message[ExpectationValue::incompatible]; Abort[])]];
		       
	 (BraToRowVector[KetToBra[w * ket[bits] + kets]] . 
	  observable . 
	  KetToColumnVector[w * ket[bits] + kets]
	 )[[1,1]]  (* scalar = a 1 x 1 matrix, [[1,1]] removes the parentheses *)
	)

ExpectationValue[superposition_, observable_]:=
	ExpectationValue[Expand[superposition], observable]

ExpectationValue::notHermitian =
	"Your purported observable is not an Hermitian matrix.";
ExpectationValue::incompatible =
	"The dimensions of the state vector and observable are incompatible.";

(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Creating Eigenstates that Span a Hilbert Space
:[font = input; initialization; preserveAspect]
*)
BasisEigenstates[m_Integer]:= EigenKets[2^m]

BasisEigenstates::usage = 
  "BasisEigenstates[m] returns the complete set of \
  eigenstates that span the Hilbert space of an m-bit \
  quantum memory register.";
(*
:[font = input; initialization; preserveAspect; endGroup]
*)
EigenKets[n_]:=
	Module[{bits},
		bits = Table[Apply[ket, IntegerDigits[i,2]], 
		             {i, 0, n-1}];
		          (* last eigenket has the most bits *)
		Map[PadTo[Length[Last[bits]], #]&, bits]
	]

PadTo[nDigits_, digits_]:=
	Join[Apply[ket, Table[0,{nDigits - Length[digits]}]], 
	     digits]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Accessing Amplitudes of Superpositions and Computing Probabilities
:[font = input; initialization; preserveAspect]
*)
Options[Amplitudes] = {ShowEigenstates->False};

ShowEigenstates::usage = 
	"ShowEigenstates is an option for Amplitudes that \
	determines whether the 
output should be a list of the \
	amplitudes or a list of {eigenstate, 
amplitude} pairs.";

Amplitudes[w_. ket[bits__] + kets_., opts___]:=
	Module[{showeigen},
	showeigen = ShowEigenstates /. {opts} /. Options[Amplitudes];
	Which[showeigen == True, 
			Map[{#, Coefficient[w ket[bits] + kets, #]}&,
		        BasisEigenstates[ Length[{bits}] ]
		       ],
		  showeigen == False,
		    Map[Coefficient[w ket[bits] + kets, #]&,
		        BasisEigenstates[ Length[{bits}] ]
		       ]
		  ]
	]

(* This clause catches cases like 1/Sqrt[2] (ket[0] + ket[1]) etc *)	
Amplitudes[c_ (w_. ket[bits__] + kets_.)]:=
	Amplitudes[ Expand[c (w ket[bits] + kets)] ]

Amplitudes::usage = 
  "Amplitudes[superposition] returns the amplitudes of the \
  eigenstates in a superposition or ket vectors.";
(*
:[font = input; initialization; preserveAspect]
*)
Options[Probabilities] = {ShowEigenstates->False};

Probabilities[w_. ket[bits__] + kets_., opts___]:=
	Module[{showeigen, amplitudes, symbols, sumOfSquares},
		showeigen    = ShowEigenstates /. {opts} /. Options[Probabilities];
		amplitudes   = Amplitudes[w ket[bits] + kets];
		symbols      = SymbolicCoefficients[amplitudes]; (*see below*)
		sumOfSquares = Simplify[
		                Apply[Plus, 
		                      Map[ComplexExpand[Abs[#]^2, symbols]&, 
		                          amplitudes]]];
		amplitudes   = If[sumOfSquares=!=1,  (* renormalize amplitudes
		                                         if necessary *)
		                  amplitudes/Sqrt[sumOfSquares],
		                  amplitudes];
		Which[showeigen == True,  
		       MapThread[{#1, ComplexExpand[Abs[#2]^2, symbols]}&, 
		                 {BasisEigenstates[Length[{bits}]], amplitudes}
		                ],
			  showeigen == False, 
			   Map[ComplexExpand[Abs[#]^2, symbols]&, amplitudes]
	    ]
	]

Probabilities[c_ (w_. ket[bits__] + kets_.)]:=
	Probabilities[ Expand[c (w ket[bits] + kets)] ]

Probabilities::usage =
	"Probabilities[superposition] returns the probabilities of \
	 finding a system in a state described by superposition in \
	 each of its possible eigenstates upon being measured (observed). \
	 If Probabilities is given the option ShowEigenstates->True \
	 the function returns a list of {eigenstate, probability} pairs.";
(*
:[font = input; initialization; preserveAspect; endGroup]
*)
SymbolicCoefficients[amplitudes_List]:=
	Select[Union[Flatten[Map[Variables, amplitudes]]], 
		   Not[MatchQ[#, Abs[_]]]&]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Testing Whether a Ket is Properly Normalized
:[font = input; initialization; preserveAspect; endGroup]
*)
Needs["Algebra`ReIm`"];

NormalizedKetQ[ket_]:=
	Module[{columnVector},
		columnVector = KetToColumnVector[ket];
		(Inner[Times, 
		       Conjugate[Transpose[columnVector]], 
               columnVector,
               Plus
              ] == {{1}} // N ) /. z_ Conjugate[z_] :> Abs[z]^2
    ]
   
NormalizedKetQ::usage =
	"NormalizedKetQ[ket] returns True if the square \
	moduli of the amplitudes of the eigenkets in the \
	superposition \"ket\" sum to 1. If \"ket\" has non-numeric \
	amplitudes, the normalization cannot always be determined.";
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
NormalizeKet
:[font = input; initialization; preserveAspect; endGroup]
*)
NormalizeKet[superposition_]:=
	superposition /; NormalizedKetQ[superposition]
NormalizeKet[superposition_]:=
	Expand[superposition / 
	       Sqrt[Apply[Plus, 
	                  Map[Abs[#]^2&, 
	                      Amplitudes[superposition, 
	                                 ShowEigenstates->False]
	                     ]
	                 ]
	           ]
	]
	      
NormalizeKet::usage =
	"NormalizeKet[superposition] is used to normalize a given \
	superposition of
 ket vectors. That is, if the sum of the squares \
	of the absolute values of 
the amplitudes of the eigenstates in \
	the superposition do not sum to 1, 
NormalizeKet rescales the \
	amplitudes so that they squared moduli will sum 
to 1.";
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Direct Product
:[font = input; initialization; preserveAspect; endGroup]
*)
(* Last modified 09/07/96 *)
Needs["LinearAlgebra`MatrixManipulation`"];

Direct[op1_, op2_]:=
	BlockMatrix[Outer[Times, op1, op2]] /; MatrixQ[op1] && MatrixQ[op2]
	
Direct[ket_, bra_]:=
	Direct[KetToColumnVector[ket], BraToRowVector[bra]] /; IsKetQ[ket] && IsBraQ[
bra]
	
Direct[ket1_, ket2_]:=
	ColumnVectorToKet[
		Direct[KetToColumnVector[ket1],
	           KetToColumnVector[ket2]]
	]/; IsKetQ[ket1] && IsKetQ[ket2]

Direct[bra1_, bra2_]:=
	RowVectorToBra[
		Direct[BraToRowVector[bra1],
			   BraToRowVector[bra2]]
	] /; IsBraQ[bra1] && IsBraQ[bra2]
	
Direct[bra_, ket_]:=
	(Message[Direct::braket];
	 Direct[BraToRowVector[bra], KetToColumnVector[ket]]) /; IsBraQ[bra] && 
IsKetQ[ket]

Direct[bra_, op_]:=
	(Message[Direct::braop];
	 Direct[BraToRowVector[bra], op]) /; IsBraQ[bra] && MatrixQ[op]
	
Direct[op_, bra_]:=
	(Message[Direct::opbra];
	 Direct[op, BraToRowVector[bra]]) /; MatrixQ[op] && IsBraQ[bra]
	
Direct[ket_, op_]:=
	(Message[Direct::ketop];
	 Direct[KetToColumnVector[ket], op]) /; IsKetQ[ket] && MatrixQ[op]
	
Direct[op_, ket_]:=
	(Message[Direct::opket];
	 Direct[op, KetToColumnVector[ket]]) /; MatrixQ[op] && IsKetQ[ket]

Direct[matrices__]:=
	Fold[Direct, First[{matrices}], Rest[{matrices}]]

Direct::braket =
	"Warning - You are taking the DIRECT product of a bra \
	and a ket. This is 
unusual. Perhaps you meant to use \
	the DOT product?";
	
Direct::braop =
	"Warning - You are taking the DIRECT product of a bra \
	with an operator. 
This is unusual. Perhaps you meant to use \
	the DOT product?";
	
Direct::opbra =
	"Warning - You are taking the DIRECT product of an operator \
	with a bra. 
This is unusual. Perhaps you meant to use \
	the DOT product?";

Direct::ketop =
	"Warning - You are taking the DIRECT product of a ket \
	with an operator. 
This is unusual. Perhaps you meant to use \
	the DOT product?";

Direct::opket =
	"Warning - You are taking the DIRECT product of an operator \
	with a ket. 
This is unusual. Perhaps you meant to use \
	the DOT product?";


IsKetQ[w_. ket[__] + kets_.]:= True
IsKetQ[_]:=False
	
IsBraQ[w_. bra[__] + bras_.]:= True
IsBraQ[_]:=False
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Truth Table of a Logic Gate
:[font = input; initialization; preserveAspect; endGroup]
*)
TruthTable[gate_]:=
	Module[{n,m},
		{n,m} = Dimensions[gate];
		Which[Not[n==m && IntegerQ[n] && IntegerQ[m]],
		      Message[TruthTable::notsquare]; Abort[],
		      Not[IntegerQ[Log[2, n]]],
		      Message[TruthTable::powerof2]; Abort[]
		     ];
		Map[(# -> ColumnVectorToKet[gate . KetToColumnVector[#]])&, 
		    EigenKets[n]
		   ]  // ColumnForm
	]
	
TruthTable::notsquare = 
  "Your input is not a square matrix and cannot, therefore, represent a \
  
reversible logic gate.";

TruthTable::powerof2 = 
  "Your input is not a matrix of dimensions (2^m) x (2^m) for integer m \
  
and cannot, therefore, represent a reversible logic gate that operates \
  on 
m bits.";
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Types of Operators (Matrices)
:[font = input; preserveAspect]
HermitianQ[matrix_]:=
	matrix == Conjugate[Transpose[matrix]]
:[font = input; initialization; preserveAspect; endGroup]
*)
UnitaryQ[matrix_]:=
  Module[{rows, cols},
	{rows, cols} = Dimensions[matrix];
	If[Not[IntegerQ[rows]] || 
	   Not[IntegerQ[cols]] || 
	   rows != cols, Message[UnitaryQ::notsquarematrix]];
	
	   Chop[Simplify[ComplexExpand[Conjugate[Transpose[matrix]]] - 
	                 ComplexExpand[Inverse[matrix]]
	                ]
	       ] == ZeroMatrix[rows, cols]
  ]

UnitaryQ::notsquarematrix =
  "Your input is not a square matrix.";
  
ZeroMatrix[rows_, cols_]:=
	Table[0, {rows}, {cols}]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Tools for Making Test Superpositions
:[font = input; initialization; preserveAspect]
*)
SymbolicSuperposition[m_]:=
	Apply[Plus,
	      MapThread[(#1 #2)&, 
	                {SymbolicAmplitudes[m], BasisEigenstates[m]}]
	]
	
SymbolicSuperposition::usage =
	"SymbolicSuperposition[m] creates a superposition of 2^m \
	eigenstates whose
 amplitudes are uninstantiated symbols. These \
	eigenstates represent the 
possible states of an m-bit memory \
	register of a quantum computer. This 
function is useful for \
	exploring the effects of quantum mechanical 
operations on \
	arbitrary superpositions. Note that the general form does not

	guarentee that the superposition is normalized.";
	
SymbolicAmplitudes[m_]:=
	(Clear[w];
	 Map[ToExpression["w"<>ToString[#]]&, Table[i,{i,0,2^m - 1}]]
	)
(*
:[font = input; initialization; preserveAspect; endGroup]
*)
Options[RandomSuperposition] = {Normalized->True};

RandomSuperposition[m_, opts___]:=
	Module[{normalized},
		normalized = Normalized /. {opts} /. Options[RandomSuperposition];
		superposition = Apply[Plus,
	                          MapThread[(#1 #2)&, 
	                                    {RandomAmplitudes[m],
	                                     BasisEigenstates[m]}
	                                   ]
	                         ];
		Which[normalized==True, NormalizeKet[superposition],
	          normalized==False, superposition
	         ]
	]

RandomSuperposition::usage =
	"RandomSuperposition[m] creates a normalized superposition \
	of 2^m eigenstates whose amplitudes are random complex numbers. \
	These eigenstates represent the possible states of an m-bit \
	memory register of a quantum computer. You can generate an \
	un-normalized superposition by setting the option Normalized->False.";
	
(* You can pick the amplitudes according to whatever distribution
   you like. In the current case we pick random complex numbers
   uniformly from the square in the complex plane bounded by a lower
   left corner at (-1,-I) and an upper right corner at (1,I).
*)
RandomAmplitudes[m_]:=
	Table[Random[Complex, {-1-I, 1+I}], {2^m}]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
Arbitrary State
:[font = input; preserveAspect; endGroup; endGroup]
Options[ArbitraryState] = {Numeric->False};

ArbitraryState[m_, opts___]:=
	Module[{coeffs, numericQ},
		Clear[w];
		numericQ = Numeric /. {opts} /. Options[ArbitraryState];
		Switch[numericQ, 
		        True, RandomSuperposition[m],
		        False, (coeffs = Table[c[i], {i,0,2^m -1}];
		                Apply[Plus, 
		                      MapThread[#1 #2&, 
		                                {coeffs, BasisEigenstates[m]}]]
		               )
		      ]
	]
:[font = subsection; inactive; preserveAspect; startGroup]
Quantum Memory Registers
:[font = subsubsection; inactive; preserveAspect; startGroup]
ReadMemoryRegister
:[font = input; initialization; preserveAspect; endGroup]
*)
(*====================*)
(* ReadMemoryRegister *)
(*====================*)
(* Given a superposition representing the state of the memory of a
   quantum computer, return the result of measuring the memory.
*)
Options[ReadMemoryRegister] = {TraceProgress->False};

ReadMemoryRegister[w_. ket[bits__] + kets_., opts___]:=
	Module[{nBits, superposition, resultsPerStep, traceQ},
		traceQ = TraceProgress /. {opts} /. Options[ReadMemoryRegister];
		nBits = Length[{bits}]; (* figure out number of bits in memory *)
		superposition = {"BeforeAnyMeasurements", w ket[bits] + kets};
		resultsPerStep = FoldList[MeasureIthBit[#2,#1,nBits]&,
		                          superposition,
		                          Range[nBits]
		                         ];
		Which[traceQ===False, 
		        {Rest[Map[#[[1]]&, resultsPerStep]],  (* results for bits *)
		         Last[resultsPerStep][[2]]            (* projected state *)
		        },
		      traceQ===True,
		        ColumnForm[resultsPerStep] (*list of {results,projectedStates}*)
		     ]
	]
	
ReadMemoryRegister::usage =
	"ReadMemoryRegister[superposition] reads the state of each bit \
	in the memory register. As the i-th and j-th bit measurement \
	operators commute (for any i and j), it does not matter in what \
	order you measure the bits.";
	
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
MeasureIthBit
:[font = input; initialization; preserveAspect; endGroup]
*)
MeasureIthBit[i_, {_, superposition_}, nBits_]:=
	Module[{p1, zeroOrOne, projectedState},
		p1 = ProbabilityIthBitIs1[i, superposition];
		zeroOrOne = BiasedSelect[{0,1}, {1-p1, p1}];
		projectedState = 
		  SuperpositionWithIthBitFixed[i, zeroOrOne, nBits, superposition];
		{zeroOrOne, projectedState}
	]
		       
KetWithIthBitZeroOrOne[i_, zeroOrOne_, nBits_]:=
	ReplacePart[Apply[ket, Table[_,{nBits}]], zeroOrOne, i]

SuperpositionWithIthBitFixed[_, _, _, w_. ket[bits__]]:=
  NormalizeKet[w ket[bits]]
SuperpositionWithIthBitFixed[i_, zeroOrOne_, nBits_, superposition_]:=
  NormalizeKet[Select[superposition,
		              MatchQ[#, _. KetWithIthBitZeroOrOne[i,zeroOrOne,nBits]
		                    ]&
		             ]
		      ]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
ProbabilityIthBitIs1
:[font = input; initialization; preserveAspect; endGroup]
*)
ProbabilityIthBitIs1[i_, w_. ket[bits__]]:=
	If[ket[bits][[i]] == 1, Abs[w]^2, 1-Abs[w]^2]  (* Abs[w]^2 == 1 or 0 only *)
	
ProbabilityIthBitIs1[i_, w_. ket[bits__] + kets_.]:=
	Module[{nBits, terms},
		nBits = Length[{bits}];
		terms = Select[w ket[bits] + kets, 
		               MatchQ[#, _. KetWithIthBit1[i,nBits]]&];
		N[Apply[Plus, Map[Abs[#]^2&, Amplitudes[terms]]]/
		  Apply[Plus, Map[Abs[#]^2&, Amplitudes[w ket[bits] + kets]]]
		 ]
	]
	
ProbabilityIthBitIs1[i_, c_. (w_. ket[bits__] + kets_.)]:=
	ProbabilityIthBitIs1[i, Expand[c (w ket[bits] + kets)]]

ProbabilityIthBitIs1::usage =
	"The state of the memory register of a quantum computer \
	(that is comprised of m 2-state particles) is represented by a \
	superposition 2^m eigenstates. The function \
	ProbabilityIthBitIs1[i, superposition] computes the probability \
	that, upon being measured, the i-th, of the m, bits will be a 1.";
	
KetWithIthBit1[i_, nBits_]:=
	ReplacePart[Apply[ket, Table[_,{nBits}]], 1, i]
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
BiasedSelect
:[font = input; initialization; preserveAspect; endGroup]
*)
(* The list of probabilities should sum to 1. The call to Partition 
   constructs a set of probability intervals whose width is proportional
   to the probability with which the corresponding element in list 
   is selected.
*)
BiasedSelect[list_, probabilities_]:=
	Module[{random},
		random  = Random[];
		Apply[Part[list, #]&,
		      Flatten[
		       Position[Map[InRangeQ[random, #]&, 
		                    Partition[FoldList[Plus,0,probabilities],2,1]
		                   ],
		                True
		       ]
		      ]
		     ]
	] (* /;CheckProbabilitiesQ[probabilities] *)

BiasedSelect::usage =
	"BiasedSelect[{e1,e2,...,en}, {p1,p2,...,pn}] returns element ei of \
	the first list with probability given in the second list pi.";

BiasedSelect::probabilityLeak =
	"You have a probability leak. The probabilities you specified do \
	not add up to 1.";

BiasedSelect::excess =
	"The probabilities you specified sum to greater than 1.";

CheckProbabilitiesQ[probabilities_]:=
	Module[{psum = Apply[Plus, probabilities]},
		Which[psum<1,  Message[BiasedSelect::probabilityLeak],
		      psum>1,  Message[BiasedSelect::excess],
		      psum==1, True
		     ]
	]
	
InRangeQ[n_, {lb_, 1}]:=   lb <= n <= 1
InRangeQ[n_, {lb_, ub_}]:= lb <= n < ub
(*
:[font = subsubsection; inactive; preserveAspect; startGroup]
ReadPartOfMemoryRegister
:[font = input; initialization; preserveAspect; endGroup; endGroup; endGroup; endGroup]
*)
(*==========================*)
(* ReadPartOfMemoryRegister *)
(*==========================*)
(* Given a superposition representing the state of the memory of a
   quantum computer, return the result of measuring a specific
   subset of the qubits in the memory.
   
   >>> This function is used in error correcting codes <<<
*)
Options[ReadPartOfMemoryRegister] = {TraceProgress->False};

ReadPartOfMemoryRegister[w_. ket[bits__] + kets_., bitsToRead_, opts___]:=
	Module[{nBits, superposition, resultsPerStep, traceQ},
		traceQ = TraceProgress /. {opts} /. Options[ReadPartOfMemoryRegister];
		nBits = Length[{bits}]; (* figure out number of bits in memory *)
		superposition = {"BeforeAnyMeasurements", w ket[bits] + kets};
		resultsPerStep = FoldList[MeasureIthBit[#2,#1,nBits]&,
		                          superposition,
		                          bitsToRead
		                         ];
		Which[traceQ===False, 
		        {Rest[Map[#[[1]]&, resultsPerStep]],  (* results for bits *)
		         Last[resultsPerStep][[2]]            (* projected state *)
		        },
		      traceQ===True,
		        ColumnForm[resultsPerStep] (*list of {results,projectedStates}*)
		     ]
	]
	
ReadPartOfMemoryRegister::usage =
	"ReadPartOfMemoryRegister[superposition, bitsToRead] reads the state \
	of selected bits in the memory register. As the i-th and j-th bit \
	measurement operators commute (for any i and j), it does not matter \
	in what order you measure the bits.";
(*
^*)
